<?php
/**
 * Moony - the tiny web application framework
 *
 * @package   Moony
 * @author    YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @link      http://moony.sourceforge.jp/
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license   http://opensource.org/licenses/bsd-license.php The BSD License
 */

/**
 * Smartyを利用してテンプレートを扱うためのクラスです。
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @access public
 */
class Moony_Template_Smarty extends Moony_Template
{
    /**
     * Smartyのインスタンス
     * @var object Smarty
     */
    var $smarty;

    /**
     * コンストラクタです。
     *
     * @access public
     */
    function Moony_Template_Smarty()
    {
        if (!class_exists('Smarty')) {
            trigger_error('Not loaded: Smarty', E_USER_ERROR);
            header('HTTP/1.0 500 Internal Server Error');
            exit;
        }
        $smarty =& new Smarty();
        // プラグイン登録
        $smarty->register_function('moony_token', array($this, 'token'));
        $smarty->register_function('moony_errors', array($this, 'errors'));
        $smarty->register_block('moony_has_error', array($this, 'hasError'));
        $this->smarty = &$smarty;
    }

    /**
     * テンプレートに値を割り当てます。
     * $varが配列の場合、展開して全ての値を割り当てます。
     *
     * @access public
     * @param string|array $var 値の名称、または割り当てる値の連想配列
     * @param mixed $value 割り当てる値
     */
    function assign($var, $value = null)
    {
        $this->smarty->assign($var, $value);
    }

    /**
     * テンプレートの処理結果を取得します。
     * テンプレートファイルが存在しない場合、falseを返します。
     *
     * @access public
     * @param string $template テンプレートファイル名
     * @return string|false テンプレートの処理結果
     */
    function fetch($template)
    {
        if (!$this->smarty->template_exists($template)) {
            return false;
        }
        return $this->smarty->fetch($template);
    }

    /**
     * 設定を行います。
     * 個々の要素をSmartyのプロパティとして設定します。
     *
     * @access public
     * @param array $configs 設定内容の連想配列
     */
    function configure($configs)
    {
        if (!is_array($configs)) {
            return;
        }
        $prop_keys = array_keys(get_object_vars($this->smarty));
        foreach ($configs as $name => $value) {
            if (!in_array($name, $prop_keys)) {
                continue;
            }
            if (is_array($this->smarty->$name)) {
                // 配列の場合要素として追加
                $this->smarty->{$name}[] = $value;
            } else {
                $this->smarty->$name = $value;
            }
        }
    }

    /**
     * hiddenフィールドとしてトランザクショントークンを描画します。
     *
     * @access public
     * @param array $params パラメータ配列
     * @param object $smarty Smarty
     */
    function token($params, &$smarty)
    {
        $xhtml = $this->_getParam('xhtml', $params, true);
        $close_tag = $xhtml ? ' />' : '>';
        if (isset($_SESSION[MOONY_TOKEN_KEY])) {
            $token = $_SESSION[MOONY_TOKEN_KEY];
        }
        echo '<input type="hidden" value="', $token, '"', $close_tag, "\n";
    }

    /**
     * 検証エラーメッセージを描画します。
     *
     * @access public
     * @param array $params パラメータ配列
     * @param object $smarty Smarty
     */
    function errors($params, &$smarty)
    {
        $header = $this->_getParam('header', $params, '<ul>');
        $footer = $this->_getParam('footer', $params, '</ul>');
        $prefix = $this->_getParam('prefix', $params, '<li>');
        $postfix = $this->_getParam('postfix', $params, '</li>');

        $error_messages = Moony_Registry::get(MOONY_ERROR_MESSAGES_KEY);
        if (!is_null($error_messages)) {
            echo $header;
            foreach ($error_messages as $error_message) {
                echo $prefix, $error_message, $postfix;
            }
            echo $footer;
        }
    }

    /**
     * 検証エラーの有無を判断し、
     * エラーが存在する場合はブロックの内容を描画します。
     *
     * @access public
     * @param array $params パラメータ配列
     * @param string $content ブロックの内容
     * @param object $smarty Smarty
     * @return エラーが存在する場合、ブロックの内容
     */
    function hasError($params, $content, &$smarty)
    {
        $has_error = Moony_Registry::get(MOONY_HAS_ERROR_KEY);
        $has_error = is_null($has_error) ? false : $has_error;
        if ($has_error) {
            return $content;
        }
    }

    /**
     * プラグインに渡されたパラメータを取得します。
     * 該当するパラメータが存在しない場合、$defaultを返します。
     *
     * @access protected
     * @param string $name パラメータの名称
     * @param array $params 渡されたパラメータの配列
     * @param mixed $default パラメータが存在しない場合に返す値
     * @return mixed パラメータの値、または$default
     */
    function _getParam($name, $params, $default = null)
    {
        if (isset($params[$name])) {
            return $params[$name];
        }
        return $default;
    }
}
?>
