<?php
/**
 * Moony - the tiny web application framework
 *
 * @package   Moony
 * @author    YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @link      http://moony.sourceforge.jp/
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license   http://opensource.org/licenses/bsd-license.php The BSD License
 */

/**
 * 値の検証を行うクラスです。
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @access public
 */
class Moony_Validator
{
    /**
     * 内部エンコーディング
     * @var string
     */
    var $_internal_encoding;

    /**
     * エラーの有無
     * @var boolean
     */
    var $_has_error = false;

    /**
     * エラーメッセージの配列
     * @var array
     */
    var $_error_messages = array();

    /**
     * コンストラクタです。
     *
     * @access public
     * @param string $internal_encoding 内部エンコーディング
     */
    function Moony_Validator($internal_encoding)
    {
        $this->_internal_encoding = $internal_encoding;
        mb_regex_encoding($internal_encoding);
    }

    /**
     * エラーを追加登録します。
     *
     * @access public
     * @param string $message エラーメッセージ
     */
    function addError($message = null)
    {
        $this->_has_error = true;
        if (!is_null($message)) {
            $this->_error_messages[] = $message;
        }
        // エラー情報を保管
        Moony_Registry::set(MOONY_HAS_ERROR_KEY, true);
        Moony_Registry::set(MOONY_ERROR_MESSAGES_KEY, $this->_error_messages);
    }

    /**
     * エラーの存在有無を調べます。
     *
     * @access public
     * @return boolean エラーの存在有無
     */
    function hasError()
    {
        return $this->_has_error;
    }

    /**
     * 全てのエラーメッセージの配列を取得します。
     *
     * @access public
     * @return array 全てのエラーメッセージの配列
     */
    function getErrorMessages()
    {
        return $this->_error_messages;
    }

    /**
     * 必須項目の検証を行います。
     *
     * @access public
     * @param string|array $var 検証対象の値
     * @param string $message エラーメッセージ
     * @return boolean エラーがなければtrue
     */
    function validateRequired($var, $message = null)
    {
        if (is_null($var)) {
            $this->addError($message);
            return false;
        }
        $var = (array) $var;
        foreach ($var as $each) {
            if (is_null($each) || strlen($each) == 0) {
                $this->addError($message);
                return false;
            }
        }
        return true;
    }

    /**
     * 正規表現を用いてパターンの検証を行います。
     * $mbがtrueの場合、mb_ereg関数を用いて検証します。
     * $mbがfalseの場合、preg_match関数を用いて検証します。
     * $varがnullの場合、無条件にエラーなしとみなします。
     *
     * @param string|array $var 検証対象の値
     * @param string $pattern 正規表現パターン
     * @param string $message エラーメッセージ
     * @param boolean $mb mb_ereg関数を使用するかどうか
     * @return boolean エラーがなければtrue
     */
    function validatePattern($var, $pattern, $message = null, $mb = false)
    {
        if (is_null($var)) {
            return true;
        }
        $var = (array) $var;
        foreach ($var as $each) {
            if ($mb) {
                if (mb_ereg($pattern, $each) === false) {
                    $this->addError($message);
                    return false;
                }
            } else {
                // 0 or false
                if (preg_match($pattern, $each) == false) {
                    $this->addError($message);
                    return false;
                }
            }
        }
        return true;
    }

    /**
     * 文字列長の検証を行います。
     * $mbがtrueの場合、mb_strlen関数を用いて検証します。
     * $mbがfalseの場合、strlen関数を用いて検証します。
     * $varがnullの場合、無条件にエラーなしとみなします。
     *
     * @param string|array $var 検証対象の値
     * @param integer $min 許容される最小長
     * @param integer $max 許容される最大長
     * @param string $message エラーメッセージ
     * @param boolean $inclusive 境界値を許容するかどうか
     * @param boolean $mb mb_strlen関数を使用するかどうか
     * @return boolean エラーがなければtrue
     */
    function validateLength($var, $min, $max,
        $message = null, $inclusive = true, $mb = false)
    {
        if (is_null($var)) {
            return true;
        }
        $var = (array) $var;
        foreach ($var as $each) {
            $length = $mb ? mb_strlen($each,
                $this->_internal_encoding) : strlen($each);
            if ($length < $min || $length > $max) {
                $this->addError($message);
                return false;
            }
            if (!$inclusive && ($length == $min || $length == $max)) {
                $this->addError($message);
                return false;
            }
        }
        return true;
    }

    /**
     * 値の範囲の検証を行います。
     * $varがnullの場合、無条件にエラーなしとみなします。
     *
     * @param string|array $var 検証対象の値
     * @param integer $min 許容される最小値
     * @param integer $max 許容される最大値
     * @param string $message エラーメッセージ
     * @param boolean $inclusive 境界値を許容するかどうか
     * @return boolean エラーがなければtrue
     */
    function validateRange(
        $var, $min, $max, $message = null, $inclusive = true)
    {
        if (is_null($var)) {
            return true;
        }
        $var = (array) $var;
        foreach ($var as $each) {
            if ($each < $min || $each > $max) {
                $this->addError($message);
                return false;
            }
            if (!$inclusive && ($each == $min || $each == $max)) {
                $this->addError($message);
                return false;
            }
        }
        return true;
    }

    /**
     * 半角英字かどうかの検証を行います。
     * ctype_alpha関数を用いて検証します。
     * $varがnullの場合、無条件にエラーなしとみなします。
     *
     * @param string|array $var 検証対象の値
     * @param string $message エラーメッセージ
     * @return boolean エラーがなければtrue
     */
    function validateAlpha($var, $message = null)
    {
        return $this->validateUserFunc($var, 'ctype_alpha', $message);
    }

    /**
     * 半角英数字かどうかの検証を行います。
     * ctype_alnum関数を用いて検証します。
     * $varがnullの場合、無条件にエラーなしとみなします。
     *
     * @param string|array $var 検証対象の値
     * @param string $message エラーメッセージ
     * @return boolean エラーがなければtrue
     */
    function validateAlnum($var, $message = null)
    {
        return $this->validateUserFunc($var, 'ctype_alnum', $message);
    }

    /**
     * 半角数字かどうかの検証を行います。
     * ctype_digit関数を用いて検証します。
     * $varがnullの場合、無条件にエラーなしとみなします。
     *
     * @param string|array $var 検証対象の値
     * @param string $message エラーメッセージ
     * @return boolean エラーがなければtrue
     */
    function validateDigit($var, $message = null)
    {
        return $this->validateUserFunc($var, 'ctype_digit', $message);
    }

    /**
     * 数値かどうかの検証を行います。
     * is_numeric関数を用いて検証します。
     * $varがnullの場合、無条件にエラーなしとみなします。
     *
     * @param string|array $var 検証対象の値
     * @param string $message エラーメッセージ
     * @return boolean エラーがなければtrue
     */
    function validateNumeric($var, $message = null)
    {
        return $this->validateUserFunc($var, 'is_numeric', $message);
    }

    /**
     * ひらがなかどうかの検証を行います。
     * $varがnullの場合、無条件にエラーなしとみなします。
     *
     * @param string|array $var 検証対象の値
     * @param string $message エラーメッセージ
     * @return boolean エラーがなければtrue
     */
    function validateHiragana($var, $message = null)
    {
        if (is_null($var)) {
            return true;
        }
        if ($this->_internal_encoding != 'UTF-8') {
            mb_convert_variables('UTF-8', $this->_internal_encoding, $var);
        }
        return $this->validatePattern($var,
            '/^(\xe3(\x81[\x81-\xbf]|\x82[\x80-\x93]|\x83\xbc))*$/', $message);
    }

    /**
     * 全角カタカナかどうかの検証を行います。
     * $varがnullの場合、無条件にエラーなしとみなします。
     *
     * @param string|array $var 検証対象の値
     * @param string $message エラーメッセージ
     * @return boolean エラーがなければtrue
     */
    function validateKatakana($var, $message = null)
    {
        if (is_null($var)) {
            return true;
        }
        if ($this->_internal_encoding != 'UTF-8') {
            mb_convert_variables('UTF-8', $this->_internal_encoding, $var);
        }
        return $this->validatePattern($var,
            '/^(\xe3(\x82[\xa1-\xbf]|\x83[\x80-\xb6]|\x83\xbc))*$/', $message);
    }

    /**
     * ユーザ関数を用いた検証を行います。
     * $callbackは引数として個々の入力値をとり、
     * 検証結果としてboolean値を返す必要があります。
     * $varがnullの場合、無条件にエラーなしとみなします。
     *
     * @param string|array $var 検証対象の値
     * @param callback $callback 検証用のコールバック関数
     * @param string $message エラーメッセージ
     * @return boolean エラーがなければtrue
     */
    function validateUserFunc($var, $callback, $message = null)
    {
        if (is_null($var)) {
            return true;
        }
        $var = (array) $var;
        foreach ($var as $each) {
            if (call_user_func($callback, $each) === false) {
                $this->addError($message);
                return false;
            }
        }
        return true;
    }
}
?>
