/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.framework.instance;

import java.sql.Connection;
import java.util.Comparator;

import jp.mosp.framework.base.BaseBeanHandlerInterface;
import jp.mosp.framework.base.BaseBeanInterface;
import jp.mosp.framework.base.BaseDaoInterface;
import jp.mosp.framework.base.MospException;
import jp.mosp.framework.base.MospParams;
import jp.mosp.framework.constant.ExceptionConst;
import jp.mosp.framework.utils.LogUtility;
import jp.mosp.framework.utils.MospUtility;

/**
 * インスタンス生成クラス
 */
public class InstanceFactory {
	
	/**
	 * 他クラスからのインスタンス化を防止する。<br>
	 */
	private InstanceFactory() {
		// 処理無し
	}
	
	/**
	 * クラスローダーからインスタンスを生成する。
	 * @param className	 対象クラス名
	 * @return 対象クラスインスタンス
	 * @throws MospException インスタンスの生成に失敗した場合
	 */
	public static Object loadInstance(String className) throws MospException {
		Object obj = null;
		try {
			obj = Thread.currentThread().getContextClassLoader().loadClass(className).newInstance();
		} catch (NullPointerException e) {
			throw new MospException(e, ExceptionConst.EX_NO_CLASS_NAME, null);
		} catch (ClassNotFoundException e) {
			throw new MospException(e, ExceptionConst.EX_NO_CLASS, className);
		} catch (InstantiationException e) {
			throw new MospException(e, ExceptionConst.EX_FAIL_INSTANTIATE, className);
		} catch (IllegalAccessException e) {
			throw new MospException(e, ExceptionConst.EX_FAIL_INSTANTIATE, className);
		}
		return obj;
	}
	
	/**
	 * クラス名からインスタンスを生成する。
	 * @param className 対象クラス名
	 * @return 対象クラスインスタンス
	 * @throws MospException インスタンスの生成に失敗した場合
	 */
	public static Object simplifiedInstance(String className) throws MospException {
		Object obj = null;
		try {
			obj = Class.forName(className).newInstance();
		} catch (NullPointerException e) {
			throw new MospException(e, ExceptionConst.EX_NO_CLASS_NAME, null);
		} catch (ClassNotFoundException e) {
			throw new MospException(e, ExceptionConst.EX_NO_CLASS, className);
		} catch (InstantiationException e) {
			throw new MospException(e, ExceptionConst.EX_FAIL_INSTANTIATE, className);
		} catch (IllegalAccessException e) {
			throw new MospException(e, ExceptionConst.EX_FAIL_INSTANTIATE, className);
		}
		return obj;
	}
	
	/**
	 * クラスローダーからインスタンスを生成する。<br>
	 * 対象インターフェースからキーを生成し、
	 * MosP処理情報から対象クラス名を取得する。<br>
	 * @param cls        対象インターフェース
	 * @param mospParams MosP処理情報
	 * @return 対象Beanクラスインスタンス
	 * @throws MospException インスタンスの生成に失敗した場合
	 */
	public static Object loadInstance(Class<?> cls, MospParams mospParams) throws MospException {
		// クラス名取得
		String className = mospParams.getApplicationProperty(MospUtility.getBeanKey(cls));
		// インスタンス生成
		return loadInstance(className);
	}
	
	/**
	 * BeanHandlerインスタンスを生成し、初期化する。<br>
	 * 但し、DBコネクションは取得しない。
	 * @param cls        対象BeanHandlerインターフェース
	 * @param mospParams MosP処理情報
	 * @return 初期化されたBeanインスタンス
	 * @throws MospException Beanインスタンスの生成及び初期化に失敗した場合
	 */
	public static BaseBeanHandlerInterface loadBeanHandler(Class<?> cls, MospParams mospParams) throws MospException {
		// BeanHandlerインスタンス取得
		BaseBeanHandlerInterface beanHandler = (BaseBeanHandlerInterface)loadInstance(cls, mospParams);
		// MosPパラメータ設定
		beanHandler.setMospParams(mospParams);
		// デバッグメッセージ
		LogUtility.debug(mospParams, beanHandler.toString());
		return beanHandler;
	}
	
	/**
	 * Beanインスタンスを生成し、初期化する。<br>
	 * @param cls        対象Beanインターフェース
	 * @param mospParams MosP処理情報
	 * @param connection コネクション
	 * @return 初期化されたBeanインスタンス
	 * @throws MospException Beanインスタンスの生成及び初期化に失敗した場合
	 */
	public static BaseBeanInterface loadBean(Class<?> cls, MospParams mospParams, Connection connection)
			throws MospException {
		// Beanインスタンス取得
		BaseBeanInterface bean = (BaseBeanInterface)loadInstance(cls, mospParams);
		// MosP処理情報及びコネクションを設定
		bean.setParams(mospParams, connection);
		// 初期化処理を実行
		bean.initBean();
		// デバッグメッセージ
		LogUtility.debug(mospParams, bean.toString());
		return bean;
	}
	
	/**
	 * Beanインスタンスを生成し、初期化する。<br>
	 * @param className  クラス名
	 * @param mospParams MosP処理情報
	 * @param connection コネクション
	 * @return 初期化されたBeanインスタンス
	 * @throws MospException Beanインスタンスの生成及び初期化に失敗した場合
	 */
	public static BaseBeanInterface loadBean(String className, MospParams mospParams, Connection connection)
			throws MospException {
		// Beanインスタンス取得
		BaseBeanInterface bean = (BaseBeanInterface)loadInstance(className);
		// MosP処理情報及びコネクションを設定
		bean.setParams(mospParams, connection);
		// 初期化処理を実行
		bean.initBean();
		// デバッグメッセージ
		LogUtility.debug(mospParams, bean.toString());
		return bean;
	}
	
	/**
	 * DAOインスタンスを生成し、初期化する。<br>
	 * @param cls        対象DAOインターフェース
	 * @param mospParams MosP処理情報
	 * @param connection コネクション
	 * @return 初期化されたDAOインスタンス
	 * @throws MospException DAOインスタンスの生成及び初期化に失敗した場合
	 */
	public static BaseDaoInterface loadDao(Class<?> cls, MospParams mospParams, Connection connection)
			throws MospException {
		// DAOインスタンス取得
		BaseDaoInterface dao = (BaseDaoInterface)loadInstance(cls, mospParams);
		// MosP処理情報及びコネクションを設定
		dao.setInitParams(mospParams, connection);
		// 初期化処理を実行
		dao.initDao();
		// デバッグメッセージ
		LogUtility.debug(mospParams, dao.toString());
		return dao;
	}
	
	/**
	 * 比較クラスインスタンスを生成し、初期化する。<br>
	 * @param className 比較クラス名
	 * @return 比較クラスインスタンス
	 * @throws MospException 比較クラスインスタンスの生成に失敗した場合
	 */
	@SuppressWarnings("unchecked")
	public static Comparator<Object> loadComparator(String className) throws MospException {
		// 比較クラスインスタンス取得
		return (Comparator<Object>)loadInstance(className);
	}
	
}
