/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.platform.bean.system.impl;

import java.sql.Connection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import jp.mosp.framework.base.BaseDto;
import jp.mosp.framework.base.MospException;
import jp.mosp.framework.base.MospParams;
import jp.mosp.platform.base.PlatformBean;
import jp.mosp.platform.bean.system.PositionReferenceBeanInterface;
import jp.mosp.platform.comparator.base.PositionCodeComparator;
import jp.mosp.platform.comparator.system.PositionMasterGradeLevelComparator;
import jp.mosp.platform.dao.system.PositionDaoInterface;
import jp.mosp.platform.dto.system.PositionDtoInterface;

/**
 * 職位マスタ参照クラス。
 */
public class PositionReferenceBean extends PlatformBean implements PositionReferenceBeanInterface {
	
	/**
	 * 職位マスタDAO。
	 */
	private PositionDaoInterface	dao;
	

	/**
	 * {@link PlatformBean#PlatformBean()}を実行する。<br>
	 */
	public PositionReferenceBean() {
		super();
	}
	
	/**
	 * {@link PlatformBean#PlatformBean(MospParams, Connection)}を実行する。<br>
	 * @param mospParams MosP処理情報
	 * @param connection DBコネクション
	 */
	protected PositionReferenceBean(MospParams mospParams, Connection connection) {
		super(mospParams, connection);
	}
	
	@Override
	public void initBean() throws MospException {
		dao = (PositionDaoInterface)createDao(PositionDaoInterface.class);
		
	}
	
	@Override
	public List<PositionDtoInterface> getPositionHistory(String positionCode) throws MospException {
		return dao.findForHistory(positionCode);
	}
	
	@Override
	public PositionDtoInterface getPositionInfo(String positionCode, Date targetDate) throws MospException {
		return dao.findForInfo(positionCode, targetDate);
	}
	
	@Override
	public String getPositionName(String positionCode, Date targetDate) throws MospException {
		PositionDtoInterface dto = getPositionInfo(positionCode, targetDate);
		if (dto == null) {
			return "";
		}
		return dto.getPositionName();
	}
	
	@Override
	public String getPositionAbbr(String positionCode, Date targetDate) throws MospException {
		PositionDtoInterface dto = getPositionInfo(positionCode, targetDate);
		if (dto == null) {
			return "";
		}
		return dto.getPositionAbbr();
	}
	
	@Override
	public List<PositionDtoInterface> getPositionList(Date targetDate, String operationType) throws MospException {
		// 操作範囲準備
		String[] rangeArray = new String[0];
		// 操作区分確認
		if (operationType != null && operationType.isEmpty() == false) {
			// 操作範囲取得
			rangeArray = getRangePosition(operationType, targetDate);
		}
		// リストを取得
		return dao.findForActivateDate(targetDate, rangeArray);
	}
	
	@Override
	public Map<String, PositionDtoInterface> getPositionMap(Date targetDate, String operationType) throws MospException {
		// リストを取得しマップに変換
		return getMap(getPositionList(targetDate, operationType));
	}
	
	@Override
	public PositionDtoInterface getLeaderPositionInfo(List<String> positionCodeList, Date targetDate)
			throws MospException {
		// 職位マスタセットを生成。
		Map<String, PositionDtoInterface> positionMap = new HashMap<String, PositionDtoInterface>();
		for (String positionCode : positionCodeList) {
			positionMap.put(positionCode, getPositionInfo(positionCode, targetDate));
		}
		// 職位マスタセットから等級が最も大きい数字のものを取得
		// 等級が同じ場合、号数が大きいものを採用。
		int grade = 0;
		int level = 0;
		String positionCode = "";
		Set<Entry<String, PositionDtoInterface>> entrySet = positionMap.entrySet();
		for (Entry<String, PositionDtoInterface> entry : entrySet) {
			PositionDtoInterface dto = entry.getValue();
			if (dto != null) {
				if (dto.getPositionGrade() > grade) {
					grade = dto.getPositionGrade();
					level = dto.getPositionLevel();
					positionCode = entry.getKey();
				} else if (dto.getPositionGrade() == grade) {
					if (dto.getPositionLevel() > level) {
						level = dto.getPositionLevel();
						positionCode = entry.getKey();
					}
				}
			}
		}
		return positionMap.get(positionCode);
	}
	
	@Override
	public PositionDtoInterface findForkey(long id) throws MospException {
		BaseDto dto = findForKey(dao, id, false);
		if (dto != null) {
			return (PositionDtoInterface)dto;
		}
		return null;
	}
	
	@Override
	public boolean hasAdvantage(PositionDtoInterface dto, int grade) {
		return grade > dto.getPositionGrade() == hasLowGradeAdvantage();
	}
	
	@Override
	public boolean hasLowGradeAdvantage() {
		return dao.hasLowGradeAdvantage();
	}
	
	@Override
	public String[][] getSelectArray(Date targetDate, boolean needBlank, String operationType) throws MospException {
		// プルダウン用配列取得(略称)
		return getSelectArray(targetDate, needBlank, operationType, false, false, false);
	}
	
	@Override
	public String[][] getNameSelectArray(Date targetDate, boolean needBlank, String operationType) throws MospException {
		// プルダウン用配列取得(名称)
		return getSelectArray(targetDate, needBlank, operationType, true, false, false);
	}
	
	@Override
	public String[][] getCodedSelectArray(Date targetDate, boolean needBlank, String operationType)
			throws MospException {
		// プルダウン用配列取得(略称)
		return getSelectArray(targetDate, needBlank, operationType, true, true, false);
	}
	
	@Override
	public String[][] getCodedAbbrSelectArray(Date targetDate, boolean needBlank, String operationType)
			throws MospException {
		// プルダウン用配列取得(コード+名称)
		return getSelectArray(targetDate, needBlank, operationType, false, true, false);
	}
	
	@Override
	public String[][] getGradedSelectArray(Date targetDate, boolean needBlank, String operationType)
			throws MospException {
		// プルダウン用配列取得(等級+名称)
		return getSelectArray(targetDate, needBlank, operationType, true, false, true);
	}
	
	/**
	 * プルダウン用配列を取得する。<br>
	 * viewCode、viewGrade共にtrueとした場合は、コードのみが表示される。<br>
	 * @param targetDate    対象年月日
	 * @param needBlank     空白行要否(true：空白行要、false：空白行不要)
	 * @param operationType 操作範囲権限
	 * @param isName        名称表示(true：名称表示、false：略称表示)
	 * @param viewCode      コード表示(true：コード表示、false：コード非表示)
	 * @param viewGrade     等級表示(true：等級表示、false：等級非表示)
	 * @return プルダウン用配列
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected String[][] getSelectArray(Date targetDate, boolean needBlank, String operationType, boolean isName,
			boolean viewCode, boolean viewGrade) throws MospException {
		// 一覧取得
		List<PositionDtoInterface> list = getPositionList(targetDate, operationType);
		// 一覧件数確認
		if (list.size() == 0) {
			// 対象データ無し
			return getNoObjectDataPulldown();
		}
		// ソート(コード)
		sortList(list, PositionCodeComparator.class, false);
		// 等級表示確認
		if (viewGrade) {
			// ソート(等級、号数)
			sortList(list, PositionMasterGradeLevelComparator.class, hasLowGradeAdvantage());
		}
		// コード最大長取得
		int length = getMaxCodeLength(list, viewCode, viewGrade);
		// プルダウン用配列及びインデックス準備
		String[][] array = prepareSelectArray(list.size(), needBlank);
		int idx = needBlank ? 1 : 0;
		// プルダウン用配列作成
		for (PositionDtoInterface dto : list) {
			// コード設定
			array[idx][0] = dto.getPositionCode();
			// 表示内容設定
			if (isName && viewCode) {
				// コード+名称
				array[idx++][1] = getCodedName(dto.getPositionCode(), dto.getPositionName(), length);
			} else if (isName && viewGrade) {
				// 等級+名称
				array[idx++][1] = getCodedName(String.valueOf(dto.getPositionGrade()), dto.getPositionName(), length);
			} else if (isName) {
				// 名称
				array[idx++][1] = dto.getPositionName();
			} else if (viewCode) {
				// コード+略称
				array[idx++][1] = getCodedName(dto.getPositionCode(), dto.getPositionAbbr(), length);
			} else if (viewGrade) {
				// 等級+略称
				array[idx++][1] = getCodedName(String.valueOf(dto.getPositionGrade()), dto.getPositionAbbr(), length);
			} else {
				// 略称
				array[idx++][1] = dto.getPositionAbbr();
			}
		}
		return array;
	}
	
	/**
	 * リスト中のDTOにおけるコード等最大文字数を取得する。<br>
	 * @param list     対象リスト
	 * @param viewCode コード表示(true：コード表示、false：コード非表示)
	 * @param viewGrade 等級表示(true：等級表示、false：等級非表示)
	 * @return リスト中のDTOにおけるコード最大文字数
	 */
	protected int getMaxCodeLength(List<PositionDtoInterface> list, boolean viewCode, boolean viewGrade) {
		// コード等表示確認
		if (viewCode == false && viewGrade == false) {
			return 0;
		}
		// コード等最大文字数
		int length = 0;
		// コード等最大文字数確認
		for (PositionDtoInterface dto : list) {
			if (viewCode) {
				if (dto.getPositionCode().length() > length) {
					length = dto.getPositionCode().length();
				}
			} else if (viewGrade) {
				if (String.valueOf(dto.getPositionGrade()).length() > length) {
					length = String.valueOf(dto.getPositionGrade()).length();
				}
			}
		}
		return length;
	}
	
	/**
	 * 職位マスタリストから職位マスタマップを取得する。<br>
	 * マップのキーは職位コード。<br>
	 * @param list 職位マスタリスト
	 * @return 職位マスタマップ
	 */
	protected Map<String, PositionDtoInterface> getMap(List<PositionDtoInterface> list) {
		// マップ準備
		Map<String, PositionDtoInterface> map = new HashMap<String, PositionDtoInterface>();
		// マスタリストからマスタマップを作成
		for (PositionDtoInterface dto : list) {
			map.put(dto.getPositionCode(), dto);
		}
		return map;
	}
	
}
