/*
 * block_partition.c
 *
 * Copyright 2007, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 *Գס
 *
 *
 *
 *ԸƤ
 */


#include <sys/config.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/errno.h>
#include <sys/conf.h>
#include <sys/fcntl.h>
#include <sys/disklabel.h>
#include <sys/diskslice.h>
#include <sys/stat.h>
#include <sys/systm.h>
#include <sys/buf.h>
#include <machine/lib.h>
#include <kern/kmalloc.h>
#include <kern/dev_common.h>
#include <kern/device.h>
#include <kern/devfs.h>

#include <kern/debug.h>


//#define DEBUG_BLOCK_PARTITION 1
#ifdef DEBUG_BLOCK_PARTITION
	#define STATIC
	#define INLINE
#else
	#define STATIC	static
	#define INLINE	inline
#endif


//=====================================  ===================================================

//===================================== Х륤ݡ =======================================

//===================================== PRIVATE ====================================================

enum{
	/* Partition table */
	PARTITION_TBL_MAX		= 4,		/* 1κѡƥơ֥ */
	PARTITION_BEGIN			= 0x1be,	/* ѡƥơ֥볫ϥեå */
	PARTITION_MAGIC_ADDR	= 0x1fe,	/* ѡƥơ֥ޥåʥСեå */
	PARTITION_MAGIC_NUMBER	= 0xaa55,	/* ѡƥơ֥ޥåʥС */

	/* Partition type */
	PARTITION_TYPE_NULL		= 0,	/* NULL partition */
	PARTITION_TYPE_EXT		= 0x5,	/* ĥ */
};

/* Partition table */
typedef struct{
	uint8_t		active;		/* Boot active flag=0x80 */
	uint8_t		chsBeg[3];	/* CHS begin block */
	uint8_t		type;		/* Partition type */
	uint8_t		chsEnd[3];	/* CHS last block */
	uint32_t	lbaBeg;		/* LBA begin block */
	uint32_t	size;		/* Partition size by sector */
} PARTITION_TBL;

/*
 * ѡƥơ֥򸡺
 * return : NOERR
 *          ENODEV=Ѥ or ĥѡƥ
 *          ENOENT=ĥѡƥ󥯤κǸã
 */
STATIC int readPartition(
	const dev_t i_dev,
	const int partNum,
	PARTITION_TBL *o_partTbl)
{
	char *sectBuf;
	PARTITION_TBL *partTbl;
	uint extBeg, extNext;
	int c_part;
	int i;
	int error;
	size_t dummy;

	sectBuf = kmalloc(i_dev->sectSize);
	if (sectBuf == NULL) {
		return -ENOMEM;
	}

	// MBRɤ
	error = read_direct(i_dev, sectBuf, 1, 0, &dummy);
	if (error < 0){
		goto END;
	}

	/* ѡƥơ֥ǧ */
	if (*(uint16_t*) (sectBuf + PARTITION_MAGIC_ADDR) != PARTITION_MAGIC_NUMBER) {
		error = -ENODEV;
		goto END;
	}
	partTbl = (PARTITION_TBL*) (sectBuf + PARTITION_BEGIN);

	if (partNum <= PARTITION_TBL_MAX){
		/* ܥѡƥ򸡺 */
		c_part = partNum - 1;

		switch (partTbl[c_part].type) {
		// ΰγĥѡƥϳʤ
		case PARTITION_TYPE_EXT:
			// 
		case PARTITION_TYPE_NULL:
			error = -ENODEV;
			goto END;
		default:
			// ѥѡƥơ֥
			o_partTbl->type		= partTbl[c_part].type;
			o_partTbl->lbaBeg	= partTbl[c_part].lbaBeg;
			o_partTbl->size		= partTbl[c_part].size;
		}
	}
	else {
		/*
		 * MBRγĥѡƥ򸡺
		 * out : c_part = ĥѡƥֹ
		 */
		for (i = 0; ; ++i) {
			if (PARTITION_TBL_MAX <= i) {
				error = -ENOENT;
				goto END;
			}
			if (partTbl[i].type == PARTITION_TYPE_EXT) {
				c_part = i;
				break;
			}
		}

		/*
		 * ĥѡƥ󥯤򤿤ɤ
		 * out : extNext = ĥΰϤޤ꤫Υ֥å sectBuf = ĥΰΥѡƥ󥻥
		 */
		extBeg = extNext = partTbl[c_part].lbaBeg;
		for (i = PARTITION_TBL_MAX; ;) {
			error = read_direct(i_dev, sectBuf, 1, extNext, &dummy);
			if (error < 0) {
				goto END;
			}

			// ѡƥȯ
			if (++i == partNum) {
				break;
			}

			// ĥѡƥ󥯤νλ
			if (partTbl[1].type != PARTITION_TYPE_EXT) {
				error = -ENOENT;
				goto END;
			}

			extNext = extBeg + partTbl[1].lbaBeg;
		}

		// Ѥߥѡƥ
		if (partTbl[0].type == PARTITION_TYPE_NULL) {
			error = -ENODEV;
			goto END;
		}
		else {
			// ѥѡƥơ֥
			o_partTbl->type		= partTbl[0].type;
			o_partTbl->lbaBeg	= partTbl[0].lbaBeg + extNext;
			o_partTbl->size		= partTbl[0].size;
		}
	}

	error = NOERR;
END:
	kfree(sectBuf);

	return error;
}

/*
 *FreeBSD
 */
STATIC char *dsname(
	dev_t	dev,
	int		unit,
	int		slice,
	int		part,
	char	*partname)
{
	static char name[32];
	const char *dname;

	dname = devsw(dev)->d_name;
	if (MAX_DEVICE_NAME < strlen(dname)) {
		dname = "nametoolong";
	}
	snprintf(name, sizeof(name), "%s%d", dname, unit);
	partname[0] = '\0';
	if (slice != WHOLE_DISK_SLICE || part != RAW_PART) {
		partname[0] = 'a' + part;
		partname[1] = '\0';
		if (slice != COMPATIBILITY_SLICE){
			snprintf(name + strlen(name), sizeof(name) - strlen(name), "s%d", slice - 1);
		}
	}
	return (name);
}

//===================================== PUBLIC =====================================================

/*
 * Attempt to read a disk label from a device using the indicated strategy
 * routine.  The label must be partly set up before this: secpercyl, secsize
 * and anything required in the strategy routine (e.g., dummy bounds for the
 * partition containing the label) must be filled in before calling us.
 * Returns NULL on success and an error string on failure.
 */
char *readdisklabel(dev_t dev, struct disklabel *lp)
{
	char *buf;
	struct disklabel *dlp;
	char *msg = NULL;
	size_t size;

	buf = kmalloc(lp->d_secsize);
	if (buf == NULL){
		return "No memory";
	}

	if (read_direct(dev, buf, 1, LABELSECTOR, &size) != NOERR) {
		msg = "I/O error";
	}
	else{
		for (dlp = (struct disklabel *)buf;
			dlp <= (struct disklabel *)(buf + lp->d_secsize - sizeof(*dlp));
			dlp = (struct disklabel *)((char *)dlp + sizeof(long))) {
			if (dlp->d_magic != DISKMAGIC || dlp->d_magic2 != DISKMAGIC) {
				if (msg == NULL){
					msg = "no disk label";
			   }
			}
			else if (MAXPARTITIONS < dlp->d_npartitions || dkcksum(dlp) != 0){
				msg = "disk label corrupted";
			}
			else{
				*lp = *dlp;
				msg = NULL;
				break;
			}
		}
	}
	kfree(buf);

	return (msg);
}

/*
 *FreeBSD
 * Check new disk label for sensibility before setting it.
 */
int setdisklabel(struct disklabel *olp, struct disklabel *nlp, u_long openmask)
{
	int i;
	struct partition *opp, *npp;

	/*
	* Check it is actually a disklabel we are looking at.
	*/
	if (nlp->d_magic != DISKMAGIC || nlp->d_magic2 != DISKMAGIC || dkcksum(nlp) != 0){
		return -EINVAL;
	}
	/*
	* For each partition that we think is open,
	*/
	while ((i = ffs((long)openmask)) != 0) {
		i--;
		/*
		* Check it is not changing....
		*/
		openmask &= ~(1 << i);
		if (nlp->d_npartitions <= i){
			return (EBUSY);
		}
		opp = &olp->d_partitions[i];
		npp = &nlp->d_partitions[i];
		if (npp->p_offset != opp->p_offset || npp->p_size < opp->p_size){
			return (EBUSY);
		}
		/*
		* Copy internally-set partition information
		* if new label doesn't include it.		XXX
		* (If we are using it then we had better stay the same type)
		* This is possibly dubious, as someone else noted (XXX)
		*/
		if (npp->p_fstype == FS_UNUSED && opp->p_fstype != FS_UNUSED) {
			npp->p_fstype = opp->p_fstype;
			npp->p_fsize = opp->p_fsize;
			npp->p_frag = opp->p_frag;
			npp->p_cpg = opp->p_cpg;
		}
	}
	nlp->d_checksum = 0;
	nlp->d_checksum = dkcksum(nlp);
	*olp = *nlp;

	return (0);
}

/*
 * Write disk label back to device after modification.
 * return : error number
 */
int writedisklabel(dev_t dev, struct disklabel *lp)
{
	char *buf;
	struct disklabel *dlp;
	size_t size;
	int error;

	if (lp->d_partitions[RAW_PART].p_offset != 0){
		return -EXDEV;			/* not quite right */
	}

	buf = kmalloc(lp->d_secsize);
	if (buf == NULL){
		return -ENOMEM;
	}
	memset(buf, 0, lp->d_secsize);

	dlp = (struct disklabel*)buf;
	*dlp = *lp;
	error = write_direct(dev, dlp, 1, LABELSECTOR, &size);
	if (error != NOERR){
		return error;
	}
	kfree(buf);

	return NOERR;
}

/*
 *FreeBSD
 * Disk error is the preface to plaintive error messages
 * about failing disk transfers.  It prints messages of the form
 * hp0g: hard error reading fsbn 12345 of 12344-12347 (hp0 bn %d cn %d tn %d sn %d)
 * if the offset of the error in the transfer and a disk label
 * are both available.  blkdone should be -1 if the position of the error
 * is unknown; the disklabel pointer may be null from drivers that have not
 * been converted to use them.  The message is printed with printf
 * if pri is LOG_PRINTF, otherwise it uses log at the specified priority.
 * The message should be completed (with at least a newline) with printf
 * or addlog, respectively.  There is no trailing space.
 */
void diskerr(
	struct buf *bp,
	char *what,
	int pri,
	int blkdone,
	struct disklabel *lp)
{
	int unit = dkunit(bp->b_dev);
	int slice = dkslice(bp->b_dev);
	int part = dkpart(bp->b_dev);
	char partname[2];
	char *sname;
	daddr_t sn;

	sname = dsname(bp->b_dev, unit, slice, part, partname);
	printf("%s%s: %s %sing fsbn ", sname, partname, what, bp->b_flags & B_READ ? "read" : "writ");
	sn = bp->b_blkno;
	if (bp->b_bcount <= DEV_BSIZE){
		printf("%ld", (long)sn);
	}
	else {
		if (blkdone >= 0) {
			sn += blkdone;
			printf("%ld of ", (long)sn);
		}
		printf("%ld-%ld", (long)bp->b_blkno, (long)(bp->b_blkno + (bp->b_bcount - 1) / DEV_BSIZE));
	}
	if (lp && (blkdone >= 0 || bp->b_bcount <= lp->d_secsize)) {
		sn += lp->d_partitions[part].p_offset;
		/*
		* XXX should add slice offset and not print the slice,
		* but we don't know the slice pointer.
		* XXX should print bp->b_pblkno so that this will work
		* independent of slices, labels and bad sector remapping,
		* but some drivers don't set bp->b_pblkno.
		*/
		printf(" (%s bn %ld; cn %ld", sname, (long)sn, (long)(sn / lp->d_secpercyl));
		sn %= (long)lp->d_secpercyl;
		printf(" tn %ld sn %ld)", (long)(sn / lp->d_nsectors), (long)(sn % lp->d_nsectors));
	}
}

/*
 * ֥åǥХѡƥǥХե륷ƥϿ
 *եǥХǥХեϿ줿ƤӽФ롣
 * return : error number
 */
int registBlockPartition(
	struct specinfo *m_dev)
{
	DEV_STAT devStat;
	int partNum;
	int error;
	void *dummy;

	// ֥åǥХ򥪡ץ󤹤
	error = openDev(m_dev, O_RDONLY, &dummy);
	if (error != NOERR) {
		return error;
	}

	devStatDev(m_dev, &devStat);

	// ΰθ
	for (partNum = 1; partNum <= PARTITION_TBL_MAX; ++partNum) {
		PARTITION_TBL partTbl;

		if (readPartition(m_dev, partNum, &partTbl) == NOERR) {
			struct specinfo *dev = makeBlkPartDev(
				m_dev->si_devsw,
				partTbl.lbaBeg,
				partTbl.size,
				partTbl.type,
				devStat.major,
				partNum,
				S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP);
			if (dev != NULL) {
				makeDevf(dev->si_name, dev, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP);
			}
		}
	}

	// ĥΰθ
	for (partNum = PARTITION_TBL_MAX + 1; ; ++partNum){
		PARTITION_TBL partTbl;

		switch (error = readPartition(m_dev, partNum, &partTbl)) {
		case NOERR: {
			struct specinfo *dev = makeBlkPartDev(
				m_dev->si_devsw,
				partTbl.lbaBeg,
				partTbl.size,
				partTbl.type,
				devStat.major,
				partNum,
				S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP);
			if (dev != NULL) {
				makeDevf(dev->si_name, dev, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP);
			}

			break;
		}
		case -ENODEV:
			continue;
		case -ENOENT:
			goto END;
		default:
			printk("%s registBlockPartition() unkown return value!:%d", __FILE__, error);
			ASSERT(0);
		}
	}
END:
	return NOERR;
}
