/*
 * console.c
 *
 * Copyright 2002, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 */


#include <sys/types.h>
#include <sys/param.h>
#include <sys/conf.h>
#include <lib/lib.h>
#include <kern/proc.h>
#include <kern/Thread.h>
#include <kern/term.h>
#include <kern/errno.h>
#include <kern/vm.h>
#include <kern/devfs.h>
#include <kern/device.h>
#include <kern/Wait.h>
#include <dev/console/display.h>
#include <dev/console/console.h>

#include <kern/debug.h>


//#define DEBUG_CONSOLE 1
#ifdef DEBUG_CONSOLE
	#define STATIC
	#define INLINE
#else
	#define STATIC static
	#define INLINE	inline
#endif

/***************************************************************************************
 *
 * üǽ
 *
 ***************************************************************************************/

//================================== PRIVATE ============================================

enum {
	TAB_SIZE	= 8,
	ESC			= 0x1b,		// ץ
};

static struct termios termio = {
	ICRNL | IXOFF | IXON,	/* c_iflag. */
	OPOST,					/* c_oflag. */
	CS8 | CREAD | B9600,	/* c_cflag. */
	ISIG | ICANON | ECHO,	/* c_lflag. */
};

static LINE_BUF lbuf;

/*
 * ߥʥν
 */
STATIC void initTerm()
{
	/* termiosꡣ */
	termio.c_cc[VINTR]	= CINTR;
	termio.c_cc[VQUIT]	= CQUIT;
	termio.c_cc[VSTART]	= CSTART;
	termio.c_cc[VSTOP]	= CSTOP;
	termio.c_cc[VSUSP]	= CDSUSP;
	termio.c_cc[VEOF]	= CEOF;
	termio.c_cc[VEOL]	= CEOL;
	termio.c_cc[VERASE]	= CERASE;
	termio.c_cc[VKILL]	= CKILL;
}

/*
 * ֤ʣʸɽ
 */
STATIC void putNumScreen(
	const char *str,		// ʸ
	const size_t size)		// 
{
	int i;

	for (i = 0; i < size; ++i) {
		putDisplay(getCursorPos() + i, str[i], ATTR_NORMAL);
	}
	updateCursor(getCursorPos() + size);
}

/*
 * ֤ʣʸ
 */
STATIC void deleteBackScreen(
	const size_t size)		// 
{
	int i;
	
	for (i = 0; i < size; ++i) {
		putDisplay(getCursorPos() - i, '\0', ATTR_NORMAL);
	}
	updateCursor(getCursorPos() - size);
}

/*
 * ֤ʣʸ
 */
STATIC void deleteForwardScreen(
	const size_t size)		// 
{
	int i;
	
	for (i = 0; i < size; ++i) {
		putDisplay(getCursorPos() + i, '\0', ATTR_NORMAL);
	}
}

/*
 * ̤õ
 */
STATIC void deleteScreen()
{
	int pos = getScreenTop();
	int i;
	
	for (i = 0; i < SCREEN_COLUMNS * SCREEN_LINES; ++i) {
		putDisplay(pos + i, '\0', ATTR_NORMAL);
	}
	updateCursor(getCursorPos() + (SCREEN_COLUMNS * SCREEN_LINES));
}

/*
 * 򺸤ذư
 */
STATIC void leftCursor(
	int len)		// ư
{
	updateCursor(getCursorPos() - len);
}

/*
 * 򱦤ذư
 */
STATIC void rightCursor(
	int len)		// ư
{
	updateCursor(getCursorPos() + len);
}

/*
 * Ԥ
 */
STATIC void newLine()
{
	updateCursor(ROUNDUP(getCursorPos() + 1, SCREEN_COLUMNS));
}

/*
 * Ƭذư
 */
STATIC void carriageRetun()
{
	updateCursor(ROUNDDOWN(getCursorPos(), SCREEN_COLUMNS));
}

/*
 * ֥ʬذư
 */
STATIC void tab()
{
	int columns = ROUNDUP(getCursorPos() % SCREEN_COLUMNS + 1, TAB_SIZE);

	if (SCREEN_COLUMNS < columns) {
		columns = SCREEN_COLUMNS;
	}
	updateCursor(columns + ROUNDDOWN(getCursorPos(), SCREEN_COLUMNS));
}

/*
 * üǽɽؿ
 */

STATIC void leftCursorTerm(int n)
{
	if (lbuf.crt < lbuf.pastCrt)
		leftCursor(n);
}

STATIC void rightCursorTerm(int n)
{
	if (lbuf.pastCrt < lbuf.crt)
		rightCursor(n);
}

STATIC void inputStrTerm(void)
{
	putNumScreen(lbuf.buf + lbuf.pastCrt, lbuf.last - lbuf.pastCrt);
}

STATIC void eraseLineTerm(void)
{
	putNumScreen(lbuf.buf + lbuf.lineTop, lbuf.pastLast - lbuf.lineTop);
	leftCursor(lbuf.pastCrt - lbuf.lineTop);
}

STATIC void deleteTerm(void)
{
	putNumScreen(lbuf.buf + lbuf.crt, lbuf.pastLast - lbuf.crt);
}

STATIC void backSpaceTerm(void)
{
	putNumScreen(lbuf.buf + lbuf.crt, lbuf.pastLast - lbuf.crt);
	leftCursor(lbuf.pastCrt - lbuf.crt);
}

static TERM_CTL termCtl = {
	NULL,
	0,
	NULL,
	leftCursorTerm,
	rightCursorTerm,
	inputStrTerm,
	newLine,
	carriageRetun,
	eraseLineTerm,
	deleteTerm,
	backSpaceTerm,
	tab
};

/*
 * ץ
 */

STATIC void setCursolInside(int x, int y)
{
	int position = SCREEN_COLUMNS * (y - 1) + (x - 1);

	if (SCREEN_LINES * SCREEN_COLUMNS <= position) {
		position = SCREEN_COLUMNS * SCREEN_LINES - 1;
	}
	updateCursor(getScreenTop() + position);
}

STATIC void leftCursolInside(int x)
{
	int position = getCursorPos() - x;

	if (position < getScreenTop()) {
		position = getScreenTop();
	}
	updateCursor(position);
}

STATIC void rightCursolInside(int x)
{
	int position = getCursorPos() + x;

	if (getScreenTop() + SCREEN_COLUMNS * SCREEN_LINES <= position) {
		position = getScreenTop() + SCREEN_COLUMNS * SCREEN_LINES - 1;
	}
	updateCursor(position);
}

STATIC void leftCursolLine(int x)
{
	int position = getCursorPos() - x;

	if (position < ROUNDDOWN(getCursorPos(), SCREEN_COLUMNS)) {
		position = ROUNDDOWN(getCursorPos(), SCREEN_COLUMNS);
	}
	updateCursor(position);
}

STATIC void rightCursolLine(int x)
{
	int position = getCursorPos() + x;

	if (ROUNDUP(getCursorPos(), SCREEN_COLUMNS) <= position) {
		position = ROUNDUP(getCursorPos(), SCREEN_COLUMNS) - 1;
	}
	updateCursor(position);
}

STATIC void copyUpLine(int y)
{
	int src = ROUNDUP(getCursorPos(), SCREEN_COLUMNS);
	int dst = src - y * SCREEN_COLUMNS;
	
	if (dst  < getScreenTop()) {
		dst = getScreenTop();
	}
	copyDisplay(dst, src, getScreenTop() + SCREEN_LINES * SCREEN_COLUMNS - src);
}

STATIC void copyDownLine(int y)
{
	int size = ROUNDDOWN(getCursorPos(), SCREEN_COLUMNS) - getScreenTop();
	int startSize = y * SCREEN_COLUMNS;
	
	if (SCREEN_LINES * SCREEN_COLUMNS - size < startSize) {
		startSize = SCREEN_LINES * SCREEN_COLUMNS - size;
	}
	copyDisplay(getScreenTop() + startSize, getScreenTop(), size);
}

// ץѴơ֥	
static const uchar sequence[]={
//	 0   1   2   3   4   5   6   7   8   9   a   b   c   d   e   f
	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,	// 0
	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,	// 1
	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0, '*', 0,  0,  0,  0,  0,	// 2
	'n','n','n','n','n','n','n','n','n','n', 0, ';', 0,  0,  0,  0,	// 3
	 0, 'A','B','C','D','E', 0,  0, 'H', 0, 'J','K','L','M', 0,  0,	// 4
	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,	// 5
	 0,  0,  0,  0,  0,  0, 'H', 0,  0,  0,  0,  0,  0, 'm', 0,  0,	// 6
	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,	// 7
	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,	// 8
	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,	// 9
	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,	// a
	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,	// b
	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,	// c
	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,	// d
	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,	// e
	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,	// f
};

// escape_sequence()λ
enum{
	ESC_DONE = 0,	// Ѵλ
	ESC_ERR,		// Ѵ顼
	ESC_NOTDONE,	// Ѵ
};

/*
 * ץѴ
 */
STATIC int escapeSequence(const uchar *str, const int num)
{
	uint xy[2] = {0, 0};
	int xy_idx = 0;
	int c_str;

	if (num == 0){
		return ESC_NOTDONE;
	}
	
	switch (*str){
	case '[':
		break;
	case 'c':
		deleteScreen();
		return ESC_DONE;
	case 'D':	// 饤󡢥뤢
		rightCursor(SCREEN_COLUMNS);
		return ESC_DONE;
	case 'E':	// 饤󡢥뤢
		rightCursor(ROUNDUP(getCursorPos() + 1,SCREEN_COLUMNS));
		return ESC_DONE;
	case 'M':	// 饤󥢥åסܡ饤
		leftCursolInside(SCREEN_COLUMNS);
		copyDownLine(1);
		return ESC_DONE;
	default:
		return ESC_ERR;
	}
	for (c_str = 1; c_str < num; ++c_str) {
		switch (sequence[(int)str[c_str]]){
		case 'n':
			xy[xy_idx] *= 10;
			xy[xy_idx] += str[c_str] - '0';
			break;
		case ';':
			if (1 < ++xy_idx){
				return ESC_ERR;
			}
			break;
		case 'A':	// 饤󥢥å
			if (xy_idx == 0){
				leftCursolInside(((xy[0] == 0)? 1 : xy[0]) * SCREEN_COLUMNS);
				return ESC_DONE;
			}
			return ESC_ERR;
		case 'B':	// 饤
			if (xy_idx == 0){
				rightCursolInside(((xy[0] == 0)? 1 : xy[0]) * SCREEN_COLUMNS);
				return ESC_DONE;
			}
			return ESC_ERR;
		case 'C':	// 饤
			if (xy_idx == 0){
				rightCursolLine((xy[0] == 0)? 1 : xy[0]);
				return ESC_DONE;
			}
			return ESC_ERR;
		case 'D':	// ե
			if (xy_idx == 0){
				leftCursolLine((xy[0] == 0)? 1 : xy[0]);
				return ESC_DONE;
			}
			return ESC_ERR;
		case 'L':	// 饤ư
			if (xy_idx == 0){
				copyUpLine((xy[0] == 0)? 1 : xy[0]);
				return ESC_DONE;
			}
			return ESC_ERR;
		case 'M':	// 饤󲼰ư
			if (xy_idx == 0){
				copyDownLine((xy[0] == 0)? 1 : xy[0]);
				return ESC_DONE;
			}
			return ESC_ERR;
		case 'J':
			if (xy_idx == 0){
				switch (xy[0]){
				case 0:	// 뤫̺ǸޤǾõ
					deleteForwardScreen(getScreenTop() + SCREEN_LINES * SCREEN_COLUMNS - getCursorPos());
					break;
				case 1:	// ̺ǽ餫饫ޤǾõ
					deleteBackScreen(getCursorPos() - getScreenTop());
					break;
				case 2:	// õ
					deleteScreen();
					break;
				}
				return ESC_DONE;
			}
			return ESC_ERR;
		case 'K':
			if (xy_idx == 0){
				switch (xy[0]){
				case 0:	// 뤫鱦üޤǾõ
					deleteForwardScreen(ROUNDUP(getCursorPos() + 1, SCREEN_COLUMNS) - getCursorPos());
					break;
				case 1:	// ü饫ޤǾõ
					deleteBackScreen(getCursorPos() - ROUNDDOWN(getCursorPos(), SCREEN_COLUMNS));
					break;
				case 2:	// 饤õ
					deleteForwardScreen(ROUNDUP(getCursorPos() + 1, SCREEN_COLUMNS) - getCursorPos());
					deleteBackScreen(getCursorPos() - ROUNDDOWN(getCursorPos(), SCREEN_COLUMNS));
					break;
				}
				return ESC_DONE;
			}
			return ESC_ERR;
		case 'H':	// 륻å
			if (xy_idx == 1){
				setCursolInside((xy[1] == 0)? 1 : xy[1],(xy[0] == 0)? 1 : xy[0]);
				return ESC_DONE;
			}
			return ESC_ERR;
		case '*':	// õ
			if (xy[0] == 0 && xy_idx == 0){
				deleteScreen();
				return ESC_DONE;
			}
			return ESC_ERR;
		case 'm':
			// ̤
			// 
		default:
			return ESC_ERR;
		}
	}

	return ESC_NOTDONE;
}

/*
 * Ѵɽ
 * return : FALSE or ESCɤ = TRUE
 */
STATIC INLINE int printScreen(const char c)
{
	switch(c){
	case '\b':
		leftCursor(1);
		break;
	case '\n':
		newLine();
		break;
	case '\r':
		carriageRetun();
		break;
	case '\t':
		tab();
		break;
	case '\a':
		// ̤
		break;
	case ESC:
		return TRUE;
	default:
		putNumScreen(&c, 1);
	}
	
	return FALSE;
}

/*
 * 饯򥳥󥽡ɽ
 */
STATIC void printConsole(const char chr)
{
	enum {
		ESC_BUF_SIZE = 10,			// ץѥХåե
	};
	static uchar escBuf[ESC_BUF_SIZE];	// ץѥХåե
	static int escSw = FALSE;			// ץ󥹽񤭽Ф
	static int escNum = 0;				// ץѥХåե¸ʸ

	if (escSw == TRUE) {
		int i;

		if (escNum < ESC_BUF_SIZE) {
			// ץ󥹥Хåե˳Ǽ
			escBuf[escNum++] = chr;
			switch (escapeSequence(escBuf, escNum)){
			case ESC_DONE:
				escNum = 0;
				escSw = FALSE;
				break;
			case ESC_ERR:
				// Хåեߤʸ򥳥󥽡ǤФ
				for (i = 0; i < escNum; ++i) {
					escSw = printScreen(escBuf[i]);
				}
				escNum = 0;
				escSw = FALSE;
				break;
			case ESC_NOTDONE:
				break;
			default:
				ASSERT(0);
			}
		}
		else{
			// Хåեߤʸ򥳥󥽡ǤФ
			for (i = 0; i < escNum; ++i){
				printScreen(escBuf[i]);
			}
			escNum = 0;
			escSw = FALSE;
			escSw = printScreen(chr);
		}
	}
	else{
		escSw = printScreen(chr);
	}
}

//================================== PUBLIC =============================================

/*
 * 󥽡˥Хåեñ̤ǽ񤭹
 * return : write size
 */
int writeConsole(const char *str, size_t size)
{
	int i;

	for (i = 0; i < size; ++i){
		printConsole(str[i]);
	}

	return size;
}

/*
 * 󥽡˳ϡʥܡɡˤ줿饯
 * return : YES = 桼ɤ߹߲ǽˤʤä or NO = 桼ɤ߹ߤǤʤ
 */
int sendConsole(uchar cha)
{
	// ܡɥ⥸塼ENTERͤ'\r'ʤΤ촹
	if (cha == '\r') {
		cha = '\n';
	}
	return writeTerm(cha, &termio, &termCtl, &lbuf);
}

/***************************************************************************************
 *
 * ƥॳ
 *
 ***************************************************************************************/

STATIC int open(dev_t dev, int oflags, int devtype, d_thread_t *proc)
{
	return 0;
}

STATIC int close(dev_t dev, int fflag, int devtype, d_thread_t *proc)
{
	return 0;
}

/* size = žǽХȿ */
STATIC int read(dev_t dev, struct uio *uio, int ioflag)
{
	/* ե饦ɥ롼פ? */
	while (isForeground(getCurrentProc()) == NO){
		waitTaskSignal();
	}

	// 󥽡Хåեɤ
	return readTerm(uio->uio_iov->iov_base, uio->uio_iov->iov_len, &termio, &termCtl, &lbuf);
}

STATIC int write(dev_t dev, struct uio *uio, int ioflag)
{
	return writeConsole(uio->uio_iov->iov_base, uio->uio_iov->iov_len);
}

STATIC int ioctl(dev_t dev, u_long cmd, caddr_t data, int fflag, d_thread_t *p)
{
	void *proc;
	void *grp_proc;
	pid_t *ppgid;
	struct winsize *win;
	struct termios *tio;

	switch (cmd){
	case TIOCSETD:			// 浬§
		break;
	case TIOCGETD:			// 浬§μ
		break;
	case TIOCSBRK:			// BREAK֤
		break;
	case TIOCCBRK:			// BREAK֤γ
		break;
	case TIOCSDTR:			// ǡüǥ(DTR)
		break;
	case TIOCCDTR:			// ǡüǥ(DTR)Υꥢ
		break;
	case TIOCGPGRP:			// ե饦ɥץ롼פμ
		if(getCtltermProc(getCurrentProc()) != &termCtl){
			return -ENOTTY;
		}
		ppgid = (pid_t*) data;
		if (checkMem(ppgid,sizeof(pid_t)) == ERR) {
			return -EFAULT;
		}
		*ppgid = termCtl.foregrnd;
		break;
	case TIOCSPGRP:			// ե饦ɥץ롼פ
		ppgid = (gid_t*) data;
		if (checkMem(ppgid, sizeof(gid_t)) == ERR){
			return -EFAULT;
		}
		proc = getCurrentProc();

		/* üץ */
		if (proc != termCtl.ctlProc){
			return -ENOTTY;
		}
		if (getPgid(proc) != *ppgid){
			/* Ϳ줿롼פƱåΤΤ */
			grp_proc = procGetProcPid(*ppgid);
			if (grp_proc == NULL){
				return -EINVAL;
			}
			if (getSid(grp_proc) != getSid(proc)){
				return -EPERM;
			}
		}
		termCtl.foregrnd = *ppgid;
		break;
	case TIOCGETA:			// termiosμ
		tio = (struct termios*)data;
		if (checkMem(tio,sizeof(struct termios)) == ERR){
			return -EFAULT;
		}
		*tio = termio;
		break;
		break;
	case TIOCSETA:			// ľtermiosꤹ
		// 
	case TIOCSETAW:			// ٤ƤνϤδλԤätermiosꤹ
		// 
	case TIOCSETAF:			// αƤϤ򥯥ꥢ٤ƤνϤδλԤätermiosꤹ
		tio = (struct termios*)data;
		if (checkMem(tio,sizeof(struct termios)) == ERR){
			return -EFAULT;
		}
		termio = *tio;
		break;
	case TIOCOUTQ:			// ϥ塼θߤʸμ
		break;
	case TIOCSTI:			// ѥ᡼ʸ
		break;
//	case TIOCNOTTY:
//		break;
	case TIOCSTOP:			// üνϤ
		break;
	case TIOCSTART:			// üνϤγ
		break;
	case TIOCSCTTY:			// üץ
		if (termCtl.ctlProc == NULL){
			termCtl.ctlProc = getCurrentProc();
			termCtl.foregrnd = getPgid(termCtl.ctlProc);
			setCtltermProc(termCtl.ctlProc, &termCtl);
		}
		else{
			return -EPERM;
		}
		break;
	case TIOCDRAIN:			// ϤˤʤޤԤ
		break;
	case TIOCEXCL:			// üѻѤ
		break;
	case TIOCNXCL:			// üѻѤβ
		break;
	case TIOCFLUSH:			// ϡϥ塼Υꥢ
		break;
	case TIOCGWINSZ:		// ɥμ
		win=(struct winsize*)data;
		if (checkMem(win,sizeof(struct winsize)) == ERR){
			return -EFAULT;
		}
		win->ws_row=SCREEN_LINES;
		win->ws_col=SCREEN_COLUMNS;
		break;
	case TIOCSWINSZ:		// ɥ
		break;
	case TIOCCONS:			// ѹ
		break;
	case TIOCMSET:			// ǥ֤
		break;
	case TIOCMGET:			// ǥ֤μ
		break;
	case TIOCMBIS:			// ǥ֤ɲ
		break;
	case TIOCMBIC:			// ǥ֤ɲåꥢ
		break;
	default:
	 	printk("ioctl() failed command : %d : %s line=%d \n", cmd, __FILE__, __LINE__);
		break;
	}

	return 0;
}

STATIC int poll(dev_t dev, int events, d_thread_t *p)
{
	if (canReadTerm(&termio, &termCtl, &lbuf) == YES) {
		return POLLIN | POLLRDNORM | POLLOUT | POLLWRNORM;
	}
	else {
		return POLLOUT | POLLWRNORM;
	}
}

/***************************************************************************************
 *
 * 
 *
 ***************************************************************************************/

/* Device interface */
static struct cdevsw devInfo = {
	open, 
	close, 
	read, 
	write, 
	ioctl, 
	poll, 
	NULL, 
	NULL, 
	"console", 
	0,
	NULL, 
	NULL, 
	D_TTY,
};

//================================== PUBLIC =============================================

/*
 * ǥХϿ򤹤롣
 */
int registConsole()
{
	dev_t dev;

	dev = make_dev(&devInfo, 0, 0, 0, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP, "console");
	if (dev == NODEV) {
		return -ENODEV;
	}
	
	// ǥХե
	makeDevf(devInfo.d_name, dev, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP);

	return 0;
}

/*
 * 󥽡ν
 */
int initConsole()
{
	// ǥץ쥤ν
	initDisplay();

	// üǽν
	initTerm();

	return 0;
}

//*************************************************************************************************
void test_console()
{
	int i;
	
	for (i = 0; i < 16; ++i) {
		
	}
}
