/*
 * device.c
 *
 * Copyright 2002, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 */


#include <sys/config.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/errno.h>
#include <sys/systm.h>
#include <sys/uio.h>
#include <sys/AggregateList.h>
#include <lib/lib.h>
#include <lib/lib_path.h>
#include <machine/interrupt.h>
#include <kern/kmalloc.h>
#include <kern/lock.h>
#include <kern/timer.h>
#include <kern/ProcSignal.h>
#include <kern/block_cache.h>
#include <kern/devfs.h>
#include <kern/dev_common.h>
#include <kern/Wait.h>
#include <kern/device.h>

#include <kern/debug.h>


//#define DEBUG_DEVICE 1
#ifdef DEBUG_DEVICE
	#define STATIC
	#define INLINE
#else
	#define STATIC static
	#define INLINE inline
#endif


//================================== PRIVATE ============================================

/*
 * ǥХ¤Υꥹȥإå
 */
typedef struct {
	AggregateList	aggregate;		// ǥХꥹȥإå
	int				lock;			// ԥå
} DEV_SPEC_HEAD;

static DEV_SPEC_HEAD devSpecHead;

/*
 * ǥХ̾õ
 * return : struct specinfo or NULL
 */
STATIC struct specinfo *searchDevByName(
	const char *i_name)
{
	AggregateList *aggregate = &devSpecHead.aggregate;
	IteratorList iterator;
	struct specinfo *dev;

	enter_spinlock(&devSpecHead.lock);
	{
		aggregate->iterator(aggregate, &iterator);
		for (; ; ) {
			if (iterator.hasNext(&iterator) == BOOL_FALSE) {
				dev = NULL;
				break;
			}

			dev = iterator.next(&iterator);
			if (strcmp(dev->si_name, i_name) == 0) {
				break;
			}
		}
	}
	exit_spinlock(&devSpecHead.lock);
	
	return dev;
}

/*
 * ǥХ̾õ
 * return : struct specinfo or NULL
 */
STATIC struct specinfo *searchDevByDevnum(
	const int i_devNum)
{
	struct specinfo *dev;
	AggregateList *aggregate = &devSpecHead.aggregate;
	IteratorList iterator;

	enter_spinlock(&devSpecHead.lock);
	{
		aggregate->iterator(aggregate, &iterator);
		for (; ; ) {
			if (iterator.hasNext(&iterator) == BOOL_FALSE) {
				dev = NULL;
				break;
			}

			dev = iterator.next(&iterator);
			if (dev->si_udev == i_devNum) {
				break;
			}
		}
	}
	exit_spinlock(&devSpecHead.lock);
	
	return dev;
}

/*
 * ǥХϿ
 * return : struct specinfo or NULL
 */
STATIC struct specinfo *registDev(
	struct cdevsw *i_cdevsw,
	const size_t i_blockBeg,
	const size_t i_blocks,
	const int i_partType,
	const int i_major,
	const int i_minor,
	const int i_perms,
	const char *i_name)
{
	AggregateList *aggregate = &devSpecHead.aggregate;
	struct specinfo *dev;

	dev = kmalloc(sizeof(*dev));
	if (dev == NULL) {
		return NULL;
	}
	memset(dev, 0, sizeof(*dev));

	// ƱǥХ̾ʤ
	if (searchDevByName(dev->si_name) != NULL) {
		kfree(dev);
		return NULL;
	}

	// ǥХꤹ
	dev->si_udev		= makeudev(i_major, i_minor);
	dev->si_devsw		= i_cdevsw;
	dev->beginSector	= i_blockBeg;
	dev->sectors		= i_blocks;
	dev->partType		= i_partType;
	memcpy(dev->si_name, i_name, strlen(i_name));
	if (i_cdevsw->d_flags & (D_TAPE | D_TTY | D_MEM)) {		// 饯ǥХ
		dev->si_flags = S_IFCHR | i_perms;
	}
	else if ((i_cdevsw->d_flags & D_DISK) == D_DISK) {		// ֥åǥХ
		dev->si_flags = S_IFBLK | i_perms;
	}
	else{
		// Ȥꤢ饯ǥХˤƤ
		dev->si_flags = S_IFCHR | i_perms;
	}

	// ꥹȤ³
	listConstructor(&dev->list, dev);
	enter_spinlock(&devSpecHead.lock);
	{
		aggregate->insertHead(aggregate, &dev->list);
	}
	exit_spinlock(&devSpecHead.lock);

	return dev;
}

/*
 * ǥХϿ
 */
STATIC void unregistDev(
	struct specinfo *m_dev)
{
	AggregateList *aggregate = &devSpecHead.aggregate;
	
	enter_spinlock(&devSpecHead.lock);
	{
		aggregate->removeEntry(aggregate, &m_dev->list);
	}
	exit_spinlock(&devSpecHead.lock);

	kfree(m_dev);
}

/*
 * cdevsw¤ΤNULLؿѴ롣
 */
STATIC void compile_devsw(
	struct cdevsw *devsw)
{
	if (devsw->d_open == NULL) {
		devsw->d_open = noopen;
	}
	if (devsw->d_close == NULL) {
		devsw->d_close = noclose;
	}
	if (devsw->d_read == NULL) {
		devsw->d_read = noread;
	}
	if (devsw->d_write == NULL) {
		devsw->d_write = nowrite;
	}
	if (devsw->d_ioctl == NULL) {
		devsw->d_ioctl = noioctl;
	}
	if (devsw->d_poll == NULL) {
		devsw->d_poll = nopoll;
	}
	if (devsw->d_mmap == NULL) {
		devsw->d_mmap = nommap;
	}
	if (devsw->d_strategy == NULL) {
		devsw->d_strategy = nostrategy;
	}
	if (devsw->d_dump == NULL) {
		devsw->d_dump = nodump;
	}
	if (devsw->d_psize == NULL) {
		devsw->d_psize = nopsize;
	}
	if (devsw->d_kqfilter == NULL) {
		devsw->d_kqfilter = nokqfilter;
	}
}

//----------------------------------------------------------
// ǥХ
//----------------------------------------------------------

static void (*intrHandler[IRQ_ENTRY])(void *);
static void *intrArg[IRQ_ENTRY];

STATIC int deviceIntr(int irq)
{
	intrHandler[irq](intrArg[irq]);

	return 1;
}

//----------------------------------------------------------
// ǥХڥ졼
//----------------------------------------------------------

/*
 * ǥХڥ졼˥СȤ롣
 * return : error number
 */
STATIC int convReadDev(
	struct specinfo *i_dev,
	void *buf,
	const size_t size,	// READ֥å
	const size_t begin,	// Ƭ֥å
	const int ioflag,
	size_t *o_size)		// READѥ֥å
{
	struct iovec *iovec;
	struct uio uio;
	int error;

	ASSERT(0 <= i_dev->sectSize);

	iovec = kmalloc(sizeof(*iovec));
	iovec->iov_base = buf;
	iovec->iov_len = size * i_dev->sectSize;

	setUio(
		1, 
		UIO_READ, 
		begin * i_dev->sectSize,
		iovec->iov_len,
		iovec,
		&uio);
	error = i_dev->si_devsw->d_read(i_dev, &uio, ioflag);
	*o_size = size - (uio.uio_resid / i_dev->sectSize);
	
	kfree(iovec);
	
	return error;
}

/*
 * ǥХڥ졼˥СȤ롣
 * return : error number
 */
STATIC int convWriteDev(
	struct specinfo *dev,
	void *buf,
	const size_t size,	// WRITE֥å
	const size_t begin,	// Ƭ֥å
	const int ioflag,
	size_t *o_size)		// WRITEѥ֥å
{
	struct iovec *iovec;
	struct uio uio;
	int error;

	ASSERT(0 <= dev->sectSize);

	iovec = kmalloc(sizeof(*iovec));
	iovec->iov_base = buf;
	iovec->iov_len = size * dev->sectSize;

	setUio(
		1, 
		UIO_WRITE, 
		begin * dev->sectSize,
		iovec->iov_len,
		iovec,
		&uio);
	error = dev->si_devsw->d_write(dev, &uio, ioflag);
	*o_size = size - (uio.uio_resid / dev->sectSize);
	
	kfree(iovec);
	
	return error;
}

//================================== PUBLIC =============================================

/*
 * ǥХƥν
 * return : error number
 */
int initDevice()
{
	AggregateListConstructor(&devSpecHead.aggregate);

	return NOERR;
}

/*
 * ǥХϿ
 * return : struct specinfo or NODEV
 */
struct specinfo *make_dev(
	struct cdevsw *devsw, 
	int minor,				// ޥʡֹ
	uid_t uid,
	gid_t gid,
	int perms,				// ե⡼
	const char *fmt, ...)
{
	char name[MAX_DEVICE_NAME + 1];

	// cdevsw¤ΤNULLؿ
	compile_devsw(devsw);
	
	// ǥХ̾
	snprintf(name, MAX_DEVICE_NAME + 1, "%s%d", devsw->d_name, minor);
	name[MAX_DEVICE_NAME] = '\0';

	return registDev(
		devsw,
		0,
		0,
		0,
		getNewMajor(),
		minor,
		perms,
		name);
}

/*
 * ֥åǥХϿ
 * return : struct specinfo or NULL
 */
struct specinfo *makeBlkDev(
	struct cdevsw *m_cdevsw,
	const size_t i_blockBeg,
	const size_t i_blocks,
	const int i_perms)
{
	// cdevsw¤ΤNULLؿ
	compile_devsw(m_cdevsw);

	return registDev(
		m_cdevsw,
		i_blockBeg,
		i_blocks,
		0,
		getNewMajor(),
		0,
		i_perms,
		m_cdevsw->d_name);
}

/*
 * ֥åѡƥϿ
 * return : struct specinfo or NULL
 */
struct specinfo *makeBlkPartDev(
	struct cdevsw *i_cdevsw,
	const size_t i_blockBeg,	// ϥ֥åֹ
	const size_t i_blocks,		// ֥å
	const int i_partType,		// ѡƥ󥿥
	const int i_major,
	const int i_partNum,		// ѡƥֹ
	const int i_perms)
{
	char name[MAX_DEVICE_NAME + 1];

	// ǥХ̾
	snprintf(name, MAX_DEVICE_NAME + 1, "%s%d", i_cdevsw->d_name, i_partNum);
	name[MAX_DEVICE_NAME] = '\0';

	return registDev(
		i_cdevsw,
		i_blockBeg,
		i_blocks,
		i_partType,
		i_major,
		i_partNum,
		i_perms,
		name);
}

/*
 * Delete from regist table
 * return : error number
 */
void destroy_dev(
	struct specinfo *dev)
{
	unregistDev(dev);
}

/*
 * ǥХֹõ
 * return : error number
 */
struct specinfo *getDevByDevnum(
	const int i_devnum)
{
	return searchDevByDevnum(i_devnum);
}

//----------------------------------------------------------
// ǥХڥ졼
//----------------------------------------------------------

/*
 * UIOꤹ
 * return : error number
 */
void setUio(
	const int iovecCnt,		// struct iovec
	enum uio_rw uioMode,	// UIO_READ or UIO_WRITE
	const uint offset,		// Ƭեå
	const int resid,		// IO
	struct iovec *iovec,
	struct uio *m_uio)		// ꤹstruct uio
{
	m_uio->uio_iov = iovec;
	m_uio->uio_iovcnt = iovecCnt;
	m_uio->uio_offset = offset;
	m_uio->uio_resid = resid;
	m_uio->uio_segflg = UIO_SYSSPACE;
	m_uio->uio_rw = uioMode;
//	m_uio->uio_procp = NULL;
}

int noopen(struct specinfo *dev, int flags, int fmt, struct proc *p)
{
	return (-ENODEV);
}

int noclose(struct specinfo *dev, int flags, int fmt, struct proc *p)
{
	return (-ENODEV);
}

int noread(struct specinfo *dev, struct uio *uio, int ioflag)
{
	return (-ENODEV);
}

int nowrite(struct specinfo *dev, struct uio *uio, int ioflag)
{
	return (-ENODEV);
}

int noioctl(struct specinfo *dev, u_long cmd, caddr_t data, int flags, struct proc *p)
{
	return (-ENODEV);
}

int nokqfilter(struct specinfo *dev, struct knote *kn)
{
	return (-ENODEV);
}

int nommap(struct specinfo *dev, vm_offset_t offset, int nprot)
{
	/* Don't return ENODEV.  That would allow mapping address ENODEV! */
	return (-1);
}

int nodump(struct specinfo *dev)
{
	return (-ENODEV);
}

/*
 * nopoll
 */
int seltrue(struct specinfo *dev, int events, struct proc *p)
{
	return (events & (POLLIN | POLLOUT | POLLRDNORM | POLLWRNORM));
}

/*
 *FreeBSD
 * return : cdevsw
 */
struct cdevsw *devsw(
	struct specinfo *dev)
{
	if (dev->si_devsw != NULL) {
		return dev->si_devsw;
	}
	else {
		ASSERT(0);
		return NULL;
	}
}

/*
 *FreeBSD
 * cdevsw¤ΤåϿ롣
 * return : error number
 */
int cdevsw_add(struct cdevsw *newentry)
{
	compile_devsw(newentry);
	if (newentry->d_maj < 0 || newentry->d_maj >= NUMCDEVSW) {
		printf("%s: ERROR: driver has bogus cdevsw->d_maj = %d\n", newentry->d_name, newentry->d_maj);
		return -EINVAL;
	}
	if (newentry->d_bmaj >= NUMCDEVSW) {
		printf("%s: ERROR: driver has bogus cdevsw->d_bmaj = %d\n",newentry->d_name, newentry->d_bmaj);
		return -EINVAL;
	}
	if (newentry->d_bmaj >= 0 && (newentry->d_flags & D_DISK) == 0) {
		printf("ERROR: \"%s\" bmaj but is not a disk\n", newentry->d_name);
		return -EINVAL;
	}
	if (newentry->d_bmaj < 0){
		return (0);
	}

	return (0);
}

//----------------------------------------------------------
// ǥХ
//----------------------------------------------------------

/*
 * ǥХߤ
 * return : error number
 */
void setupIntr(
	struct resource *irq,
	void (*handler)(void *),
	void *softc)
{
	irq_entry[irq->r_start] = deviceIntr;
	intrHandler[irq->r_start] = handler;
	intrArg[irq->r_start] = softc;
}

//----------------------------------------------------------
// ǥХԤ
//----------------------------------------------------------

/*
 * ϥɥ
 */
STATIC void handleDevWake(
	void *i_task)
{
	addToSchedule(i_task, TASK_DEVICE_WAIT);
}

/*
 *ɬȯθȤʤ륳ޥɤ˸ƤӽФ
 */
void devInitWait(WAIT_INTR *wait)
{
	wait->task = getCurrentTask();
	wait->isIntr = NO;
	wait->lock = 0;
}

/*
 * ǥХԤ
 *WAIT_INTRdevInitWait()ǽƤ뤳
 *        ߤƱCPUȯ뤳
 */
void devWaitIntr(WAIT_INTR *wait, uint time)
{
	void *timer;
	int error = setTaskTimer(time, 0, handleDevWake, getCurrentTask(), &timer);
	int eflag;

	eflag = enterCli();
	{
		enter_spinlock(&wait->lock);

		if (wait->isIntr == NO) {
			delFromSchedule(TASK_DEVICE_WAIT);
			exit_spinlock(&wait->lock);
			wait_task();
		}
		else {
			exit_spinlock(&wait->lock);
		}
	}
	exitCli(eflag);

	if (error == NOERR){
		freeTimer(timer);
	}
}

/*
 *Գ
 * return : YES = å
 */
int devWakeIntr(WAIT_INTR *wait)
{
	int eflag;

	eflag = enterCli();
	enter_spinlock(&wait->lock);
	{
		wait->isIntr = YES;
		addScheduleSoon(wait->task, TASK_DEVICE_WAIT);
	}
	exit_spinlock(&wait->lock);
	exitCli(eflag);

	return YES;
}

//----------------------------------------------------------
// ľžؿ
//----------------------------------------------------------

/*
 * return : error number
 */
int read_direct(
	void *i_dev,		// ǥХ
	void *buf,
	size_t i_sectors,
	size_t begin,
	size_t *o_sectors)	// READѥ֥å
{
	struct specinfo *dev = i_dev;
	size_t sectors;

	/* ǥ¤Υå */
	if (dev->sectors < begin){
		return -EIO;
	}
	if ((begin + i_sectors) <= dev->sectors){
		sectors = i_sectors;
	}
	else{
		sectors = dev->sectors - begin;
	}

	return convReadDev(dev, buf, sectors, begin + dev->beginSector, 0, o_sectors);
}

/*
 * return : error number
 */
int write_direct(
	void *i_dev,		// ǥХ
	void *buf,
	size_t i_sectors,
	size_t begin,
	size_t *o_sectors)	// WRITEѥ֥å
{
	struct specinfo *dev = i_dev;
	size_t sectors;

	/* ǥ¤Υå */
	if (dev->sectors < begin){
		return -EIO;
	}
	if ((begin + i_sectors) <= dev->sectors){
		sectors = i_sectors;
	}
	else{
		sectors = dev->sectors - begin;
	}

	return convWriteDev(dev, buf, sectors, begin + dev->beginSector, 0, o_sectors);
}

//--------------------------------------------------------------------------------------------------
// եե󥯥
//--------------------------------------------------------------------------------------------------

/*
 * ǥХ򥪡ץ󤹤
 * return : error number
 */
int devOpenDev(
	struct specinfo *m_dev,
	const int i_oflag,
	void **o_privateData)
{
	if ((m_dev->si_flags & S_IFMT) == S_IFCHR) {
		m_dev->sectSize = 1;
	}

	return m_dev->si_devsw->d_open(m_dev, i_oflag, 0, NULL);
}

/*
 * ǥХ򥯥
 * return : error number
 */
int devCloseDev(
	struct specinfo *m_dev,
	void *i_privateData)
{
	return m_dev->si_devsw->d_close(m_dev, 0, 0, NULL);
}

/*
 * ǥХɤ߹
 * return : read bytes or error number
 */
int devReadDev(
	struct specinfo *i_dev,
	void *m_buf,				// ߥХåե
	const size_t i_bytes,		// ɤ߹ߥХȿ
	const size_t i_offset,		// ɤ߹߳ϥХȥեå
	void *i_privateData)
{
	char *readBuf = m_buf;
	size_t ioSize;				// žѥ
	int error;

	switch (i_dev->si_flags & S_IFMT) {
	// 饯ǥХ
	case S_IFCHR:
		error = convReadDev(i_dev, readBuf, i_bytes, 0, 0, &ioSize);
		if (error != NOERR){
			return error;
		}
		else{
			return ioSize;
		}
	// ֥åǥХ
	case S_IFBLK:{
		char *sectBuf = NULL;
		int sectBeg = i_offset / i_dev->sectSize;	// žϤΥֹ
		size_t restBytes = i_bytes;					// ĤžХȿ
		size_t sectors;

		/* ǽΥ̤ž */
		if (0 < (i_offset % i_dev->sectSize)){
			int forward = i_offset % i_dev->sectSize;
			int back;

			sectBuf = kmalloc(i_dev->sectSize);
			if (sectBuf == NULL){
				return -ENOMEM;
			}
			error = read_cache(i_dev, sectBuf, 1, sectBeg);
			if (error != NOERR){
				goto ERR;
			}
			back = i_dev->sectSize - forward;
			if (i_bytes < back){
				back = i_bytes;
			}
			memcpy(readBuf, sectBuf + forward, back);
			readBuf += back;
			sectBeg += 1;
			restBytes -= back;
		}

		sectors = restBytes / i_dev->sectSize;
		error = read_cache(i_dev, readBuf, sectors, sectBeg);
		if (error != NOERR){
			goto ERR;
		}
		restBytes -= sectors * i_dev->sectSize;
		sectBeg += sectors;
		readBuf += sectors * i_dev->sectSize;

		/* Ĥ꤬̤ž */
		if (0 < restBytes){
			if (sectBuf == NULL){
				sectBuf = kmalloc(i_dev->sectSize);
				if (sectBuf == NULL){
					return -ENOMEM;
				}
			}
			error = read_cache(i_dev, sectBuf, 1, sectBeg);
			if (error != NOERR){
				goto ERR;
			}
			memcpy(readBuf, sectBuf, restBytes);
			restBytes = 0;
		}

		kfree(sectBuf);
		return i_bytes - restBytes;
ERR:
		kfree(sectBuf);
		return error;
	}
	// root directory
	case S_IFDIR:
		return 0;
	default:
		ASSERT(0);
		return -ENOENT;
	}
}

/*
 * ǥХ˽񤭹
 * return : write bytes or error number
 */
int devWriteDev(
	struct specinfo *i_dev,
	void *i_buf,				// ɤ߹ߥХåե
	const size_t i_bytes,		// ߥХȿ
	const size_t i_offset,		// ߳ϥХȥեå
	void *i_privateData)
{
	char *writeBuf = i_buf;
	size_t ioSize;				// žѥ
	int error;

	switch (i_dev->si_flags & S_IFMT){
	// 饯ǥХ
	case S_IFCHR:
		error = convWriteDev(i_dev, writeBuf, i_bytes, 0, 0, &ioSize);
		if (error != NOERR){
			return error;
		}
		else{
			return ioSize;
		}
	// ֥åǥХ
	case S_IFBLK:{
		char *sectBuf = NULL;
		int sectBeg = i_offset / i_dev->sectSize;	// žϤΥֹ
		size_t restSize = i_bytes;					// ĤžХȿ
		size_t sectors;

		/* ǽΥ̤ž */
		if (0 < (i_offset % i_dev->sectSize)){
			int forward = i_offset % i_dev->sectSize;
			int back;

			sectBuf = kmalloc(i_dev->sectSize);
			if (sectBuf == NULL){
				return -ENOMEM;
			}
			error = read_cache(i_dev, sectBuf, 1, sectBeg);
			if (error != NOERR){
				goto ERR;
			}
			back = i_dev->sectSize - forward;
			if (i_bytes < back){
				back = i_bytes;
			}
			memcpy(sectBuf + forward, writeBuf, back);
			error = write_cache(i_dev, sectBuf, 1, sectBeg);
			if (error != NOERR){
				goto ERR;
			}
			writeBuf += back;
			sectBeg += 1;
			restSize -= back;
		}

		sectors = restSize / i_dev->sectSize;
		error = write_cache(i_dev, writeBuf, sectors, sectBeg);
		if (error != NOERR){
			goto ERR;
		}
		restSize -= sectors * i_dev->sectSize;
		sectBeg += sectors;
		writeBuf += sectors * i_dev->sectSize;

		/* Ĥ꤬̤ž */
		if (0 < restSize){
			if (sectBuf == NULL){
				sectBuf = kmalloc(i_dev->sectSize);
				if (sectBuf == NULL){
					return -ENOMEM;
				}
			}
			error = read_cache(i_dev, sectBuf, 1, sectBeg);
			if (error != NOERR){
				goto ERR;
			}
			memcpy(sectBuf, writeBuf, restSize);
			error = write_cache(i_dev, sectBuf, 1, sectBeg);
			if (error != NOERR){
				goto ERR;
			}
			restSize = 0;
		}

		kfree(sectBuf);
		return i_bytes - restSize;
ERR:
		kfree(sectBuf);
		return error;
	}
	/* root directory */
	case S_IFDIR:
		return 0;
	default:
		ASSERT(0);
		return -ENOENT;
	}
}

/*
 * ǥХIOCTL
 * return : error number
 */
int devIoctlDev(
	struct specinfo *i_dev,
	const int cmd,
	caddr_t param,
	const int fflag,
	void *i_privateData)
{
	return i_dev->si_devsw->d_ioctl(i_dev, cmd, param, fflag, NULL);
}

/*
 * ǥХPOLL
 * return : error number
 */
int devPollDev(
	struct specinfo *i_dev,
	const int events,
	void *i_privateData)
{
	return i_dev->si_devsw->d_poll(i_dev, events, NULL);
}

/*
 * ǥХơȤμ
 * return : error number
 */
void devStatDev(
	struct specinfo *i_dev,
	DEV_STAT *m_devStat)
{
	ASSERT(i_dev != NULL);

	m_devStat->major		= major(i_dev);
	m_devStat->minor		= minor(i_dev);
	m_devStat->prt_type		= i_dev->partType;
	m_devStat->all_sect		= i_dev->sectors;
	m_devStat->sect_size	= i_dev->sectSize;
	m_devStat->flags		= i_dev->si_flags;
	m_devStat->name			= i_dev->si_name;
}

/****************************************************************************/
void test_device()
{
}
