/*
 * proc.c
 *
 * Copyright 2002, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * ץ
 */


#include <sys/config.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/limits.h>
#include <sys/times.h>
#include <sys/wait.h>
#include <sys/AggregateList.h>
#include <machine/interrupt.h>
#include <kern/kmalloc.h>
#include <kern/time.h>
#include <kern/term.h>
#include <kern/Hierarchy.h>
#include <kern/ProcSignal.h>
#include <kern/TaskSignal.h>
#include <kern/Wait.h>
#include <kern/Thread.h>

#include <kern/debug.h>


//#define DEBUG_PROC 1
#ifdef DEBUG_PROC
	#define STATIC
	#define INLINE
#else
	#define STATIC static
	#define INLINE inline
#endif


//=====================================  ===================================================

enum{
	// ץ
	PROC_RUNNING = 1,
	PROC_EXIT = 2,
	
	MAX_CHILD = 8,		/* Max number of child processes */
};

/*
 * ץ¤
 */
struct proc {
	int					state;			// ץξ
	int					childCount;		// Number of child processes
	uint				ctime[2];		// ҥץ׻ system = 0,user = 1
	uchar				exit_state;		// Exit state.
	uchar				signum;			// Exit signal number.

	// ץID
	pid_t				pid;			// process ID.
	pid_t				pgid;			// process groupe ID.
	uid_t				uid;			// user ID.
	gid_t				gid;			// user groupe ID.
	pid_t				sid;			// session ID.

	// ץ
	HierarchyObj		hierarchy;		// ץع¤

	// å
	AggregateList		aggrThread;		// åɥꥹȽ

	ITIMER				itimerVirtual;	// itimer
	ITIMER				itimerProf;		// itimer
	void				*ctlterm;		// ߥʥ湽¤
	VmProcObj			vmProcObj;		// ץ꡼¤
	void				*fileStruct;	// ե빽¤Υɥ쥹
	int					spinLock;		// ԥå
	ProcSignalObj		procSignalObj;	// ʥ¤Ρ
};

typedef struct proc PROC;

//===================================== Х륤ݡ =======================================

extern int registThread(void*, List*);

//===================================== PRIVATE ====================================================

/*
 * ¾ΥץȻΥå
 */
static int spinLock;

/*
 * ɥץ
 */
static PROC *idleProc[MAX_CPU];

/*
 * Initץ
 */
static PROC *initProc;

/*
 * Get new process ID.
 *¡ĹϢ³Ư硢Сեˤʤǽꡣ
 * return : Process ID.
 */
STATIC INLINE int getNewPid()
{
	static int pid = 1;
	return pid++;
}

/*
 * 󥹥ȥ饯
 */
STATIC void procConstructor(
	PROC *this,
	PROC *i_parent)
{
	memset(this, 0, sizeof(*this));

	HierarchyConstructor(&this->hierarchy, this);
	AggregateListConstructor(&this->aggrThread);

	if (i_parent != NULL) {
		this->pid = getNewPid();
		this->pgid = i_parent->pgid;
		this->uid = 0;
		this->gid = i_parent->gid;
		this->sid = i_parent->sid;
	}

}

//--------------------------------------------------------------------------------------------------
// åɴ
//--------------------------------------------------------------------------------------------------

/*
 * åɤ롣
 * return : error number
 */
STATIC int getThread(
	PROC *i_proc,
	const pthread_t i_id,
	pthread_t (*getThreadId)(ThreadObj*),
	ThreadObj **o_thread)
{
	AggregateList *aggrThread = &i_proc->aggrThread;
	IteratorList iterator;
	int error = NOERR;
	int eflags;

	// åɤθ
	aggrThread->iterator(aggrThread, &iterator);
	eflags = enterCli();
	enter_spinlock(&i_proc->spinLock);
	{
		for (;;) {
			ThreadObj *thread;

			if (iterator.hasNext(&iterator) == BOOL_FALSE) {
				error = -ESRCH;
				break;
			}
			thread = iterator.next(&iterator);
			if (getThreadId(thread) == i_id) {
				*o_thread = thread;
				break;
			}
		}
	}
	exit_spinlock(&i_proc->spinLock);
	exitCli(eflags);

	return error;
}

/*
 * ٤ƤΥåɤ˥å
 *Գ
 * return : error number
 */
STATIC int sendTaskMessage(
	PROC *proc,
	void *(*getObj)(void*),
	int (*method)(void*, uint, uint, uint, uint, uint, uint),
	uint param1,
	uint param2,
	uint param3,
	uint param4,
	uint param5,
	uint param6)
{
	AggregateList *aggrThread = &proc->aggrThread;
	IteratorList iterator;
	int eflags;
	int error = NOERR;

	aggrThread->iterator(aggrThread, &iterator);

	eflags = enterCli();
	enter_spinlock(&proc->spinLock);
	{
		while (iterator.hasNext(&iterator) == BOOL_TRUE) {
			void *thread = iterator.next(&iterator);

			error = method(getObj(thread), param1, param2, param3, param4, param5, param6);
		}
	}
	exit_spinlock(&proc->spinLock);
	exitCli(eflags);

	return error;
}

//--------------------------------------------------------------------------------------------------
// ץ
//--------------------------------------------------------------------------------------------------

/*
 * CPU֤
 */
STATIC void getProcTime(
	PROC *i_proc,
	int *o_sysTime,		// ƥॿ
	int *o_userTime)	// 桼
{
	AggregateList *aggregate = &i_proc->aggrThread;
	IteratorList iterator;
	int sysTime = 0;
	int userTime = 0;

	aggregate->iterator(aggregate, &iterator);
	while (iterator.hasNext(&iterator) == BOOL_TRUE) {
		void *thread = iterator.next(&iterator);
		sysTime += calcMiliSecondFromClock(getSysClocks(thread)) / TASK_TIME;
		userTime += calcMiliSecondFromClock(getUserClocks(thread)) / TASK_TIME;
	}
	*o_sysTime = sysTime;
	*o_userTime = userTime;
}

//--------------------------------------------------------------------------------------------------
// ץ
//--------------------------------------------------------------------------------------------------

/*
 * λҥץξ֤𤹤롣
 * ӥåȥޥåװ֤ΰ̣"sys/wait.h"򻲾ȤΤ
 * return : 0 or not exit or stop=-1
 */
STATIC void setState(
	PROC *proc,
	int *stat_loc)		// state address
{
	union{
		int i;
		char c[4];
	} stat;
	
	if (stat_loc == NULL)
		return;

	stat.i = 0;
	stat.c[0] = proc->signum;
	stat.c[1] = proc->exit_state;
	if (proc->state == TASK_SIGNAL_STOP) {
		stat.c[0] = 0x7F;
		stat.c[1] = proc->signum;
	}
	*stat_loc = stat.i;
}

/*
 * EXITҥץ
 *ոƤӽФ¾򤫤뤳
 */
STATIC void releaseChild(
	PROC *this)
{
	HierarchyObj *hierarchy = &this->hierarchy;
	HierarchyObj *childHierarchy;
	HierarchyObj *next;

	for (childHierarchy = HierarchyGetFirstChild(hierarchy); childHierarchy != NULL; childHierarchy = next) {
		next = HierarchyGetNextChild(hierarchy, childHierarchy);
		PROC *child = HierarchyGetProc(childHierarchy);
		if (child->state == PROC_EXIT) {
			// åɤγ
			AggregateList *aggregate = &child->aggrThread;
			void *thread;
			while ((thread = aggregate->refHead(aggregate)) != NULL) {
				if (getTaskState(thread) == TASK_EXIT) {
					freeThread(aggregate->getHead(aggregate), &child->vmProcObj);
				}
				else {
					wait_task();
				}
			}

			// ʥγ
			ProcSignalDestructor(&child->procSignalObj);

			// ץؤ
			HierarchyRemoveChild(hierarchy, childHierarchy);

			kfree(child);
		}
	}
}

/*
 * ҥץinitץϤ
 *ոƤӽФ¾򤫤뤳
 */
STATIC void addChildProc1(
	PROC *this)
{
	HierarchyObj *childHierarchy;
	HierarchyObj *next;

	for (childHierarchy = HierarchyGetFirstChild(&this->hierarchy); childHierarchy != NULL; childHierarchy = next) {
		next = HierarchyGetNextChild(&this->hierarchy, childHierarchy);

		HierarchyRemoveChild(&this->hierarchy, childHierarchy);
		HierarchyAddChild(&initProc->hierarchy, childHierarchy);
	}
}

/*
 * λҥץξ֤𤹤롣
 *ոƤӽФ¾򤫤뤳
 * return : pid or 0 or error number
 */
STATIC int searchExitState(
	PROC *i_parent, 
	pid_t i_pid, 
	pid_t i_pgid, 
	int i_state, 
	int *o_state)
{
	HierarchyObj *hierarchy;
	HierarchyObj *childHierar;
	int rest = -ECHILD;

	if (o_state != NULL){
		*o_state = 0;
	}

	if (i_pid != 0) {
		hierarchy = &i_parent->hierarchy;
		for (childHierar = HierarchyGetFirstChild(hierarchy); childHierar != NULL; childHierar = HierarchyGetNextChild(hierarchy, childHierar)) {
			PROC *child = HierarchyGetProc(childHierar);
			if (child->pid == i_pid) {
				rest = 0;
				if (child->state & i_state) {
					setState(child, o_state);
					rest = i_pid;
				}
				break;
			}
		}
	}
	else if (i_pgid != 0) {
		hierarchy = &i_parent->hierarchy;
		for (childHierar = HierarchyGetFirstChild(hierarchy); childHierar != NULL; childHierar = HierarchyGetNextChild(hierarchy, childHierar)) {
			PROC *child = HierarchyGetProc(childHierar);
			if (child->pgid == i_pgid) {
				rest = 0;
				if (child->state & i_state) {
					setState(child, o_state);
					rest = child->pid;
					break;
				}
			}
		}
	}
	else {
		hierarchy = &i_parent->hierarchy;
		for (childHierar = HierarchyGetFirstChild(hierarchy); childHierar != NULL; childHierar = HierarchyGetNextChild(hierarchy, childHierar)) {
			PROC *child = HierarchyGetProc(childHierar);
			rest = 0;
			if (child->state & i_state) {
				setState(child, o_state);
				rest = child->pid;
				break;
			}
		}
	}

	return rest;
}

/*
 * ץե
 */
STATIC int fork(
	int isKernelThread)		// isKernelThreadˤϥͥ륹åɤȤ"0"ʳ
{
	PROC *this = getCurrentProc();
	PROC *child;
	ThreadObj *newThread;
	AggregateList *parentAggregate;
	int error;

	if (MAX_CHILD <= this->childCount) {
		return -EAGAIN;
	}

	// ҥץƤ
	child = kmalloc(sizeof(PROC));
	if (child == NULL) {
		return -ENOMEM;
	}
	procConstructor(child, this);

	// ץؤϿ
	HierarchyAddChild(&this->hierarchy, &child->hierarchy);

	// å󥢥ɥ쥹(ü¤Υɥ쥹)򥳥ԡ
	child->ctlterm = this->ctlterm;

	// եط¤Τγ
	child->fileStruct = copyFileStruct(getFileStructProc(this));
	if (getFileStructProc(child) == NULL) {
		error = -ENOMEM;
		goto ERR;
	}

	/* ʥꤹ롣 */
	ProcSignalCopyConstructor(&child->procSignalObj, &this->procSignalObj);

	/* ޡν */
	itimerConstructor(&child->itimerVirtual);
	itimerConstructor(&child->itimerProf);

	// Ƥβۥץ¤Τ򥳥ԡ
	vmProcFork(&child->vmProcObj, &this->vmProcObj);

	// åɤκ
	parentAggregate = &this->aggrThread;
	error = forkThread(
		isKernelThread, 
		child, 
		parentAggregate->refEnd(parentAggregate), 
		&child->vmProcObj, 
		&newThread);
	if (error == 1) {
		// ҥץ¹Գ
		child->state = PROC_RUNNING;
		return 0;
	}
	else if (error != NOERR) {
		goto ERR;
	}

	// åɤץϿ
	registThread(child, getProcList(newThread));

	// 塼˲ä
	addNewTask(newThread);

	++this->childCount;

	return child->pid;
ERR:
	kfree(child);

	return error;
}

//--------------------------------------------------------------------------------------------------
// ʥ
//--------------------------------------------------------------------------------------------------

/*
 * Ʊץ롼פ˥ʥ
 *ոƤӽФ¾򤫤뤳
 */
STATIC void sendSignalPgid(
	PROC *i_parent,
	const pid_t i_pgid,		// PGID 0ʤ餹٤Ƥ˥ʥ
	const int i_signal)
{
	HierarchyObj *hierarchy = &i_parent->hierarchy;
	HierarchyObj *childHierar;

	for (childHierar = HierarchyGetFirstChild(hierarchy); childHierar != NULL; childHierar = HierarchyGetNextChild(hierarchy, childHierar)) {
		PROC *child = HierarchyGetProc(childHierar);
		if ((i_pgid == 0) || (child->pgid == i_pgid)) {
			procSendSignal(child, i_signal);
		}

		// ƵƤӽФ
		sendSignalPgid(child, i_pgid, i_signal);
	}
}

//--------------------------------------------------------------------------------------------------
// 
//--------------------------------------------------------------------------------------------------

/*
 * PIDץ
 *ոƤӽФ¾򤫤뤳
 * return : PROC or NULL
 */
STATIC PROC *searchProcPid(
	PROC *i_parent,
	const pid_t i_pid)
{
	HierarchyObj *hierarchy = &i_parent->hierarchy;
	HierarchyObj *childHierar;
	PROC *child = NULL;

	for (childHierar = HierarchyGetFirstChild(hierarchy); childHierar != NULL; childHierar = HierarchyGetNextChild(hierarchy, childHierar)) {
		child = HierarchyGetProc(childHierar);
		if (child->pid == i_pid) {
			break;
		}
		
		// ƵƤӽФ
		child = searchProcPid(child, i_pid);
		if (child != NULL) {
			break;
		}
	}

	return child;
}

//================================== PUBLIC =============================================

//--------------------------------------------------------------------------------------------------
// ᥽å
//--------------------------------------------------------------------------------------------------

/*
 * Process group ID 
 */
pid_t getPgid(
	const void *i_proc)
{
	return ((PROC*) i_proc)->pgid;
}

/*
 * User ID 
 */
uid_t getUid(
	const void *i_proc)
{
	return ((PROC*) i_proc)->uid;
}

/*
 * User group ID 
 */
gid_t getGid(
	const void *i_proc)
{
	return ((PROC*) i_proc)->gid;
}

/*
 * Session ID 
 */
pid_t getSid(
	const void *i_proc)
{
	return ((PROC*) i_proc)->sid;
}

/*
 * ꡼¤Υɥ쥹
 */
VmProcObj *getVmProc(
	void *i_proc)
{
	return &((PROC*) i_proc)->vmProcObj;
}

/*
 * ߥʥ湽¤Υɥ쥹
 */
void *getCtltermProc(
	const void *i_proc)
{
	return ((PROC*) i_proc)->ctlterm;
}

/*
 * ʥ¤Υɥ쥹
 */
ProcSignalObj *procGetProcSignal(
	const void *i_proc)
{
	return &((PROC*) i_proc)->procSignalObj;
}

/*
 * ե¤Υɥ쥹
 */
void *getFileStructProc(
	const void *i_proc)
{
	return ((PROC*) i_proc)->fileStruct;
}

/*
 * Exit signal number ꤹ
 */
void procSetExitSignum(
	void *m_proc,
	const int i_signum)
{
	((PROC*) m_proc)->signum = i_signum;
}

/*
 * ե¤Υɥ쥹ꤹ
 */
void setFileStructProc(
	void *m_proc,
	void *i_fileStruct)
{
	((PROC*) m_proc)->fileStruct = i_fileStruct;
}

/*
 * ߥʥ湽¤Υɥ쥹ꤹ
 */
void setCtltermProc(
	void *m_proc,
	void *ctlterm)
{
	((PROC*) m_proc)->ctlterm = ctlterm;
}

/*
 * PIDץ
 * return : PROC
 */
void *procGetProcPid(
	const pid_t i_pid)
{
	PROC *proc;
	int eflag;

	eflag = enterCli();
	enter_spinlock(&spinLock);
	{
		proc = searchProcPid(initProc, i_pid);
	}
	exit_spinlock(&spinLock);
	exitCli(eflag);
	
	return proc;
}

/*
 * ߼¹Υץμ
 */
void *getCurrentProc()
{
	return ThreadGetProc(getCurrentTask());
}

//--------------------------------------------------------------------------------------------------
// ץ󥿡Х륿ޡ
//--------------------------------------------------------------------------------------------------

/*
 * ȥץΥץitimer
 * return : ITIMER
 */
ITIMER *getItimerVirtual()
{
	return &((PROC*) getCurrentProc())->itimerVirtual;
}

/*
 * ȥץΥƥitimer
 * return : ITIMER
 */
ITIMER *getItimerProc()
{
	return &((PROC*) getCurrentProc())->itimerProf;
}

//--------------------------------------------------------------------------------------------------
// ץ
//--------------------------------------------------------------------------------------------------

/*
 * ץλ
 *Գ
 */
void exit(
	int state,		// ƥץϤ
	int signum)
{
	static int exitSpinLock = 0;
	int eflag;

	eflag = enterCli();
	enter_spinlock(&exitSpinLock);
	{
		PROC *this = getCurrentProc();

		if (this->state != PROC_EXIT) {
			// åɤ˽λʥ
			sendTaskMessage(this, (void*) TaskGetTaskSignal, (void*) sendSignal, SIGKILL, 0, 0, 0, 0, 0);

			// üץγ
			releaseTerm(this);

			// եǥץγ
			releaseFileStruct(getFileStructProc(this));

			this->exit_state = state & 0xff;
			this->signum = signum & 0xff;
			
			enter_spinlock(&spinLock);
			{
				PROC *parent;

				// ҥץγ
				releaseChild(this);

				// ƤҥץINITץλҤɲä
				addChildProc1(this);

				parent = HierarchyGetParent(&this->hierarchy);
				--parent->childCount;

				// CPUƥץ­
				getProcTime(this, &parent->ctime[0], &parent->ctime[1]);

				procSendSignal(parent, SIGCHLD);
				activeTaskSignal(getWaitTask(getMainThread(parent)));
			}
			exit_spinlock(&spinLock);

			this->state = PROC_EXIT;
		}
	}
	exit_spinlock(&exitSpinLock);
	exitCli(eflag);
}

/*
 * ɥץʺǽΥץˤ
 * returns : error number
 */
int createIdleProc(
	const int cpu)
{
	ThreadObj *newThread;
	int error;

	idleProc[cpu] = kmalloc(sizeof(PROC));
	if (idleProc[cpu] == NULL) {
		return -ENOMEM;
	}
	procConstructor(idleProc[cpu], NULL);

	error = initFileStruct(idleProc[cpu]);
	if (error != NOERR) {
		return error;
	}

	// init signal.
	ProcSignalConstructor(&idleProc[cpu]->procSignalObj);

	// ۥץ¤Τν
	initVmProc(&idleProc[cpu]->vmProcObj);

	// åɤκ
	error = createIdleThread(cpu, idleProc[cpu], &newThread);
	if (error != NOERR) {
		return error;
	}

	// åɤץϿ
	registThread(idleProc[cpu], getProcList(newThread));

	/* add to schedule queue */
	setFirstTask(cpu, newThread);

	idleProc[cpu]->state = PROC_RUNNING;

	return NOERR;
}

/*
 * ǽΥץꡣ
 * return : error number
 */
void setInitProc()
{
	initProc = getCurrentProc();
}

/*
 * initProcwait
 */
void waitProc1()
{
	ProcSignalSetMask(SIGCHLD);

	for(;;) {
		int eflag;
		waitTaskSignal();

		eflag = enterCli();
		enter_spinlock(&spinLock);
		{
			releaseChild(initProc);
		}
		exit_spinlock(&spinLock);
		exitCli(eflag);
	}
}

/*
 * ͥ륹åɤfork
 */
int forkKernelThread()
{
	return fork(1);
}

//--------------------------------------------------------------------------------------------------
// åɴ
//--------------------------------------------------------------------------------------------------

/*
 * åɤϿ
 * return : error number
 */
int registThread(
	void *m_proc,
	List *m_threadList)
{
	PROC *proc = (PROC*) m_proc;
	AggregateList *aggregate = &proc->aggrThread;
	int error = NOERR;

	enter_spinlock(&proc->spinLock);
	{
		if (aggregate->getCount(aggregate) < PTHREAD_THREADS_MAX) {
			aggregate->insertHead(aggregate, m_threadList);
		}
		else {
			error = -EAGAIN;
		}
	}
	exit_spinlock(&proc->spinLock);
	
	return error;
}

/*
 * åɤ롣
 * return : error number
 */
int procGetThread(
	const pthread_t i_id,
	ThreadObj **o_thread)
{
	return getThread(getCurrentProc(), i_id, threadGetId, o_thread);
}

/*
 * λԤåɤ롣
 * return : error number
 */
int procGetWaitThread(
	const pthread_t i_id,
	ThreadObj **o_thread)
{
	return getThread(getCurrentProc(), i_id, threadGetJoinId, o_thread);
}

/*
 * ᥤ󥹥åɤ
 * return : thread
 */
ThreadObj *getMainThread(
	const void *i_proc)
{
	AggregateList *aggregate = &((PROC*) i_proc)->aggrThread;
	return aggregate->refEnd(aggregate);
}

//--------------------------------------------------------------------------------------------------
// ʥ
//--------------------------------------------------------------------------------------------------

/*
 * Ʊץ롼פ˥ʥ
 */
void procSendSignalPgid(
	const pid_t i_pgid,
	const int i_signal)
{
	int eflag;
	eflag = enterCli();
	enter_spinlock(&spinLock);
	{
		sendSignalPgid(initProc, i_pgid, i_signal);
	}
	exit_spinlock(&spinLock);
	exitCli(eflag);
}

/*
 * ץ˥ʥ
 */
void sendSignalAllProc(
	const int i_signal)
{
	int eflag;
	eflag = enterCli();
	enter_spinlock(&spinLock);
	{
		sendSignalPgid(initProc, 0, i_signal);
	}
	exit_spinlock(&spinLock);
	exitCli(eflag);
}

/*
 * ٤ƤΥåɤ˥ʥ
 *Գ
 * return : error number
 */
int procSendSignal(
	void *i_this,
	const int i_signal)
{
	return sendTaskMessage(i_this, (void*) TaskGetTaskSignal, (void*) sendSignal, i_signal, 0, 0, 0, 0, 0);
}

/*
 * ٤ƤΥåɤ˥ʥ
 *Գ
 */
void procForceSendSignal(
	void *i_this,
	const int i_signal)
{
	sendTaskMessage(i_this, (void*) TaskGetTaskSignal, (void*) forceSendSignal, i_signal, 0, 0, 0, 0, 0);
}

/*
 * åɤΥʥޥ
 */
sigset_t *procGetSignalMask(
	void *i_this)
{
	// Ȥꤢᥤ󥹥åɤΥʥޥ֤
	void *thread = getMainThread(i_this);
	return TaskSignalGetMask(TaskGetTaskSignal(thread));
}

/*
 * ٤ƤΥåɤ˥ʥޥꤹ
 */
void procSetSignalMask(
	void *i_this,
	const sigset_t *i_mask)
{
	sendTaskMessage(i_this, (void*) TaskGetTaskSignal, (void*) TaskSignalSetMask, (int) i_mask, 0, 0, 0, 0, 0);
}

/*
 * åɤΥʥڥǥ󥰤
 */
sigset_t *procGetSignalPending(
	void *i_this)
{
	// Ȥꤢᥤ󥹥åɤΥʥڥǥ󥰤֤
	void *thread = getMainThread(i_this);
	return TaskSignalGetPending(TaskGetTaskSignal(thread));
}

/*
 * ٤ƤΥåɤ˥ʥڥǥ󥰤ꤹ
 */
void procSetSignalPending(
	void *i_this,
	const sigset_t *i_pending)
{
	sendTaskMessage(i_this, (void*) TaskGetTaskSignal, (void*) TaskSignalSetPending, (int) i_pending, 0, 0, 0, 0, 0);
}

/*
 * ٤ƤΥåɤ˥ʥϥɥѲۥ꡼ꤹ
 */
int procCreatUserHandlerStack(
	void *i_this)
{
	return sendTaskMessage(i_this, (void*) getVmTask, (void*) vmCreatSignalHandlerStack, 0, 0, 0, 0, 0, 0);
}

/*
 * ץinitץλҥץˤ
 */
void setParentToInit()
{
	PROC *this = getCurrentProc();
	PROC *parent;
	int eflag;

	eflag = enterCli();
	enter_spinlock(&spinLock);
	{
		parent = HierarchyGetParent(&this->hierarchy);
		HierarchyRemoveChild(&parent->hierarchy, &this->hierarchy);
		HierarchyAddChild(&initProc->hierarchy, &this->hierarchy);
	}
	exit_spinlock(&spinLock);
	exitCli(eflag);
}

/*
 * Хå饦ɤǼ¹ԤƤ뤫
 *ԸƤդ⤷̤λҥץԤä顩
 * return : YES or NO
 */
/*int isBackgroundProc()
{
	PROC *this = getCurrentProc();
	PROC *parent = HierarchyGetParent(&this->hierarchy);

	ASSERT(parent != NULL);

	return  (getTaskState(getMainThread(parent)) & TASK_SIGNAL_WAIT) ? NO : YES;
}*/

//--------------------------------------------------------------------------------------------------
// ƥॳ
//--------------------------------------------------------------------------------------------------

/*
 * return : ƥץʤҥץID,ҥץʤ0,error=error number
 * todoꥹ
 *  ǥ쥯ȥꥹȥ꡼򥳥ԡ롣
 */
int sys_fork()
{
	return fork(0);
}

int sys_exit(int state)
{
	exit(state,0);
	taskExit();
	// äƤʤ

	return 0;
}

int sys_wait(int pid, int *stat_loc, int options)
{
	PROC *this = getCurrentProc();
	int state;
	int eflag;
	int rest;

	// Ĵ٤ץ֤
	state = PROC_EXIT;

	eflag = enterCli();
	enter_spinlock(&spinLock);
	{
		if (0 < pid) {
			rest = searchExitState(this, pid, 0, state, stat_loc);
		}
		else if (pid == 0) {
			rest = searchExitState(this, 0, this->pgid, state, stat_loc);
		}
		else if (pid == -1) {
			rest = searchExitState(this, 0, 0, state, stat_loc);
		}
		else {
			rest = searchExitState(this, 0, -pid, state, stat_loc);
		}
		releaseChild(this);
	}
	exit_spinlock(&spinLock);
	exitCli(eflag);

	if (0 < rest) {
		return rest;
	}

	if ((rest < 0) && (options & WNOHANG)) {
		return rest;
	}

	waitTaskSignal();

	eflag = enterCli();
	enter_spinlock(&spinLock);
	{
		if (0 < pid) {
			rest = searchExitState(this, pid, 0, state, stat_loc);
		}
		else if (pid == 0) {
			rest = searchExitState(this, 0, this->pgid, state, stat_loc);
		}
		else if (pid == -1) {
			rest = searchExitState(this, 0, 0, state, stat_loc);
		}
		else {
			rest = searchExitState(this, 0, -pid, state, stat_loc);
		}
		releaseChild(this);
	}
	exit_spinlock(&spinLock);
	exitCli(eflag);

	return rest;
}

int sys_getpid()
{
	PROC *proc = getCurrentProc();
	return proc->pid;
}

int sys_getgid()
{
	PROC *proc = getCurrentProc();
	return proc->gid;
}

int sys_getpgid(
	pid_t pid)
{
	if (pid == 0) {
		PROC *proc = getCurrentProc();
		return proc->pgid;
	}
	else {
		PROC *proc;
		PROC *this = getCurrentProc();
		int eflag;

		eflag = enterCli();
		enter_spinlock(&spinLock);
		{
			proc = searchProcPid(initProc, pid);
		}
		exit_spinlock(&spinLock);
		exitCli(eflag);
		
		if (proc == NULL) {
			return -ESRCH;
		}
		if (proc->sid != this->sid) {
			return -EPERM;
		}
		return proc->pgid;
	}
}

int sys_getppid()
{
	PROC *this = getCurrentProc();
	PROC *parent = HierarchyGetParent(&this->hierarchy);
	return parent->pid;
}

int sys_getuid()
{
	PROC *proc = getCurrentProc();
	return proc->uid;
}

int sys_getsid(pid_t pid)
{
	if (pid == 0) {
		PROC *proc = getCurrentProc();
		return proc->sid;
	}
	else {
		PROC *proc;
		PROC *this = getCurrentProc();
		int eflag;

		eflag = enterCli();
		enter_spinlock(&spinLock);
		{
			proc = searchProcPid(initProc, pid);
		}
		exit_spinlock(&spinLock);
		exitCli(eflag);
		
		if (proc == NULL) {
			return -ESRCH;
		}
		if (proc->sid != this->sid) {
			return -EPERM;
		}
		return proc->sid;
	}
}

int sys_setgid(gid_t gid)
{
	PROC *proc = getCurrentProc();

	if (proc->uid != 0) {
		return -EPERM;
	}
	proc->gid = gid;
	return 0;
}

int sys_setpgid(pid_t pid, pid_t pgid)
{
	PROC *this;
	PROC *proc;
	PROC *pgproc;
	int eflag;

	if (pid == 0) {
		proc = getCurrentProc();
	}
	else {
		PROC *parent;

		eflag = enterCli();
		enter_spinlock(&spinLock);
		{
			proc = searchProcPid(initProc, pid);
			if (proc == NULL) {
				exit_spinlock(&spinLock);
				exitCli(eflag);
				return -ESRCH;
			}
			parent = HierarchyGetParent(&proc->hierarchy);
		}
		exit_spinlock(&spinLock);
		exitCli(eflag);

		if ((proc != getCurrentProc()) && (parent != getCurrentProc())) {
			return -ESRCH;
		}
	}

	if (pgid < 0) {
		return -EINVAL;
	}
	if (pgid == 0) {
		pgid = proc->pid;
	}
	
	/* Session leader */
	if (proc->sid == proc->pid) {
		return -EPERM;
	}

	this = getCurrentProc();
	if (proc->sid != this->sid) {
		return -EPERM;
	}
	
	/* ץۤʤ륻åΥץ롼פ˰ư褦Ȥ */
	eflag = enterCli();
	enter_spinlock(&spinLock);
	{
		pgproc = searchProcPid(initProc, pgid);
	}
	exit_spinlock(&spinLock);
	exitCli(eflag);
	if (pgproc == NULL) {
		return -EPERM;
	}
	if (pgproc->sid != proc->sid) {
		return -EPERM;
	}

	proc->pgid = pgid;

	return 0;
}

int sys_setuid(
	uid_t uid)
{
	PROC *proc = getCurrentProc();
	if (proc->uid != 0) {
		return -EPERM;
	}
	proc->uid = uid;

	return 0;
}

int sys_setsid()
{
	PROC *proc = getCurrentProc();
	
	if (proc->pgid != proc->pid) {
		proc->sid = proc->pid;
		proc->pgid = proc->pid;
		proc->ctlterm = NULL;
		return proc->sid;
	}
	else {
		return -EPERM;
	}
}

int sys_times(struct tms *tms)
{
	PROC *proc;

	if (checkMem(tms, sizeof(struct tms)) == ERR) {
		return -EFAULT;
	}

	proc = getCurrentProc();
	getProcTime(proc, &tms->tms_stime, &tms->tms_utime);
	tms->tms_cstime = proc->ctime[0];
	tms->tms_cutime = proc->ctime[1];

	return 0;
}
