/*
 * Copyright (c) 1997, Stefan Esser <se@freebsd.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice unmodified, this list of conditions, and the following
 *    disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $FreeBSD: src/sys/pci/pci.c,v 1.141.2.15 2002/04/30 17:48:18 tmm Exp $
 *
 */


#define FREEBSD_DRIVER


#include <sys/param.h>
#include <sys/systm.h>
#include <sys/malloc.h>
#include <sys/module.h>
#include <sys/fcntl.h>
#include <sys/conf.h>
#include <sys/kernel.h>
#include <sys/queue.h>
#include <sys/types.h>
#include <sys/buf.h>
#include <sys/bus.h>
#include <sys/queue.h>
#include <sys/pciio.h>
#include <sys/rman.h>
#include <sys/errno.h>
#include <machine/pcibus.h>
#include <machine/DevInterrupt.h>
#include <pci/pcireg.h>
#include <pci/pcivar.h>
#include <sys/config.h>
#include <kern/device.h>
#include <machine/mp.h>
#include <kern/debug.h>


//#define DEBUG_PCI 1
#ifdef DEBUG_PCI
	#define STATIC
	#define INLINE
#else
	#define STATIC	static
	#define INLINE	inline
#endif

//#define PCI_DEBUG


/****************************************************************************
*Уãɥ᥽å
*****************************************************************************/


struct pci_devinfo {
	struct { 
		struct pci_devinfo *stqe_next; 
	} pci_links;
	struct resource_list resources;
	pcicfgregs			cfg;
	struct pci_conf		conf;
};

struct pci_quirk {
	u_int32_t devid;			/* Vendor/device of the card */
	int	type;
#define PCI_QUIRK_MAP_REG	1 /* PCI map register in wierd place */
	int	arg1;
	int	arg2;
};


//================================== PRIVATE ============================================


struct pci_quirk pci_quirks[] = {
	/*
	 * The Intel 82371AB and 82443MX has a map register at offset 0x90.
	 */
	{ 0x71138086, PCI_QUIRK_MAP_REG,	0x90,	 0 },
	{ 0x719b8086, PCI_QUIRK_MAP_REG,	0x90,	 0 },

	{ 0 }
};

static STAILQ_HEAD(devlist, pci_devinfo) pci_devq;
static u_int32_t pci_numdevs = 0;
static u_int32_t pci_generation = 0;


/*
 * Some convenience functions for PCI device drivers.
 */
STATIC INLINE void pci_set_command_bit(device_t dev, device_t child, u_int16_t bit)
{
	u_int16_t	command;

	command = PCI_READ_CONFIG(dev, child, PCIR_COMMAND, 2);
	command |= bit;
	PCI_WRITE_CONFIG(dev, child, PCIR_COMMAND, command, 2);
}

STATIC INLINE void pci_clear_command_bit(device_t dev, device_t child, u_int16_t bit)
{
	u_int16_t	command;

	command = PCI_READ_CONFIG(dev, child, PCIR_COMMAND, 2);
	command &= ~bit;
	PCI_WRITE_CONFIG(dev, child, PCIR_COMMAND, command, 2);
}


/* adjust some values from PCI 1.0 devices to match 2.0 standards ... */
STATIC void pci_fixancient(pcicfgregs *cfg)
{
	if (cfg->hdrtype != 0){
		return;
	}

	// PCI to PCI bridges use header type 1
	if (cfg->baseclass == PCIC_BRIDGE && cfg->subclass == PCIS_BRIDGE_PCI){
		cfg->hdrtype = 1;
	}
}


/* read config data specific to header type 1 device (PCI to PCI bridge) */
STATIC void *pci_readppb(pcicfgregs *cfg)
{
	pcih1cfgregs *p;

	p = malloc(sizeof (pcih1cfgregs), M_DEVBUF, M_WAITOK);
	if (p == NULL){
		return (NULL);
	}

	bzero(p, sizeof *p);

	p->secstat = pci_cfgread(cfg, PCIR_SECSTAT_1, 2);
	p->bridgectl = pci_cfgread(cfg, PCIR_BRIDGECTL_1, 2);
	p->seclat = pci_cfgread(cfg, PCIR_SECLAT_1, 1);
	p->iobase = PCI_PPBIOBASE (pci_cfgread(cfg, PCIR_IOBASEH_1, 2), pci_cfgread(cfg, PCIR_IOBASEL_1, 1));
	p->iolimit = PCI_PPBIOLIMIT (pci_cfgread(cfg, PCIR_IOLIMITH_1, 2), pci_cfgread(cfg, PCIR_IOLIMITL_1, 1));
	p->membase = PCI_PPBMEMBASE (0, pci_cfgread(cfg, PCIR_MEMBASE_1, 2));
	p->memlimit = PCI_PPBMEMLIMIT (0, pci_cfgread(cfg, PCIR_MEMLIMIT_1, 2));
	p->pmembase = PCI_PPBMEMBASE ((pci_addr_t)pci_cfgread(cfg, PCIR_PMBASEH_1, 4), pci_cfgread(cfg, PCIR_PMBASEL_1, 2));
	p->pmemlimit = PCI_PPBMEMLIMIT ((pci_addr_t)pci_cfgread(cfg, PCIR_PMLIMITH_1, 4), pci_cfgread(cfg, PCIR_PMLIMITL_1, 2));

	return (p);
}


// read config data specific to header type 2 device (PCI to CardBus bridge)
// return : pcih2cfgregs or NULL
STATIC void *pci_readpcb(pcicfgregs *cfg)
{
	pcih2cfgregs *p;

	p = malloc(sizeof (pcih2cfgregs), M_DEVBUF, M_WAITOK);
	if (p == NULL){
		return (NULL);
	}
	bzero(p, sizeof *p);

	p->secstat		= pci_cfgread(cfg, PCIR_SECSTAT_2, 2);
	p->bridgectl	= pci_cfgread(cfg, PCIR_BRIDGECTL_2, 2);
	p->seclat		= pci_cfgread(cfg, PCIR_SECLAT_2, 1);
	p->membase0		= pci_cfgread(cfg, PCIR_MEMBASE0_2, 4);
	p->memlimit0	= pci_cfgread(cfg, PCIR_MEMLIMIT0_2, 4);
	p->membase1		= pci_cfgread(cfg, PCIR_MEMBASE1_2, 4);
	p->memlimit1	= pci_cfgread(cfg, PCIR_MEMLIMIT1_2, 4);
	p->iobase0		= pci_cfgread(cfg, PCIR_IOBASE0_2, 4);
	p->iolimit0		= pci_cfgread(cfg, PCIR_IOLIMIT0_2, 4);
	p->iobase1		= pci_cfgread(cfg, PCIR_IOBASE1_2, 4);
	p->iolimit1		= pci_cfgread(cfg, PCIR_IOLIMIT1_2, 4);
	p->pccardif		= pci_cfgread(cfg, PCIR_PCCARDIF_2, 4);

	return p;
}


/* extract header type specific config data */
STATIC void pci_hdrtypedata(pcicfgregs *cfg)
{
	switch (cfg->hdrtype) {
		case 0:
			cfg->subvendor		= pci_cfgread(cfg, PCIR_SUBVEND_0, 2);
			cfg->subdevice		= pci_cfgread(cfg, PCIR_SUBDEV_0, 2);
			cfg->nummaps		= PCI_MAXMAPS_0;
			break;
		case 1:
			cfg->subvendor		= pci_cfgread(cfg, PCIR_SUBVEND_1, 2);
			cfg->subdevice		= pci_cfgread(cfg, PCIR_SUBDEV_1, 2);
			cfg->secondarybus	= pci_cfgread(cfg, PCIR_SECBUS_1, 1);
			cfg->subordinatebus	= pci_cfgread(cfg, PCIR_SUBBUS_1, 1);
			cfg->nummaps		= PCI_MAXMAPS_1;
			cfg->hdrspec		= pci_readppb(cfg);
			break;
		case 2:
			cfg->subvendor		= pci_cfgread(cfg, PCIR_SUBVEND_2, 2);
			cfg->subdevice		= pci_cfgread(cfg, PCIR_SUBDEV_2, 2);
			cfg->secondarybus	= pci_cfgread(cfg, PCIR_SECBUS_2, 1);
			cfg->subordinatebus	= pci_cfgread(cfg, PCIR_SUBBUS_2, 1);
			cfg->nummaps		= PCI_MAXMAPS_2;
			cfg->hdrspec		= pci_readpcb(cfg);
			break;
	}
}


STATIC void pci_read_extcap(pcicfgregs *cfg)
{
	int	ptr, nextptr, ptrptr;

	switch (cfg->hdrtype) {
		case 0:
			ptrptr = 0x34;
			break;
		case 2:
			ptrptr = 0x14;
			break;
		default:
			return;		// no extended capabilities support
	}
	nextptr = pci_cfgread(cfg, ptrptr, 1);	// sanity check?

	// Read capability entries.
	while (nextptr != 0) {
		// Sanity check
		if (nextptr > 255) {
			printf("illegal PCI extended capability offset %d\n", nextptr);
			return;
		}
		// Find the next entry
		ptr = nextptr;
		nextptr = pci_cfgread(cfg, ptr + 1, 1);

		// Process this entry
		switch (pci_cfgread(cfg, ptr, 1)) {
		case 0x01:		// PCI power management
			if (cfg->pp_cap == 0) {
				cfg->pp_cap = pci_cfgread(cfg, ptr + PCIR_POWER_CAP, 2);
				cfg->pp_status = ptr + PCIR_POWER_STATUS;
				cfg->pp_pmcsr = ptr + PCIR_POWER_PMCSR;
				if (PCIR_POWER_DATA < (nextptr - ptr)){
					cfg->pp_data = ptr + PCIR_POWER_DATA;
				}
			}
			break;
		default:
			break;
		}
	}
}


//================================== PROTECTED ==========================================


STATIC u_int32_t pci_read_config_method(device_t dev, device_t child, int reg, int width)
{
	struct pci_devinfo *dinfo = device_get_ivars(child);

	pcicfgregs *cfg = &dinfo->cfg;
	return pci_cfgread(cfg, reg, width);
}


STATIC void pci_write_config_method(device_t dev, device_t child, int reg, u_int32_t val, int width)
{
	struct pci_devinfo *dinfo = device_get_ivars(child);

	pcicfgregs *cfg = &dinfo->cfg;
	pci_cfgwrite(cfg, reg, val, width);
}


STATIC void pci_enable_busmaster_method(device_t dev, device_t child)
{
	pci_set_command_bit(dev, child, PCIM_CMD_BUSMASTEREN);
}


STATIC void pci_disable_busmaster_method(device_t dev, device_t child)
{
	pci_clear_command_bit(dev, child, PCIM_CMD_BUSMASTEREN);
}


STATIC void pci_enable_io_method(device_t dev, device_t child, int space)
{
	switch(space) {
	case SYS_RES_IOPORT:
		pci_set_command_bit(dev, child, PCIM_CMD_PORTEN);
		break;
	case SYS_RES_MEMORY:
		pci_set_command_bit(dev, child, PCIM_CMD_MEMEN);
		break;
	}
}


STATIC void pci_disable_io_method(device_t dev, device_t child, int space)
{
	switch(space) {
	case SYS_RES_IOPORT:
		pci_clear_command_bit(dev, child, PCIM_CMD_PORTEN);
		break;
	case SYS_RES_MEMORY:
		pci_clear_command_bit(dev, child, PCIM_CMD_MEMEN);
		break;
	}
}


// read configuration header into pcicfgrect structure
// return : pci_devinfo¤ or NULL = 顼
STATIC struct pci_devinfo *pci_readcfg(
	pcicfgregs *probe)				// ե幽¤
{
	struct pci_devinfo *devlist_entry = NULL;
	struct devlist *devlist_head = &pci_devq;

	if (pci_cfgread(probe, PCIR_DEVVENDOR, 4) != -1) {
		pcicfgregs *cfg;
		devlist_entry = malloc(sizeof(struct pci_devinfo), M_DEVBUF, M_WAITOK);
		if (devlist_entry == NULL){
			return (NULL);
		}
		bzero(devlist_entry, sizeof(*devlist_entry));
		cfg = &devlist_entry->cfg;

		cfg->hose		= probe->hose;
		cfg->bus		= probe->bus;
		cfg->slot		= probe->slot;
		cfg->func		= probe->func;
		cfg->vendor		= pci_cfgread(cfg, PCIR_VENDOR, 2);
		cfg->device		= pci_cfgread(cfg, PCIR_DEVICE, 2);
		cfg->cmdreg		= pci_cfgread(cfg, PCIR_COMMAND, 2);
		cfg->statreg	= pci_cfgread(cfg, PCIR_STATUS, 2);
		cfg->baseclass	= pci_cfgread(cfg, PCIR_CLASS, 1);
		cfg->subclass	= pci_cfgread(cfg, PCIR_SUBCLASS, 1);
		cfg->progif		= pci_cfgread(cfg, PCIR_PROGIF, 1);
		cfg->revid		= pci_cfgread(cfg, PCIR_REVID, 1);
		cfg->hdrtype	= pci_cfgread(cfg, PCIR_HEADERTYPE, 1);
		cfg->cachelnsz	= pci_cfgread(cfg, PCIR_CACHELNSZ, 1);
		cfg->lattimer	= pci_cfgread(cfg, PCIR_LATTIMER, 1);
		cfg->intpin		= pci_cfgread(cfg, PCIR_INTPIN, 1);
		cfg->intline	= pci_cfgread(cfg, PCIR_INTLINE, 1);
		cfg->mingnt		= pci_cfgread(cfg, PCIR_MINGNT, 1);
		cfg->maxlat		= pci_cfgread(cfg, PCIR_MAXLAT, 1);
		cfg->mfdev		= (cfg->hdrtype & PCIM_MFDEV) != 0;
		cfg->hdrtype	&= ~PCIM_MFDEV;
		pci_fixancient(cfg);
		pci_hdrtypedata(cfg);

		if (pci_cfgread(probe, PCIR_STATUS, 2) & PCIM_STATUS_CAPPRESENT){
			pci_read_extcap(cfg);
		}

		// pci_devq˿ȥ꡼³
		STAILQ_INSERT_TAIL(devlist_head, devlist_entry, pci_links);

		devlist_entry->conf.pc_sel.pc_bus	= cfg->bus;
		devlist_entry->conf.pc_sel.pc_dev	= cfg->slot;
		devlist_entry->conf.pc_sel.pc_func	= cfg->func;
		devlist_entry->conf.pc_hdr			= cfg->hdrtype;
		devlist_entry->conf.pc_subvendor	= cfg->subvendor;
		devlist_entry->conf.pc_subdevice	= cfg->subdevice;
		devlist_entry->conf.pc_vendor		= cfg->vendor;
		devlist_entry->conf.pc_device		= cfg->device;
		devlist_entry->conf.pc_class		= cfg->baseclass;
		devlist_entry->conf.pc_subclass		= cfg->subclass;
		devlist_entry->conf.pc_progif		= cfg->progif;
		devlist_entry->conf.pc_revid		= cfg->revid;
		pci_numdevs++;
		pci_generation++;
	}

	return (devlist_entry);
}


/*
 * New style pci driver.  Parent device is either a pci-host-bridge or a
 * pci-pci-bridge.  Both kinds are represented by instances of pcib.
 */
STATIC void pci_print_verbose(struct pci_devinfo *dinfo)
{
	if (bootverbose) {
		pcicfgregs *cfg = &dinfo->cfg;

		printf("found->\tvendor=0x%04x, dev=0x%04x, revid=0x%02x\n", 
			cfg->vendor, 
			cfg->device, 
			cfg->revid);
		printf("\tclass=%02x-%02x-%02x, hdrtype=0x%02x, mfdev=%d\n", 
			cfg->baseclass, 
			cfg->subclass, 
			cfg->progif, 
			cfg->hdrtype, 
			cfg->mfdev);
		printf("\tsubordinatebus=%x \tsecondarybus=%x\n", 
			cfg->subordinatebus, 
			cfg->secondarybus);
#ifdef PCI_DEBUG
		printf("\tcmdreg=0x%04x, statreg=0x%04x, cachelnsz=%d (dwords)\n", 
			cfg->cmdreg, 
			cfg->statreg, 
			cfg->cachelnsz);
		printf("\tlattimer=0x%02x (%d ns), mingnt=0x%02x (%d ns), maxlat=0x%02x (%d ns)\n",
		       cfg->lattimer, 
		       cfg->lattimer * 30, 
		       cfg->mingnt, 
		       cfg->mingnt * 250, 
		       cfg->maxlat, 
		       cfg->maxlat * 250);
#endif // PCI_DEBUG
		if (cfg->intpin > 0){
			printf("\tintpin=%c, irq=%d\n", 
				cfg->intpin +'a' -1, 
				cfg->intline);
		}
	}
}


/****************************************************************************
*Х᥽å
*****************************************************************************/


//================================== PRIVATE ============================================


STATIC int pci_print_resources(struct resource_list *rl, const char *name, int type, const char *format)
{
	struct resource_list_entry *rle;
	int printed, retval;

	printed = 0;
	retval = 0;
	/* Yes, this is kinda cheating */
	SLIST_FOREACH(rle, rl, link) {
		if (rle->type == type) {
			if (printed == 0)
				retval += printf(" %s ", name);
			else if (printed > 0)
				retval += printf(",");
			printed++;
			retval += printf(format, rle->start);
			if (rle->count > 1) {
				retval += printf("-");
				retval += printf(format, rle->start + rle->count - 1);
			}
		}
	}
	return retval;
}


// ֹ
// return : struct resource or NULL
STATIC struct resource *pci_alloc_resource(
	device_t dev,			// ƥǥХ
	device_t child, 
	int type,				// ꥽
	int *rid,
	u_long start, 
	u_long end, 
	u_long count, 
	u_int flags)
{
	struct pci_devinfo *dinfo = device_get_ivars(child);
	struct resource_list *rl = &dinfo->resources;
	struct resource_list_entry *rle;
	struct resource *res;
	pcicfgregs *cfg = &dinfo->cfg;

	/*
	 * Perform lazy resource allocation
	 *
	 * XXX add support here for SYS_RES_IOPORT and SYS_RES_MEMORY
	 */
	if (device_get_parent(child) == dev) {
		/*
		 * If device doesn't have an interrupt routed, and is
		 * deserving of an interrupt, try to assign it one.
		 */
		if ((type == SYS_RES_IRQ) && 
			(cfg->intline == 255 || cfg->intline == 0) && 
			(cfg->intpin != 0) && 
			(start == 0) && 
			(end == ~0UL)) {
			cfg->intline = pci_cfgintr(pci_get_bus(child), pci_get_slot(child), cfg->intpin);
			if (cfg->intline != 255) {
				pci_write_config(child, PCIR_INTLINE, cfg->intline, 1);
				resource_list_add(rl, SYS_RES_IRQ, 0, cfg->intline, cfg->intline, 1);
			}
		}
	}

	rle = resource_list_find(rl, type, *rid);
	if (rle == NULL){
		return NULL;
	}
	res = malloc(sizeof(*res), M_RMAN, M_NOWAIT);
	if (res == NULL){
		return NULL;
	}
	memset(res, 0, sizeof(*res));
	res->r_start = rle->start;
	res->r_end = rle->end;
	res->r_flags = flags;

	switch (type){
	case SYS_RES_IOPORT:
		rman_set_bustag(res, I386_BUS_SPACE_IO);
		rman_set_bushandle(res, res->r_start);
		break;
	case SYS_RES_MEMORY:
		rman_set_bustag(res, I386_BUS_SPACE_MEM);
		rman_set_bushandle(res, res->r_start);
		break;
	case SYS_RES_IRQ:
		break;
	case SYS_RES_DRQ:
		break;
	default:
		printf("type = %d\n", type);
		ASSERT(0);
	}

	rle->res = res;

	return rle->res;
}


STATIC int pci_release_resource(
	device_t dev, 
	device_t child, 
	int type, 
	int rid,
	struct resource *r)
{
	struct pci_devinfo *dinfo = device_get_ivars(child);
	struct resource_list *rl = &dinfo->resources;

	return resource_list_release(rl, dev, child, type, rid, r);
}


/* return base address of memory or port map */
STATIC u_int32_t pci_mapbase(unsigned mapreg)
{
	int mask = 0x03;
	if ((mapreg & 0x01) == 0){
		mask = 0x0f;
	}
	return (mapreg & ~mask);
}


/* return map type of memory or port map */
STATIC int pci_maptype(unsigned mapreg)
{
	static u_int8_t maptype[0x10] = {
		PCI_MAPMEM,					PCI_MAPPORT,
		PCI_MAPMEM,					0,
		PCI_MAPMEM,					PCI_MAPPORT,
		0,							0,
		PCI_MAPMEM | PCI_MAPMEMP,	PCI_MAPPORT,
		PCI_MAPMEM | PCI_MAPMEMP,	0,
		PCI_MAPMEM | PCI_MAPMEMP,	PCI_MAPPORT,
		0,							0,
	};

	return maptype[mapreg & 0x0f];
}


/* return log2 of map size decoded for memory or port map */
STATIC int pci_mapsize(unsigned testval)
{
	int ln2size;

	testval = pci_mapbase(testval);
	ln2size = 0;
	if (testval != 0) {
		while ((testval & 1) == 0)
		{
			ln2size++;
			testval >>= 1;
		}
	}
	return (ln2size);
}


/* return log2 of address range supported by map register */
STATIC int pci_maprange(unsigned mapreg)
{
	int ln2range = 0;
	switch (mapreg & 0x07) {
	case 0x00:
	case 0x01:
	case 0x05:
		ln2range = 32;
		break;
	case 0x02:
		ln2range = 20;
		break;
	case 0x04:
		ln2range = 64;
		break;
	}
	return (ln2range);
}


STATIC int pci_porten(pcicfgregs *cfg)
{
	return ((cfg->cmdreg & PCIM_CMD_PORTEN) != 0);
}


STATIC int pci_memen(pcicfgregs *cfg)
{
	return ((cfg->cmdreg & PCIM_CMD_MEMEN) != 0);
}


/*
 * Add a resource based on a pci map register. Return 1 if the map
 * register is a 32bit map register or 2 if it is a 64bit register.
 */
STATIC int pci_add_map(device_t dev, pcicfgregs* cfg, int reg)
{
	struct pci_devinfo *dinfo = device_get_ivars(dev);
	struct resource_list *rl = &dinfo->resources;
	u_int32_t map;
	u_int64_t base;
	u_int8_t ln2size;
	u_int8_t ln2range;
	u_int32_t testval;
	int type;

	map = pci_cfgread(cfg, reg, 4);

	if (map == 0 || map == 0xffffffff){
		return 1; /* skip invalid entry */
	}

	pci_cfgwrite(cfg, reg, 0xffffffff, 4);
	testval = pci_cfgread(cfg, reg, 4);
	pci_cfgwrite(cfg, reg, map, 4);

	base = pci_mapbase(map);

	if (pci_maptype(map) & PCI_MAPMEM){
		type = SYS_RES_MEMORY;

		// ޥåץIO١ɥ쥹γǧ
		ASSERT(IOMAP_BEG <= base);
	}
	else{
		type = SYS_RES_IOPORT;
	}
	ln2size = pci_mapsize(testval);
	ln2range = pci_maprange(testval);
	if (ln2range == 64) {
		/* Read the other half of a 64bit map register */
		base |= (u_int64_t) pci_cfgread(cfg, reg + 4, 4) << 32;
	}

	/*
	 * This code theoretically does the right thing, but has
	 * undesirable side effects in some cases where
	 * peripherals respond oddly to having these bits
	 * enabled.  Leave them alone by default.
	 */
#ifdef PCI_ENABLE_IO_MODES
	if (type == SYS_RES_IOPORT && !pci_porten(cfg)) {
		cfg->cmdreg |= PCIM_CMD_PORTEN;
		pci_cfgwrite(cfg, PCIR_COMMAND, cfg->cmdreg, 2);
	}
	if (type == SYS_RES_MEMORY && !pci_memen(cfg)) {
		cfg->cmdreg |= PCIM_CMD_MEMEN;
		pci_cfgwrite(cfg, PCIR_COMMAND, cfg->cmdreg, 2);
	}
#else
	if (type == SYS_RES_IOPORT && !pci_porten(cfg)){
		return 1;
	}
	if (type == SYS_RES_MEMORY && !pci_memen(cfg)){
		return 1;
	}
#endif

	resource_list_add(rl, type, reg, base, base + (1 << ln2size) - 1, (1 << ln2size));

	if (bootverbose) {
		printf("\tmap[%02x]: type %x, range %2d, base %08x, size %2d\n", reg, pci_maptype(base), ln2range, (unsigned int) base, ln2size);
	}

	return (ln2range == 64) ? 2 : 1;
}


//================================== PROTECTED ==========================================


STATIC void pci_add_resources(device_t dev, pcicfgregs* cfg)
{
	struct pci_devinfo *dinfo = device_get_ivars(dev);
	struct resource_list *rl = &dinfo->resources;
	struct pci_quirk *q;
	int i;

	for (i = 0; i < cfg->nummaps;) {
		i += pci_add_map(dev, cfg, PCIR_MAPS + i * 4);
	}

	for (q = &pci_quirks[0]; q->devid; q++) {
		if (q->devid == ((cfg->device << 16) | cfg->vendor) && q->type == PCI_QUIRK_MAP_REG){
			pci_add_map(dev, cfg, q->arg1);
		}
	}

	if (cfg->intpin > 0 && cfg->intline != 255){
		resource_list_add(rl, SYS_RES_IRQ, 0, cfg->intline, cfg->intline, 1);
	}
}


STATIC int pci_print_child(device_t dev, device_t child)
{
	struct pci_devinfo *dinfo;
	struct resource_list *rl;
	pcicfgregs *cfg;
	int retval = 0;

	dinfo = device_get_ivars(child);
	cfg = &dinfo->cfg;
	rl = &dinfo->resources;

	retval += bus_print_child_header(dev, child);

	retval += pci_print_resources(rl, "port", SYS_RES_IOPORT, "%#lx");
	retval += pci_print_resources(rl, "mem", SYS_RES_MEMORY, "%#lx");
	retval += pci_print_resources(rl, "irq", SYS_RES_IRQ, "%ld");
	if (device_get_flags(dev))
		retval += printf(" flags %#x", device_get_flags(dev));

	retval += printf(" at device %d.%d", pci_get_slot(child), pci_get_function(child));

	retval += bus_print_child_footer(dev, child);

	return (retval);
}


STATIC void pci_probe_nomatch(device_t dev, device_t child)
{
	struct pci_devinfo *dinfo;
	pcicfgregs *cfg;
	const char *desc;
	int unknown;

	unknown = 0;
	dinfo = device_get_ivars(child);
	cfg = &dinfo->cfg;
	desc = pci_ata_match(child);
	if (!desc) desc = pci_usb_match(child);
	if (!desc) desc = pci_vga_match(child);
	if (!desc) {
		desc = "unknown card";
		unknown++;
	}
	device_printf(dev, "<%s>", desc);
	if (bootverbose || unknown) {
		printf(" (vendor=0x%04x, dev=0x%04x)", cfg->vendor, cfg->device);
	}
	printf(" at %d.%d", pci_get_slot(child), pci_get_function(child));
	if (cfg->intpin > 0 && cfg->intline != 255) {
		printf(" irq %d", cfg->intline);
	}
	printf("\n");

	return;
}


STATIC int pci_read_ivar(device_t dev, device_t child, int which, u_long *result)
{
	struct pci_devinfo *dinfo;
	pcicfgregs *cfg;

	dinfo = device_get_ivars(child);
	cfg = &dinfo->cfg;

	switch (which) {
		case PCI_IVAR_SUBVENDOR:
			*result = cfg->subvendor;
			break;
		case PCI_IVAR_SUBDEVICE:
			*result = cfg->subdevice;
			break;
		case PCI_IVAR_VENDOR:
			*result = cfg->vendor;
			break;
		case PCI_IVAR_DEVICE:
			*result = cfg->device;
			break;
		case PCI_IVAR_DEVID:
			*result = (cfg->device << 16) | cfg->vendor;
			break;
		case PCI_IVAR_CLASS:
			*result = cfg->baseclass;
			break;
		case PCI_IVAR_SUBCLASS:
			*result = cfg->subclass;
			break;
		case PCI_IVAR_PROGIF:
			*result = cfg->progif;
			break;
		case PCI_IVAR_REVID:
			*result = cfg->revid;
			break;
		case PCI_IVAR_INTPIN:
			*result = cfg->intpin;
			break;
		case PCI_IVAR_IRQ:
			*result = cfg->intline;
			break;
		case PCI_IVAR_BUS:
			*result = cfg->bus;
			break;
		case PCI_IVAR_SLOT:
			*result = cfg->slot;
			break;
		case PCI_IVAR_FUNCTION:
			*result = cfg->func;
			break;
		case PCI_IVAR_SECONDARYBUS:
			*result = cfg->secondarybus;
			break;
		case PCI_IVAR_SUBORDINATEBUS:
			*result = cfg->subordinatebus;
			break;
		case PCI_IVAR_HOSE:
			/*
			 * Pass up to parent bridge.
			 */
			ASSERT(0);
//			*result = pcib_get_hose(dev);
			break;
		default:
			return ENOENT;
	}

	return 0;
}


STATIC int pci_write_ivar(device_t dev, device_t child, int which, uintptr_t value)
{
	struct pci_devinfo *dinfo;
	pcicfgregs *cfg;

	dinfo = device_get_ivars(child);
	cfg = &dinfo->cfg;

	switch (which) {
		case PCI_IVAR_SUBVENDOR:
		case PCI_IVAR_SUBDEVICE:
		case PCI_IVAR_VENDOR:
		case PCI_IVAR_DEVICE:
		case PCI_IVAR_DEVID:
		case PCI_IVAR_CLASS:
		case PCI_IVAR_SUBCLASS:
		case PCI_IVAR_PROGIF:
		case PCI_IVAR_REVID:
		case PCI_IVAR_INTPIN:
		case PCI_IVAR_IRQ:
		case PCI_IVAR_BUS:
		case PCI_IVAR_SLOT:
		case PCI_IVAR_FUNCTION:
			return EINVAL;	/* disallow for now */
		case PCI_IVAR_SECONDARYBUS:
			cfg->secondarybus = value;
			break;
		case PCI_IVAR_SUBORDINATEBUS:
			cfg->subordinatebus = value;
			break;
		default:
			return ENOENT;
	}
	return 0;
}


STATIC int pci_setup_intr(
	device_t parent,
	device_t dev,
	struct resource *irq,
	int intrFlag,
	void (*handler)(void *),
	void *softc,
	void *opt)
{
	setupIntr(irq, handler, softc);

	if (isMp() == YES){
		// IOAPIC
		struct pci_devinfo *dinfo = device_get_ivars(dev);
		pcicfgregs *cfg = &dinfo->cfg;
		setPciIrqInIoapic(cfg->bus, cfg->slot, cfg->intpin, irq->r_start);
	}
	else{
		release_irq_mask(irq->r_start);
	}

	return NOERR;
}


/****************************************************************************
*ǥХ᥽å
*****************************************************************************/


//================================== PRIVATE ============================================


STATIC void pci_add_children(
	device_t dev, 
	int busno)
{
	pcicfgregs probe;

	bzero(&probe, sizeof probe);
	probe.hose = 0;
	probe.bus = busno;

	for (probe.slot = 0; probe.slot <= PCI_SLOTMAX; probe.slot++) {
		int pcifunchigh = 0;
		for (probe.func = 0; probe.func <= pcifunchigh; probe.func++) {
			struct pci_devinfo *dinfo = pci_readcfg(&probe);
			if (dinfo != NULL) {
				if (dinfo->cfg.mfdev){
					pcifunchigh = 7;
				}
				pci_print_verbose(dinfo);
				dinfo->cfg.dev = device_add_child(dev, NULL, -1);
				device_set_ivars(dinfo->cfg.dev, dinfo);
				pci_add_resources(dinfo->cfg.dev, &dinfo->cfg);
			}
		}
	}
}


STATIC int pci_new_probe(device_t dev)
{
//	static int once;

//	device_set_desc(dev, "PCI bus");
	pci_add_children(dev, device_get_unit(dev));
//	if (!once) {
		// ǥХեϿϤʤ
//		make_dev(&pcicdev, 0, UID_ROOT, GID_WHEEL, 0644, "pci");
//		once++;
//	}

	return 0;
}


/****************************************************************************
*
*****************************************************************************/


//================================== PRIVATE ============================================


struct device pciDevice;

static device_method_t pci_methods[] = {
	/* Device interface */
	DEVMETHOD(device_probe,			pci_new_probe),
	DEVMETHOD(device_attach,		bus_generic_attach),
	DEVMETHOD(device_shutdown,		bus_generic_shutdown),
	DEVMETHOD(device_suspend,		bus_generic_suspend),
	DEVMETHOD(device_resume,		bus_generic_resume),

	/* Bus interface */
	DEVMETHOD(bus_print_child,		pci_print_child),
	DEVMETHOD(bus_probe_nomatch,	pci_probe_nomatch),
	DEVMETHOD(bus_read_ivar,		pci_read_ivar),
	DEVMETHOD(bus_write_ivar,		pci_write_ivar),
	DEVMETHOD(bus_driver_added,		bus_generic_driver_added),
	DEVMETHOD(bus_alloc_resource,	pci_alloc_resource),
	DEVMETHOD(bus_release_resource,	pci_release_resource),
	DEVMETHOD(bus_setup_intr,		pci_setup_intr),

	/* PCI interface */
	DEVMETHOD(pci_read_config,		pci_read_config_method),
	DEVMETHOD(pci_write_config,		pci_write_config_method),
	DEVMETHOD(pci_enable_busmaster,	pci_enable_busmaster_method),
	DEVMETHOD(pci_disable_busmaster, pci_disable_busmaster_method),
	DEVMETHOD(pci_enable_io,		pci_enable_io_method),
	DEVMETHOD(pci_disable_io,		pci_disable_io_method),

	{ 0, 0 }
};
static driver_t pci_driver = {
	"pci",
	pci_methods,
	1,			/* no softc */
};


//================================== PUBLIC =============================================


// return : error number
int initPciNew()
{
	device_t device;
	const char *busName;
	int busnum;
	int found = 0;
	int error;

	// PCIǥХθ
	while (pci_identify(&busName, &busnum) == TRUE){
		// ХǥХκ
		device = make_bus_device("pci", busnum);
		if (device == NULL){
			printf("make_bus_device(): Failed make device!\n");
			continue;
		}
		device->ops = pci_driver.ops;
		device->driver = &pci_driver;
		device_set_desc(device, busName);
		device->state = DS_ATTACHED;

		// ǥХΥå
		error = pci_new_probe(device);
		if (error == NOERR){
			found++;
		}
		else{
			printf("device_probe_and_attach(): Failed probe and attach! error=%d \n", error);
		}
	}
	if (found == 0){
		printf("pci_identify(): no PCI bridge found!\n");
		return -ENXIO;
	}

	STAILQ_INIT(&pci_devq);

	return NOERR;
}
