/*-
 * Copyright (c) 1997,1998 Doug Rabson
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $FreeBSD: src/sys/sys/bus.h,v 1.30.2.5 2004/03/17 17:54:25 njl Exp $
 */


#ifndef _SYS_BUS_H_
#define _SYS_BUS_H_


#include <sys/module.h>


typedef int (*devop_t)(void);
typedef struct device			*device_t;
typedef struct device_ops		*device_ops_t;
typedef struct driver			driver_t;
typedef struct devclass			*devclass_t;
typedef enum device_state {
    DS_NOTPRESENT,		/* not probed or probe failed */
    DS_ALIVE,			/* probe succeeded */
    DS_ATTACHED,		/* attach method called */
    DS_BUSY				/* device is open */
} device_state_t;


#include <sys/queue.h>
#include <sys/bus_private.h>
#include <sys/rman.h>
#include <machine/bus.h>
#include <machine/syscall.h>
#include <kern/dev_method.h>


struct device;

typedef struct device_method	device_method_t;
typedef void driver_intr_t(void*);


// ǥХɥ饤Сѹ¤
struct driver {
    const char		*name;		/* driver name */
    device_method_t	*methods;	/* method table */
    size_t			softc;		/* size of device softc struct */
    void			*priv;		/* driver private data */
    device_ops_t	ops;		/* compiled method table */
    int				refs;		/* # devclasses containing driver */
};

struct device_method {
    int		index;
    devop_t	func;
};

struct resource_list_entry {
    struct { 
    	struct resource_list_entry *sle_next; 
    } link;
    int				type;		/* type argument to alloc_resource */
    int				rid;		/* resource identifier */
    struct resource	*res;		/* the real resource when allocated */
    u_long			start;		/* start of resource range */
    u_long			end;		/* end of resource range */
    u_long			count;		/* count within range */
};
SLIST_HEAD(resource_list, resource_list_entry);

struct driver_module_data {
	int			(*dmd_chainevh)(struct module *, int, void *);
	void		*dmd_chainarg;
	const char	*dmd_busname;
	driver_t	**dmd_drivers;
	int			dmd_ndrivers;
	devclass_t	*dmd_devclass;
};


/*
 * We define this in terms of bits because some devices may belong
 * to multiple classes (and therefore need to be included in
 * multiple interrupt masks, which is what this really serves to
 * indicate.  Buses which do interrupt remapping will want to
 * change their type to reflect what sort of devices are underneath.
 */
enum intr_type {
    INTR_TYPE_TTY	= 1,
    INTR_TYPE_BIO	= 2,
    INTR_TYPE_NET	= 4,
    INTR_TYPE_CAM	= 8,
    INTR_TYPE_MISC	= 16,
    INTR_TYPE_FAST	= 128
};

#define DEVMETHOD(NAME, FUNC) { NAME##_desc, (devop_t) FUNC }

#define DRIVER_MODULE(name, busname, driver, devclass, evh, arg)							\
static driver_t *name##_##busname##_driver_list[] = { &driver };							\
static struct driver_module_data name##_##busname##_driver_mod = {							\
	evh, 																					\
	arg,																					\
	#busname,																				\
	name##_##busname##_driver_list,															\
	(sizeof name##_##busname##_driver_list) / (sizeof name##_##busname##_driver_list[0]),	\
	&devclass																				\
};																							\
int name##_##busname##InitDevice()															\
{																							\
	return initDev(&name##_##busname##_driver_mod);											\
}

extern struct			resource *bus_alloc_resource(device_t dev, int type, int *rid, u_long start, u_long end, u_long count, u_int flags);
extern int				resource_query_string(int i, const char *resname, const char *value);
extern char	*			resource_query_name(int i);
extern int				resource_query_unit(int i);
extern int				resource_int_value(const char *name, int unit, const char *resname, int *result);
extern void				resource_list_add(struct resource_list *rl, int type, int rid, u_long start, u_long end, u_long count);
extern int				resource_list_release(struct resource_list *, device_t, device_t, int, int, struct resource *);
extern struct resource_list_entry *	resource_list_find(struct resource_list *rl, int type, int rid);
extern int				bus_deactivate_resource(device_t dev, int type, int rid, struct resource *r);
extern int				bus_release_resource(device_t dev, int type, int rid, struct resource *r);
extern u_long			bus_get_resource_count(device_t dev, int type, int rid);
extern int				bus_set_resource(device_t dev, int type, int rid, u_long start, u_long count);
extern int				bus_setup_intr(device_t dev, struct resource *r, int flags, driver_intr_t handler, void *arg, void **cookiep);
extern int				bus_teardown_intr(device_t dev, struct resource *r, void *cookie);
extern int				bus_print_child_header(device_t dev, device_t child);
extern int				bus_print_child_footer(device_t dev, device_t child);
extern int				bus_generic_attach(device_t dev);
extern int				bus_generic_detach(device_t dev);
extern int				bus_generic_shutdown(device_t dev);
extern int				bus_generic_suspend(device_t dev);
extern int				bus_generic_resume(device_t dev);
extern int				bus_generic_print_child(device_t dev, device_t child);
extern void				bus_generic_driver_added(device_t dev, driver_t *driver);
extern void	*			devclass_get_softc(devclass_t dc, int unit);
extern device_t			devclass_get_device(devclass_t dc, int unit);
extern int				device_is_enabled(device_t dev);
extern device_state_t	device_get_state(device_t dev);
extern device_t			device_get_parent(device_t dev);
extern void *			device_get_softc(device_t dev);
extern u_int32_t		device_get_flags(device_t dev);
extern int				device_detach(device_t dev);
extern int				device_get_unit(device_t dev);
extern device_t			device_add_child(device_t dev, const char *name, int unit);
extern device_t			device_add_child_ordered(device_t dev, int order, const char *name, int unit);
extern int				device_delete_child(device_t dev, device_t child);
extern int				device_printf(device_t dev, const char *, ...);
extern void				device_quiet(device_t dev);
extern const char *		device_get_desc(device_t dev);
extern void				device_set_desc(device_t dev, const char* desc);
extern void				device_set_desc_copy(device_t dev, const char* desc);
extern void	*			device_get_ivars(device_t dev);
extern void				device_set_ivars(device_t dev, void *ivars);
extern int				device_set_driver(device_t dev, driver_t *driver);
extern int				device_set_devclass(device_t dev, const char *classname);
extern void				device_busy(device_t dev);
extern void				device_unbusy(device_t dev);
extern void				device_enable(device_t dev);
extern void				device_disable(device_t dev);
extern const char *		device_get_name(device_t dev);
extern const char *		device_get_nameunit(device_t dev);
extern int				device_is_alive(device_t dev);		/* did probe succeed? */
extern int				device_probe_and_attach(device_t dev);
extern device_t			make_bus_device(const char *, int);
extern int				initDev(struct driver_module_data *);


#endif /* !_SYS_BUS_H_ */
