/*-
 * Copyright (c) 1991, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * 2004: modified by minoru murashma.
 */


#include <sys/types.h>
#include <errno.h>
#include <math.h>
#include "mathimpl.h"


static const ushort msign=0x7fff;
static const ushort mexp =0x7ff0;
static const short  prep1=54;
static const short  gap=4;
static const short  bias=1023;
static const double novf=1.7E308;
static const double nunf=3.0E-308;
static const double zero=0.0, one=1.0, two=2.0, negone= -1.0, half=1.0/2.0;


static double copysign(double x,double y)
{
	unsigned short  *px=(unsigned short *) &x,
                    *py=(unsigned short *) &y;

	*px = ( *px & msign ) | ( *py & ~msign );
	return(x);
}


static double logb(double x)
{
	short *px=(short *) &x, k;


	if( (k= *px & mexp ) != mexp )
	{
		if ( k != 0 )return ( (k>>gap) - bias );
		else if( x != zero)return ( -1022.0 );
		else return(-(1.0/zero)); 
	}
	else if(x != x)return(x);
	else
	{
		*px &= msign;
		return(x);
	}
}


static int finite(double x)
{
	return( (*((short *) &x ) & mexp ) != mexp );
}


static double drem(double x,double p)
{
	short sign;
	double hp,dp,tmp;
	ushort  k;
	unsigned short *px=(unsigned short *) &x  ,
	               *pp=(unsigned short *) &p  ,
	               *pd=(unsigned short *) &dp ,
	               *pt=(unsigned short *) &tmp;


	*pp &= msign ;

	if( ( *px & mexp ) == mexp )return  (x-p)-(x-p);	/* create nan if x is inf */
	if (p == zero)return zero/zero;

	if( ( *pp & mexp ) == mexp )
	{
		if (p != p) return p; else return x;
	}
	else  if ( ((*pp & mexp)>>gap) <= 1 ) 		/* subnormal p, or almost subnormal p */
	{
		double b;

		b=scalb(1.0,(int)prep1);
		p *= b;
		x = drem(x,p);
		x *= b;
		return(drem(x,p)/b);
	}
	else  if ( p >= novf/2)
	{
		p /= 2 ;
		x /= 2;
		return(drem(x,p)*2);
	}
	else
	{
		dp=p+p;
		hp=p/2;
		sign= *px & ~msign ;
		*px &= msign;
		while ( x > dp )
		{
			k=(*px & mexp) - (*pd & mexp) ;
			tmp = dp ;
			*pt += k ;
			if( x < tmp ) *pt -= 16 ;
			x -= tmp ;
		}
		if ( x > hp )
		{
			x -= p ;
			if ( x >= hp ) x -= p ;
		}

		*px ^= sign;

		return( x);
	}
}


double scalb(double x, int N)
{
	int k;
	ushort *px=(unsigned short *) &x;


	if( x == zero ) return(x);

	/* defined(vax)||defined(tahoe) */
	if( (k= *px & mexp ) != mexp )
	{
		if( N<-2100) return(nunf*nunf);
		else if(N>2100) return(novf+novf);

		if( k == 0 )
		{
			x *= scalb(1.0,(int)prep1);
			N -= prep1;
			return(scalb(x,N));
		}

		if((k = (k>>gap)+ N) > 0 )
		{
			if( k < (mexp>>gap) ) *px = (*px&~mexp) | (k<<gap);
			else x=novf+novf;               /* overflow */
		}
		else
		{
			if( k > -prep1 ) 		/* gradual underflow */
			{
				*px=(*px&~mexp)|(short)(1<<gap);
				x *= scalb(1.0,k-1);
			}
			else return(nunf*nunf);
		}
	}

	return(x);
}


vc(p1, 1.5150724356786683059E-2 ,3abe,3d78,066a,67e1,  -6, .F83ABE67E1066A)
vc(p2, 6.3112487873718332688E-5 ,5b42,3984,0173,48cd, -13, .845B4248CD0173)
vc(p3,     6.6137563214379341918E-5  ,b355,398a,f15f,792e, -13, .8AB355792EF15F)
vc(p4,    -1.6533902205465250480E-6  ,ea0e,b6dd,5f84,2e93, -19,-.DDEA0E2E935F84)
vc(p5,     4.1381367970572387085E-8  ,bb4b,3431,2683,95f5, -24, .B1BB4B95F52683)
vc(q1, 1.1363478204690669916E-1 ,b95a,3ee8,ec45,44a2,  -3, .E8B95A44A2EC45)
vc(q2, 1.2624568129896839182E-3 ,7905,3ba5,f5e7,72e4,  -9, .A5790572E4F5E7)
vc(q3, 1.5021856115869022674E-6 ,9eb4,36c9,c395,604a, -19, .C99EB4604AC395)
vc(ln2hi,  6.9314718055829871446E-1  ,7217,4031,0000,f7d0,   0, .B17217F7D00000)
vc(ln2lo,  1.6465949582897081279E-12 ,bcd5,2ce7,d9cc,e4f1, -39, .E7BCD5E4F1D9CC)
vc(lnhuge, 9.4961163736712506989E1   ,ec1d,43bd,9010,a73e,   7, .BDEC1DA73E9010)
vc(invln2, 1.4426950408889634148E0   ,aa3b,40b8,17f1,295c,   1, .B8AA3B295C17F1)
vc(lntiny,-9.5654310917272452386E1   ,4f01,c3bf,33af,d72e,   7,-.BF4F01D72E33AF)

#ifdef vccast
#define       p1    vccast(p1)
#define       p2    vccast(p2)
#define       p3    vccast(p3)
#define       p4    vccast(p4)
#define       p5    vccast(p5)
#define       q1    vccast(q1)
#define       q2    vccast(q2)
#define       q3    vccast(q3)
#define    ln2hi    vccast(ln2hi)
#define    ln2lo    vccast(ln2lo)
#define   lnhuge    vccast(lnhuge)
#define   lntiny    vccast(lntiny)
#define   invln2    vccast(invln2)
#endif

ic(p1, 1.3887401997267371720E-2,  -7, 1.C70FF8B3CC2CF)
ic(p2, 3.3044019718331897649E-5, -15, 1.15317DF4526C4)
ic(p3,     6.6137563214379343612E-5, -14,  1.1566AAF25DE2C)
ic(p4,    -1.6533902205465251539E-6, -20, -1.BBD41C5D26BF1)
ic(p5,     4.1381367970572384604E-8, -25,  1.6376972BEA4D0)
ic(q1, 1.1110813732786649355E-1,  -4, 1.C719538248597)
ic(q2, 9.9176615021572857300E-4, -10, 1.03FC4CB8C98E8)
ic(ln2hi,  6.9314718036912381649E-1,   -1, 1.62E42FEE00000)
ic(ln2lo,  1.9082149292705877000E-10, -33, 1.A39EF35793C76)
ic(lnhuge, 7.1602103751842355450E2,     9, 1.6602B15B7ECF2)
ic(invln2, 1.4426950408889633870E0,     0, 1.71547652B82FE)
ic(lntiny,-7.5137154372698068983E2,    9, -1.77AF8EBEAE354)


double __exp__D(double x,double c)
{
	double  z,hi,lo;
	int k;


	if (x!=x) return(x);	/* x is NaN */

	if ( x <= lnhuge )
	{
		if ( x >= lntiny )
		{
		    /* argument reduction : x --> x - k*ln2 */
			z = invln2*x;
			k = z + copysign(.5, x);

		    /* express (x+c)-k*ln2 as hi-lo and let x=hi-lo rounded */
			hi=(x-k*ln2hi);			/* Exact. */
			x= hi - (lo = k*ln2lo-c);
			z=x*x;
			c= x - z*(p1+z*(p2+z*(p3+z*(p4+z*p5))));
			c = (x*c)/(2.0-c);

			return  scalb(1.+(hi-(lo - c)), k);
		}
		else
		{
		     if(finite(x))return(scalb(1.0,-5000));		/* exp(-big#) underflows to zero */
		     else return(0.0);		/* exp(-INF) is zero */
		}
	}
	else
		/* exp(INF) is INF, exp(+big#) overflows to INF */
	    return( finite(x) ?  scalb(1.0,5000)  : x);
}


double __exp__E(double x,double c)
{
	const static double small=1.0E-19;
	double z,p,q,xp,xh,w;


	if(copysign(x,one)>small)
	{
		z = x*x;
		p = z*( p1 +z* p2 );
		q = z*( q1 +z*  q2 );
		xp= x*p;
		xh= x*half;
		w = xh-(q-xp);
		p = p+p;
		c += x*((xh*w-(q-(p+xp)))/(one-w)+c);
		return(z*half+c);
	}
	else
	{
/*		if(x!=zero) one+small;*/	/* raise the inexact flag */
		return(copysign(zero,x));
	}
}


static double expm1(double x)
{
	static int prec=53;

	double  z,hi,lo,c;
	int k;


	if(x!=x) return(x);	/* x is NaN */

	if( x <= lnhuge )
	{
		if( x >= -40.0 )
		{
			/* argument reduction : x - k*ln2 */
			k= invln2 *x+copysign(0.5,x);	/* k=NINT(x/ln2) */
			hi=x-k*ln2hi ;
			z=hi-(lo=k*ln2lo);
			c=(hi-z)-lo;

			if(k==0) return(z+__exp__E(z,c));
			if(k==1)
			{
			    if(z< -0.25)
			    {
					x=z+half;
					x +=__exp__E(z,c);
					return(x+x);
				}
			    else
				{
					z+=__exp__E(z,c);
					x=half+z;
					return(x+x);
				}
			}
			else
			{
			    if(k<=prec)
			      { x=one-scalb(one,-k); z += __exp__E(z,c);}
			    else if(k<100)
			      { x = __exp__E(z,c)-scalb(one,-k); x+=z; z=one;}
			    else
			      { x = __exp__E(z,c)+z; z=one;}

			    return (scalb(x+z,k));
			}
		}
		else
		{
		     /* expm1(-big#) rounded to -1 (inexact) */
		     if(finite(x))
			 {
/*			 	ln2hi+ln2lo;*/
			 	return(-one);
			 }
			 else return(-one);		/* expm1(-INF) is -1 */
		}
	}
	else
	/*  expm1(INF) is INF, expm1(+big#) overflows to INF */
	    return( finite(x) ?  scalb(one,5000) : x);
}


double frexp(double value,int *eptr)
{
	union {
		double v;
		struct{
			unsigned u_mant2 : 32;
			unsigned u_mant1 : 20;
			unsigned   u_exp : 11;
			unsigned  u_sign :  1;
		}s;
	} u;


	if (value)
	{
		u.v = value;
		*eptr = u.s.u_exp - 1022;
		u.s.u_exp = 1022;

		return(u.v);
	}
	else
	{
		*eptr = 0;

		return((double)0);
	}
}


double ldexp (double value, int exp)
{
	double temp, texp, temp2;


	texp = exp;
#ifdef __GNUC__
#if    __GNUC__ >= 2
	asm ("fscale "
		: "=u" (temp2), "=t" (temp)
		: "0" (texp), "1" (value));
#else
	asm ("fscale ; fxch %%st(1) ; fstp%L1 %1 "
		: "=f" (temp), "=0" (temp2)
		: "0" (texp), "f" (value));
#endif
#else
error unknown asm
#endif

	return (temp);
}


#define _IEEE		1
#define endian		(((*(int *) &one)) ? 1 : 0)
#define TRUNC(x) 	*(((int *) &x)+endian) &= 0xf8000000
#define infnan(x)	0.0
#define N 128

static double	A1 = 	  .08333333333333178827;
static double	A2 = 	  .01250000000377174923;
static double	A3 =	 .002232139987919447809;
static double	A4 =	.0004348877777076145742;
static double logF_head[N+1] = {
	0.,
	.007782140442060381246,
	.015504186535963526694,
	.023167059281547608406,
	.030771658666765233647,
	.038318864302141264488,
	.045809536031242714670,
	.053244514518837604555,
	.060624621816486978786,
	.067950661908525944454,
	.075223421237524235039,
	.082443669210988446138,
	.089612158689760690322,
	.096729626458454731618,
	.103796793681567578460,
	.110814366340264314203,
	.117783035656430001836,
	.124703478501032805070,
	.131576357788617315236,
	.138402322859292326029,
	.145182009844575077295,
	.151916042025732167530,
	.158605030176659056451,
	.165249572895390883786,
	.171850256926518341060,
	.178407657472689606947,
	.184922338493834104156,
	.191394852999565046047,
	.197825743329758552135,
	.204215541428766300668,
	.210564769107350002741,
	.216873938300523150246,
	.223143551314024080056,
	.229374101064877322642,
	.235566071312860003672,
	.241719936886966024758,
	.247836163904594286577,
	.253915209980732470285,
	.259957524436686071567,
	.265963548496984003577,
	.271933715484010463114,
	.277868451003087102435,
	.283768173130738432519,
	.289633292582948342896,
	.295464212893421063199,
	.301261330578199704177,
	.307025035294827830512,
	.312755710004239517729,
	.318453731118097493890,
	.324119468654316733591,
	.329753286372579168528,
	.335355541920762334484,
	.340926586970454081892,
	.346466767346100823488,
	.351976423156884266063,
	.357455888922231679316,
	.362905493689140712376,
	.368325561158599157352,
	.373716409793814818840,
	.379078352934811846353,
	.384411698910298582632,
	.389716751140440464951,
	.394993808240542421117,
	.400243164127459749579,
	.405465108107819105498,
	.410659924985338875558,
	.415827895143593195825,
	.420969294644237379543,
	.426084395310681429691,
	.431173464818130014464,
	.436236766774527495726,
	.441274560805140936281,
	.446287102628048160113,
	.451274644139630254358,
	.456237433481874177232,
	.461175715122408291790,
	.466089729924533457960,
	.470979715219073113985,
	.475845904869856894947,
	.480688529345570714212,
	.485507815781602403149,
	.490303988045525329653,
	.495077266798034543171,
	.499827869556611403822,
	.504556010751912253908,
	.509261901790523552335,
	.513945751101346104405,
	.518607764208354637958,
	.523248143765158602036,
	.527867089620485785417,
	.532464798869114019908,
	.537041465897345915436,
	.541597282432121573947,
	.546132437597407260909,
	.550647117952394182793,
	.555141507540611200965,
	.559615787935399566777,
	.564070138285387656651,
	.568504735352689749561,
	.572919753562018740922,
	.577315365035246941260,
	.581691739635061821900,
	.586049045003164792433,
	.590387446602107957005,
	.594707107746216934174,
	.599008189645246602594,
	.603290851438941899687,
	.607555250224322662688,
	.611801541106615331955,
	.616029877215623855590,
	.620240409751204424537,
	.624433288012369303032,
	.628608659422752680256,
	.632766669570628437213,
	.636907462236194987781,
	.641031179420679109171,
	.645137961373620782978,
	.649227946625615004450,
	.653301272011958644725,
	.657358072709030238911,
	.661398482245203922502,
	.665422632544505177065,
	.669430653942981734871,
	.673422675212350441142,
	.677398823590920073911,
	.681359224807238206267,
	.685304003098281100392,
	.689233281238557538017,
	.693147180560117703862
};
static double logF_tail[N+1] = {
	0.,
	-.00000000000000543229938420049,
	 .00000000000000172745674997061,
	-.00000000000001323017818229233,
	-.00000000000001154527628289872,
	-.00000000000000466529469958300,
	 .00000000000005148849572685810,
	-.00000000000002532168943117445,
	-.00000000000005213620639136504,
	-.00000000000001819506003016881,
	 .00000000000006329065958724544,
	 .00000000000008614512936087814,
	-.00000000000007355770219435028,
	 .00000000000009638067658552277,
	 .00000000000007598636597194141,
	 .00000000000002579999128306990,
	-.00000000000004654729747598444,
	-.00000000000007556920687451336,
	 .00000000000010195735223708472,
	-.00000000000017319034406422306,
	-.00000000000007718001336828098,
	 .00000000000010980754099855238,
	-.00000000000002047235780046195,
	-.00000000000008372091099235912,
	 .00000000000014088127937111135,
	 .00000000000012869017157588257,
	 .00000000000017788850778198106,
	 .00000000000006440856150696891,
	 .00000000000016132822667240822,
	-.00000000000007540916511956188,
	-.00000000000000036507188831790,
	 .00000000000009120937249914984,
	 .00000000000018567570959796010,
	-.00000000000003149265065191483,
	-.00000000000009309459495196889,
	 .00000000000017914338601329117,
	-.00000000000001302979717330866,
	 .00000000000023097385217586939,
	 .00000000000023999540484211737,
	 .00000000000015393776174455408,
	-.00000000000036870428315837678,
	 .00000000000036920375082080089,
	-.00000000000009383417223663699,
	 .00000000000009433398189512690,
	 .00000000000041481318704258568,
	-.00000000000003792316480209314,
	 .00000000000008403156304792424,
	-.00000000000034262934348285429,
	 .00000000000043712191957429145,
	-.00000000000010475750058776541,
	-.00000000000011118671389559323,
	 .00000000000037549577257259853,
	 .00000000000013912841212197565,
	 .00000000000010775743037572640,
	 .00000000000029391859187648000,
	-.00000000000042790509060060774,
	 .00000000000022774076114039555,
	 .00000000000010849569622967912,
	-.00000000000023073801945705758,
	 .00000000000015761203773969435,
	 .00000000000003345710269544082,
	-.00000000000041525158063436123,
	 .00000000000032655698896907146,
	-.00000000000044704265010452446,
	 .00000000000034527647952039772,
	-.00000000000007048962392109746,
	 .00000000000011776978751369214,
	-.00000000000010774341461609578,
	 .00000000000021863343293215910,
	 .00000000000024132639491333131,
	 .00000000000039057462209830700,
	-.00000000000026570679203560751,
	 .00000000000037135141919592021,
	-.00000000000017166921336082431,
	-.00000000000028658285157914353,
	-.00000000000023812542263446809,
	 .00000000000006576659768580062,
	-.00000000000028210143846181267,
	 .00000000000010701931762114254,
	 .00000000000018119346366441110,
	 .00000000000009840465278232627,
	-.00000000000033149150282752542,
	-.00000000000018302857356041668,
	-.00000000000016207400156744949,
	 .00000000000048303314949553201,
	-.00000000000071560553172382115,
	 .00000000000088821239518571855,
	-.00000000000030900580513238244,
	-.00000000000061076551972851496,
	 .00000000000035659969663347830,
	 .00000000000035782396591276383,
	-.00000000000046226087001544578,
	 .00000000000062279762917225156,
	 .00000000000072838947272065741,
	 .00000000000026809646615211673,
	-.00000000000010960825046059278,
	 .00000000000002311949383800537,
	-.00000000000058469058005299247,
	-.00000000000002103748251144494,
	-.00000000000023323182945587408,
	-.00000000000042333694288141916,
	-.00000000000043933937969737844,
	 .00000000000041341647073835565,
	 .00000000000006841763641591466,
	 .00000000000047585534004430641,
	 .00000000000083679678674757695,
	-.00000000000085763734646658640,
	 .00000000000021913281229340092,
	-.00000000000062242842536431148,
	-.00000000000010983594325438430,
	 .00000000000065310431377633651,
	-.00000000000047580199021710769,
	-.00000000000037854251265457040,
	 .00000000000040939233218678664,
	 .00000000000087424383914858291,
	 .00000000000025218188456842882,
	-.00000000000003608131360422557,
	-.00000000000050518555924280902,
	 .00000000000078699403323355317,
	-.00000000000067020876961949060,
	 .00000000000016108575753932458,
	 .00000000000058527188436251509,
	-.00000000000035246757297904791,
	-.00000000000018372084495629058,
	 .00000000000088606689813494916,
	 .00000000000066486268071468700,
	 .00000000000063831615170646519,
	 .00000000000025144230728376072,
	-.00000000000017239444525614834
};


struct Double __log__D(double x)
{
	int m, j;
	double F, f, g, q, u, v, u2;
	volatile double u1;
	struct Double r;


	/* Argument reduction: 1 <= g < 2; x/2^m = g;	*/
	/* y = F*(1 + f/F) for |f| <= 2^-8		*/

	m = logb(x);
	g = ldexp(x, -m);
	if (_IEEE && m == -1022)
	{
		j = logb(g), m += j;
		g = ldexp(g, -j);
	}
	j = N*(g-1) + .5;
	F = (1.0/N) * j + 1;
	f = g - F;

	g = 1/(2*F+f);
	u = 2*f*g;
	v = u*u;
	q = u*v*(A1 + v*(A2 + v*(A3 + v*A4)));
	if (m | j)u1 = u + 513, u1 -= 513;
	else u1 = u, TRUNC(u1);
	u2 = (2.0*(f - F*u1) - u1*f) * g;
	u1 += m*logF_head[N] + logF_head[j];
	u2 +=  logF_tail[j]; u2 += q;
	u2 += logF_tail[N]*m;
	r.a = u1 + u2;			/* Only difference is here */
	TRUNC(r.a);
	r.b = (u1 - r.a) + u2;

	return (r);
}


vc(mln2hi, 8.8029691931113054792E1   ,0f33,43b0,2bdb,c7e2,   7, .B00F33C7E22BDB)
vc(mln2lo,-4.9650192275318476525E-16 ,1b60,a70f,582a,279e, -50,-.8F1B60279E582A)
vc(lnovfl, 8.8029691931113053016E1   ,0f33,43b0,2bda,c7e2,   7, .B00F33C7E22BDA)

ic(mln2hi, 7.0978271289338397310E2,    10, 1.62E42FEFA39EF)
ic(mln2lo, 2.3747039373786107478E-14, -45, 1.ABC9E3B39803F)
ic(lnovfl, 7.0978271289338397310E2,     9, 1.62E42FEFA39EF)


static int max = 1023;


double sinh(double x)
{
	double t, sign;


	if(x!=x) return(x);	/* x is NaN */

	sign=copysign(one,x);
	x=copysign(x,one);
	if(x<lnovfl)
	{
		t=expm1(x);
		return(copysign((t+t/(one+t))*half,sign));
	}

	else if(x <= lnovfl+0.7)
		/* subtract x by ln(2^(max+1)) and return 2^max*exp(x) to avoid unnecessary overflow */
	    return(copysign(scalb(one+expm1((x-mln2hi)-mln2lo),max),sign));

	else  /* sinh(+-INF) = +-INF, sinh(+-big no.) overflow to +-INF */
	    return( expm1(x)*sign );
}


double cosh(double x)
{
	static const double small=1.0E-18; /* fl(1+small)==1 */
	double t;


	if(x!=x) return(x);	/* x is NaN */

	if((x=copysign(x,one)) <= 22)
	{
	    if(x<0.3465)
		if(x<small) return(one+x);
		else {t=x+__exp__E(x,0.0);x=t+t; return(one+t*t/(2.0+x)); }

	    else /* for x lies in [0.3465,22] */
	        { t=exp(x); return((t+one/t)*half); }
	}
	if( lnovfl <= x && x <= (lnovfl+0.7))
        /* for x lies in [lnovfl, lnovfl+ln2], decrease x by ln(2^(max+1)) and return 2^max*exp(x) to avoid unnecessary overflow. */
	    return(scalb(exp((x-mln2hi)-mln2lo), max));
	else return(exp(x)*half);	/* for large x,  cosh(x)=exp(x)/2 */
}


double tanh(double x)
{
	static double small = 1.0e-10/*, big = 1.0e10*/;
	double expm1(), t, copysign(), sign;
	int finite();


	if(x!=x) return(x);	/* x is NaN */

	sign=copysign(one,x);
	x=copysign(x,one);
	if(x < 22.0)
	{
	    if( x > one )return(copysign(one-two/(expm1(x+x)+two),sign));
	    else if ( x > small )
		{
			t= -expm1(-(x+x));
			return(copysign(t/(two-t),sign));
		}
	    else		/* raise the INEXACT flag for non-zero x */
		{
			/*big+x;*/
			return(copysign(x,sign));
		}
	}
	else if(finite(x))return (sign+1.0E-37); /* raise the INEXACT flag */
	else return(sign);	/* x is +- INF */
}


static double pow_P(double x, double y)
{
	struct Double s, t, __log__D();
	double  __exp__D();
	volatile double huge = 1e300, tiny = 1e-300;


	if (x == zero)
	{
		if (y > zero)return (zero);
		else if (_IEEE)return (huge*huge);
		else return (infnan(ERANGE));
	}
	if (x == one)return (one);
	if (!finite(x))
	{
		if (y < zero)return (zero);
		else if (_IEEE)return (huge*huge);
		else return (infnan(ERANGE));
	}
	if (y >= 7e18)		/* infinity */
	{
		if (x < 1)return(tiny*tiny);
		else if (_IEEE)return (huge*huge);
		else return (infnan(ERANGE));
	}

	/* Return exp(y*log(x)), using simulated extended */
	/* precision for the log and the multiply.	  */

	s = __log__D(x);
	t.a = y;
	TRUNC(t.a);
	t.b = y - t.a;
	t.b = s.b*y + t.b*s.a;
	t.a *= s.a;
	s.a = t.a + t.b;
	s.b = (t.a - s.a) + t.b;

	return (__exp__D(s.a, s.b));
}

double pow(double x,double y)
{
	double t;


	if (y==zero)return (one);
	else if(y==one||(_IEEE && x != x))return (x);		/* if x is NaN or y=1 */
	else if (_IEEE && y!=y)return (y);		/* if y is NaN */
	else if (!finite(y))		/* if y is INF */
	{
		if ((t=fabs(x))==one)return (y - y);	/* +-1 ** +-INF is NaN */
		else if (t>one)return ((y<0)? zero : ((x<zero)? y-y : y));
		else return ((y>0)? zero : ((x<0)? y-y : -y));
	}
	else if (y==two)return (x*x);
	else if (y==negone)return (one/x);
    /* x > 0, x == +0 */
	else if (copysign(one, x) == one)return (pow_P(x, y));
    /* sign(x)= -1 */
	/* if y is an even integer */
	else if ( (t=drem(y,two)) == zero)return (pow_P(-x, y));
	/* if y is an odd integer */
	else if (copysign(t,one) == one)return (-pow_P(-x, y));
	/* Henceforth y is not an integer */
	else if (x==zero)return ((y>zero)? -x : one/(-x));	/* x is -0 */
	else if (_IEEE)return (zero/zero);
	else return (infnan(EDOM));
}
