/*
 * Copyright (c) 1997, Stefan Esser <se@freebsd.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice unmodified, this list of conditions, and the following
 *    disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $FreeBSD: src/sys/i386/intr_machdep.c,v 1.29.2.6 2003/08/18 20:22:22 jhb Exp $
 *
 * 2008: modified by minoru murashima.
 */


#include <sys/types.h>
#include <stddef.h>
#include <stdlib.h>


//=====================================  ===================================================

typedef struct intrec {
	intrmask_t		mask;
	inthand2_t		*handler;
	void			*argument;
	struct intrec	*next;
	char			*name;
	int				intr;
	intrmask_t		*maskptr;
	int				flags;
} intrec;

//===================================== Х륤ݡ =======================================

extern int _inthand_add(const uint, inthand2_t, void*, const int, void*);
extern int _inthand_remove(const uint);
extern void *getReturnMethod();

//===================================== PRIVATE ====================================================

//===================================== PUBLIC =====================================================

/*
 * Create and activate an interrupt handler descriptor data structure.
 *
 * The dev_instance pointer is required for resource management, and will
 * only be passed through to resource_claim().
 *
 * There will be functions that derive a driver and unit name from a
 * dev_instance variable, and those functions will be used to maintain the
 * interrupt counter label array referenced by systat and vmstat to report
 * device interrupt rates (->update_intrlabels).
 *
 * Add the interrupt handler descriptor data structure created by an
 * earlier call of create_intr() to the linked list for its irq and
 * adjust the interrupt masks if necessary.
 *
 * WARNING: This is an internal function and not to be used by device
 * drivers.  It is subject to change without notice.
 */
intrec *inthand_add(
	const char *name, 
	int irq, 
	inthand2_t handler, 
	void *arg,
	intrmask_t *maskptr, 
	int flags)
{
	intrec *idesc;

	if (_inthand_add(irq, handler, arg, flags, getReturnMethod()) != 0) {
		return NULL;
	}

	idesc = malloc(sizeof *idesc);
	if (idesc == NULL) {
		return NULL;
	}

	bzero(idesc, sizeof *idesc);
	if (name == NULL) {
		name = "???";
	}
	idesc->name = malloc(strlen(name) + 1);
	if (idesc->name == NULL) {
		free(idesc);
		return NULL;
	}
	strcpy(idesc->name, name);
	idesc->handler  = handler;
	idesc->argument = arg;
	idesc->maskptr  = maskptr;
	idesc->intr     = irq;
	idesc->flags    = flags;

	return (idesc);
}

/*
 * Deactivate and remove the interrupt handler descriptor data connected
 * created by an earlier call of intr_connect() from the linked list and
 * adjust theinterrupt masks if necessary.
 *
 * Return the memory held by the interrupt handler descriptor data structure
 * to the system. Make sure, the handler is not actively used anymore, before.
 */
int inthand_remove(
	intrec *idesc)
{
	intrec **hook, *head;
	int irq;
	int errcode = 0;
	intrmask_t oldspl;

	if (idesc == NULL)
		return (-1);

	return _inthand_remove(idesc->intr);
}
