/*-
 * Copyright (c) 1997 Doug Rabson
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $FreeBSD: src/sys/kern/kern_module.c,v 1.21 1999/11/08 06:53:30 peter Exp $
 *
 * 2008: modified by minoru murashima.
 */


#include <sys/types.h>
#include <sys/param.h>
#include <sys/errno.h>
#include <sys/queue.h>
#include <sys/module.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>


//=====================================  ===================================================

//*********************************************************
typedef	long				intfptr_t;
typedef	unsigned long		uintfptr_t;
//*********************************************************

typedef TAILQ_HEAD(, module) modulelist_t;

struct module {
    TAILQ_ENTRY(module)	link;		/* chain together all modules */
    TAILQ_ENTRY(module)	flink;		/* all modules in a file */
    int					refs;		/* reference count */
    int					id;			/* unique id number */
    char				*name;		/* module name */
    modeventhand_t		handler;	/* event handler */
    void				*arg;		/* argument for handler */
    modspecific_t		data;		/* module specific data */
};

#define MOD_EVENT(mod, type) (mod)->handler((mod), (type), (mod)->arg)

//===================================== Х륤ݡ =======================================

//===================================== PRIVATE ====================================================

static modulelist_t modules;
static int nextid = 1;

static int modevent_nop(
	module_t mod, 
	int what, 
	void* arg)
{
	return 0;
}

/*
 * return : error number
 */
static int module_register(
	const moduledata_t *data)
{
	size_t namelen;
	module_t newmod;

	newmod = module_lookupbyname(data->name);
	if (newmod != NULL) {
		printf("module_register: module %s already exists!\n", data->name);
		return EEXIST;
	}
	namelen = strlen(data->name) + 1;
	newmod = (module_t) malloc(sizeof(struct module) + namelen);
	if (newmod == 0) {
		return ENOMEM;
	}

	newmod->refs = 1;
	newmod->id = nextid++;
	newmod->name = (char *) (newmod + 1);
	strcpy(newmod->name, data->name);
	newmod->handler = data->evhand ? data->evhand : modevent_nop;
	newmod->arg = data->priv;
	bzero(&newmod->data, sizeof(newmod->data));
	TAILQ_INSERT_TAIL(&modules, newmod, link);

	return 0;
}

static void module_release(
	module_t mod)
{
	if (mod->refs <= 0) {
		panic("module_release: bad reference count");
	}

	mod->refs--;
	if (mod->refs == 0) {
		TAILQ_REMOVE(&modules, mod, link);
		free(mod);
	}
}

//===================================== PUBLIC =====================================================

void module_init()
{
    TAILQ_INIT(&modules);
}

/*
 * return : module or NULL
 */
module_t module_lookupbyname(
	const char* name)
{
	module_t mod;

	for (mod = TAILQ_FIRST(&modules); mod; mod = TAILQ_NEXT(mod, link)) {
		if (!strcmp(mod->name, name)) {
			return mod;
		}
	}

	return NULL;
}

void module_register_init(
	const void *arg)
{
	const moduledata_t* data = (const moduledata_t*) arg;
	int error;
	module_t mod;

	mod = module_lookupbyname(data->name);
	if (mod == NULL) {
		/* temporary kludge until kernel `file' attachment registers modules */
		error = module_register(data);
		if (error) {
			panic("module_register_init: register of module failed! %d", error);
		}
		mod = module_lookupbyname(data->name);
		if (mod == NULL) {
			panic("module_register_init: module STILL not found!");
		}
	}
	error = mod->handler(mod, (MOD_LOAD), mod->arg);
	if (error) {
		mod->handler(mod, (MOD_UNLOAD), mod->arg);
		module_release(mod);
		printf("module_register_init: MOD_LOAD (%s, %lx, %p) error %d\n", 
			data->name, (u_long)(uintfptr_t)data->evhand, data->priv, error);
	}
}

int module_unload(
	module_t mod)
{
    return MOD_EVENT(mod, MOD_UNLOAD);
}

