/*
 * Copyright (c) 1997,1998 Doug Rabson
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $FreeBSD: src/sys/kern/subr_bus.c,v 1.54.2.9 2002/10/10 15:13:32 jhb Exp $
 *
 * 2008: modified by minoru murashima.
 */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/errno.h>
#include <sys/queue.h>
#include <sys/libkern.h>
#include <sys/bus.h>
#include <sys/bus_private.h>
#include <lib/dev_method.h>
#include <stdarg.h>
#include <string.h>

//=====================================  ===================================================

//===================================== Х륤ݡ =======================================

//===================================== PRIVATE ====================================================

//--------------------------------------------------------------------------------------------------
// ꥽
//--------------------------------------------------------------------------------------------------

extern struct config_device devtab[];
extern int devtab_count;

static int resource_match_string(
	int i, 
	const char *resname, 
	const char *value)
{
	int j;
	struct config_resource *res;

	for (j = 0, res = devtab[i].resources; j < devtab[i].resource_count; j++, res++) {
		if (!strcmp(res->name, resname)
		    && res->type == RES_STRING
		    && !strcmp(res->u.stringval, value)) {
			return j;
		}
	}
	return -1;
}

static int resource_find(
	const char *name, 
	int unit, 
	const char *resname, 
	struct config_resource **result)
{
	int i, j;
	struct config_resource *res;

	/*
	 * First check specific instances, then generic.
	 */
	for (i = 0; i < devtab_count; i++) {
		if (devtab[i].unit < 0) {
			continue;
		}
		if (!strcmp(devtab[i].name, name) && devtab[i].unit == unit) {
			res = devtab[i].resources;
			for (j = 0; j < devtab[i].resource_count; j++, res++)
				if (!strcmp(res->name, resname)) {
					*result = res;
					return 0;
				}
		}
	}
	for (i = 0; i < devtab_count; i++) {
		if (devtab[i].unit >= 0) {
			continue;
		}

		/* XXX should this `&& devtab[i].unit == unit' be here? */
		/* XXX if so, then the generic match does nothing */
		if (!strcmp(devtab[i].name, name) && devtab[i].unit == unit) {
			res = devtab[i].resources;
			for (j = 0; j < devtab[i].resource_count; j++, res++)
				if (!strcmp(res->name, resname)) {
					*result = res;
					return 0;
				}
		}
	}
	return ENOENT;
}

//--------------------------------------------------------------------------------------------------
// ǥХ饹
//--------------------------------------------------------------------------------------------------

static devclass_list_t devclasses = TAILQ_HEAD_INITIALIZER(devclasses);

static driverlink_t devclass_find_driver_internal(
	devclass_t dc, 
	const char *classname)
{
	driverlink_t dl;

	for (dl = TAILQ_FIRST(&dc->drivers); dl; dl = TAILQ_NEXT(dl, link)) {
		if (!strcmp(dl->driver->name, classname))
			return dl;
	}

	return NULL;
}

static int devclass_alloc_unit(
	devclass_t dc, 
	int *unitp)
{
	int unit = *unitp;

	/* If we have been given a wired unit number, check for existing device */
	if (unit != -1) {
		if (unit >= 0 && unit < dc->maxunit && dc->devices[unit] != NULL) {
			/* find the next available slot */
			while (++unit < dc->maxunit && dc->devices[unit] != NULL);
		}
	}
	else {
		/* Unwired device, find the next available slot for it */
		unit = 0;
		while (unit < dc->maxunit && dc->devices[unit] != NULL) {
			unit++;
		}
	}

	/*
	 * We've selected a unit beyond the length of the table, so let's extend
	 * the table to make room for all units up to and including this one.
	 */
	if (dc->maxunit <= unit) {
		device_t *newlist;
		int newsize;

		newsize = roundup((unit + 1), MINALLOCSIZE / sizeof(device_t));
		newlist = malloc(sizeof(device_t) * newsize);
		if (!newlist) {
			return ENOMEM;
		}
		bcopy(dc->devices, newlist, sizeof(device_t) * dc->maxunit);
		bzero(newlist + dc->maxunit, sizeof(device_t) * (newsize - dc->maxunit));
		if (dc->devices) {
			free(dc->devices);
		}
		dc->devices = newlist;
		dc->maxunit = newsize;
	}
	*unitp = unit;
	return 0;
}

static int devclass_add_device(
	devclass_t dc, 
	device_t dev)
{
	int buflen, error;

	buflen = strlen(dc->name) + 5;
	dev->nameunit = malloc(buflen);
	if (!dev->nameunit) {
		return ENOMEM;
	}
	bzero(dev->nameunit, buflen);

	if ((error = devclass_alloc_unit(dc, &dev->unit)) != 0) {
		free(dev->nameunit);
		dev->nameunit = NULL;
		return error;
	}
	dc->devices[dev->unit] = dev;
	dev->devclass = dc;
	snprintf(dev->nameunit, buflen, "%s%d", dc->name, dev->unit);

	return 0;
}

static int devclass_delete_device(
	devclass_t dc, 
	device_t dev)
{
	if (!dc || !dev) {
		return 0;
	}

	if (dev->devclass != dc || dc->devices[dev->unit] != dev) {
		panic("devclass_delete_device: inconsistent device class");
	}
	dc->devices[dev->unit] = NULL;
	if (dev->flags & DF_WILDCARD) {
		dev->unit = -1;
	}
	dev->devclass = NULL;
	free(dev->nameunit);
	dev->nameunit = NULL;

	return 0;
}

//--------------------------------------------------------------------------------------------------
// ɥ饤С
//--------------------------------------------------------------------------------------------------

static driverlink_t first_matching_driver(
	devclass_t dc, 
	device_t dev)
{
	if (dev->devclass) {
		return devclass_find_driver_internal(dc, dev->devclass->name);
	}
	else {
		return TAILQ_FIRST(&dc->drivers);
	}
}

static driverlink_t next_matching_driver(
	devclass_t dc, 
	device_t dev, 
	driverlink_t last)
{
	if (dev->devclass) {
		driverlink_t dl;
		for (dl = TAILQ_NEXT(last, link); dl; dl = TAILQ_NEXT(dl, link)) {
			if (!strcmp(dev->devclass->name, dl->driver->name)) {
				return dl;
			}
		}
		return NULL;
	} 
	else
		return TAILQ_NEXT(last, link);
}

//--------------------------------------------------------------------------------------------------
// ǥХ
//--------------------------------------------------------------------------------------------------

static int error_method(void)
{
	return ENXIO;
}

static struct device_ops null_ops = {
	1, 
	{ error_method }
};

static void device_set_desc_internal(
	device_t dev, 
	const char* desc, 
	int copy)
{
	if (dev->desc && (dev->flags & DF_DESCMALLOCED)) {
		free(dev->desc);
		dev->flags &= ~DF_DESCMALLOCED;
		dev->desc = NULL;
	}

	if (copy && desc) {
		dev->desc = malloc(strlen(desc) + 1);
		if (dev->desc) {
			strcpy(dev->desc, desc);
			dev->flags |= DF_DESCMALLOCED;
		}
	} 
	else {
		/* Avoid a -Wcast-qual warning */
		dev->desc = (char *)(uintptr_t) desc;
	}
}

static int device_probe_child(
	device_t dev, 
	device_t child)
{
	devclass_t dc;
	driverlink_t best = 0;
	driverlink_t dl;
	int result, pri = 0;
	int hasclass = (child->devclass != 0);

	dc = dev->devclass;
	if (!dc) {
		panic("device_probe_child: parent device has no devclass");
	}

	if (child->state == DS_ALIVE) {
		return 0;
	}

	for (dl = first_matching_driver(dc, child); dl; dl = next_matching_driver(dc, child, dl)) {
		device_set_driver(child, dl->driver);
		if (!hasclass) {
			device_set_devclass(child, dl->driver->name);
		}
		result = DEVICE_PROBE(child);
		if (!hasclass) {
			device_set_devclass(child, 0);
		}

		/*
		 * If the driver returns SUCCESS, there can be no higher match
		 * for this device.
		 */
		if (result == 0) {
			best = dl;
			pri = 0;
			break;
		}

		/*
		 * The driver returned an error so it certainly doesn't match.
		 */
		if (result > 0) {
			device_set_driver(child, 0);
			continue;
		}

		/*
		 * A priority lower than SUCCESS, remember the best matching
		 * driver. Initialise the value of pri for the first match.
		 */
		if (best == 0 || result > pri) {
			best = dl;
			pri = result;
			continue;
		}
	}

	/*
	 * If we found a driver, change state and initialise the devclass.
	 */
	if (best) {
		if (!child->devclass) {
			device_set_devclass(child, best->driver->name);
		}
		device_set_driver(child, best->driver);
		if (pri < 0) {
			/*
			 * A bit bogus. Call the probe method again to make sure
			 * that we have the right description.
			 */
			DEVICE_PROBE(child);
		}
		child->state = DS_ALIVE;
		return 0;
	}

	return ENXIO;
}

static int device_print_child(
	device_t dev, 
	device_t child)
{
	int retval = 0;

	if (device_is_alive(child)) {
		retval += BUS_PRINT_CHILD(dev, child);
	} 
	else {
		retval += device_printf(child, " not found\n");
	}

	return (retval);
}

//===================================== PUBLIC =====================================================

//--------------------------------------------------------------------------------------------------
// ꥽
//--------------------------------------------------------------------------------------------------

void resource_list_init(
	struct resource_list *rl)
{
	SLIST_INIT(rl);
}

struct resource *resource_list_alloc(
	struct resource_list *rl,
	device_t bus, 
	device_t child,
	int type, 
	int *rid,
	u_long start, 
	u_long end,
	u_long count, 
	u_int flags)
{
	struct resource_list_entry *rle = 0;
	int passthrough = (device_get_parent(child) != bus);
	int isdefault = (start == 0UL && end == ~0UL);

	if (passthrough) {
		return BUS_ALLOC_RESOURCE(device_get_parent(bus), child, type, rid, start, end, count, flags);
	}

	rle = resource_list_find(rl, type, *rid);

	if (!rle) {
		return 0;		/* no resource of that type/rid */
	}
	if (rle->res) {
		panic("resource_list_alloc: resource entry is busy");
	}

	if (isdefault) {
		start = rle->start;
		count = max(count, rle->count);
		end = max(rle->end, start + count - 1);
	}

	rle->res = BUS_ALLOC_RESOURCE(device_get_parent(bus), child, type, rid, start, end, count, flags);

	/*
	 * Record the new range.
	 */
	if (rle->res) {
		rle->start = rman_get_start(rle->res);
		rle->end = rman_get_end(rle->res);
		rle->count = count;
	}

	return rle->res;
}

void resource_list_add(
	struct resource_list *rl,
	int type, 
	int rid,
	u_long start, 
	u_long end, 
	u_long count)
{
	struct resource_list_entry *rle;

	rle = resource_list_find(rl, type, rid);
	if (!rle) {
		rle = malloc(sizeof(struct resource_list_entry));
		if (!rle) {
			panic("resource_list_add: can't record entry");
		}
		SLIST_INSERT_HEAD(rl, rle, link);
		rle->type = type;
		rle->rid = rid;
		rle->res = NULL;
	}

	if (rle->res) {
		panic("resource_list_add: resource entry is busy");
	}

	rle->start = start;
	rle->end = end;
	rle->count = count;
}

int resource_list_release(
	struct resource_list *rl,
	device_t bus, 
	device_t child,
	int type, 
	int rid, 
	struct resource *res)
{
	struct resource_list_entry *rle = 0;
	int passthrough = (device_get_parent(child) != bus);
	int error;

	if (passthrough) {
		return BUS_RELEASE_RESOURCE(device_get_parent(bus), child, type, rid, res);
	}

	rle = resource_list_find(rl, type, rid);

	if (!rle) {
		panic("resource_list_release: can't find resource");
	}
	if (!rle->res) {
		panic("resource_list_release: resource entry is not busy");
	}

	error = BUS_RELEASE_RESOURCE(device_get_parent(bus), child, type, rid, res);
	if (error) {
		return error;
	}

	rle->res = NULL;
	return 0;
}

struct resource_list_entry *resource_list_find(
	struct resource_list *rl,
	int type, 
	int rid)
{
	struct resource_list_entry *rle;

	SLIST_FOREACH(rle, rl, link)
	if (rle->type == type && rle->rid == rid) {
		return rle;
	}
	return NULL;
}

int resource_list_print_type(
	struct resource_list *rl, 
	const char *name, 
	int type,
	const char *format)
{
	struct resource_list_entry *rle;
	int printed, retval;

	printed = 0;
	retval = 0;
	/* Yes, this is kinda cheating */
	SLIST_FOREACH(rle, rl, link) {
		if (rle->type == type) {
			if (printed == 0)
				retval += printf(" %s ", name);
			else
				retval += printf(",");
			printed++;
			retval += printf(format, rle->start);
			if (rle->count > 1) {
				retval += printf("-");
				retval += printf(format, rle->start +
						 rle->count - 1);
			}
		}
	}
	return (retval);
}

void resource_list_delete(
	struct resource_list *rl,
	int type, 
	int rid)
{
	struct resource_list_entry *rle = resource_list_find(rl, type, rid);

	if (rle) {
		SLIST_REMOVE(rl, rle, resource_list_entry, link);
		free(rle);
	}
}

int resource_query_string(
	int i, 
	const char *resname, 
	const char *value)
{
	if (i < 0) {
		i = 0;
	}
	else {
		i = i + 1;
	}
	for (; i < devtab_count; i++) {
		if (0 <= resource_match_string(i, resname, value)) {
			return i;
		}
	}
	return -1;
}

int resource_query_unit(
	int i)
{
	return devtab[i].unit;
}

int resource_int_value(
	const char *name, 
	int unit, 
	const char *resname, 
	int *result)
{
	int error;
	struct config_resource *res;

	if ((error = resource_find(name, unit, resname, &res)) != 0) {
		return error;
	}
	if (res->type != RES_INT) {
		return EFTYPE;
	}
	*result = res->u.intval;
	return 0;
}

char *resource_query_name(
	int i)
{
	return devtab[i].name;
}

int resource_locate(
	int i, 
	const char *resname)
{
	if (i < 0) {
		i = 0;
	}
	else {
		i = i + 1;
	}
	for (; i < devtab_count; i++) {
		if (!strcmp(devtab[i].name, resname)) {
			return i;
		}
	}
	return -1;
}

//--------------------------------------------------------------------------------------------------
// ǥХ饹
//--------------------------------------------------------------------------------------------------

const char *devclass_get_name(
	devclass_t dc)
{
	return dc->name;
}

device_t devclass_get_device(
	devclass_t dc, 
	int unit)
{
	if (dc == NULL || unit < 0 || dc->maxunit <= unit) {
		return NULL;
	}
	return dc->devices[unit];
}

int device_set_devclass(
	device_t dev, 
	const char *classname)
{
	devclass_t dc;

	if (!classname) {
		if (dev->devclass) {
			devclass_delete_device(dev->devclass, dev);
		}
		return 0;
	}

	if (dev->devclass) {
		printf("device_set_devclass: device class already set\n");
		return EINVAL;
	}

	dc = devclass_find_internal(classname, TRUE);
	if (!dc) {
		return ENOMEM;
	}

	return devclass_add_device(dc, dev);
}

devclass_t devclass_find_internal(
	const char *classname, 
	int create)
{
	devclass_t dc;

	if (!classname) {
		return NULL;
	}

	for (dc = TAILQ_FIRST(&devclasses); dc; dc = TAILQ_NEXT(dc, link)) {
		if (!strcmp(dc->name, classname)) {
			return dc;
		}
	}

	if (create) {
		dc = malloc(sizeof(struct devclass) + strlen(classname) + 1);
		if (!dc) {
			return NULL;
		}
		bzero(dc, sizeof(struct devclass) + strlen(classname) + 1);
		dc->name = (char*) (dc + 1);
		strcpy(dc->name, classname);
		dc->devices = NULL;
		dc->maxunit = 0;
		TAILQ_INIT(&dc->drivers);
		TAILQ_INSERT_TAIL(&devclasses, dc, link);
	}

	return dc;
}

devclass_t devclass_find(
	const char *classname)
{
	return devclass_find_internal(classname, FALSE);
}

int devclass_add_driver(
	devclass_t dc,			// ƥ饹
	driver_t *driver)		// ҥɥ饤С
{
	driverlink_t dl;
	int i;

	dl = malloc(sizeof *dl);
	if (!dl) {
		return ENOMEM;
	}
	bzero(dl, sizeof *dl);

	/*
	 * Compile the driver's methods.
	 */
	if (!driver->ops) {
		compile_methods(driver);
	}

	/*
	 * Make sure the devclass which the driver is implementing exists.
	 */
	devclass_find_internal(driver->name, TRUE);

	dl->driver = driver;
	TAILQ_INSERT_TAIL(&dc->drivers, dl, link);
	driver->refs++;

	/*
	 * Call BUS_DRIVER_ADDED for any existing busses in this class.
	 */
	for (i = 0; i < dc->maxunit; i++) {
		if (dc->devices[i]) {
			BUS_DRIVER_ADDED(dc->devices[i], driver);
		}
	}

	return 0;
}

int devclass_delete_driver(
	devclass_t busclass, 
	driver_t *driver)
{
	devclass_t dc = devclass_find(driver->name);
	driverlink_t dl;
	device_t dev;
	int i;
	int error;

	if (!dc) {
		return 0;
	}

	/*
	 * Find the link structure in the bus' list of drivers.
	 */
	for (dl = TAILQ_FIRST(&busclass->drivers); dl; dl = TAILQ_NEXT(dl, link)) {
		if (dl->driver == driver) {
			break;
		}
	}

	if (!dl) {
		return ENOENT;
	}

	/*
	 * Disassociate from any devices.  We iterate through all the
	 * devices in the devclass of the driver and detach any which are
	 * using the driver and which have a parent in the devclass which
	 * we are deleting from.
	 *
	 * Note that since a driver can be in multiple devclasses, we
	 * should not detach devices which are not children of devices in
	 * the affected devclass.
	 */
	for (i = 0; i < dc->maxunit; i++) {
		if (dc->devices[i]) {
			dev = dc->devices[i];
			if (dev->driver == driver && dev->parent && dev->parent->devclass == busclass) {
				if ((error = device_detach(dev)) != 0) {
					return error;
				}
				device_set_driver(dev, NULL);
			}
		}
	}

	TAILQ_REMOVE(&busclass->drivers, dl, link);
	free(dl);

	driver->refs--;
	if (driver->refs == 0) {
		free_methods(driver);
	}

	return 0;
}

//--------------------------------------------------------------------------------------------------
// ǥХ
//--------------------------------------------------------------------------------------------------

const char *device_get_name(
	device_t dev)
{
	if (dev->devclass) {
		return devclass_get_name(dev->devclass);
	}
	return NULL;
}

const char *device_get_nameunit(
	device_t dev)
{
	return dev->nameunit;
}

void *device_get_ivars(
	device_t dev)
{
	return dev->ivars;
}

const char *device_get_desc(
	device_t dev)
{
	return dev->desc;
}

int device_get_unit(
	device_t dev)
{
	return dev->unit;
}

driver_t *device_get_driver(
	device_t dev)
{
	return dev->driver;
}

u_int32_t device_get_flags(
	device_t dev)
{
	return dev->devflags;
}

device_state_t device_get_state(
	device_t dev)
{
	return dev->state;
}

device_t device_get_parent(
	device_t dev)
{
	return dev->parent;
}

void *device_get_softc(
	device_t dev)
{
	return dev->softc;
}

void device_set_ivars(
	device_t dev, 
	void * ivars)
{
	if (!dev) {
		return;
	}
	dev->ivars = ivars;
	return;
}

void device_set_desc(
	device_t dev, 
	const char* desc)
{
	device_set_desc_internal(dev, desc, FALSE);
}

void device_set_desc_copy(
	device_t dev, 
	const char* desc)
{
	device_set_desc_internal(dev, desc, TRUE);
}

int device_is_quiet(
	device_t dev)
{
	return (dev->flags & DF_QUIET) != 0;
}

int device_is_enabled(
	device_t dev)
{
	return (dev->flags & DF_ENABLED) != 0;
}

void device_disable(
	device_t dev)
{
	dev->flags &= ~DF_ENABLED;
}

device_t make_device(
	device_t parent, 
	const char *name, 
	int unit)
{
	device_t dev;
	devclass_t dc;

	if (name) {
		dc = devclass_find_internal(name, TRUE);
		if (!dc) {
			printf("make_device: can't find device class %s\n", name);
			return NULL;
		}
	}
	else {
		dc = NULL;
	}

	dev = malloc(sizeof(struct device));
	if (!dev) {
		return 0;
	}
	bzero(dev, sizeof(struct device));

	dev->parent = parent;
	TAILQ_INIT(&dev->children);
	dev->ops = &null_ops;
	dev->driver = NULL;
	dev->devclass = NULL;
	dev->unit = unit;
	dev->nameunit = NULL;
	dev->desc = NULL;
	dev->busy = 0;
	dev->devflags = 0;
	dev->flags = DF_ENABLED;
	dev->order = 0;
	if (unit == -1) {
		dev->flags |= DF_WILDCARD;
	}
	if (name) {
		dev->flags |= DF_FIXEDCLASS;
		devclass_add_device(dc, dev);
	}
	dev->ivars = NULL;
	dev->softc = NULL;

	dev->state = DS_NOTPRESENT;

	return dev;
}

int device_detach(
	device_t dev)
{
	int error;

	if (dev->state == DS_BUSY) {
		return EBUSY;
	}
	if (dev->state != DS_ATTACHED) {
		return 0;
	}

	if ((error = DEVICE_DETACH(dev)) != 0) {
		return error;
	}
	device_printf(dev, "detached\n");
	if (dev->parent) {
		BUS_CHILD_DETACHED(dev->parent, dev);
	}

	if (!(dev->flags & DF_FIXEDCLASS)) {
		devclass_delete_device(dev->devclass, dev);
	}

	dev->state = DS_NOTPRESENT;
	device_set_driver(dev, NULL);

	return 0;
}

int device_shutdown(
	device_t dev)
{
	if (dev->state < DS_ATTACHED) {
		return 0;
	}
	return DEVICE_SHUTDOWN(dev);
}

void device_quiet(
	device_t dev)
{
	dev->flags |= DF_QUIET;
}

int device_probe_and_attach(
	device_t dev)
{
	device_t bus = dev->parent;
	int error = 0;
	int hasclass = (dev->devclass != 0);

	if (dev->state >= DS_ALIVE) {
		return 0;
	}

	if (dev->flags & DF_ENABLED) {
		error = device_probe_child(bus, dev);
		if (!error) {
			if (!device_is_quiet(dev)) {
				device_print_child(bus, dev);
			}
			error = DEVICE_ATTACH(dev);
			if (!error) {
				dev->state = DS_ATTACHED;
			}
			else {
				printf("device_probe_and_attach: %s%d attach returned %d\n", dev->driver->name, dev->unit, error);
				/* Unset the class that was set in device_probe_child */
				if (!hasclass) {
					device_set_devclass(dev, 0);
				}
				device_set_driver(dev, NULL);
				dev->state = DS_NOTPRESENT;
			}
		} 
		else {
			if (!(dev->flags & DF_DONENOMATCH)) {
				BUS_PROBE_NOMATCH(bus, dev);
				dev->flags |= DF_DONENOMATCH;
			}
		}
	} 
	else {
	    device_print_prettyname(dev);
	    printf("not probed (disabled)\n");
	}

	return error;
}

device_t device_add_child_ordered(
	device_t dev, 
	int order, 
	const char *name, 
	int unit)
{
	device_t child;
	device_t place;

	child = make_device(dev, name, unit);
	if (child == NULL) {
		return NULL;
	}
	child->order = order;

	TAILQ_FOREACH(place, &dev->children, link)
	if (place->order > order) {
		break;
	}

	if (place) {
		/*
		 * The device 'place' is the first device whose order is
		 * greater than the new child.
		 */
		TAILQ_INSERT_BEFORE(place, child, link);
	} 
	else {
		/*
		 * The new child's order is greater or equal to the order of
		 * any existing device. Add the child to the tail of the list.
		 */
		TAILQ_INSERT_TAIL(&dev->children, child, link);
	}

	return child;
}

device_t device_add_child(
	device_t dev, 
	const char *name, 
	int unit)
{
	return device_add_child_ordered(dev, 0, name, unit);
}

int device_set_driver(
	device_t dev, 
	driver_t *driver)
{
	if (dev->state >= DS_ATTACHED) {
		return EBUSY;
	}

	if (dev->driver == driver) {
		return 0;
	}

	if (dev->softc && !(dev->flags & DF_EXTERNALSOFTC)) {
		free(dev->softc);
		dev->softc = NULL;
	}
	dev->ops = &null_ops;
	dev->driver = driver;
	if (driver) {
		dev->ops = driver->ops;
		if (!(dev->flags & DF_EXTERNALSOFTC)) {
			dev->softc = malloc(driver->softc);
			if (!dev->softc) {
				dev->ops = &null_ops;
				dev->driver = NULL;
				return ENOMEM;
			}
			bzero(dev->softc, driver->softc);
		}
	}
	return 0;
}

int device_get_children(
	device_t dev, 
	device_t **devlistp, 
	int *devcountp)
{
	int count;
	device_t child;
	device_t *list;
	
	count = 0;
	for (child = TAILQ_FIRST(&dev->children); child; child = TAILQ_NEXT(child, link)) {
		count++;
	}

	list = malloc(count * sizeof(device_t));
	if (!list) {
		return ENOMEM;
	}
	bzero(list, count * sizeof(device_t));

	count = 0;
	for (child = TAILQ_FIRST(&dev->children); child; child = TAILQ_NEXT(child, link)) {
		list[count] = child;
		count++;
	}

	*devlistp = list;
	*devcountp = count;

	return 0;
}

int device_is_alive(
	device_t dev)
{
	return dev->state >= DS_ALIVE;
}

int device_print_prettyname(
	device_t dev)
{
	const char *name = device_get_name(dev);

	if (name == 0) {
		return printf("unknown: ");
	}
	else {
		return printf("%s%d: ", name, device_get_unit(dev));
	}
}

int device_printf(
	device_t dev, 
	const char * fmt,
	...)
{
	va_list ap;
	int retval;

	retval = device_print_prettyname(dev);
	va_start(ap, fmt);
	retval += vprintf(fmt, ap);
	va_end(ap);
	return retval;
}

//--------------------------------------------------------------------------------------------------
// Х
//--------------------------------------------------------------------------------------------------

/*
 * Call DEVICE_IDENTIFY for each driver.
 */
int bus_generic_probe(
	device_t dev)
{
	devclass_t dc = dev->devclass;
	driverlink_t dl;

	for (dl = TAILQ_FIRST(&dc->drivers); dl; dl = TAILQ_NEXT(dl, link)) {
		DEVICE_IDENTIFY(dl->driver, dev);
	}

	return 0;
}

int bus_generic_attach(
	device_t dev)
{
	device_t child;

	for (child = TAILQ_FIRST(&dev->children); child; child = TAILQ_NEXT(child, link)) {
		device_probe_and_attach(child);
	}

	return 0;
}

int bus_generic_detach(
	device_t dev)
{
	device_t child;
	int error;

	if (dev->state != DS_ATTACHED) {
		return EBUSY;
	}

	for (child = TAILQ_FIRST(&dev->children); child; child = TAILQ_NEXT(child, link)) {
		if ((error = device_detach(child)) != 0) {
			return error;
		}
	}

	return 0;
}

int bus_generic_shutdown(
	device_t dev)
{
	device_t child;

	for (child = TAILQ_FIRST(&dev->children); child; child = TAILQ_NEXT(child, link))
		device_shutdown(child);

	return 0;
}

int bus_generic_suspend(
	device_t dev)
{
	int error;
	device_t child, child2;

	for (child = TAILQ_FIRST(&dev->children); child; child = TAILQ_NEXT(child, link)) {
		error = DEVICE_SUSPEND(child);
		if (error) {
			for (child2 = TAILQ_FIRST(&dev->children); child2 && child2 != child; child2 = TAILQ_NEXT(child2, link)) {
				DEVICE_RESUME(child2);
			}
			return (error);
		}
	}
	return 0;
}

int bus_generic_resume(
	device_t dev)
{
	device_t	child;

	for (child = TAILQ_FIRST(&dev->children); child; child = TAILQ_NEXT(child, link)) {
		DEVICE_RESUME(child);
		/* if resume fails, there's nothing we can usefully do... */
	}
	return 0;
}

int bus_generic_setup_intr(
	device_t dev, 
	device_t child, 
	struct resource *irq, 
	int flags, 
	driver_intr_t *intr, 
	void *arg,
	void **cookiep)
{
	/* Propagate up the bus hierarchy until someone handles it. */
	if (dev->parent) {
		return (BUS_SETUP_INTR(dev->parent, child, irq, flags, intr, arg, cookiep));
	}
	else {
		return (EINVAL);
	}
}

int bus_generic_teardown_intr(
	device_t dev, 
	device_t child, 
	struct resource *irq,
	void *cookie)
{
	/* Propagate up the bus hierarchy until someone handles it. */
	if (dev->parent) {
		return (BUS_TEARDOWN_INTR(dev->parent, child, irq, cookie));
	}
	else {
		return (EINVAL);
	}
}

struct resource *bus_generic_alloc_resource(
	device_t dev, 
	device_t child, 
	int type, 
	int *rid,
	u_long start, 
	u_long end, 
	u_long count, 
	u_int flags)
{
	/* Propagate up the bus hierarchy until someone handles it. */
	if (dev->parent) {
		return (BUS_ALLOC_RESOURCE(dev->parent, child, type, rid, start, end, count, flags));
	}
	else {
		return (NULL);
	}
}

int bus_generic_release_resource(
	device_t dev, 
	device_t child, 
	int type, 
	int rid,
	struct resource *r)
{
	/* Propagate up the bus hierarchy until someone handles it. */
	if (dev->parent) {
		return (BUS_RELEASE_RESOURCE(dev->parent, child, type, rid, r));
	}
	else {
		return (EINVAL);
	}
}

int bus_generic_activate_resource(
	device_t dev, 
	device_t child, 
	int type, 
	int rid,
	struct resource *r)
{
	/* Propagate up the bus hierarchy until someone handles it. */
	if (dev->parent) {
		return (BUS_ACTIVATE_RESOURCE(dev->parent, child, type, rid, r));
	}
	else {
		return (EINVAL);
	}
}

int bus_generic_deactivate_resource(
	device_t dev, 
	device_t child, 
	int type,
	int rid, 
	struct resource *r)
{
	/* Propagate up the bus hierarchy until someone handles it. */
	if (dev->parent) {
		return (BUS_DEACTIVATE_RESOURCE(dev->parent, child, type, rid, r));
	}
	else {
		return (EINVAL);
	}
}

int bus_print_child_header(
	device_t dev, 
	device_t child)
{
	int retval = 0;

	if (device_get_desc(child)) { 
		retval += device_printf(child, "<%s>", device_get_desc(child));
	} 
	else {
		retval += printf("%s", device_get_nameunit(child));
	}

	return (retval);
}

int bus_print_child_footer(
	device_t dev, 
	device_t child)
{
	return(printf(" on %s\n", device_get_nameunit(dev)));
}

int bus_generic_read_ivar(
	device_t dev, 
	device_t child, 
	int index, 
	uintptr_t * result)
{
	return ENOENT;
}

int bus_generic_write_ivar(
	device_t dev, 
	device_t child, 
	int index, 
	uintptr_t value)
{
	return ENOENT;
}

/*
 * Some convenience functions to make it easier for drivers to use the
 * resource-management functions.  All these really do is hide the
 * indirection through the parent's method table, making for slightly
 * less-wordy code.  In the future, it might make sense for this code
 * to maintain some sort of a list of resources allocated by each device.
 */
struct resource *bus_alloc_resource(
	device_t dev, 
	int type, 
	int *rid, 
	u_long start, 
	u_long end,
	u_long count, 
	u_int flags)
{
	if (dev->parent == 0) {
		return (0);
	}
	return (BUS_ALLOC_RESOURCE(dev->parent, dev, type, rid, start, end, count, flags));
}

int bus_release_resource(
	device_t dev, 
	int type, 
	int rid, 
	struct resource *r)
{
	if (dev->parent == 0) {
		return (EINVAL);
	}
	return (BUS_RELEASE_RESOURCE(dev->parent, dev, type, rid, r));
}

int bus_activate_resource(
	device_t dev, 
	int type, 
	int rid, 
	struct resource *r)
{
	if (dev->parent == 0) {
		return (EINVAL);
	}
	return (BUS_ACTIVATE_RESOURCE(dev->parent, dev, type, rid, r));
}

int bus_deactivate_resource(
	device_t dev, 
	int type, 
	int rid, 
	struct resource *r)
{
	if (dev->parent == 0) {
		return (EINVAL);
	}
	return (BUS_DEACTIVATE_RESOURCE(dev->parent, dev, type, rid, r));
}

void bus_delete_resource(
	device_t dev, 
	int type, 
	int rid)
{
	BUS_DELETE_RESOURCE(device_get_parent(dev), dev, type, rid);
}

u_long bus_get_resource_start(
	device_t dev, 
	int type, 
	int rid)
{
	u_long start, count;
	int error;

	error = BUS_GET_RESOURCE(device_get_parent(dev), dev, type, rid, &start, &count);
	if (error) {
		return 0;
	}
	return start;
}

int bus_set_resource(
	device_t dev, 
	int type, 
	int rid,
	u_long start, 
	u_long count)
{
	return BUS_SET_RESOURCE(device_get_parent(dev), dev, type, rid, start, count);
}

int driver_module_handler(
	module_t mod, 
	int what, 
	void *arg)
{
	int error, i;
	struct driver_module_data *dmd;
	devclass_t bus_devclass;

	dmd = (struct driver_module_data *)arg;
	bus_devclass = devclass_find_internal(dmd->dmd_busname, TRUE);
	error = 0;

	switch (what) {
	case MOD_LOAD:
		if (dmd->dmd_chainevh) {
			error = dmd->dmd_chainevh(mod,what,dmd->dmd_chainarg);
		}

		for (i = 0; !error && i < dmd->dmd_ndrivers; i++) {
			error = devclass_add_driver(bus_devclass, dmd->dmd_drivers[i]);
		}
		if (error) {
			break;
		}

		/*
		 * The drivers loaded in this way are assumed to all
		 * implement the same devclass.
		 */
		*dmd->dmd_devclass = devclass_find_internal(dmd->dmd_drivers[0]->name, TRUE);
		break;

	case MOD_UNLOAD:
		for (i = 0; !error && i < dmd->dmd_ndrivers; i++) {
			error = devclass_delete_driver(bus_devclass, dmd->dmd_drivers[i]);
		}

		if (!error && dmd->dmd_chainevh) {
			error = dmd->dmd_chainevh(mod,what,dmd->dmd_chainarg);
		}
		break;
	}

	return (error);
}
