/*-
 * Copyright (c) 1997,1998 Doug Rabson
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $FreeBSD: src/sys/sys/bus.h,v 1.30.2.5 2004/03/17 17:54:25 njl Exp $
 *
 * 2008: modified by minoru murashima.
 */


#ifndef bus_h
#define bus_h

#include <sys/module.h>
#include <sys/queue.h>
#include <sys/rman.h>

struct device;
struct devclass;
struct device_ops;
struct device_op_desc;

typedef struct devclass			*devclass_t;
typedef struct device			*device_t;
typedef struct device_method	device_method_t;
typedef struct device_ops		*device_ops_t;
typedef struct device_op_desc	*device_op_desc_t;
typedef struct driver			driver_t;
typedef void driver_intr_t(void*);
typedef int (*devop_t)(void);

/*
 * Module support for automatically adding drivers to busses.
 */
struct driver_module_data {
	int			(*dmd_chainevh)(struct module *, int, void *);
	void		*dmd_chainarg;
	const char	*dmd_busname;
	driver_t	**dmd_drivers;
	int			dmd_ndrivers;
	devclass_t	*dmd_devclass;
};

typedef enum device_state {
    DS_NOTPRESENT,			/* not probed or probe failed */
    DS_ALIVE,				/* probe succeeded */
    DS_ATTACHED,			/* attach method called */
    DS_BUSY					/* device is open */
} device_state_t;

struct driver {
    const char		*name;		/* driver name */
    device_method_t	*methods;	/* method table */
    size_t			softc;		/* size of device softc struct */
    void			*priv;		/* driver private data */
    device_ops_t	ops;		/* compiled method table */
    int				refs;		/* # devclasses containing driver */
};

struct device_method {
    int			desc;
    devop_t		func;
};

struct resource_list_entry {
    SLIST_ENTRY(resource_list_entry) link;
    int			type;			/* type argument to alloc_resource */
    int			rid;			/* resource identifier */
    struct resource	*res;		/* the real resource when allocated */
    u_long		start;			/* start of resource range */
    u_long		end;			/* end of resource range */
    u_long		count;			/* count within range */
};
SLIST_HEAD(resource_list, resource_list_entry);

/*
 * We define this in terms of bits because some devices may belong
 * to multiple classes (and therefore need to be included in
 * multiple interrupt masks, which is what this really serves to
 * indicate.  Buses which do interrupt remapping will want to
 * change their type to reflect what sort of devices are underneath.
 */
enum intr_type {
    INTR_TYPE_TTY = 1,
    INTR_TYPE_BIO = 2,
    INTR_TYPE_NET = 4,
    INTR_TYPE_CAM = 8,
    INTR_TYPE_MISC = 16,
    INTR_TYPE_FAST = 128
};
#define INTR_TYPE_AV INTR_TYPE_TTY	/* for source compatability with 5.x */

#define DEVMETHOD(NAME, FUNC) {NAME##_desc, (devop_t) FUNC }
#define	MINALLOCSIZE	(1 << MINBUCKET)

#define DRIVER_MODULE(name, busname, driver, devclass, evh, arg)	\
static driver_t *name##_##busname##_driver_list[] = { &driver };	\
static struct driver_module_data name##_##busname##_driver_mod = {	\
	evh,															\
	arg,															\
	#busname,														\
	name##_##busname##_driver_list,									\
	(sizeof name##_##busname##_driver_list) /						\
		(sizeof name##_##busname##_driver_list[0]),					\
	&devclass														\
};																	\
moduledata_t name##_##busname##_mod = {								\
	#busname "/" #name,												\
	driver_module_handler,											\
	&name##_##busname##_driver_mod									\
};

extern device_t root_bus;
extern devclass_t root_devclass;

extern void			resource_list_init(struct resource_list *rl);
extern struct resource *resource_list_alloc(struct resource_list*, device_t, device_t, int, int*, u_long, u_long, u_long, u_int);
extern void			resource_list_add(struct resource_list *rl, int type, int rid, u_long start, u_long end, u_long count);
extern int			resource_list_release(struct resource_list *rl, device_t bus, device_t child, int type, int rid, struct resource *res);
extern struct resource_list_entry *resource_list_find(struct resource_list *rl, int type, int rid);
extern void			resource_list_delete(struct resource_list *rl, int type, int rid);
extern int			resource_list_print_type(struct resource_list *rl, const char *name, int type, const char *format);
extern int			resource_int_value(const char *name, int unit, const char *resname, int *result);
extern char			*resource_query_name(int i);
extern int			resource_query_unit(int i);
extern int			resource_locate(int i, const char *resname);

extern const char 	*devclass_get_name(devclass_t dc);
extern device_t		devclass_get_device(devclass_t dc, int unit);
extern int			devclass_get_devices(devclass_t dc, device_t **listp, int *countp);
extern devclass_t	devclass_find_internal(const char*, int);
extern devclass_t	devclass_find(const char *classname);
extern int			devclass_add_driver(devclass_t dc, driver_t *driver);

extern device_t		device_add_child(device_t dev, const char *name, int unit);
extern device_t		device_add_child_ordered(device_t dev, int order, const char *name, int unit);
extern int			device_get_children(device_t dev, device_t **listp, int *countp);
extern device_t		make_device(device_t, const char*, int);
extern const char	*device_get_name(device_t dev);
extern const char	*device_get_nameunit(device_t dev);
extern void			*device_get_ivars(device_t dev);
extern const char 	*device_get_desc(device_t dev);
extern driver_t		*device_get_driver(device_t dev);
extern int			device_get_unit(device_t dev);
extern u_int32_t	device_get_flags(device_t dev);
extern device_state_t	device_get_state(device_t dev);
extern device_t		device_get_parent(device_t dev);
extern void			*device_get_softc(device_t dev);
extern void			device_set_ivars(device_t dev, void *ivars);
extern int			device_set_devclass(device_t dev, const char *classname);
extern void			device_set_desc(device_t dev, const char* desc);
extern int			device_is_quiet(device_t dev);
extern int			device_is_enabled(device_t dev);
extern void			device_disable(device_t dev);
extern int			device_detach(device_t dev);
extern int			device_shutdown(device_t dev);
extern int			device_probe_and_attach(device_t dev);
extern int			device_set_driver(device_t dev, driver_t *driver);
extern int			device_is_alive(device_t dev);
extern int			device_printf(device_t dev, const char *, ...);
extern int			device_print_prettyname(device_t dev);

extern int			bus_generic_probe(device_t dev);
extern int			bus_generic_shutdown(device_t dev);
extern int			bus_generic_suspend(device_t dev);
extern int			bus_generic_resume(device_t dev);
extern int			bus_generic_setup_intr(device_t dev, device_t child, struct resource *irq, int flags, driver_intr_t *intr, void *arg, void **cookiep);
extern int			bus_generic_teardown_intr(device_t dev, device_t child, struct resource *irq, void *cookie);
extern struct resource *bus_generic_alloc_resource(device_t bus, device_t child, int type, int *rid, u_long start, u_long end, u_long count, u_int flags);
extern int			bus_generic_release_resource(device_t bus, device_t child, int type, int rid, struct resource *r);
extern int			bus_generic_activate_resource(device_t dev, device_t child, int type, int rid, struct resource *r);
extern int			bus_generic_deactivate_resource(device_t dev, device_t child, int type, int rid, struct resource *r);
extern int			bus_print_child_header(device_t dev, device_t child);
extern int			bus_print_child_footer(device_t dev, device_t child);
extern int			bus_generic_read_ivar(device_t dev, device_t child, int which, uintptr_t *result);
extern int			bus_generic_write_ivar(device_t dev, device_t child, int which, uintptr_t value);
extern int			bus_generic_attach(device_t dev);
extern int			bus_generic_detach(device_t dev);
extern struct resource *bus_alloc_resource(device_t dev, int type, int *rid, u_long start, u_long end, u_long count, u_int flags);
extern int			bus_activate_resource(device_t dev, int type, int rid, struct resource *r);
extern int			bus_deactivate_resource(device_t dev, int type, int rid, struct resource *r);
extern int			bus_release_resource(device_t dev, int type, int rid, struct resource *r);
extern u_long		bus_get_resource_start(device_t dev, int type, int rid);
extern int			bus_set_resource(device_t dev, int type, int rid, u_long start, u_long count);

extern int			driver_module_handler(struct module *, int, void *);

#endif
