/* uio.c
   Copyright (C) 2006 Free Software Foundation, Inc.

This file is part of GNU Classpath.

GNU Classpath is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; version 2 of the License.

GNU Classpath is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Classpath; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA.

Linking this library statically or dynamically with other modules is
making a combined work based on this library.  Thus, the terms and
conditions of the GNU General Public License cover the whole
combination.

As a special exception, the copyright holders of this library give you
permission to link this library with independent modules to produce an
executable, regardless of the license terms of these independent
modules, and to copy and distribute the resulting executable under
terms of your choice, provided that you also meet, for each linked
independent module, the terms and conditions of the license of that
module.  An independent module is a module which is not derived from
or based on this library.  If you modify this library, you may extend
this exception to your version of the library, but you are not
obligated to do so.  If you do not wish to do so, delete this
exception statement from your version. */

#include <sys/uio.h>
#include <windows.h>
#include <errno.h>

ssize_t read(int fd, void *buf, size_t count) {
	DWORD bytesRead;
	BOOL result = ReadFile((HANDLE) fd,
				buf,
				count,
				&bytesRead,
				NULL);
	if (! result) {
		DWORD error = GetLastError();
		switch (error) {
			case ERROR_MORE_DATA:
				// ͋炭G[ł͂Ȃ
				break;

			case ERROR_INVALID_HANDLE:
				errno = EBADF;
				bytesRead = (DWORD) -1;
				break;

			default:
				errno = EIO;
				bytesRead = (DWORD) -1;
				break;
		}
	}
	return (ssize_t) bytesRead;
}

ssize_t write(int fd, const void *buf, size_t count) {
	DWORD bytesWritten;
	BOOL result = WriteFile((HANDLE) fd,
				buf,
				count,
				&bytesWritten,
				NULL);
	if (! result) {
		DWORD error = GetLastError();
		switch (error) {
			case ERROR_INVALID_HANDLE:
				errno = EBADF;
				bytesWritten = (DWORD) -1;
				break;

			case ERROR_DISK_FULL:
				errno = ENOSPC;
				bytesWritten = (DWORD) -1;
				break;

			default:
				errno = EIO;
				bytesWritten = (DWORD) -1;
				break;
		}
	}
	return (ssize_t) bytesWritten;
}

ssize_t readv(int fd, const struct iovec *vector, int count) {
	size_t tmpbuf_len = 0;
	void* tmpbuf;
	char* charbuf;
	ssize_t result;
	int i;

	if (count < 0) {
		errno = EINVAL;
		return (ssize_t) -1;
	}

	// e|obt@̃TCYvZ
	for (i = 0; i < count; ++i) {
		tmpbuf_len += vector[i].iov_len;
	}
	
	// e|obt@mۂ
	tmpbuf = malloc(tmpbuf_len);
	if (! tmpbuf) {
		errno = ENOMEM;
		return -1;
	}
	result = read(fd, tmpbuf, tmpbuf_len);
	charbuf = (char*) tmpbuf;
	if (result >= 0) {
		// e|obt@ vector ɃRs[
		ssize_t left = result;
		for (i = 0; i < count; ++i) {
			const size_t iov_len = vector[i].iov_len;
			const size_t copy_len = (left < (ssize_t) iov_len)
						? left
						: iov_len;
			memcpy(vector[i].iov_base, charbuf, copy_len);
			charbuf += copy_len;
			left -= copy_len;
			if (left <= 0) {
				// Rs[f[^Ȃ
				break;
			}
		}
	}
	// e|obt@
	free(tmpbuf);

	return result;
}

ssize_t writev(int fd, const struct iovec *vector, int count) {
	ssize_t tmpbuf_len;
	ssize_t result;
	void* tmpbuf;
	char* charbuf;
	int i;

	if (count < 0) {
		errno = EINVAL;
		return (ssize_t) -1;
	}

	// e|obt@̃TCYvZ
	tmpbuf_len = 0;
	for (i = 0; i < count; ++i) {
		tmpbuf_len += vector[i].iov_len;
	}
	
	// e|obt@mۂ
	tmpbuf = malloc(tmpbuf_len);
	if (! tmpbuf) {
		errno = ENOMEM;
		return -1;
	}

	// e|obt@Ƀf[^Rs[
	charbuf = (char*) tmpbuf;
	for (i = 0; i < count; ++i) {
		const size_t iov_len = vector[i].iov_len;
		memcpy(charbuf, vector[i].iov_base, iov_len);
		charbuf += iov_len;
	}

	// e|obt@̓eo
	result = write(fd, tmpbuf, tmpbuf_len);

	// e|obt@
	free(tmpbuf);

	return result;
}
