/* $Id: MySQLProcessDAO.java,v 1.17 2005/08/26 04:54:43 ysahara Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.dataaccess.mysql;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;

import jp.co.argo21.nautica.workflow.dataaccess.AbstractDAO;
import jp.co.argo21.nautica.workflow.dataaccess.DAOException;
import jp.co.argo21.nautica.workflow.dataaccess.ProcessBean;
import jp.co.argo21.nautica.workflow.dataaccess.ProcessDAO;
import jp.co.argo21.nautica.workflow.dataaccess.ProcessHistoryBean;
import jp.co.argo21.nautica.workflow.dataaccess.ProcessHistoryDAO;
import jp.co.argo21.nautica.workflow.dataaccess.WorkflowDAOFactory;
import jp.co.argo21.nautica.workflow.engine.DataAccessManager;
import jp.co.argo21.nautica.workflow.engine.LogManager;
import jp.co.argo21.nautica.workflow.filter.FilterConverter;
import jp.co.argo21.nautica.workflow.util.StringManager;
import jp.co.argo21.nautica.workflow.wfmc.Filter;
import jp.co.argo21.nautica.workflow.wfmc.InvalidFilterException;
import jp.co.argo21.nautica.workflow.wfmc.ProcessState;

/**
 * ProcessBeanf[^ƂMySQLpDAONXłB
 * <pre><tt>
 * PROCESSe[u
 * +----------------+--------------+------+-----+---------------------+-------+
 * | Field          | Type         | Null | Key | Default             | Extra |
 * +----------------+--------------+------+-----+---------------------+-------+
 * | PROC_ID        | varchar(64)  |      | PRI |                     |       |
 * | NAME           | varchar(128) | YES  |     | NULL                |       |
 * | PACKAGE_ID     | varchar(64)  |      |     |                     |       |
 * | PROC_DEF_ID    | varchar(64)  |      |     |                     |       |
 * | START_DATE     | datetime     |      |     | 0000-00-00 00:00:00 |       |
 * | STARTER_ID     | varchar(64)  |      |     |                     |       |
 * | DURATION_LIMIT | datetime     |      |     | 0000-00-00 00:00:00 |       |
 * | PROC_STATE     | int(1)       |      |     | 0                   |       |
 * | UPDATE_DATE    | datetime     | YES  |     | NULL                |       |
 * +----------------+--------------+------+-----+---------------------+-------+
 * </tt></pre>
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.17 $
 * @since   Nautica Workflow 0.9
 */
public class MySQLProcessDAO extends AbstractDAO implements ProcessDAO
{
	/** GWO */
	static protected Logger eLog = LogManager.getEngineLogger();

	/** vZXID̃J */
	static public final String PROC_ID        = "PROC_ID";
	/** vZX̃J */
	static public final String NAME           = "NAME";
	/** pbP[WID̃J */
	static public final String PACKAGE_ID     = "PACKAGE_ID";
	/** vZX`ID̃J */
	static public final String PROC_DEF_ID    = "PROC_DEF_ID";
	/** JñJ */
	static public final String START_DATE     = "START_DATE";
	/** Jn҂̃J */
	static public final String STARTER_ID     = "STARTER_ID";
	/** s̃J */
	static public final String DURATION_LIMIT = "DURATION_LIMIT";
	/** vZXԂ̃J */
	static public final String PROC_STATE     = "PROC_STATE";
	/** XṼJ */
	static public final String UPDATE_DATE    = "UPDATE_DATE";

	static private final String SQL_SELECT_BY_PRIMARY_KEY
	= "select * from PROCESS where PROC_ID = ? ";

	static private final String SQL_SELECT_BY_PID_AND_PDID
	= "select * from PROCESS where PROC_DEF_ID = ? and PROC_ID = ? ";

	static private final String SQL_SELECT_BY_PDID
	= "select * from PROCESS where PROC_DEF_ID = ? ";

	static private final String SQL_SELECT
	= "select * from PROCESS ";

	static private final String SQL_SELECT_COUNT_BY_PDID
	= "select COUNT(*) from PROCESS where PROC_DEF_ID = ? ";

	static private final String SQL_INSERT
	= "insert into PROCESS ("
		+ PROC_ID + ", "
		+ NAME + ", "
		+ PACKAGE_ID + ", "
		+ PROC_DEF_ID + ", "
		+ START_DATE + ", "
		+ STARTER_ID + ", "
		+ DURATION_LIMIT + ", "
		+ PROC_STATE + ", "
		+ UPDATE_DATE + ") values (?,?,?,?,?,?,?,?,?)";

	private static final String SQL_UPDATE_NAME
	= "UPDATE PROCESS SET NAME = ? WHERE PROC_ID = ? ";

	private static final String SQL_UPDATE_STATE
	= "UPDATE PROCESS SET PROC_STATE = ?, UPDATE_DATE = ? WHERE PROC_ID = ? ";

	private static final String SQL_UPDATE_DATE
	= "UPDATE PROCESS SET UPDATE_DATE = ? WHERE PROC_ID = ? ";

	/**
	 * vZXǗe[uɑ΂ANZXIuWFNg𐶐B
	 *
	 * @param conn RlNV
	 */
	MySQLProcessDAO(Connection conn)
	{
		super(conn);
	}
	
	/**
	 * L[ɂAvZX擾B
	 *
	 * @param pid vZXID
	 * @return vZX
	 * @throws SQLException CӂSQLO
	 * @throws DAOException DAOsɏƂĂꍇO
	 * @see jp.co.argo21.nautica.workflow.dataaccess.ProcessDAO#findByPrimaryKey(java.lang.String)
	 */
	public ProcessBean findByPrimaryKey(String pid)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT_BY_PRIMARY_KEY);
		try {
			st.setString(1, pid);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			ResultSet rs = st.executeQuery();
			
			ProcessBean bean = null;
			if (rs.next()) {
				bean = createBean(rs);
			}
			return bean;
		} finally {
			st.close();
		}
	}
	
	/**
	 * vZX`IDƃvZXIDvZX擾B
	 *
	 * @param pdid	vZX`ID
	 * @param pid vZXID
	 * @return vZX
	 * @throws SQLException CӂSQLO
	 * @throws DAOException DAOsɏƂĂꍇO
	 * @see jp.co.argo21.nautica.workflow.dataaccess.ProcessDAO#findByPDIDandPID(java.lang.String, java.lang.String)
	 */
	public ProcessBean findByPDIDandPID(String pdid, String pid)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT_BY_PID_AND_PDID);
		try {
			st.setString(1, pdid);
			st.setString(2, pid);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			ResultSet rs = st.executeQuery();
			
			ProcessBean bean = null;
			if (rs.next()) {
				bean = createBean(rs);
			}
			return bean;
		} finally {
			st.close();
		}
	}
	
	/**
	 * vZX`IDɈvvZX擾B
	 *
	 * @param pdid	vZX`ID
	 * @return vZX
	 * @throws SQLException CӂSQLO
	 * @throws DAOException DAOsɏƂĂꍇO
	 * @see jp.co.argo21.nautica.workflow.dataaccess.ProcessDAO#findByPDID(java.lang.String)
	 */
	public ProcessBean[] findByPDID(String pdid)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT_BY_PDID);
		try {
			st.setString(1, pdid);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			ResultSet rs = st.executeQuery();
			
			List list = new ArrayList();
			while (rs.next()) {
				ProcessBean bean = createBean(rs);
				list.add(bean);
			}
			return (ProcessBean[])list.toArray(new ProcessBean[list.size()]);
		} finally {
			st.close();
		}
	}
	
	/**
	 * vZX`IDɈvvZX̌ԂB
	 *
	 * @param pdid	vZX`ID
	 * @return vZX
	 * @throws SQLException CӂSQLO
	 * @throws DAOException DAOsɏƂĂꍇO
	 * @see jp.co.argo21.nautica.workflow.dataaccess.ProcessDAO#findCountByPDID(java.lang.String)
	 */
	public int findCountByPDID(String pdid)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT_COUNT_BY_PDID);
		try {
			st.setString(1, pdid);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			ResultSet rs = st.executeQuery();
			
			ProcessBean bean = null;
			int count = 0;
			if (rs.next()) {
				count = rs.getInt(1);
			}
			return count;
		} finally {
			st.close();
		}
	}
	
	/**
	 * tB^[ɈvvZXԂB
	 *
	 * @param filter tB^[
	 * @return vZX
	 * @throws SQLException CӂSQLO
	 * @throws DAOException DAOsɏƂĂꍇO
	 * @throws InvalidFilterException
	 * @see jp.co.argo21.nautica.workflow.dataaccess.ProcessDAO#findByFilter(jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	public ProcessBean[] findByFilter(Filter filter)
	throws SQLException, DAOException, InvalidFilterException
	{
		FilterConverter converter = new ProcessFilterConverter();
		String where = converter.convertToSQL(filter);
		
		if (where.equals("") == false) {
			where = " where " + where;
		}
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT + where);

		/************** DEBUG **************/
		eLog.debug(st.toString());
		/***********************************/

		try {
			ResultSet rs = st.executeQuery();
			
			List list = new ArrayList();
			while (rs.next()) {
				ProcessBean bean = createBean(rs);
				list.add(bean);
			}
			return (ProcessBean[])list.toArray(new ProcessBean[list.size()]);
		} finally {
			st.close();
		}
	}

	/**
	 * vZX`IDƃtB^[ɈvvZXԂB
	 *
	 * @param pdid	vZX`ID
	 * @param filter tB^[
	 * @return vZX
	 * @throws SQLException CӂSQLO
	 * @throws DAOException DAOsɏƂĂꍇO
	 * @see jp.co.argo21.nautica.workflow.dataaccess.ProcessDAO#findByFilter(java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	public ProcessBean[] findByFilter(String pdid, Filter filter)
	throws SQLException, DAOException, InvalidFilterException
	{
		FilterConverter converter = new ProcessFilterConverter();
		String where = converter.convertToSQL(filter);
		
		if (where.equals("")) {
			where = " where PROC_DEF_ID = '" + pdid + "' ";
		} else {
			where = " where PROC_DEF_ID = '" + pdid + "' and " + where;
		}
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT + where);

		/************** DEBUG **************/
		eLog.debug(st.toString());
		/***********************************/

		try {
			ResultSet rs = st.executeQuery();
			
			List list = new ArrayList();
			while (rs.next()) {
				ProcessBean bean = createBean(rs);
				list.add(bean);
			}
			return (ProcessBean[])list.toArray(new ProcessBean[list.size()]);
		} finally {
			st.close();
		}
	}
	
	/**
	 * vZX}B
	 *
	 * @param bean vZX
	 * @throws SQLException CӂSQLO
	 * @throws DAOException DAOsɏƂĂꍇO
	 * @see jp.co.argo21.nautica.workflow.dataaccess.ProcessDAO#insert(jp.co.argo21.nautica.workflow.dataaccess.ProcessBean)
	 */
	public void insert(ProcessBean bean)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_INSERT);
		try {
			st.setString(1, bean.getID());
			st.setString(2, bean.getName());
			st.setString(3, bean.getPackageID());
			st.setString(4, bean.getProcDefinitionID());
			st.setTimestamp(5, bean.getStartDate());
			st.setString(6, bean.getStarterID());
			st.setTimestamp(7, bean.getDurationLimit());
			st.setInt(8, bean.getProcessState());
			st.setTimestamp(9, bean.getUpdateDate());
			
			int i = st.executeUpdate();
			
			if (i != 1) {
				// vZXR[h̑}Ɏs܂B
				String errMsg = StringManager.get("E0018");
				throw new DAOException(errMsg);
			}
		} finally {
			st.close();
		}

		try {
			WorkflowDAOFactory factory = DataAccessManager.getDAOFactory();
			ProcessHistoryDAO phDAO = factory.getProcessHistoryDAO();
			ProcessHistoryBean history = new ProcessHistoryBean();
			history.setProcID(bean.getID());
			history.setPackageID(bean.getPackageID());
			history.setProcDefID(bean.getProcDefinitionID());
			history.setPrevState(-1);
			history.setNewState(bean.getProcessState());
			history.setUpdateDate(bean.getUpdateDate());
			history.setReason("NEW PROCESS CREATION");
			phDAO.insert(history);
		} catch (SQLException ex) {
			throw ex;
		} catch (DAOException ex) {
			throw ex;
		} catch (Exception ex) {
			// vZXԗ̍XVɎs܂B
			String errMsg = StringManager.get("E0019");
			throw new DAOException(errMsg, ex);
		}
	}
	
	/**
	 * vZXXVB
	 *
	 * @param bean vZX
	 * @param name vZX
	 * @throws SQLException CӂSQLO
	 * @throws DAOException DAOsɏƂĂꍇO
	 * @see jp.co.argo21.nautica.workflow.dataaccess.ProcessDAO#updateName(jp.co.argo21.nautica.workflow.dataaccess.ProcessBean, java.lang.String)
	 */
	public void updateName(ProcessBean bean, String name)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_UPDATE_NAME);
		try {
			st.setString(1, name);
			st.setString(2, bean.getID());
			
			int i = st.executeUpdate();
			
			if (i != 1) {
				// vZX̍XVɎs܂B
				String errMsg = StringManager.get("E0020");
				throw new DAOException(errMsg);
			}
			bean.setName(name);
		} finally {
			st.close();
		}
	}
	
	/**
	 * vZXԂXVB
	 *
	 * @param bean vZX
	 * @param state vZX
	 * @param ts ^CX^v
	 * @throws SQLException CӂSQLO
	 * @throws DAOException DAOsɏƂĂꍇO
	 * @see jp.co.argo21.nautica.workflow.dataaccess.ProcessDAO#updateState(jp.co.argo21.nautica.workflow.dataaccess.ProcessBean, jp.co.argo21.nautica.workflow.wfmc.ProcessState, java.sql.Timestamp)
	 */
	public void updateState(ProcessBean bean, ProcessState state, Timestamp ts)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_UPDATE_STATE);
		try {
			st.setInt(1, state.toInt());
			st.setTimestamp(2, ts);
			st.setString(3, bean.getID());
			
			int i = st.executeUpdate();
			
			if (i != 1) {
				// vZXԂ̍XVɎs܂B
				String errMsg = StringManager.get("E0021");
				throw new DAOException(errMsg);
			}
		} finally {
			st.close();
		}

		try {
			WorkflowDAOFactory factory = DataAccessManager.getDAOFactory();
			ProcessHistoryDAO phDAO = factory.getProcessHistoryDAO();
			ProcessHistoryBean history = new ProcessHistoryBean();
			history.setProcID(bean.getID());
			history.setPackageID(bean.getPackageID());
			history.setProcDefID(bean.getProcDefinitionID());
			history.setPrevState(bean.getProcessState());
			history.setNewState(state.toInt());
			history.setUpdateDate(bean.getUpdateDate());
			ProcessState old = ProcessState.getState(bean.getProcessState());
			String msg = "PROCESS STATE CHANGED FROM " + old.toString() + " TO " + state.toString();
			history.setReason(msg);
			phDAO.insert(history);
		} catch (SQLException ex) {
			throw ex;
		} catch (DAOException ex) {
			throw ex;
		} catch (Exception ex) {
			// vZXԗ̍XVɎs܂B
			String errMsg = StringManager.get("E0019");
			throw new DAOException(errMsg, ex);
		}
		bean.setUpdateDate(ts);
		bean.setProcessState(state.toInt());
	}
	
	/**
	 * vZXύXtXVB
	 *
	 * @param bean vZX
	 * @param ts ^CX^v
	 * @throws SQLException CӂSQLO
	 * @throws DAOException DAOsɏƂĂꍇO
	 * @see jp.co.argo21.nautica.workflow.dataaccess.ProcessDAO#updateDate(jp.co.argo21.nautica.workflow.dataaccess.ProcessBean, java.sql.Timestamp)
	 */
	public void updateDate(ProcessBean bean, Timestamp ts)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_UPDATE_DATE);
		try {
			st.setTimestamp(1, ts);
			st.setString(2, bean.getID());
			
			int i = st.executeUpdate();
			
			if (i != 1) {
				// vZXԂ̍XVɎs܂B
				String errMsg = StringManager.get("E0021");
				throw new DAOException(errMsg);
			}
		} finally {
			st.close();
		}

		try {
			ProcessState old = ProcessState.getState(bean.getProcessState());
			WorkflowDAOFactory factory = DataAccessManager.getDAOFactory();
			ProcessHistoryDAO phDAO = factory.getProcessHistoryDAO();
			ProcessHistoryBean history = new ProcessHistoryBean();
			history.setProcID(bean.getID());
			history.setPackageID(bean.getPackageID());
			history.setProcDefID(bean.getProcDefinitionID());
			history.setPrevState(bean.getProcessState());
			history.setNewState(old.toInt());
			history.setUpdateDate(bean.getUpdateDate());
			String msg = "PROCESS STATE NOTIFIED FROM " + old.toString();
			history.setReason(msg);
			phDAO.insert(history);
		} catch (SQLException ex) {
			throw ex;
		} catch (DAOException ex) {
			throw ex;
		} catch (Exception ex) {
			// vZXԗ̍XVɎs܂B
			String errMsg = StringManager.get("E0019");
			throw new DAOException(errMsg, ex);
		}
		bean.setUpdateDate(ts);
	}
	
	/**
	 * ʃZbgvZX쐬B
	 *
	 * @param rs ʃZbg
	 * @return vZX
	 * @throws SQLException CӂSQLO
	 */
	private ProcessBean createBean(ResultSet rs)
	throws SQLException
	{
		ProcessBean bean = new ProcessBean();

		bean.setID(rs.getString(PROC_ID));
		bean.setName(rs.getString(NAME));
		bean.setPackageID(rs.getString(PACKAGE_ID));
		bean.setProcDefinitionID(rs.getString(PROC_DEF_ID));
		bean.setStartDate(rs.getTimestamp(START_DATE));
		bean.setStarterID(rs.getString(STARTER_ID));
		bean.setDurationLimit(rs.getTimestamp(DURATION_LIMIT));
		bean.setProcessState(rs.getInt(PROC_STATE));
		bean.setUpdateDate(rs.getTimestamp(UPDATE_DATE));
		
		return bean;
	}
}
