/* $Id: DefinitionRepositoryImpl.java,v 1.30 2005/08/26 04:53:30 ysahara Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.engine;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NameAlreadyBoundException;
import javax.naming.NamingException;

import jp.co.argo21.nautica.workflow.dataaccess.DefinitionBean;
import jp.co.argo21.nautica.workflow.dataaccess.DefinitionDAO;
import jp.co.argo21.nautica.workflow.dataaccess.WorkflowDAOFactory;
import jp.co.argo21.nautica.workflow.definition.PackageDefinition;
import jp.co.argo21.nautica.workflow.definition.ProcessDefinition;
import jp.co.argo21.nautica.workflow.definition.impl.DefinitionLoader;
import jp.co.argo21.nautica.workflow.omg.WorkflowException;
import jp.co.argo21.nautica.workflow.rmi.RepositoryLoaderPO;
import jp.co.argo21.nautica.workflow.security.SessionManager;
import jp.co.argo21.nautica.workflow.util.StringManager;
import jp.co.argo21.nautica.workflow.wfmc.DefinitionRepository;
import jp.co.argo21.nautica.workflow.wfmc.Filter;
import jp.co.argo21.nautica.workflow.wfmc.InvalidFilterException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidProcessDefinitionException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidSessionException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidStateException;
import jp.co.argo21.nautica.workflow.wfmc.ProcDef;
import jp.co.argo21.nautica.workflow.wfmc.ProcessDefinitionState;
import jp.co.argo21.nautica.workflow.wfmc.TransitionNotAllowedException;

import org.apache.log4j.Logger;

/**
 * [JJavaVMœ삷`|WgNXłB
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.30 $
 * @since   Nautica Workflow 0.9
 */
public class DefinitionRepositoryImpl 
implements DefinitionRepository, RepositoryLoader
{
	/** `|WgpX */
	static private final String REPOSITORY_DIR = "repository";
	
	/** |Wg[_̌J */
	static private final String WORKFLOW_REPOSITORY_LOADER = "RepositoryLoader";

	/** GWO */
	static private Logger eLog = LogManager.getEngineLogger();
	
	/** čO */
	static private Logger audit = LogManager.getAuditLogger();

	/** `[_ */
	private DefinitionLoader loader;

	/** `LbV */
	private DefinitionCache cache;
	
	/**
	 * |WgpX擾B
	 *
	 * @return |WgpX
	 * @throws WorkflowException 擾ɗONꍇ
	 */
	static private String getRepositoryPath()
	throws WorkflowException
	{
		String home = System.getProperty(WorkflowEngineConstants.NAUTICA_HOME);
		String separator = System.getProperty("file.separator");
		String path = home + separator + REPOSITORY_DIR;
		File confFile = new File(path);
	
		if (! confFile.exists()) {
			// `|WgpX݂܂B
			String msg = StringManager.get("E0052") + "(path=" + path + ")";
			eLog.error(msg);
			throw new WorkflowException(msg);
		}
		if (! confFile.isDirectory()) {
			// `|WgpX̓fBNgł͂܂B
			String msg = StringManager.get("E0053") + "(path=" + path + ")";
			eLog.error(msg);
			throw new WorkflowException(msg);
		}
		return path;
	}

	/**
	 * DefinitionRepositoryImpl𐶐B
	 */
	DefinitionRepositoryImpl()
	throws WorkflowException
	{
		loader = new DefinitionLoader(getRepositoryPath());
		cache = new DefinitionCache();
		
		// |Wg[_J
		try {
		    String scope = System.getProperty(WorkflowEngineConstants.NAUTICA_SCOPE);
			if (scope == null) scope = "";
			scope = scope.trim();
			
			Context context = new InitialContext();
			try {
				context.createSubcontext(scope);
			} catch (NameAlreadyBoundException ex) { /* NOP */ }

			if (scope.equals("") == false) scope += "/";
		    RepositoryLoaderPO rlPO = new RepositoryLoaderPO(this);
			context.rebind(scope + WORKFLOW_REPOSITORY_LOADER, rlPO);
			eLog.info(StringManager.get("I0045")
			        + "(" + scope + WORKFLOW_REPOSITORY_LOADER + ")");
			
		} catch (Exception e) {
		    String errMsg = StringManager.get("E0072");
		    eLog.error(errMsg);
		    throw new WorkflowException(errMsg, e);
		}
	}


	/**
	 * w肳ꂽ`̓o^sAGWɔFB
	 *
	 * @param session ZbVID
	 * @param pdid [Nt[`ID
	 * @param path [Nt[`̑΃pX
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws WorkflowException vZXCX^XIDLłȂꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.DefinitionRepository#registerDefinition(java.lang.String, java.lang.String, java.lang.String)
	 */
	public void registerDefinition(String session, String pdid, String path)
	throws InvalidSessionException, WorkflowException
	{
		String argInfo = "(NAUTICA API = registerDefinition,"
		 + "session = [" + session + "],"
		 + "pdid = [" + pdid + "],"
		 + "path = [" + path + "])";

		PackageDefinition packageDefinition = null;
		try {
			DataAccessManager.begin(false);
				
			validateSession(session);
				
			if (pdid == null || pdid.trim().equals("")) {
				// [Nt[`ݒ肳Ă܂B
				String E0105 = StringManager.get("E0105");
				throw new WorkflowException(E0105);
			}

			if (path == null || path.trim().equals("")) {
				// [Nt[`̑΃pXݒ肳Ă܂B
				String E0106 = StringManager.get("E0106");
				throw new WorkflowException(E0106);
			}
		
			packageDefinition = loader.load(path);
	
			WorkflowDAOFactory daoFactory
				= DataAccessManager.getDAOFactory();
			DefinitionDAO dao = daoFactory.getDefinitionDAO();
	
			Iterator it = packageDefinition.getProcesses().iterator();
			for (int i = 0; it.hasNext(); i++) {
				ProcessDefinition def = (ProcessDefinition)it.next();
				String ipdid = def.getID();
				
				DefinitionBean bean = new DefinitionBean();
				bean.setProcessDefinitionID(pdid + "-" + i);
				bean.setPath(path);
				bean.setInternalPDID(ipdid);
				bean.setRegisterDate(new Timestamp(System.currentTimeMillis()));
				bean.setState(ProcessDefinitionState.ENABLE.toInt());
							
				dao.insert(bean);
			}
				
			DataAccessManager.commit();
	
		} catch (InvalidSessionException ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			String msg = ex.getMessage() + argInfo;
			eLog.error(msg, ex);
			throw ex;
		} catch (WorkflowException ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			String msg = ex.getMessage() + argInfo;
			eLog.error(msg, ex);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// `̓o^ŃG[N܂B
			String E0107 = StringManager.get("E0107") + argInfo;
			eLog.error(E0107, ex);
			throw new WorkflowException(E0107, ex);
		}
			
	}

	/**
	 * Ƀ}b`vZX`ꗗԂB
	 *
	 * @param session ZbVID
	 * @param filter ptB^[
	 * @return ꂽvZX`Tv̔z
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidFilterException w肳ꂽtB^[LłȂꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.DefinitionRepository#getProcessDefinitions(java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	public ProcDef[] getProcessDefinitions(
			String session, Filter filter)
	throws InvalidSessionException, InvalidFilterException
	{		
		String argInfo = "(WfMC API = getProcessDefinitions,"
			 + "session = [" + session + "],"
			 + "filter = [" + filter + "])";

		try {
			DataAccessManager.begin(false);
			
			validateSession(session);

			WorkflowDAOFactory daoFactory
				= DataAccessManager.getDAOFactory();
			DefinitionDAO dao = daoFactory.getDefinitionDAO();
			List list = dao.findByFilter(filter);

			List procDefs = new ArrayList();
			Iterator it = list.iterator();
			for (int i = 0; it.hasNext(); i++) {
				DefinitionBean bean = (DefinitionBean)it.next();
				ProcessDefinition def = cache.getProcessDefinition(bean.getProcessDefinitionID());
				ProcDefImpl impl = new ProcDefImpl();
				impl.setID(bean.getProcessDefinitionID());
				impl.setName(def.getName());
				impl.setDescription(def.getDescription());
				impl.setState(ProcessDefinitionState.getState(bean.getState()));
				procDefs.add(impl);
			}
			
			DataAccessManager.commit();

			return (ProcDef[])procDefs.toArray(new ProcDef[procDefs.size()]);
		} catch (InvalidSessionException ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			String msg = ex.getMessage() + argInfo;
			eLog.error(msg, ex);
			throw ex;
		} catch (InvalidFilterException ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			String msg = ex.getMessage() + argInfo;
			eLog.error(msg, ex);
			throw ex;
		} catch (WorkflowException ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// vZX`ꗗ̎擾ŃG[N܂B
			String E0108 = StringManager.get("E0108") + argInfo;
			eLog.error(E0108, ex);
			throw new InvalidFilterException(E0108, ex);
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// vZX`ꗗ̎擾ŃG[N܂B
			String E0108 = StringManager.get("E0108") + argInfo;
			eLog.error(E0108, ex);
			throw new InvalidFilterException(E0108, ex);
		}	
	}

	/**
	 * vZX`IDœ肳ꂽ`݂̌̏ԂJډ\
	 * vZX`ԈꗗԂB
	 *
	 * @param session ZbVID
	 * @param pdid vZX`ID
	 * @return vZX`Ԃ̔z
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidProcessDefinitionException w肳ꂽvZX`IDLłȂꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.DefinitionRepository#getProcessDefinitionStates(java.lang.String, java.lang.String)
	 */
	public ProcessDefinitionState[] getProcessDefinitionStates(
			String session,String pdid)
	throws InvalidSessionException, InvalidProcessDefinitionException
	{
		String argInfo = "(WfMC API = getProcessDefinitionStates,"
			 + "session = [" + session + "],"
			 + "pdid = [" + pdid + "])";
	
		PackageDefinition packageDefinition = null;
	
		try {
			DataAccessManager.begin(false);
			
			validateSession(session);

			checkProcessDefinitionID(pdid);
	
			DefinitionBean bean = cache.getDefinitionBean(pdid);
				
			ProcessDefinitionState[] states = new ProcessDefinitionState[1];
				
			ProcessDefinitionState currentState
				= ProcessDefinitionState.getState(bean.getState());
				
			if (ProcessDefinitionState.ENABLE.equals(currentState)) {
				states[0] = ProcessDefinitionState.DISABLE;
			} else {
				states[0] = ProcessDefinitionState.ENABLE;
			}
				
			DataAccessManager.commit();
	
			return states;
		} catch (InvalidSessionException ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			String msg = ex.getMessage() + argInfo;
			eLog.error(msg, ex);
			throw ex;
		} catch (InvalidProcessDefinitionException ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			String msg = ex.getMessage() + argInfo;
			eLog.error(msg, ex);
			throw ex;
		} catch (WorkflowException ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// vZX`Ԃ̎擾ŃG[N܂B
			String E0109 = StringManager.get("E0109") + argInfo;
			eLog.error(E0109, ex);
			throw new InvalidProcessDefinitionException(E0109, ex);
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// vZX`Ԃ̎擾ŃG[N܂B
			String E0109 = StringManager.get("E0109") + argInfo;
			eLog.error(E0109, ex);
			throw new InvalidProcessDefinitionException(E0109, ex);
		}	
	}

	/**
	 * vZX`ԂύXB
	 *
	 * @param session ZbVID
	 * @param pdid vZX`ID
	 * @param state
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidProcessDefinitionException w肳ꂽvZX`IDLłȂꍇ
	 * @throws InvalidStateException w肳ꂽԂLłȂꍇ
	 * @throws TransitionNotAllowedException Ԃ̕ύXĂȂꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.DefinitionRepository#changeProcessDefinitionState(java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.ProcessDefinitionState)
	 */
	public void changeProcessDefinitionState(String session,
		String pdid, ProcessDefinitionState state)
	throws InvalidSessionException, InvalidProcessDefinitionException,
		InvalidStateException, TransitionNotAllowedException
	{
		String argInfo = "(WfMC API = changeProcessDefinitionState,"
			 + "session = [" + session + "],"
			 + "pdid = [" + pdid + "],"
			 + "state = [" + state.toString() + "])";
	
		try {
			DataAccessManager.begin(false);
			
			validateSession(session);

			checkProcessDefinitionID(pdid);
		
			if (state == null) {
				// vZX`Ԃݒ肳Ă܂B
				String E0110 = StringManager.get("E0110");
				throw new InvalidStateException(E0110);
			}

			setProcessDefinitionState(pdid, state);
				
			DataAccessManager.commit();
			
			// čOo
		    AuditDataRecord data = new AuditDataRecord(
		            new AuditPrefix("", "WMChangedProcessDifinitionState", session),
		            new AuditSuffix());
		    data.setDataElement("ProcessDefinitionID", pdid);
		    data.setDataElement("NewProcessDefinitionState", state);		    
			audit.info(data);
	
		} catch (InvalidSessionException ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			String msg = ex.getMessage() + argInfo;
			eLog.error(msg, ex);
			throw ex;
		} catch (InvalidProcessDefinitionException ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			String msg = ex.getMessage() + argInfo;
			eLog.error(msg, ex);
			throw ex;
		} catch (InvalidStateException ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			String msg = ex.getMessage() + argInfo;
			eLog.error(msg, ex);
			throw ex;
		} catch (TransitionNotAllowedException ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			String msg = ex.getMessage() + argInfo;
			eLog.error(msg, ex);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// vZX`Ԃ̕ύXɎs܂B
			String E0186 = StringManager.get("E0186") + argInfo;
			eLog.error(E0186, ex);
			throw new TransitionNotAllowedException(E0186, ex);
		}
	}
	
	/**
     * w肳ꂽpX̒`t@C𕶎ŕԂB
     * Ȃ null ԂB
     * 炩̗Oꍇ null ԂB
     * 
     * @param session      ZbVID
     * @param adminKey ǗppX[h
     * @param path     pX
     * @return `t@C̕
     */
	public String getDefinition(String session, String adminKey, String path) 
	{
		String argInfo = "(NAUTICA API = getDefinition,"
			 + "session = [" + session + "],"
			 + "adminKey = [" + adminKey + "],"
			 + "path = [" + path + "])";
	
	    String separator = System.getProperty("file.separator");
	    String filepath = System.getProperty(WorkflowEngineConstants.NAUTICA_HOME) + separator
                        + REPOSITORY_DIR + separator
                        + path;
	    
	    BufferedReader br = null;
	    try {	        
	        validateSession(session);
		    SystemChecker.checkAdmin(adminKey);
	        
		    // t@C擾AǂݍŕԂ
		    String lineseparator = System.getProperty("line.separator");
		    br = new BufferedReader(
	    	        new InputStreamReader(new FileInputStream(filepath), "utf-8"));
	        StringBuffer definition = new StringBuffer();
	        while (br.ready()) {
		        definition.append(br.readLine());
		        definition.append(lineseparator);
		    }
		    return definition.toString();
	        
	    } catch (WorkflowException ex) {
			String W0000 = ex.getMessage() + argInfo;
			eLog.warn(W0000, ex);
	        return null;
	    } catch (IOException ex) {
			// `t@CɗOߑ܂Ap܂B
			String W0003 = StringManager.get("W0003") + argInfo;
			eLog.warn(W0003, ex);
	        return null;
	    } finally {
	        if (br != null) {
	            try { br.close(); } catch (IOException e) { /* ignore */ }
	        }
	    }
	}
	
	/**
     * w肳ꂽvZX`ɑΉpXԂB
     * Ȃ null ԂB
     * 炩̗Oꍇ null ԂB
     * 
     * @param session    ZbVID
     * @param pdname vZX`
     * @return pX
	 */
	public String getDefinitionPath(String session, String pdname)
	{
		String argInfo = "(NAUTICA API = getDefinitionPath,"
			 + "session = [" + session + "],"
			 + "pdname = [" + pdname + "])";
	
	    try {
	        validateSession(session);
	        
	        WorkflowDAOFactory daoFactory = DataAccessManager.getDAOFactory();
			DefinitionDAO defDAO = daoFactory.getDefinitionDAO();
			DefinitionBean defbean = defDAO.findByPrimaryKey(pdname);
			
			if (defbean != null) {
			    return defbean.getPath();
			}
			
			return null;

	    } catch (WorkflowException ex) {
			String W0000 = ex.getMessage() + argInfo;
			eLog.warn(W0000, ex);
	        return null;
	    } catch (Exception ex) {
			// `t@CpXɗOߑ܂Ap܂B
			String W0004 = StringManager.get("W0004") + argInfo;
			eLog.warn(W0004, ex);
	        return null;
	    }
	}
	
	/**
	 * |Wg[_AoChB
	 * 
	 * @throws NamingException l[~OOꍇ
	 */
	public void unbind() throws NamingException
	{
	    String scope = System.getProperty(WorkflowEngineConstants.NAUTICA_SCOPE);
		if (scope == null) scope = "";
		scope = scope.trim();

		Context context = new InitialContext();

		if (scope.equals("") == false) scope += "/";
		context.unbind(scope + WORKFLOW_REPOSITORY_LOADER);
		eLog.info(StringManager.get("I0046"));
	}
	
	/**
 	 * X[p[oCU[pBpbP[W`擾B
	 *
	 * @param pdid vZX`ID
	 * @return pbP[W`
	 * @throws WorkflowException 擾ɗONꍇ
	 */
	PackageDefinition getPackageDefinition(String pdid)
	throws WorkflowException
	{
		return cache.getPackageDefinition(pdid);
	}
	
	/**
 	 * X[p[oCU[pBvZX`擾B
	 *
	 * @param pdid vZX`ID
	 * @return vZX`
	 * @throws WorkflowException 擾ɗONꍇ
	 */
	ProcessDefinition getProcessDefinition(String pdid)
	throws WorkflowException
	{
		return cache.getProcessDefinition(pdid);
	}
	
	/**
 	 * X[p[oCU[pBvZX`Ԃ擾B
	 *
	 * @param pdid vZX`ID
	 * @return vZX`
	 * @throws WorkflowException 擾ɗONꍇ
	 */
	ProcessDefinitionState getProcessDefinitionState(String pdid)
	throws WorkflowException
	{
		try {
			DefinitionBean bean = cache.getDefinitionBean(pdid);
			int stateNum = bean.getState();
			return ProcessDefinitionState.getState(stateNum);
		} catch (Exception ex) {
			// vZX`Ԃ̎擾Ɏs܂B
			String E0187 = StringManager.get("E0187");
			throw new WorkflowException(E0187, ex);
		}
	}

	/**
	 * X[p[oCU[pBvZX`Ԃݒ肷B
	 *
	 * @param pdid vZX`ID
	 * @param state vZX`
	 * @throws InvalidProcessDefinitionException w肳ꂽvZX`IDLłȂꍇ
	 * @throws InvalidStateException w肳ꂽԂLłȂꍇ
	 * @throws TransitionNotAllowedException Ԃ̕ύXĂȂꍇ
	 */
	void setProcessDefinitionState(
		String pdid, ProcessDefinitionState state)
	throws InvalidProcessDefinitionException, InvalidStateException,
		TransitionNotAllowedException
	{
		try {
			ProcessDefinitionState currentState = getProcessDefinitionState(pdid);
			ProcessDefinitionState.checkTransition(currentState, state);
	
			WorkflowDAOFactory daoFactory
				= DataAccessManager.getDAOFactory();
			DefinitionDAO dao = daoFactory.getDefinitionDAO();
			dao.updateState(pdid, currentState.toInt(), state.toInt());
		} catch (InvalidProcessDefinitionException ex) {
			throw ex;
		} catch (InvalidStateException ex) {
			throw ex;
		} catch (TransitionNotAllowedException ex) {
			throw ex;
		} catch (Exception ex) {
			// vZX`Ԃ̕ύXɎs܂B
			String E0186 = StringManager.get("E0186");
			throw new TransitionNotAllowedException(E0186, ex);
		}		
	}
	
	/**
	 * vZX`ID̃`FbNsB
	 *
	 * @param pdid vZX`ID
	 * @throws InvalidProcessDefinitionException `FbNɎsꍇ
	 */
	private void checkProcessDefinitionID(String pdid)
	throws InvalidProcessDefinitionException
	{
		if (pdid == null || pdid.trim().equals("")) {
			// vZX`IDݒłB
			String E0188 = StringManager.get("E0188");
			throw new InvalidProcessDefinitionException(E0188);
		}
	}
	
	/**
	 * ZbV؂B
	 *
	 * @param session ZbVID
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 */
	private void validateSession(String session)
	throws InvalidSessionException
	{
		if (session == null || session.trim().equals("")) {
			// ZbVIDݒłB
			String E0170 = StringManager.get("E0170");
			throw new InvalidSessionException(E0170);
		}

		SessionManagerFactory factory = SessionManagerFactory.getInstance();
		SessionManager manager = factory.getSessionManager();
		manager.validateSession(session);
	}
}
