/* $Id: WorkItemHandlerImpl.java,v 1.22 2005/08/26 08:28:43 mmanabe Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.engine;

import org.apache.log4j.Logger;

import jp.co.argo21.nautica.workflow.security.SessionManager;
import jp.co.argo21.nautica.workflow.util.StringManager;
import jp.co.argo21.nautica.workflow.wfmc.Attribute;
import jp.co.argo21.nautica.workflow.wfmc.AttributeAssignmentFailedException;
import jp.co.argo21.nautica.workflow.wfmc.Filter;
import jp.co.argo21.nautica.workflow.wfmc.InvalidAttributeException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidFilterException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidSessionException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidSourceUserException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidStateException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidTargetUserException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidWorkItemException;
import jp.co.argo21.nautica.workflow.wfmc.Participant;
import jp.co.argo21.nautica.workflow.wfmc.TransitionNotAllowedException;
import jp.co.argo21.nautica.workflow.wfmc.WorkItem;
import jp.co.argo21.nautica.workflow.wfmc.WorkItemHandler;
import jp.co.argo21.nautica.workflow.wfmc.WorkItemState;

/**
 * ƍڃnh́Aƍڑɑ΂NCAgC^tF[X
 * ̎łB
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.22 $
 * @since   Nautica Workflow 0.9
 */
public class WorkItemHandlerImpl implements WorkItemHandler
{
	/** GWO */
	static private Logger eLog = LogManager.getEngineLogger();
	/** čO */
	static private Logger audit = LogManager.getAuditLogger();

	/**
	 * WorkItemHandlerImpl𐶐B
	 */
	WorkItemHandlerImpl()
	{
	}

	/**
	 * ƍڈꗗ̎擾sB
	 *
	 * @param session ZbVID
	 * @param filter ptB^[
	 * @return ƍڈꗗ
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidFilterException w肳ꂽtB^[LłȂꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkItemHandler#getWorkItems(java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	public WorkItem[] getWorkItems(String session, Filter filter)
	throws InvalidSessionException, InvalidFilterException
	{
		String argInfo = "(NAUTICA API = getWorkItems,"
			 + "session = [" + session + "],"
			 + "filter = [" + filter + "])";
	
		try {
			DataAccessManager.begin(false);
			
			validateSession(session);

			WorkItemPool pool = WorkItemPool.getInstance();
			WorkItemContainer container = pool.getWorkItemContainer(session);
			WorkItem[] workItems = container.getWorkItems(filter);
			
			DataAccessManager.commit();

			return workItems;
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidFilterException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// ƍڈꗗ̎擾Ɏs܂B
			String E0174 = StringManager.get("E0174") + argInfo;
			eLog.error(E0174, ex);
			throw new InvalidFilterException(E0174, ex);
		}
	}

	/**
	 * ƍڂ̎擾sB
	 * APIĂяoŁAYƍڂZbVɂăbNB
	 *
	 * @param session ZbVID
	 * @param wid ƍID
	 * @return ƍ
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidWorkItemException ƍIDLłȂꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkItemHandler#getWorkItem(java.lang.String, java.lang.String)
	 */
	public WorkItem getWorkItem(String session, String wid)
	throws InvalidSessionException, InvalidWorkItemException
	{
		String argInfo = "(NAUTICA API = getWorkItem,"
			 + "session = [" + session + "],"
			 + "work item id = [" + wid + "])";
	
		try {
			checkWorkItemID(wid);
			
			DataAccessManager.begin(false);
				
			validateSession(session);

			// čOo
		    AuditDataRecord data = new AuditDataRecord(
		            new AuditPrefix("", "WMSelectedWorkitem", session),
		            new AuditSuffix());
				
			WorkItemPool pool = WorkItemPool.getInstance();
			WorkItemContainer container = pool.getWorkItemContainer(session);
			WorkItem workItem = container.getWorkItem(wid);

			data.setDataElement("ActivityInstanceID", workItem.getActivityID());
				
			DataAccessManager.commit();

		    data.setDataElement("WorkItemID", wid);
		    audit.info(data);

			return workItem;
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidWorkItemException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// ƍڂ̎擾Ɏs܂B
			String E0116 = StringManager.get("E0116") + argInfo;
			eLog.error(E0116, ex);
			throw new InvalidWorkItemException(E0116, ex);
		}
	}

	/**
	 * ƍڂ̊sB
	 * APIĂяoŁAYƍڂZbVɂăAbNB
	 *
	 * @param session ZbVID
	 * @param wid ƍID
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidWorkItemException ƍIDLłȂꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkItemHandler#completeWorkItem(java.lang.String, java.lang.String)
	 */
	public void completeWorkItem(String session, String wid)
	throws InvalidSessionException, InvalidWorkItemException
	{
		String argInfo = "(NAUTICA API = completeWorkItem,"
			 + "session = [" + session + "],"
			 + "work item id = [" + wid + "])";
	
		try {
			checkWorkItemID(wid);
			
			DataAccessManager.begin(false);
				
			validateSession(session);
			// čOo
		    AuditDataRecord data = new AuditDataRecord(
		            new AuditPrefix("", "WMCompletedWorkitem", session),
		            new AuditSuffix());
				
			WorkItemPool pool = WorkItemPool.getInstance();
			WorkItemContainer container = pool.getWorkItemContainer(session);

			data.setDataElement("ActivityInstanceID", container.getWorkItem(wid).getActivityID());

			container.changeWorkItemState(session, wid, WorkItemState.CLOSED_COMPLETED);
				
			DataAccessManager.commit();

		    data.setDataElement("WorkItemID", wid);
		    data.setDataElement("WorkItemState", WorkItemState.CLOSED_COMPLETED);			    
		    audit.info(data);

		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidWorkItemException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// ƍڂ̊Ɏs܂B
			String E0175 = StringManager.get("E0175") + argInfo;
			eLog.error(E0175, ex);
			throw new InvalidWorkItemException(E0175, ex);
		}
	}

	/**
	 * ݂̍ƍڂJډ\ȏԂ̈ꗗԂB
	 * ԂȂꍇ́A̔zԂB
	 * p\ȃtB^[StateFilter݂̂ƂB
	 * API͊gꂽԂ߂̂̂łA
	 * ʏcompletê݂ԂB
	 *
	 * @param session ZbVID
	 * @param wid ƍID
	 * @param filter ptB^[
	 * @return ƍڏԈꗗ
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidWorkItemException ƍIDLłȂꍇ
	 * @throws InvalidFilterException w肳ꂽtB^[LłȂꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkItemHandler#getWorkItemStates(java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	public WorkItemState[] getWorkItemStates(String session, String wid, Filter filter)
	throws InvalidSessionException, InvalidWorkItemException,
		InvalidFilterException
	{
		String argInfo = "(NAUTICA API = getWorkItemStates,"
			 + "session = [" + session + "],"
			 + "work item id = [" + wid + "],"
			 + "filter = [" + filter + "])";
	
		try {
			checkWorkItemID(wid);
			
			DataAccessManager.begin(false);
				
			validateSession(session);
				
			WorkItemPool pool = WorkItemPool.getInstance();
			WorkItemContainer container = pool.getWorkItemContainer(session);
			WorkItemState[] states = container.getWorkItemStates(wid, filter);
				
			DataAccessManager.commit();
	
			return states;
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidWorkItemException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidFilterException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// ƍڏԂ̎擾Ɏs܂B
			String E0176 = StringManager.get("E0176") + argInfo;
			eLog.error(E0176, ex);
			throw new InvalidWorkItemException(E0176, ex);
		}
	}
	
	/**
	 * ƍڏԂ̕ύXsB
	 *
	 * @param session ZbVID
	 * @param wid ƍID
	 * @param state ƍڏ
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidWorkItemException ƍIDLłȂꍇ
	 * @throws InvalidStateException w肳ꂽԂLłȂꍇ
	 * @throws TransitionNotAllowedException Ԃ̕ύXĂȂꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkItemHandler#changeWorkItemState(java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.WorkItemState)
	 */
	public void changeWorkItemState(String session, String wid, WorkItemState state)
	throws InvalidSessionException, InvalidWorkItemException,
		InvalidStateException, TransitionNotAllowedException
	{
		String argInfo = "(NAUTICA API = changeWorkItemState,"
			 + "session = [" + session + "],"
			 + "work item id = [" + wid + "],"
			 + "state = [" + state + "])";
	
		try {
			checkWorkItemID(wid);
			
			if (state == null) {
				// ƍڏԂݒłB
				String E0191 = StringManager.get("E0191");
				throw new InvalidStateException(E0191);
			}
		
			DataAccessManager.begin(false);
		
			validateSession(session);
				
			WorkItemPool pool = WorkItemPool.getInstance();
			WorkItemContainer container = pool.getWorkItemContainer(session);
			String aid = container.getWorkItem(wid).getActivityID();
			container.changeWorkItemState(session, wid, state);
				
			DataAccessManager.commit();

			// čOo
		    AuditDataRecord data = new AuditDataRecord(
		            new AuditPrefix("", "WMChangedWorkitemState", session),
		            new AuditSuffix());
		    data.setDataElement("ActivityInstanceID", aid);
		    data.setDataElement("WorkItemID", wid);
		    data.setDataElement("WorkItemState", state);
		    audit.info(data);
	
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidWorkItemException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidStateException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (TransitionNotAllowedException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			// ƍڏԂ̕ύXɎs܂B
			String E0177 = StringManager.get("E0177") + argInfo;
			eLog.error(E0177, ex);
			throw new InvalidWorkItemException(E0177, ex);
		}
	}
	
	/**
	 * ƍڂ̍Ċ蓖ĂsB
	 *
	 * @param session ZbVID
	 * @param sourceUser 蓖đÕ[U
	 * @param targetUser 蓖Č̃[U
	 * @param wid ƍID
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidWorkItemException ƍIDLłȂꍇ
	 * @throws InvalidSourceUserException 蓖đO[ULłȂꍇ
	 * @throws InvalidTargetUserException 蓖Đ惆[ULłȂꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkItemHandler#reassignWorkItem(java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Participant, jp.co.argo21.nautica.workflow.wfmc.Participant, java.lang.String)
	 */
	public void reassignWorkItem(String session,
			Participant sourceUser, Participant targetUser, String wid)
	throws InvalidSessionException, InvalidWorkItemException,
	InvalidSourceUserException, InvalidTargetUserException
	{
		String argInfo = "(NAUTICA API = reassignWorkItem,"
			 + "session = [" + session + "],"
			 + "sourceUser = [" + sourceUser + "],"
			 + "targetUser = [" + targetUser + "],"
			 + "work item id = [" + wid + "])";
	
		try {
			if (sourceUser == null) {
				// 蓖đO[UݒłB
				String E0195 = StringManager.get("E0195");
				throw new InvalidSourceUserException(E0195);
			}

			if (targetUser == null) {
				// 蓖Čテ[UݒłB
				String E0196 = StringManager.get("E0196");
				throw new InvalidTargetUserException(E0196);
			}

			checkWorkItemID(wid);

			DataAccessManager.begin(false);
				
			validateSession(session);

			// čOo
		    AuditDataRecord data = new AuditDataRecord(
		            new AuditPrefix("", "WMReassignedWorkitem", session),
		            new AuditSuffix());
				
			WorkItemPool pool = WorkItemPool.getInstance();
			WorkItemContainer container = pool.getWorkItemContainer(session);

			data.setDataElement("ActivityInstanceID", container.getWorkItem(wid).getActivityID());

			container.reassignWorkItem(wid, sourceUser, targetUser);
				
			DataAccessManager.commit();

		    data.setDataElement("WorkItemID", wid);
		    data.setDataElement("TargetDomainID", "nautica");
		    data.setDataElement("TargetNodeID", System.getProperty(WorkflowEngineConstants.NAUTICA_SCOPE));
		    data.setDataElement("SourceUserID", sourceUser.getID());
		    data.setDataElement("TargetUserID", targetUser.getID());
		    audit.info(data);

		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidWorkItemException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidTargetUserException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// ƍڂ̊蓖ĂɎs܂B
			String E0178 = StringManager.get("E0178") + argInfo;
			eLog.error(E0178, ex);
			throw new InvalidWorkItemException(E0178, ex);
		}
	}
	
	/**
	 * ƍڑꗗ̎擾sB
	 * p\ȃtB^[AttributeFilter݂̂ƂB
	 *
	 * @param session ZbVID
	 * @param wid ƍID
	 * @param filter ptB^[
	 * @return ƍڑꗗ
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidWorkItemException ƍIDLłȂꍇ
	 * @throws InvalidFilterException w肳ꂽtB^[LłȂꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkItemHandler#getWorkItemAttributes(java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	public Attribute[] getWorkItemAttributes(String session, String wid, Filter filter)
	throws InvalidSessionException, InvalidWorkItemException,
		InvalidFilterException
	{
		String argInfo = "(NAUTICA API = getWorkItemAttributes,"
			 + "session = [" + session + "],"
			 + "work item id = [" + wid + "],"
			 + "filter = [" + filter + "])";
	
		try {
			checkWorkItemID(wid);

			DataAccessManager.begin(false);
				
			validateSession(session);
				
			WorkItemPool pool = WorkItemPool.getInstance();
			WorkItemContainer container = pool.getWorkItemContainer(session);
			Attribute[] attrs = container.getWorkItemAttributes(session, wid, filter);
				
			DataAccessManager.commit();
	
			return attrs;
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidWorkItemException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidFilterException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// ƍڑ̎擾Ɏs܂B
			String E0179 = StringManager.get("E0179") + argInfo;
			eLog.error(E0179, ex);
			throw new InvalidWorkItemException(E0179, ex);
		}
	}
	
	/**
	 * ƍڑ̎擾sB
	 *
	 * @param session ZbVID
	 * @param wid ƍID
	 * @param name 
	 * @return 
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidWorkItemException ƍIDLłȂꍇ
	 * @throws InvalidAttributeException LłȂꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkItemHandler#getWorkItemAttributeValue(java.lang.String, java.lang.String, java.lang.String)
	 */
	public Attribute getWorkItemAttributeValue(String session, String wid, String name)
	throws InvalidSessionException, InvalidWorkItemException,
		InvalidAttributeException
	{
		String argInfo = "(NAUTICA API = getWorkItemAttributeValue,"
			 + "session = [" + session + "],"
			 + "work item id = [" + wid + "],"
			 + "attr name = [" + name + "])";
	
		try {
			checkWorkItemID(wid);
			
			if (name == null || name.trim().equals("")) {
				// ƍڑݒłB
				String E0194 = StringManager.get("E0194");
				throw new InvalidAttributeException(E0194);
			}
			
			DataAccessManager.begin(false);
				
			validateSession(session);
			
			WorkItemPool pool = WorkItemPool.getInstance();
			WorkItemContainer container = pool.getWorkItemContainer(session);
				
			DataAccessManager.commit();
	
			return container.getWorkItemAttributeValue(wid, name);
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidWorkItemException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidAttributeException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// ƍڑ̎擾Ɏs܂B
			String E0179 = StringManager.get("E0179") + argInfo;
			eLog.error(E0179, ex);
			throw new InvalidAttributeException(E0179, ex);
		}
	}

	/**
	 * ƍڑ̊蓖ĂsB
	 *
	 * @param session ZbVID
	 * @param wid ƍID
	 * @param attr 
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidWorkItemException ƍIDLłȂꍇ
	 * @throws InvalidAttributeException LłȂꍇ
	 * @throws AttributeAssignmentFailedException ̊蓖ĂɎsꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkItemHandler#assignWorkItemAttribute(java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Attribute)
	 */
	public void assignWorkItemAttribute(String session, String wid, Attribute attr)
	throws InvalidSessionException, InvalidWorkItemException,
		InvalidAttributeException, AttributeAssignmentFailedException
	{
		String argInfo = "(NAUTICA API = assignWorkItemAttribute,"
			 + "session = [" + session + "],"
			 + "work item id = [" + wid + "],"
			 + "attr = [" + attr + "])";
	
		try {
			checkWorkItemID(wid);
			
			if (attr == null) {
				// ƍڑݒłB
				String E0193 = StringManager.get("E0193");
				throw new InvalidAttributeException(E0193);
			}
			
			DataAccessManager.begin(false);
				
			validateSession(session);

			// čOo
		    AuditDataRecord data = new AuditDataRecord(
		            new AuditPrefix("", "WMAssignedWorkitemAttributes", session),
		            new AuditSuffix());
				
			WorkItemPool pool = WorkItemPool.getInstance();
			WorkItemContainer container = pool.getWorkItemContainer(session);
			container.assignWorkItemAttribute(session, wid, attr);

			data.setDataElement("ActivityInstanceID", container.getWorkItem(wid).getActivityID());
				
			DataAccessManager.commit();

			data.setDataElement("WorkItemID", wid);
		    data.setDataElement("AttributeName", attr.getName());
		    data.setDataElement("AttributeType", attr.getType());
		    data.setDataElement("NewAttributeValue", attr.getValue());
		    audit.info(data);
	
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidWorkItemException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidAttributeException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (AttributeAssignmentFailedException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// ƍڑ̊蓖ĂɎs܂B
			String E0180 = StringManager.get("E0180") + argInfo;
			eLog.error(E0180, ex);
			throw new InvalidWorkItemException(E0180, ex);
		}
	}
	
	/**
	 * [obNƃG[Ȍo͂sB
	 *
	 * @param ex O
	 * @param argInfo 
	 */
	private void rollbackAndLogging(Exception ex, String argInfo)
	{
		try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
		String msg = ex.getMessage() + argInfo;
		eLog.error(msg, ex);
	}
	
	/**
	 * ƍID؂B
	 *
	 * @param wid ƍID
	 * @throws InvalidWorkItemException ȍƍID̏ꍇ
	 */
	private void checkWorkItemID(String wid)
	throws InvalidWorkItemException
	{
		if (wid == null || wid.trim().equals("")) {
			// ƍIDݒłB
			String E0192 = StringManager.get("E0192");
			throw new InvalidWorkItemException(E0192);
		}
	}

	/**
	 * ZbV؂B
	 *
	 * @param session ZbVID
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 */
	private void validateSession(String session)
	throws InvalidSessionException
	{
		if (session == null || session.trim().equals("")) {
			// ZbVIDݒłB
			String E0170 = StringManager.get("E0170");
			throw new InvalidSessionException(E0170);
		}

		SessionManagerFactory factory = SessionManagerFactory.getInstance();
		SessionManager manager = factory.getSessionManager();
		manager.validateSession(session);
	}
}
