/* $Id: WorkflowEngineHandlerImpl.java,v 1.34 2005/08/26 04:53:06 ysahara Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.engine;

import java.util.Iterator;
import java.util.List;

import org.apache.log4j.Logger;

import jp.co.argo21.nautica.workflow.dataaccess.ProcessBean;
import jp.co.argo21.nautica.workflow.dataaccess.ProcessDAO;
import jp.co.argo21.nautica.workflow.dataaccess.WorkflowDAOFactory;
import jp.co.argo21.nautica.workflow.definition.ActivityDefinition;
import jp.co.argo21.nautica.workflow.definition.ProcessDefinition;
import jp.co.argo21.nautica.workflow.filter.StateFilter;
import jp.co.argo21.nautica.workflow.jms.WorkflowMessage;
import jp.co.argo21.nautica.workflow.omg.AlreadyRunningException;
import jp.co.argo21.nautica.workflow.omg.NotRunningException;
import jp.co.argo21.nautica.workflow.omg.WfActivity;
import jp.co.argo21.nautica.workflow.omg.WfProcess;
import jp.co.argo21.nautica.workflow.omg.WfProcessManager;
import jp.co.argo21.nautica.workflow.omg.WfRequester;
import jp.co.argo21.nautica.workflow.omg.WfState;
import jp.co.argo21.nautica.workflow.omg.WorkflowException;
import jp.co.argo21.nautica.workflow.security.SessionManager;
import jp.co.argo21.nautica.workflow.util.StateHelper;
import jp.co.argo21.nautica.workflow.util.StringManager;
import jp.co.argo21.nautica.workflow.wfmc.Activity;
import jp.co.argo21.nautica.workflow.wfmc.ActivityState;
import jp.co.argo21.nautica.workflow.wfmc.Attribute;
import jp.co.argo21.nautica.workflow.wfmc.AttributeAssignmentFailedException;
import jp.co.argo21.nautica.workflow.wfmc.Filter;
import jp.co.argo21.nautica.workflow.wfmc.InvalidActivityInstanceException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidAttributeException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidFilterException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidProcessDefinitionException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidProcessInstanceException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidSessionException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidStateException;
import jp.co.argo21.nautica.workflow.wfmc.Process;
import jp.co.argo21.nautica.workflow.wfmc.ProcessState;
import jp.co.argo21.nautica.workflow.wfmc.TransitionNotAllowedException;
import jp.co.argo21.nautica.workflow.wfmc.WorkflowEngineHandler;

/**
 * WorkflowEngineHandlerImpĺA[Nt[GWɑ΂
 * NCAgC^tF[X̋̎NXłB
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.34 $
 * @since   Nautica Workflow 0.9
 */
public class WorkflowEngineHandlerImpl implements WorkflowEngineHandler
{
	/** GWO */
	static private Logger eLog = LogManager.getEngineLogger();
	
	/** čO */
	static private Logger audit = LogManager.getAuditLogger();

	/**
	 * WorkflowEngineHandlerImpl𐶐B
	 */
	WorkflowEngineHandlerImpl()
	{
	}

	/**
	 * vZXCX^X̐sB
	 *
	 * @param session ZbVID
	 * @param pdid vZX`ID
	 * @param name vZXCX^X
	 * @return vZXCX^XID
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidProcessDefinitionException w肳ꂽvZX`IDLłȂꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkflowEngineHandler#createProcess(java.lang.String, java.lang.String, java.lang.String)
	 */
	public String createProcess(String session, String pdid, String name)
	throws InvalidSessionException, InvalidProcessDefinitionException
	{
		String argInfo = "(NAUTICA API = createProcess,"
			 + "session = [" + session + "],"
			 + "pdid = [" + pdid + "],"
			 + "name = [" + name + "])";
	
		try {
			if (pdid == null || pdid.trim().equals("")) {
				// vZX`IDݒłB
				String E0188 = StringManager.get("E0188");
				throw new InvalidProcessDefinitionException(E0188);
			}
		
			DataAccessManager.begin(false);
				
			validateSession(session);
	
			RequesterManager rqm = RequesterManager.getInstance();
			WfRequester requester = rqm.getRequesterBySession(session);
			ProcessManagerFactory pmf = ProcessManagerFactory.getInstance();
			WfProcessManager pm = pmf.getProcessManager(pdid);
			String pid = null;
			synchronized (this) {
				WfProcess process = pm.createProcess(requester, name);
				pid = process.getID(); 
			}
			
			DataAccessManager.commit();
				
			// čOo
		    AuditDataRecord data = new AuditDataRecord(
		            new AuditPrefix("", "WMCreatedProcessInstance", session),
		            new AuditSuffix());
		    data.setDataElement("ProcessDefinitionID", pdid);
		    data.setDataElement("ProcessDefinitionBusinessName", name);		    
		    audit.info(data);
	
			return pid;
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidProcessDefinitionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// vZX̐Ɏs܂B
			String E0153 = StringManager.get("E0153") + argInfo;
			eLog.error(E0153, ex);
			throw new InvalidProcessDefinitionException(E0153, ex);
		}
	}

	/**
	 * vZXCX^X̊JnsB
	 * JnꍇɁAvZXCX^XIDXV悤
	 * \łB
	 *
	 * @param session ZbVID
	 * @param pid vZXCX^XID
	 * @return XVꂽvZXCX^XID
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidProcessInstanceException vZXCX^XIDLłȂꍇ
	 * @throws InvalidAttributeException LłȂꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkflowEngineHandler#startProcess(java.lang.String, java.lang.String)
	 */
	public String startProcess(String session, String pid)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidAttributeException
	{
		String argInfo = "(NAUTICA API = startProcess,"
			 + "session = [" + session + "],"
			 + "pid = [" + pid + "])";
	
		try {
			checkProcessID(pid);
			
			DataAccessManager.begin(false);
				
			validateSession(session);

			//vZX̑݃`FbN
			WorkflowDAOFactory daoFactory = DataAccessManager.getDAOFactory();
			ProcessDAO procDAO = daoFactory.getProcessDAO();
			ProcessBean bean = procDAO.findByPrimaryKey(pid);
			if (bean == null) {
				// vZX݂܂B
				String E0164 = StringManager.get("E0164");
				throw new InvalidProcessInstanceException(E0164);
			}
			ProcessState state = ProcessState.getState(bean.getProcessState());
			if (ProcessState.CLOSED_COMPLETED.equals(state)
					|| ProcessState.CLOSED_TERMINATED.equals(state)
					|| ProcessState.CLOSED_ABORTED.equals(state)) {
				// vZX͂łɏIĂ܂B
				String E0128 = StringManager.get("E0128");
				throw new NotRunningException(E0128);
			} else if (ProcessState.OPEN_NOT_RUNNING_NOT_STARTED.equals(state) == false) {
				// vZX͂łɊJnĂ܂B
				String E0129 = StringManager.get("E0129");
				throw new AlreadyRunningException(E0129);
			}
	
			//bZ[WL[ɓo^
			WorkflowMessage msg = new StartProcessRequestMessage(pid);
			WorkflowEngine engine = WorkflowEngine.getInstance();
			MessageTransporter mt = engine.getMessageTransporter(session, StartProcessRequestWatcher.NAME);
			mt.sendMessage(msg);
				
			DataAccessManager.commit();
				
			// čOo
		    AuditDataRecord data = new AuditDataRecord(
		            new AuditPrefix(pid, "WMStartedProcessInstance", session),
		            new AuditSuffix());
		    audit.info(data);
	
			return pid;
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidProcessInstanceException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// vZX̊JnɎs܂B
			String E0132 = StringManager.get("E0132") + argInfo;
			eLog.error(E0132, ex);
			throw new InvalidProcessInstanceException(E0132, ex);
		}
	}

	/**
	 * vZXCX^X̒~sB
	 *
	 * @param session ZbVID
	 * @param pid vZXCX^XID
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidProcessInstanceException vZXCX^XIDLłȂꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkflowEngineHandler#terminateProcess(java.lang.String, java.lang.String)
	 */
	public void terminateProcess(String session, String pid)
	throws InvalidSessionException, InvalidProcessInstanceException
	{
		String argInfo = "(NAUTICA API = terminateProcess,"
			 + "session = [" + session + "],"
			 + "pid = [" + pid + "])";
	
		try {
			checkProcessID(pid);
			
			DataAccessManager.begin(false);
				
			validateSession(session);
				
			ProcessManagerFactory pmf = ProcessManagerFactory.getInstance();
			ProcessManager pm = pmf.getOwnerProcessManager(pid);
			ProcessInternal proc = (ProcessInternal)pm.getProcess(pid);
	
			proc.terminate();
							
			DataAccessManager.commit();
				
			// čOo
		    AuditDataRecord data = new AuditDataRecord(
		            new AuditPrefix(pid, "WMTerminatedProcessInstance", session),
		            new AuditSuffix());
		    data.setDataElement("NewProcessState", ProcessState.CLOSED_TERMINATED);
		    audit.info(data);

		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidProcessInstanceException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// vZX̒~Ɏs܂B
			String E0136 = StringManager.get("E0136") + argInfo;
			eLog.error(E0136, ex);
			throw new InvalidProcessInstanceException(E0136, ex);
		}
	}

	/**
	 * w肳ꂽvZXCX^Xp\ȏԈꗗ̎擾sB
	 * p\ȃtB^[StateFilter݂̂ƂB
	 *
	 * @param session ZbVID
	 * @param pid vZXCX^XID
	 * @param filter ptB^[
	 * @return vZXCX^XԈꗗ
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidProcessInstanceException vZXCX^XIDLłȂꍇ
	 * @throws InvalidFilterException w肳ꂽtB^[LłȂꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkflowEngineHandler#getProcessStates(java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	public ProcessState[] getProcessStates(String session, 
			String pid, Filter filter)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidFilterException
	{
		String argInfo = "(NAUTICA API = getProcessStates,"
			 + "session = [" + session + "],"
			 + "pid = [" + pid + "],"
			 + "filter = [" + filter + "])";

		ProcessState state = null;
		try {
			checkProcessID(pid);
			
			DataAccessManager.begin(false);
				
			validateSession(session);
	
			//StateFilter݂̂ΏۂƂB
			if (filter != null) {
				if (filter instanceof StateFilter) {
					WfState st = ((StateFilter)filter).getState();
					if (st instanceof ProcessState) {
						state = (ProcessState)st;
					} else {
						//ݒ肳ĂԂ́AvZXԂł͂܂B
						String E0205 = StringManager.get("E0205");
						throw new InvalidFilterException(E0205);
					}
				} else {
					// ԃtB^[ȊO͎gpł܂B
					String E0184 = StringManager.get("E0184");
					throw new InvalidFilterException(E0184);
				}
			}
	
			ProcessManagerFactory pmf = ProcessManagerFactory.getInstance();
			WfProcessManager pm = pmf.getOwnerProcessManager(pid);
			WfProcess process = pm.getProcess(pid);
			ProcessState cstate = (ProcessState)process.getState();
			ProcessState[] tstates = null;
			if (state == null) {
				tstates = (ProcessState[])process.getTransitableStates();
			} else {
				if (StateHelper.containsTransitable(cstate, state)) {
					tstates = new ProcessState[1];
					tstates[0] = state;
				} else {
					tstates = new ProcessState[0];
				}
			}
			
			DataAccessManager.commit();
	
			return tstates;
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidProcessInstanceException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidFilterException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// vZXԂ̎擾Ɏs܂B
			String E0154 = StringManager.get("E0154") + argInfo;
			eLog.error(E0154, ex);
			throw new InvalidProcessInstanceException(E0154, ex);
		}
	} 

	/**
	 * vZXCX^XԂ̕ύXsB
	 *
	 * @param session ZbVID
	 * @param pid vZXCX^XID
	 * @param state vZXCX^X
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidProcessInstanceException vZXCX^XIDLłȂꍇ
	 * @throws InvalidStateException w肳ꂽԂLłȂꍇ
	 * @throws TransitionNotAllowedException Ԃ̕ύXĂȂꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkflowEngineHandler#changeProcessState(java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.ProcessState)
	 */
	public void changeProcessState(String session,
			String pid, ProcessState state)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidStateException, TransitionNotAllowedException
	{
		String argInfo = "(NAUTICA API = changeProcessState,"
			 + "session = [" + session + "],"
			 + "pid = [" + pid + "],"
			 + "state = [" + state + "])";

		try {
			DataAccessManager.begin(false);
				
			validateSession(session);

			checkProcessID(pid);
				
			if (state == null) {
				// vZX̏Ԃݒ肳Ă܂B
				String E0063 = StringManager.get("E0063");
				throw new InvalidStateException(E0063);
			} else if (ProcessState.OPEN_NOT_RUNNING_NOT_STARTED.equals(state)) {
				// vZX̏Ԃ𖢊Jnɂ邱Ƃ͂ł܂B
				String E0064 = StringManager.get("E0064");
				throw new InvalidStateException(E0064);
			} else if (ProcessState.CLOSED_COMPLETED.equals(state)) {
				// vZX̏ԂIɐI邱Ƃ͂ł܂B
				String E0065 = StringManager.get("E0065");
				throw new InvalidStateException(E0065);
			}

			ProcessManagerFactory pmf = ProcessManagerFactory.getInstance();
			WfProcessManager pm = pmf.getOwnerProcessManager(pid);
			WfProcess process = pm.getProcess(pid);
			process.changeState(state);
			
			DataAccessManager.commit();
				
			// čOo
		    AuditDataRecord data = new AuditDataRecord(
		            new AuditPrefix(pid, "WMChangedProcessInstanceState", session),
		            new AuditSuffix());
		    data.setDataElement("NewProcessState", state);		    
		    audit.info(data);
	
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidProcessInstanceException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidStateException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (TransitionNotAllowedException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// vZXԂ̕ύXɎs܂B
			String E0156 = StringManager.get("E0156") + argInfo;
			eLog.error(E0156, ex);
			throw new InvalidProcessInstanceException(E0156, ex);
		}
	}

	/**
	 * vZXCX^Xꗗ̎擾sB
	 * p\ȃtB^[NameFilter, AttributeFilter, GroupFilterƂB
	 *
	 * @param session ZbVID
	 * @param pid vZXCX^XID
	 * @param filter ptB^[
	 * @return ꗗ
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidProcessInstanceException vZXCX^XIDLłȂꍇ
	 * @throws InvalidFilterException w肳ꂽtB^[LłȂꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkflowEngineHandler#getProcessAttributes(java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	public Attribute[] getProcessAttributes(String session,
			String pid, Filter filter)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidFilterException
	{
		String argInfo = "(NAUTICA API = getProcessAttributes,"
			 + "session = [" + session + "],"
			 + "pid = [" + pid + "],"
			 + "filter = [" + filter + "])";

		try {
			checkProcessID(pid);
			
			DataAccessManager.begin(false);
			
			validateSession(session);
	
			ProcessManagerFactory pmf = ProcessManagerFactory.getInstance();
			WfProcessManager pm = pmf.getOwnerProcessManager(pid);
			ProcessInternal process = (ProcessInternal)pm.getProcess(pid);
			Attribute[] attrs = (Attribute[])process.getContext(filter);
				
			DataAccessManager.commit();
	
			return attrs;
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidProcessInstanceException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidFilterException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// vZXϐ̎擾Ɏs܂B
			String E0032 = StringManager.get("E0032") + argInfo;
			eLog.error(E0032, ex);
			throw new InvalidProcessInstanceException(E0032, ex);
		}
	}

	/**
	 * vZXCX^X̑l̎擾sB
	 *
	 * @param session ZbVID
	 * @param pid vZXCX^XID
	 * @param name 
	 * @return 
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidAttributeException LłȂꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkflowEngineHandler#getProcessAttributeValue(java.lang.String, java.lang.String, java.lang.String)
	 */
	public Attribute getProcessAttributeValue(String session,
			String pid, String name)
	throws InvalidSessionException, InvalidAttributeException
	{
		String argInfo = "(NAUTICA API = getProcessAttributeValue,"
			 + "session = [" + session + "],"
			 + "pid = [" + pid + "],"
			 + "name = [" + name + "])";

		try {
			
			if (pid == null || pid.trim().equals("")) {
				// vZXIDݒłB
				String E0189 = StringManager.get("E0189");
				throw new InvalidAttributeException(E0189);
			}
		
			if (name == null || name.trim().equals("")) {
				// vZXϐݒłB
				String E0198 = StringManager.get("E0198");
				throw new InvalidAttributeException(E0198);
			}

			DataAccessManager.begin(false);
		
			validateSession(session);
	
			ProcessManagerFactory pmf = ProcessManagerFactory.getInstance();
			WfProcessManager pm = pmf.getOwnerProcessManager(pid);
			WfProcess process = pm.getProcess(pid);
			Attribute attr = (Attribute)process.getContext(name);
				
			DataAccessManager.commit();
	
			return attr;
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidAttributeException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// vZXϐ̎擾Ɏs܂B
			String E0032 = StringManager.get("E0032") + argInfo;
			eLog.error(E0032, ex);
			throw new InvalidAttributeException(E0032, ex);
		}
	}

	/**
	 * vZXCX^X̊蓖ĂsB
	 *
	 * @param session ZbVID
	 * @param pid vZXCX^XID
	 * @param attr 
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidProcessInstanceException vZXCX^XIDLłȂꍇ
	 * @throws InvalidAttributeException LłȂꍇ
	 * @throws AttributeAssignmentFailedException ̊蓖ĂɎsꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkflowEngineHandler#assignProcessAttribute(java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Attribute)
	 */
	public void assignProcessAttribute(String session,
			String pid, Attribute attr)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidAttributeException, AttributeAssignmentFailedException
	{
		String argInfo = "(NAUTICA API = assignProcessAttribute,"
			 + "session = [" + session + "],"
			 + "pid = [" + pid + "],"
			 + "attr = [" + attr + "])";

		try {
			checkProcessID(pid);
			
			if (attr == null) {
				// vZXϐݒłB
				String E0131 = StringManager.get("E0131");
				throw new InvalidAttributeException(E0131);
			}
			
			DataAccessManager.begin(false);
				
			validateSession(session);
	
			ProcessManagerFactory pmf = ProcessManagerFactory.getInstance();
			WfProcessManager pm = pmf.getOwnerProcessManager(pid);
			WfProcess process = pm.getProcess(pid);
			process.setContext(attr);
				
			DataAccessManager.commit();
				
			// čOo
		    AuditDataRecord data = new AuditDataRecord(
		            new AuditPrefix(pid, "WMAssignedProcessInstanceAttributes", session),
		            new AuditSuffix());
		    data.setDataElement("AttributeName", attr.getName());
		    data.setDataElement("AttributeType", attr.getType());
		    data.setDataElement("NewAttributeValue", attr.getValue());
		    audit.info(data);
	
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidProcessInstanceException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidAttributeException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (AttributeAssignmentFailedException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// vZXϐ̐ݒɎs܂B
			String E0080 = StringManager.get("E0080") + argInfo;
			eLog.error(E0080, ex);
			throw new InvalidAttributeException(E0080, ex);
		}
	}

	/**
	 * w肳ꂽANeBreBCX^Xp\ȏԈꗗ̎擾sB
	 * p\ȃtB^[StateFilter݂̂ƂB
	 *
	 * @param session ZbVID
	 * @param pid vZXCX^XID
	 * @param aid ANeBreBCX^XID
	 * @param filter ptB^[
	 * @return ANeBreBCX^XԈꗗ
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidProcessInstanceException vZXCX^XIDLłȂꍇ
	 * @throws InvalidActivityInstanceException ANeBreBCX^XIDLłȂꍇ
	 * @throws InvalidFilterException w肳ꂽtB^[LłȂꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkflowEngineHandler#getActivityStates(java.lang.String, java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	public ActivityState[] getActivityStates(String session,
			String pid, String aid, Filter filter)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidActivityInstanceException, InvalidFilterException
	{
		String argInfo = "(NAUTICA API = getActivityStates,"
			 + "session = [" + session + "],"
			 + "pid = [" + pid + "],"
			 + "aid = [" + aid + "],"
			 + "filter = [" + filter + "])";

		ActivityState state = null;
		try {
			checkProcessID(pid);
			checkActivityID(aid);
		
			DataAccessManager.begin(false);
				
			validateSession(session);
	
			//StateFilter݂̂ΏۂƂB
			if (filter != null) {
				if (filter instanceof StateFilter) {
					WfState st = ((StateFilter)filter).getState();
					if (st instanceof ActivityState) {
						state = (ActivityState)st;
					} else {
						// ݒ肳ĂԂ́AANeBreBԂł͂܂B
						String E0206 = StringManager.get("E0206");
						throw new InvalidFilterException(E0206);
					}
				} else {
					// ԃtB^[ȊO͎gpł܂B
					String E0184 = StringManager.get("E0184");
					throw new InvalidFilterException(E0184);
				}
			}
	
			ProcessManagerFactory pmf = ProcessManagerFactory.getInstance();
			WfProcessManager pm = pmf.getOwnerProcessManager(pid);
			WfProcess process = pm.getProcess(pid);
			WfActivity activity = process.getStep(aid);
			if (activity == null) {
				// YANeBreB݂܂B
				String msg = StringManager.get("E0069");
				throw new InvalidActivityInstanceException(msg);
			}
			ActivityState cstate = (ActivityState)activity.getState();
			ActivityState[] tstates = null;
			if (state == null) {
				tstates = (ActivityState[])activity.getTransitableStates();
			} else {
				if (StateHelper.containsTransitable(cstate, state)) {
					tstates = new ActivityState[1];
					tstates[0] = state;
				} else {
					tstates = new ActivityState[0];
				}
			}
				
			DataAccessManager.commit();
	
			return tstates;
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidProcessInstanceException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidActivityInstanceException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidFilterException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// ANeBreBԂ̎擾Ɏs܂B
			String E0200 = StringManager.get("E0200") + argInfo;
			eLog.error(E0200, ex);
			throw new InvalidActivityInstanceException(E0200, ex);
		}
	}

	/**
	 * ANeBreBCX^XԂ̕ύXsB
	 *
	 * @param session ZbVID
	 * @param pid vZXCX^XID
	 * @param aid ANeBreBCX^XID
	 * @param state ANeBreBCX^X
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidProcessInstanceException vZXCX^XIDLłȂꍇ
	 * @throws InvalidActivityInstanceException ANeBreBCX^XIDLłȂꍇ
	 * @throws InvalidStateException w肳ꂽԂLłȂꍇ
	 * @throws TransitionNotAllowedException Ԃ̕ύXĂȂꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkflowEngineHandler#changeActivityState(java.lang.String, java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.ActivityState)
	 */
	public void changeActivityState(String session,
			String pid, String aid, ActivityState state)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidActivityInstanceException, InvalidStateException,
		TransitionNotAllowedException
	{
		String argInfo = "(NAUTICA API = changeActivityState,"
			 + "session = [" + session + "],"
			 + "pid = [" + pid + "],"
			 + "aid = [" + aid + "],"
			 + "state = [" + state + "])";

		try {
			checkProcessID(pid);
			checkActivityID(aid);
		
			if (state == null) {
				// ANeBreB̏Ԃݒ肳Ă܂B
				String msg = StringManager.get("E0066");
				throw new InvalidStateException(msg);
			} else if (ActivityState.OPEN_NOT_RUNNING_NOT_STARTED.equals(state)) {
				// ANeBreB̏Ԃ𖢊Jnɂ邱Ƃ͂ł܂B
				String msg = StringManager.get("E0067");
				throw new InvalidStateException(msg);
			//} else if (ActivityState.CLOSED_COMPLETED.equals(state)) {
				// ANeBreB̏ԂIɐI邱Ƃ͂ł܂B
				//String msg = StringManager.get("E0068");
				//throw new InvalidStateException(msg);
			}
		
			DataAccessManager.begin(false);
				
			validateSession(session);
	
			ProcessManagerFactory pmf = ProcessManagerFactory.getInstance();
			WfProcessManager pm = pmf.getOwnerProcessManager(pid);
			WfProcess process = pm.getProcess(pid);
			WfActivity activity = process.getStep(aid);
			if (activity == null) {
				// YANeBreB݂܂B
				String msg = StringManager.get("E0069");
				throw new InvalidActivityInstanceException(msg);
			}
			activity.changeState(state);
				
			DataAccessManager.commit();
			
			// čOo
		    AuditDataRecord data = new AuditDataRecord(
		            new AuditPrefix(pid, "WMChangedActivityInstanceState", session),
		            new AuditSuffix());
		    data.setDataElement("ActivityInstanceID", aid);
		    data.setDataElement("ActivityDefinitionBusinessName", activity.getName());
		    data.setDataElement("NewActivityState", state);
		    audit.info(data);
	
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidProcessInstanceException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidActivityInstanceException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidStateException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (TransitionNotAllowedException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// ANeBreBԂ̕ύXɎs܂B
			String E0157 = StringManager.get("E0157") + argInfo;
			eLog.error(E0157, ex);
			throw new TransitionNotAllowedException(E0157, ex);
		}
	}

	/**
	 * ANeBreBCX^Xꗗ̎擾sB
	 * p\ȃtB^[NameFilter, AttributeFilter, GroupFilterƂB
	 *
	 * @param session ZbVID
	 * @param pid vZXCX^XID
	 * @param aid ANeBreBCX^XID
	 * @param filter ptB^[
	 * @return ANeBreBCX^Xꗗ
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidProcessInstanceException vZXCX^XIDLłȂꍇ
	 * @throws InvalidActivityInstanceException ANeBreBCX^XIDLłȂꍇ
	 * @throws InvalidFilterException w肳ꂽtB^[LłȂꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkflowEngineHandler#getActivityAttributes(java.lang.String, java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	public Attribute[] getActivityAttributes(String session,
			String pid, String aid, Filter filter)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidActivityInstanceException, InvalidFilterException
	{
		String argInfo = "(NAUTICA API = getActivityAttributes,"
			 + "session = [" + session + "],"
			 + "pid = [" + pid + "],"
			 + "aid = [" + aid + "],"
			 + "filter = [" + filter + "])";

		try {
			checkProcessID(pid);
			checkActivityID(aid);
		
			DataAccessManager.begin(false);
				
			validateSession(session);
	
			ProcessManagerFactory pmf = ProcessManagerFactory.getInstance();
			WfProcessManager pm = pmf.getOwnerProcessManager(pid);
			WfProcess process = pm.getProcess(pid);
			ActivityInternal activity = (ActivityInternal)process.getStep(aid);
			if (activity == null) {
				// YANeBreB݂܂B
				String msg = StringManager.get("E0069");
				throw new InvalidActivityInstanceException(msg);
			}
			Attribute[] attrs = (Attribute[])activity.getContext(filter);
				
			DataAccessManager.commit();
	
			return attrs;
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidProcessInstanceException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidActivityInstanceException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidFilterException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// ANeBreBw̃vZXϐ̎擾Ɏs܂B
			String E0201 = StringManager.get("E0201") + argInfo;
			eLog.error(E0201, ex);
			throw new InvalidProcessInstanceException(E0201, ex);
		}
	}

	/**
	 * ANeBreBCX^X̑l̎擾sB
	 *
	 * @param session ZbVID
	 * @param pid vZXCX^XID
	 * @param aid ANeBreBCX^XID
	 * @param name 
	 * @return 
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidAttributeException LłȂꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkflowEngineHandler#getActivityAttributeValue(java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	public Attribute getActivityAttributeValue(String session,
			String pid, String aid, String name)
	throws InvalidSessionException, InvalidAttributeException
	{
		String argInfo = "(NAUTICA API = getActivityAttributeValue,"
			 + "session = [" + session + "],"
			 + "pid = [" + pid + "],"
			 + "aid = [" + aid + "],"
			 + "name = [" + name + "])";

		try {
			if (pid == null || pid.trim().equals("")) {
				// vZXIDݒłB
				String E0189 = StringManager.get("E0189");
				throw new InvalidAttributeException(E0189);
			}

			if (aid == null || aid.trim().equals("")) {
				// ANeBreBIDݒłB
				String E0197 = StringManager.get("E0197");
				throw new InvalidAttributeException(E0197);
			}

			if (name == null || name.trim().equals("")) {
				// vZXϐݒłB
				String E0198 = StringManager.get("E0198");
				throw new InvalidAttributeException(E0198);
			}

			DataAccessManager.begin(false);
				
			validateSession(session);
	
			ProcessManagerFactory pmf = ProcessManagerFactory.getInstance();
			WfProcessManager pm = pmf.getOwnerProcessManager(pid);
			WfProcess process = pm.getProcess(pid);
			WfActivity activity = process.getStep(aid);
			if (activity == null) {
				// YANeBreB݂܂B
				String msg = StringManager.get("E0069");
				throw new InvalidActivityInstanceException(msg);
			}
			Attribute attr = (Attribute)activity.getContext(name);
				
			DataAccessManager.commit();
	
			return attr;
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidAttributeException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// ANeBreBw̃vZXϐ̎擾Ɏs܂B
			String E0201 = StringManager.get("E0201") + argInfo;
			eLog.error(E0201, ex);
			throw new InvalidAttributeException(E0201, ex);
		}
	}

	/**
	 * ANeBreBCX^X̊蓖ĂsB
	 *
	 * @param session ZbVID
	 * @param pid vZXCX^XID
	 * @param aid ANeBreBCX^XID
	 * @param attr 
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidProcessInstanceException vZXCX^XIDLłȂꍇ
	 * @throws InvalidActivityInstanceException ANeBreBCX^XIDLłȂꍇ
	 * @throws InvalidAttributeException LłȂꍇ
	 * @throws AttributeAssignmentFailedException ̊蓖ĂɎsꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkflowEngineHandler#assignActivityAttribute(java.lang.String, java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Attribute)
	 */
	public void assignActivityAttribute(String session,
			String pid, String aid, Attribute attr)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidActivityInstanceException, InvalidAttributeException,
		AttributeAssignmentFailedException
	{
		String argInfo = "(NAUTICA API = assignActivityAttribute,"
			 + "session = [" + session + "],"
			 + "pid = [" + pid + "],"
			 + "aid = [" + aid + "],"
			 + "attr = [" + attr + "])";

		try {
			checkProcessID(pid);
			checkActivityID(aid);

			if (attr == null) {
				// vZXϐݒłB
				String E0131 = StringManager.get("E0131");
				throw new InvalidAttributeException(E0131);
			}
		
			DataAccessManager.begin(false);
				
			validateSession(session);

			ProcessManagerFactory pmf = ProcessManagerFactory.getInstance();
			WfProcessManager pm = pmf.getOwnerProcessManager(pid);
			WfProcess process = pm.getProcess(pid);
			WfActivity activity = process.getStep(aid);
			if (activity == null) {
				// YANeBreB݂܂B
				String msg = StringManager.get("E0069");
				throw new InvalidActivityInstanceException(msg);
			}
			activity.setContext(attr);
				
			DataAccessManager.commit();
				
			// čOo
		    AuditDataRecord data = new AuditDataRecord(
		            new AuditPrefix(pid, "WMAssignedActivityInstanceAttributes", session),
		            new AuditSuffix());
		    data.setDataElement("ActivityInstanceID", aid);
		    data.setDataElement("ActivityState",activity.getState());
		    data.setDataElement("AttributeName", attr.getName());
		    data.setDataElement("AttributeType", attr.getType());
		    data.setDataElement("NewAttributeValue", attr.getValue()); 
		    audit.info(data);
	
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidProcessInstanceException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidActivityInstanceException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidAttributeException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (AttributeAssignmentFailedException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// ANeBreBw̃vZXϐ̐ݒɎs܂B
			String E0158 = StringManager.get("E0158") + argInfo;
			eLog.error(E0158, ex);
			throw new InvalidAttributeException(E0158, ex);
		}
	}

	/**
	 * vZXCX^Xꗗ̎擾sB
	 * p\ȃtB^[NameFilter, StateFilterAGroupFilterƂB
	 *
	 * @param session ZbVID
	 * @param filter ptB^[
	 * @return vZẌꗗ
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidFilterException w肳ꂽtB^[LłȂꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkflowEngineHandler#getProcesses(java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	public Process[] getProcesses(String session, Filter filter)
	throws InvalidSessionException, InvalidFilterException
	{
		String argInfo = "(NAUTICA API = getProcesses,"
			 + "session = [" + session + "],"
			 + "filter = [" + filter + "])";

		try {
			DataAccessManager.begin(false);
			
			validateSession(session);

			ProcessManagerFactory pmf = ProcessManagerFactory.getInstance();
			Process[] procs = pmf.getProcesses(filter);
			
			DataAccessManager.commit();

			return procs;
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidFilterException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// vZXꗗ̎擾Ɏs܂B
			String E0199 = StringManager.get("E0199");
			eLog.error(E0199, ex);
			throw new InvalidFilterException(E0199, ex);
		}
	}

	/**
	 * vZXCX^X̎擾sB
	 *
	 * @param session ZbVID
	 * @param pid vZXCX^XID
	 * @return vZXCX^X
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidProcessInstanceException vZXCX^XIDLłȂꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkflowEngineHandler#getProcess(java.lang.String, java.lang.String)
	 */
	public Process getProcess(String session, String pid)
	throws InvalidSessionException, InvalidProcessInstanceException
	{
		String argInfo = "(NAUTICA API = getProcess,"
			 + "session = [" + session + "],"
			 + "pid = [" + pid + "])";

		try {
			checkProcessID(pid);
			
			DataAccessManager.begin(false);
				
			validateSession(session);
	
			ProcessManagerFactory pmf = ProcessManagerFactory.getInstance();
			WfProcessManager pm = pmf.getOwnerProcessManager(pid);
			ProcessInternal internal = (ProcessInternal)pm.getProcess(pid);
				
			DataAccessManager.commit();
	
			return internal.getBean();
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidProcessInstanceException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// vZX̎擾Ɏs܂B
			String E0151 = StringManager.get("E0151") + argInfo;
			eLog.error(E0151, ex);
			throw new InvalidProcessInstanceException(E0151, ex);
		}
	}

	/**
	 * ANeBreBCX^Xꗗ̎擾sB
	 * p\ȃtB^[AttributeFilter, StateFilterAGroupFilterƂB
	 *
	 * @param session ZbVID
	 * @param filter ptB^[
	 * @return ANeBreBCX^Xꗗ
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidFilterException w肳ꂽtB^[LłȂꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkflowEngineHandler#getActivities(java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	public Activity[] getActivities(String session, Filter filter)
	throws InvalidSessionException, InvalidFilterException
	{
		String argInfo = "(NAUTICA API = getActivities,"
			 + "session = [" + session + "],"
			 + "filter = [" + filter + "])";

		try {
			DataAccessManager.begin(false);
			
			validateSession(session);

			ProcessManagerFactory pmf = ProcessManagerFactory.getInstance();
			Activity[] acts = pmf.getActivities(filter);
			
			DataAccessManager.commit();

			return acts;
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidFilterException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// ANeBreBꗗ̎擾Ɏs܂B
			String E0202 = StringManager.get("E0202") + argInfo;
			eLog.error(E0202, ex);
			throw new InvalidFilterException(E0202, ex);
		}
	}

	/**
	 * ANeBreBCX^Xꗗ̎擾sB
	 * p\ȃtB^[AttributeFilter, StateFilterAGroupFilterƂB
	 * WfMCdlłpid͂ȂA͌ł낤ƔfA
	 * pidɒǉB
	 *
	 * @param session ZbVID
	 * @param pid vZXCX^XID
	 * @param filter ptB^[
	 * @return ANeBreBCX^Xꗗ
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidFilterException w肳ꂽtB^[LłȂꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkflowEngineHandler#getActivities(java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	public Activity[] getActivities(String session, String pid, Filter filter)
	throws InvalidSessionException, InvalidFilterException
	{
		String argInfo = "(NAUTICA API = getActivities,"
			 + "session = [" + session + "],"
			 + "pid = [" + pid + "],"
			 + "filter = [" + filter + "])";

		try {
			if (pid == null || pid.trim().equals("")) {
				// vZXIDݒłB
				String E0189 = StringManager.get("E0189");
				throw new InvalidFilterException(E0189);
			}
		
			DataAccessManager.begin(false);
				
			validateSession(session);
	
			ProcessManagerFactory pmf = ProcessManagerFactory.getInstance();
			WfProcessManager pm = pmf.getOwnerProcessManager(pid);
			ProcessInternal process = (ProcessInternal)pm.getProcess(pid);
			Activity[] acts = process.getActivities(filter);
				
			DataAccessManager.commit();
	
			return acts;
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidFilterException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// ANeBreBꗗ̎擾Ɏs܂B
			String E0202 = StringManager.get("E0202") + argInfo;
			eLog.error(E0202, ex);
			throw new InvalidFilterException(E0202, ex);
		}
	}

	/**
	 * ANeBreBCX^X̎擾sB
	 *
	 * @param session ZbVID
	 * @param pid vZXCX^XID
	 * @param aid ANeBreBCX^XID
	 * @return ANeBreBCX^X
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 * @throws InvalidProcessInstanceException vZXCX^XIDLłȂꍇ
	 * @throws InvalidActivityInstanceException ANeBreBCX^XIDLłȂꍇ
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkflowEngineHandler#getActivity(java.lang.String, java.lang.String, java.lang.String)
	 */
	public Activity getActivity(String session, String pid, String aid)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidActivityInstanceException
	{
		String argInfo = "(NAUTICA API = getActivity,"
			 + "session = [" + session + "],"
			 + "pid = [" + pid + "],"
			 + "aid = [" + aid + "])";

		try {
			checkProcessID(pid);
			checkActivityID(aid);

			DataAccessManager.begin(false);
			
			validateSession(session);
	
			ProcessManagerFactory pmf = ProcessManagerFactory.getInstance();
			WfProcessManager pm = pmf.getOwnerProcessManager(pid);
			WfProcess proc = pm.getProcess(pid);
			ActivityInternal internal = (ActivityInternal)proc.getStep(aid);
			
			if (internal == null) {
				// ANeBreB݂܂B
				String E0203 = StringManager.get("E0203");
				throw new InvalidActivityInstanceException(E0203);
			}
			Activity bean = internal.getBean();

			DataAccessManager.commit();
	
			return bean;
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidProcessInstanceException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidActivityInstanceException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// ANeBreB̎擾Ɏs܂B
			String E0124 = StringManager.get("E0124") + argInfo;
			eLog.error(E0124, ex);
			throw new InvalidActivityInstanceException(E0124, ex);
		}
	}
	
	/**
	 * [obNƃG[Ȍo͂sB
	 *
	 * @param ex O
	 * @param argInfo 
	 */
	private void rollbackAndLogging(Exception ex, String argInfo)
	{
		try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
		String msg = ex.getMessage() + argInfo;
		eLog.error(msg, ex);
	}

	/**
	 * vZXID؂B
	 *
	 * @param pid vZXID
	 * @throws InvalidProcessInstanceException ȃvZXID̏ꍇ
	 */
	private void checkProcessID(String pid)
	throws InvalidProcessInstanceException
	{
		if (pid == null || pid.trim().equals("")) {
			// vZXIDݒłB
			String E0189 = StringManager.get("E0189");
			throw new InvalidProcessInstanceException(E0189);
		}
	}

	/**
	 * ANeBreBID؂B
	 *
	 * @param aid ANeBreBID
	 * @throws InvalidActivityInstanceException ȃANeBreBID̏ꍇ
	 */
	private void checkActivityID(String aid)
	throws InvalidActivityInstanceException
	{
		if (aid == null || aid.trim().equals("")) {
			// ANeBreBIDݒłB
			String E0197 = StringManager.get("E0197");
			throw new InvalidActivityInstanceException(E0197);
		}
	}
	
	/**
	 * ZbV؂B
	 *
	 * @param session ZbVID
	 * @throws InvalidSessionException w肳ꂽZbV̏ꍇ
	 */
	private void validateSession(String session)
	throws InvalidSessionException
	{
		if (session == null || session.trim().equals("")) {
			// ZbVIDݒłB
			String E0170 = StringManager.get("E0170");
			throw new InvalidSessionException(E0170);
		}

		SessionManagerFactory factory = SessionManagerFactory.getInstance();
		SessionManager manager = factory.getSessionManager();
		manager.validateSession(session);
	}

	private ActivityDefinition findActivity(ProcessDefinition pd, String adid)
	throws Exception
	{
		List acts = pd.getActivities();
		Iterator it = acts.iterator();
		
		ActivityDefinition ad = null;
		while (it.hasNext()) {
			ActivityDefinition def = (ActivityDefinition)it.next();
			if (ad.getID().equals(adid)) {
				ad = def;
				break;
			}
		}
		
		if (ad == null) {
			// ANeBreB`݂܂B
			String E0204 = StringManager.get("E0204");
			throw new WorkflowException(E0204);
		}
		
		return ad;
	}
}
