/* $Id: RelationalExpression.java,v 1.5 2005/08/26 04:54:54 ysahara Exp $
 * 
 * Copyright (c)ARGO 21, Corporation. 2005.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.expression;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import jp.co.argo21.nautica.workflow.dataaccess.VariableBean;
import jp.co.argo21.nautica.workflow.definition.VariableDefinition;
import jp.co.argo21.nautica.workflow.xpdl.RelationalExpressionDocument;

import org.apache.xmlbeans.XmlObject;

/**
 * ֌W
 * 
 * @author  Norihiro Itoh(ARGO 21 Corp.)
 * @version $Revision: 1.5 $
 * @since   Nautica Workflow 0.9
 */
public class RelationalExpression implements Expression
{
	/** VAo[WUID */
	static private final long serialVersionUID = 20050000000000001L;

	/** == */
	static public final int EQUAL         = RelationalExpressionDocument.RelationalExpression.Type.INT_EQUAL;
	/** != */
	static public final int NOT_EQUAL     = RelationalExpressionDocument.RelationalExpression.Type.INT_NOT_EQUAL;
	/** < */
	static public final int LESS_THAN     = RelationalExpressionDocument.RelationalExpression.Type.INT_LESS_THAN;
	/** <= */
	static public final int LESS_EQUAL    = RelationalExpressionDocument.RelationalExpression.Type.INT_LESS_EQUAL;
	/** > */
	static public final int GREATER_THAN  = RelationalExpressionDocument.RelationalExpression.Type.INT_GREATER_THAN;
	/** >= */
	static public final int GREATER_EQUAL = RelationalExpressionDocument.RelationalExpression.Type.INT_GREATER_EQUAL;

	/** == */
	static private final String EQUAL_STR         = "==";
	/** != */
	static private final String NOT_EQUAL_STR     = "!=";
	/** < */
	static private final String LESS_THAN_STR     = "<";
	/** <= */
	static private final String LESS_EQUAL_STR    = "<=";
	/** > */
	static private final String GREATER_THAN_STR  = ">";
	/** >= */
	static private final String GREATER_EQUAL_STR = ">=";

	/** Zq */
	private int operator;
	/** Ӓl */
	private String leftValue;
	/** EӒl */
	private String rightValue;
	
	/**
	 * ֌W𐶐B
	 * 
	 * @param lVal	Ӓl
	 * @param ope	Zq
	 * @param rVal	EӒl
	 */
	public RelationalExpression(String lVal, int ope, String rVal)
	{
		operator = ope;
		leftValue = lVal;
		rightValue = rVal;
	}
	
	/**
	 * XML֌W𐶐B
	 * 
	 * @param exp	XML
	 */
	public RelationalExpression(RelationalExpressionDocument.RelationalExpression exp)
	{
		operator = exp.getType().intValue();
		leftValue = exp.getDataFieldRef();
		rightValue = exp.getComparedValue();
	}
	
	/**
	 * ̃IuWFNgɏi[B
	 * 
	 * @param o	XMLIuWFNg
	 * @see jp.co.argo21.nautica.workflow.expression.Expression#saveTo(org.apache.xmlbeans.XmlObject)
	 */
	public void saveTo(XmlObject o)
	{
		if (o == null) return;
		if (!(o instanceof RelationalExpressionDocument.RelationalExpression)) return;

		RelationalExpressionDocument.RelationalExpression re
			= (RelationalExpressionDocument.RelationalExpression)o;

		re.setType(RelationalExpressionDocument.RelationalExpression.Type.Enum.forInt(operator));
		if (leftValue != null) re.setDataFieldRef(leftValue);
		if (rightValue != null) re.setComparedValue(rightValue);
	}

	/**
	 * ̏ɏ]B
	 * 
	 * @param list	ϐ̌ݒl
	 * @return	ꂽꍇtrue
	 * @see jp.co.argo21.nautica.workflow.expression.Expression#eval(java.util.List)
	 */
	public boolean eval(List list)
	{
		VariableBean var = getVariable(list, leftValue);

		if (var == null) return false;
		
		int type = var.getType();
		String value = var.getValue();
		if (type == VariableDefinition.BOOLEAN) {
			return evalAsBoolean(value, rightValue);
		} else if (type == VariableDefinition.INTEGER) {
			return evalAsInteger(value, rightValue);
		} else {
			return evalAsString(value, rightValue);
		}
	}

	/**
	 * ̕\ԂB
	 * 
	 * 
	 * @return	\
	 * @see jp.co.argo21.nautica.workflow.expression.Expression#getExpressionString()
	 */
	public String getExpressionString()
	{
		String ls = "null";
		String rs = "null";
		if (leftValue != null) ls = leftValue;
		if (rightValue != null) rs = rightValue;
		
		return "(" + ls +  ") " + getOpeString() + " (" + rs + ")";
	}

	/**
	 * ϐꗗ𗘗pȂ̕\ԂB
	 * 
	 * @param vars	ϐꗗ
	 * @return	\
	 * @see jp.co.argo21.nautica.workflow.expression.Expression#getExpressionString(java.util.List)
	 */
	public String getExpressionString(List vars)
	{
		if (vars == null) return getExpressionString();
		if (leftValue == null) return getExpressionString();
		
		String label = null;
		Iterator it = vars.iterator();
		while (it.hasNext()) {
			Object o = it.next();
			if (o == null) continue;
			if (!(o instanceof VariableDefinition)) continue;

			if (((VariableDefinition)o).getID().equals(leftValue)) {
				label = ((VariableDefinition)o).getName();
				break;
			}
		}
		String ls = "null";
		String rs = "null";
		if (label != null) ls = label;
		if (rightValue != null) rs = rightValue;
		
		return "(" + ls +  ") " + getOpeString() + " (" + rs + ")";
	}

	/**
	 * ̏Ɋ܂܂邷ׂĂ̏ԂB
	 * 
	 * @param exps	ꗗ
	 * @see jp.co.argo21.nautica.workflow.expression.Expression#getAllExpressions(java.util.List)
	 */
	public void getAllExpressions(List exps)
	{
		if (exps.contains(this) == false) exps.add(this);
	}

	/**
	 * Zq̕\ԂB
	 * 
	 * @return	\
	 */	
	private String getOpeString()
	{
		if (operator == NOT_EQUAL) return NOT_EQUAL_STR;
		else if (operator == LESS_THAN) return LESS_THAN_STR;
		else if (operator == LESS_EQUAL) return LESS_EQUAL_STR;
		else if (operator == GREATER_THAN) return GREATER_THAN_STR;
		else if (operator == GREATER_EQUAL) return GREATER_EQUAL_STR;
		else return EQUAL_STR;
	}

	/**
	 * ^lƂĕ]B
	 * 
	 * @param v1	Ӓl
	 * @param v2	EӒl
	 * @return	]
	 */
	private boolean evalAsBoolean(String v1, String v2)
	{
		boolean result = false;
		Boolean b1 = null;
		Boolean b2 = null;
		try { b1 = new Boolean(v1); } catch (Exception ex) {}
		try { b2 = new Boolean(v2); } catch (Exception ex) {}

		if (b1 == null) return false;
		if (b2 == null) return false;

		result = b1.equals(b2);

		if (operator == EQUAL) return result;		
		else return (! result);
	}

	/**
	 * tƂĕ]B
	 * 
	 * @param v1	Ӓl
	 * @param v2	EӒl
	 * @return	]
	 */
	private boolean evalAsDateTime(String v1, String v2)
	{
		int comp = 0;
		Date d1 = null;
		Date d2 = null;
		try {
			d1 = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss").parse(v1);
		} catch (Exception ex) {
			return false;
		}

		try {
			d2 = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss").parse(v2);
		} catch (Exception ex) {
			return false;
		}

		return evalComparedValue(d1.compareTo(d2));
	}

	/**
	 * 𐮐Ƃĕ]B
	 * 
	 * @param v1	Ӓl
	 * @param v2	EӒl
	 * @return	]
	 */
	private boolean evalAsInteger(String v1, String v2)
	{
		int comp = 0;
		BigInteger i1 = null;
		BigInteger i2 = null;
		try {
			i1 = new BigInteger(v1);
		} catch (Exception ex) {
			return false;
		}

		try {
			i2 = new BigInteger(v2);
		} catch (Exception ex) {
			return false;
		}

		return evalComparedValue(i1.compareTo(i2));
	}

	/**
	 * Ƃĕ]B
	 * 
	 * @param v1	Ӓl
	 * @param v2	EӒl
	 * @return	]
	 */
	private boolean evalAsDecimal(String v1, String v2)
	{
		int comp = 0;
		BigDecimal d1 = null;
		BigDecimal d2 = null;
		try {
			d1 = new BigDecimal(v1);
		} catch (Exception ex) {
			return false;
		}

		try {
			d2 = new BigDecimal(v2);
		} catch (Exception ex) {
			return false;
		}

		return evalComparedValue(d1.compareTo(d2));
	}

	/**
	 * l̔rʂ𔻒肷B
	 * 
	 * @param comp	r
	 * @return	
	 */	
	private boolean evalComparedValue(int comp)
	{
		if (comp < 0) {
			if (operator == LESS_THAN
			  || operator == LESS_EQUAL
			  || operator == NOT_EQUAL) {
				return true;
			} else {
				return false;
			}
		} else if (comp > 0) {
			if (operator == GREATER_THAN
			  || operator == GREATER_EQUAL
			  || operator == NOT_EQUAL) {
				return true;
			} else {
				return false;
			}
		} else { 
			if (operator == EQUAL
			  || operator == LESS_EQUAL
			  || operator == GREATER_EQUAL) {
				return true;
			} else {
				return false;
			}
		}
	}

	/**
	 * 𕶎Ƃĕ]B
	 * 
	 * @param v1	Ӓl
	 * @param v2	EӒl
	 * @return	]
	 */
	private boolean evalAsString(String v1, String v2)
	{
		boolean result = false;
		if (v1 == null && v2 == null) result = true;
		else if (v1 == null) result = false;
		else if (v2 == null) result = false;
		else result = v1.equals(v2);

		if (operator == EQUAL) return result;		
		else return (! result);
	}

	/**
	 * ϐԂB
	 * 
	 * @param list	ϐꗗ
	 * @param id	ϐID
	 * @return	ϐ
	 */
	private VariableBean getVariable(List list, String id)
	{
		Iterator it = list.iterator();
		
		while (it.hasNext()) {
			VariableBean v = (VariableBean)it.next();
			if (id.equals(v.getVariableDefinitionID())) return v;
		}
		return null;
	}
}
