/* $Id: AbstractToolAgent.java,v 1.14 2005/08/26 04:52:54 ysahara Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.ta;

import java.sql.Connection;
import java.sql.Timestamp;
import java.util.List;

import javax.sql.DataSource;

import org.apache.log4j.Logger;

import jp.co.argo21.nautica.workflow.dataaccess.AppExecutionBean;
import jp.co.argo21.nautica.workflow.dataaccess.AppExecutionDAO;
import jp.co.argo21.nautica.workflow.dataaccess.WorkflowDAOFactory;
import jp.co.argo21.nautica.workflow.engine.DataAccessManager;
import jp.co.argo21.nautica.workflow.engine.MessageTransporter;
import jp.co.argo21.nautica.workflow.engine.WorkflowEngine;
import jp.co.argo21.nautica.workflow.engine.WorkflowEngineConstants;
import jp.co.argo21.nautica.workflow.jms.MessagingException;
import jp.co.argo21.nautica.workflow.util.StringManager;
import jp.co.argo21.nautica.workflow.wfmc.ApplicationBusyException;
import jp.co.argo21.nautica.workflow.wfmc.ApplicationNotDefinedException;
import jp.co.argo21.nautica.workflow.wfmc.ApplicationNotStartedException;
import jp.co.argo21.nautica.workflow.wfmc.Attribute;
import jp.co.argo21.nautica.workflow.wfmc.ConnectionFailedException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidSessionException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidToolAgentException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidWorkItemException;
import jp.co.argo21.nautica.workflow.wfmc.ToolAgent;

/**
 * c[G[WFg̒ۃNXB
 * 
 * c[G[WFgnhisIDj̎擾A
 * AvP[Vse[uւ̃R[hǉA
 * L[ւ̎s̑MA
 * vZX̊Ǘ
 * sB
 *
 * @author  kiyoda(Argo 21, Corp.)
 * @version $Revision: 1.14 $
 * @since   Nautica Workflow 0.9
 */
public abstract class AbstractToolAgent implements ToolAgent {
    
	/** c[G[WFgpX */
	static public final String TOOLAGENT_PATH;

	/* OIuWFNg */
    protected Logger log = Logger.getLogger(this.getClass());
    
    /* bZ[Wz */
    protected MessageTransporter tp = null;
    
    /* AvP[Vs */
    protected int retryCount = 1;

    /** [UID */
    protected String userId = null;
    
    /** pX[h */
    protected String passwd = null;
    
    /** ZbVID */
    protected String session = null;

	static
	{
		String home = System.getProperty(WorkflowEngineConstants.NAUTICA_HOME);
		String separator = System.getProperty("file.separator");
		TOOLAGENT_PATH = home + separator + "toolagent";
	}
	
	/**
	 * c[G[WFgIuWFNg𐶐B
	 *
	 */
	public AbstractToolAgent() {
	    // ȂB
	}
	
    /**
     * AvP[VNc[G[WFg̃NX擾B
     * 
     * @return	AvP[VNc[G[WFg̃NX
     */
    abstract protected String getToolAgentClassName();
    
    /**
     * AvP[Vs𑗐ML[擾B
     * 
     * c[G[WFgɃL[KvŁÃL[ɖOȂĂ͂ȂȂB
     * ̃L[̓c[G[WFgŗL̕łB
     * {Iɂ̓c[G[WFggpB
     * ec[G[WFgŏdȂ悤ɒӂ邱ƁB
     *
     * @return	L[
     */
    abstract protected String getQueueName();
    
    /**
     * L[ɑMAvP[Vs擾B
     *
     * AvP[VsNXł́ȀɂāA
     * AvP[VsB 
     *
     * @param appName	`Ɏw肵AvP[V
     * @param wid		ƍID
     * @param handler	c[G[WFgnh
     * @param attrList	ꗗ
     * @return	AvP[Vs
     */
    abstract protected AppExecutionInfo getExecInfo(String appName, String wid,
            int handler, Attribute[] attrList);
    
    /**
     * AvP[V`Ă邩ǂ`FbNB
     * 
     * AvP[V`ĂȂꍇAOB
     *
     * @param appName	AvP[V
     * @throws	ApplicationNotDefinedException	AvP[V`ĂȂꍇ
     */
    abstract protected void checkApplicationDefined(String appName)
    	throws ApplicationNotDefinedException;
    
    /**
	 * AvP[V̋NsB
	 * <b></b>ƂāA{[Nt[VXe^pՂł́A
	 * Ƀ[h͊܂܂Ȃ̂ƂB
	 * 
	 * @param appName `Ɏw肵AvP[V
	 * @param wid ƍID
	 * @param attrList ꗗ
	 * @return c[G[WFgnh
	 * @throws ApplicationNotStartedException AvP[VJnłȂꍇ
	 * @throws ApplicationNotDefinedException AvP[V`ĂȂꍇ
	 * @throws ApplicationBusyException AvP[VrW[ŗv󂯂Ȃꍇ
     */
    public int invokeApplication(String appName, String wid,
            Attribute[] attrList) throws ApplicationNotStartedException,
            ApplicationNotDefinedException, ApplicationBusyException {

        // O
        String msg = StringManager.get("I2001") 
        	+ "(ApplicationName=" + appName + ", " 
        	+ "WorkItemID=" + wid + ", "
        	+ "Attributes=" + attrList + ")";
        log.info(msg);
        
        int handler = 0;
        
        // AvP[VA`Ă邩ǂ`FbN
        checkApplicationDefined(appName);
        
        java.sql.Connection sqlConn = null;
        try {
            // c[G[WFgnh擾
            handler = ToolAgentHandleCreator.generate();
        
			DataSource ds = DataAccessManager.getDataSource();
			sqlConn = ds.getConnection();
			sqlConn.setAutoCommit(false);
            // DAOFactory擾
            WorkflowDAOFactory factory = DataAccessManager.getDAOFactory();

            // AppExecutionDAO擾
            AppExecutionDAO appExecutionDao = factory.getAppExecutionDAO(sqlConn);
            
            // AvP[Vse[uɃAvP[V̏(=s)INSERT
            // STARTER_ID(sID)ɂ͍ƍIDݒ肷
            AppExecutionBean bean = new AppExecutionBean();
            bean.setAppName(appName);
            bean.setToolAgent(getToolAgentClassName());
            bean.setExecId(handler);
            bean.setStartDate(new Timestamp(System.currentTimeMillis()));
            bean.setStarterId(wid);
            bean.setAppState(ApplicationState.NOT_STARTED);
            appExecutionDao.insert(bean);
            
            // L[ɃAvP[Vs𑗐M
            AppExecutionInfo info = getExecInfo(appName, wid, handler, attrList);
            info.setRetryCount(this.retryCount);
            sendMessage(info);

            // R~bg
            sqlConn.commit();
        } catch (Exception ex) {
            try {
                // [obN
                sqlConn.rollback();
            } catch (Exception ex2) {
                /* ignore */
            }
            // O
            String errMsg = StringManager.get("E2001")
        		+ "(ApplicationName=" + appName + ", " 
        		+ "WorkItemID=" + wid + ", "
        		+ "Attributes=" + attrList + ")";
            log.error(errMsg, ex);
            throw new ApplicationNotStartedException(errMsg, ex);
        } finally {
			try {
				if (sqlConn != null && (!sqlConn.isClosed())) {
				    sqlConn.close();
				}
			} catch (Exception ex) { /* ignore */ }
        }
        
        // O
        log.info(StringManager.get("I2002"));
        
        return handler;
    }
    
    /**
     * AvP[VĎsB
     * 
     * AvP[VsL[ɓo^邱ƂɂA
     * AvP[VĎsB
     *
     * @param info	AvP[Vs
     * @throws ApplicationNotStartedException	AvP[VJnłȂꍇ
     */
    protected void retryApplication(AppExecutionInfo info) 
    throws ApplicationNotStartedException {
        try {
            String msg = StringManager.get("I2013")
            	+ "(ApplicationExecutionInfomation=" + info + ")";
            log.info(msg);
            sendMessage(info);
        } catch (Exception ex) {
            String errMsg = StringManager.get("E2001");
            throw new ApplicationNotStartedException(errMsg, ex);
        }
    }
    
    /**
     * AvP[V̎s񐔂ݒ肷B
     *
     * @param count		AvP[V̎s
     */
    protected void setRetryCount(int count) {
        this.retryCount = count;
    }

    /**
	 * AvP[V̎sԂ擾B
	 * sԂ́Aȉ̂̂pӂB
	 * <ul>
	 * <li>0FI</li>
	 * <li>1FُI</li>
	 * <li>2FJn</li>
	 * <li>3F~</li>
	 * <li>6Fs</li>
	 * <li>9Fʕs</li>
	 * </ul>
	 * <b></b>ƂāA{[Nt[VXe^pՂł́A
	 * ɂẮAc[G[WFgA܂̓AvP[V
	 * XVƑz肵Aɂ͊܂܂ȂƂƂB
	 * 
	 * @param toolAgentHandle c[G[WFgnh
	 * @param wid ƍID
	 * @return s
	 * @throws ApplicationBusyException AvP[VrW[ŗv󂯂Ȃꍇ
	 * @throws InvalidToolAgentException LłȂc[G[WFgnhw肵ꍇ
	 * @throws InvalidWorkItemException LłȂƍIDw肵ꍇ
     */
    public int requestAppStatus(int toolAgentHandle, String wid)
            throws ApplicationBusyException, InvalidToolAgentException,
            InvalidWorkItemException {
        
        // O
        String msg = StringManager.get("I2005")
        	+ "(ToolAgentHandler=" + toolAgentHandle + ", "
        	+ "WorkItemID=" + wid + ")";
        log.info(msg);
        
        int state = ApplicationState.NOT_STARTED.toInt();
        
		Connection conn = null;
        try {
			DataSource ds = DataAccessManager.getDataSource();
			conn = ds.getConnection();

			// DAOFactory擾
            WorkflowDAOFactory factory = DataAccessManager.getDAOFactory();
        
            // AppExecutionDAO擾
            AppExecutionDAO dao = factory.getAppExecutionDAO(conn);
        
            // AvP[Vse[uAvP[V̎sԂ擾
            List list = dao.findByExecID(toolAgentHandle);
            
            if (list.size() != 1) {
                // G[
                String errMsg = StringManager.get("E2003")
                	+ "(ToolAgentHandler=" + toolAgentHandle + ", "
                	+ "WorkItemID=" + wid + ")";
                throw new InvalidToolAgentException(errMsg);
            }
            AppExecutionBean bean = (AppExecutionBean)list.get(0);
            
            state = bean.getAppState().toInt();
            
        } catch (InvalidToolAgentException ex) {
            log.error(StringManager.get("E2003"), ex);
            throw ex;
        } catch (Exception ex) {
            // O
            String errMsg = StringManager.get("E2003");
            log.error(errMsg, ex);
            throw new InvalidToolAgentException(errMsg, ex);
        } finally {
			try {
				if (conn != null && (! conn.isClosed())) {
					conn.close();
				}
			} catch (Exception ex) { /* ignore */ }
        }
        
        // O
        log.info(StringManager.get("I2006") + "(ExecutionState=" + state + ")");
        
        return state;
    }
    
    /**
     * [Nt[GWƐڑB
     * 
     * [Nt[GWɐڑĂȂ΁AڑB
     * ɐڑĂꍇ́AUؒfāAĐڑB
     *
     * @return	ZbVID
     * @throws	ConnectionFailedException	[Nt[GWƂ̐ڑɎsꍇ
     * @see jp.co.argo21.nautica.workflow.ta.AbstractToolAgent#connect()
     */
    protected String connect() throws ConnectionFailedException {
        /* [Nt[GWɐڑΈUؒf */
        if (this.session != null) {
            this.disconnect(this.session);
        }
        
        WorkflowEngine engine = WorkflowEngine.getInstance();
        this.session = engine.connect(this.userId, this.passwd);
        
        return this.session;
    }

    /**
     * [Nt[GWؒfB
     *
     * @param session
     * @see jp.co.argo21.nautica.workflow.ta.AbstractToolAgent#disconnect(java.lang.String)
     */
    protected void disconnect(String session) {
        WorkflowEngine engine = WorkflowEngine.getInstance();
        try {
            engine.disconnect(this.session);
        } catch (Exception ex) {
            /* Ignore */
        }
        this.session = null;
    }
    
    /**
     * bZ[Wz҂擾B
     * 
     * @return	bZ[Wz
     * @throws InvalidSessionException
     * @throws MessagingException
     */
    private MessageTransporter getMessageTransporter() 
    	throws InvalidSessionException, MessagingException
    {
        if (tp == null) {
            WorkflowEngine engine = WorkflowEngine.getInstance();
            String session = null;
            try {
                session = connect();
                tp = engine.getMessageTransporter(session, getQueueName());
            } catch (ConnectionFailedException ex) {
                String errMsg = StringManager.get("E8001");
                throw new InvalidSessionException(errMsg, ex);
            } finally {
                disconnect(session);
            }
        }
        return tp;
    }
    
    /**
     * L[ɃbZ[Wo^B
     * 
     * @param info	AvP[Vs
     * @throws InvalidSessionException	sȃZbV̏ꍇ
     * @throws MessagingException	bZ[Wo^ŃG[ꍇ
     */
    private void sendMessage(AppExecutionInfo info) 
    throws InvalidSessionException, MessagingException {
        
        info.setRetryCount(info.getRetryCount() - 1);
        getMessageTransporter().sendMessage(info);
    }
}
