/* $Id: DefinitionCache.java,v 1.1 2007/10/19 07:29:57 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfadm.definition.loader;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;

import jp.co.argo21.nautica.tool.wfadm.pref.PreferenceManager;

/**
 * 定義ファイルのキャッシュシステムである。
 * キャッシュ用ディレクトリは、ユーザのホームディレクトリ配下の以下の場所とする。
 * (Windows)  ${HOME}/Application Data/Nautica/WorkflowAdmin/DefinitionCache/
 * (Linux)    ${HOME}/.nautica/WorkflowAdmin/DefinitionCache/ 
 * 
 * @author  mmanabe(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
class DefinitionCache {
    /** 文字エンコード */
    private static final String ENCODE = "utf-8";
    
    /** 定義ファイルを保持するキャッシュ用ディレクトリ */
    private String cachedir;

    /** セパレータ */
    private String separator = System.getProperty("file.separator");
        
    /**
     * DefinitionCache を生成する。
     */
    DefinitionCache() {
        setCacheDir();        
    }
    
    /**
     * 指定されたエンジン名とパス名でキャッシュされているファイルを
     * 読み込み、文字列を返す。
     * 該当するファイルがない場合、および入出力例外が発生した場合は
     * null を返す。
     * 
     * @param engineName エンジン名
     * @param path       パス名
     * @return 定義ファイルの文字列
     */
    String fetch(String engineName, String path) {
        String filepath = cachedir + engineName + separator + path;
        BufferedReader br = null;	    
	    try {	        
		    // ファイルを取得し、読み込んで文字列を返す 
	        br = new BufferedReader(
	    	        new InputStreamReader(new FileInputStream(filepath), ENCODE));
	        StringBuffer definition = new StringBuffer();
	        while (br.ready()) {
		        definition.append(br.readLine());
		    }
		    return definition.toString();
	        
	    } catch (IOException e){
	        return null;
	        
	    } finally {
	        if (br != null) {
	            try { br.close(); } catch (IOException e) { /* Ignore */ }
	        }
	    }
    }
    
    /**
     * 指定されたエンジン名とパス名で、文字列を定義ファイルとして
     * キャッシュ用ディレクトリに保存する。
     * 
     * @param engineName エンジン名
     * @param path       パス名
     * @param contents   定義ファイルの文字列
     * @throws IOException 入出力例外が発生した場合
     */
    void store(String engineName, String path, String contents)
    throws IOException
    {		
        String filepath = cachedir + engineName + separator + path;
        
		// ファイルを生成
		File file = new File(filepath);
		if (!file.exists()) {
		    File parent = file.getParentFile();
		    parent.mkdirs();
		    
		    file.createNewFile();
		}
		
		// ファイルに書き出す
		BufferedWriter bw = null;
		try {
		    bw = new BufferedWriter(
			        new OutputStreamWriter(new FileOutputStream(filepath), ENCODE));
			bw.write(contents);
			
		} finally {
		    if (bw != null) {
		        try { bw.close(); } catch (IOException e) { /* Ignore */ }	
		    }
		}
    }
    
    /**
     * キャッシュ用ディレクトリを設定する。
     */
    private void setCacheDir() {
    	String path = PreferenceManager.getInstance().getApplicationDataPath();
        cachedir = path + separator + "DefinitionCache" + separator;
    }
}