/* $Id: DefinitionLoader.java,v 1.1 2007/10/19 07:29:57 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfadm.definition.loader;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;

import javax.naming.NamingException;

import jp.co.argo21.nautica.tool.wfadm.pref.EngineInfo;
import jp.co.argo21.nautica.tool.wfadm.pref.PreferenceManager;
import jp.co.argo21.nautica.workflow.client.WorkflowServiceAccessor;
import jp.co.argo21.nautica.workflow.definition.PackageDefinition;
import jp.co.argo21.nautica.workflow.definition.impl.PackageDefinitionImpl;
import jp.co.argo21.nautica.workflow.engine.RepositoryLoader;
import jp.co.argo21.nautica.workflow.omg.WorkflowException;

import org.apache.xmlbeans.XmlException;
import org.apache.xmlbeans.XmlOptions;
import org.wfmc.x2002.xpdl10.PackageDocument;

/**
 * 定義ファイルを読み込むローダーである。
 * 該当する定義がキャッシュ用ディレクトリに存在しない場合、
 * エンジンに接続して定義ファイルを読み込み、キャッシュする。
 *
 * @author  mmanabe(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0    
 */
public class DefinitionLoader {
    /** DefinitionLoader のシングルトン */
    private static DefinitionLoader singleton = new DefinitionLoader();
    
    /** DefinitionCache オブジェクト */
    private DefinitionCache dc;
    
    /**
     * DefinitionLoader を生成する。
     */
    private DefinitionLoader() {
        dc = new DefinitionCache();
    }
    
    /**
     * DefinitionLoader のインスタンスを返す。
     * 
     * @return DefinitionLoader
     */
    public static DefinitionLoader getInstance() {
        return singleton;
    }

    /**
     * 定義オブジェクトを返す。
     * 定義が存在しない場合は null を返す。
     * 
     * @param  engineName エンジン名
     * @param  path       パス名
     * @return 定義オブジェクト
     * @throws WorkflowException エンジンへの接続に失敗した場合
     *                            エンジンからの切断に失敗した場合
     * @throws IOException       入出力例外が発生した場合
     * @throws XmlException      PackageDocumentの取得に失敗した場合
     * @throws NamingException   ネーミング例外が発生した場合
     */
    public PackageDefinition getDefinition(String engineName, String path) 
    throws WorkflowException, IOException, XmlException,
            NamingException
    {        
        // キャッシュに保持している場合
        String definition = dc.fetch(engineName, path);
        if (definition != null && !definition.equals("")) {
            return getPackageDefinition(definition);
        }
        
        // キャッシュに保持していない場合
        // エンジンに接続して定義を取得し、接続を切る
        PreferenceManager pm = PreferenceManager.getInstance();
        EngineInfo einfo = pm.getEngineInfo(engineName);

        if (einfo == null) {
            return null;
        }
        
		WorkflowServiceAccessor wsa = einfo.getWorkflowServiceAccessor();
        try {
            String sid = wsa.open();
            RepositoryLoader loader = 
                getRepositoryLoader(einfo);
            definition = loader.getDefinition(sid, einfo.getAdminKey(), path);
        
        } finally {
            if (wsa != null) { wsa.close(); }
        }
        
        if (definition == null || definition.equals("")) {
            return null;
        }        

        // 文字列をキャッシュ
        dc.store(engineName, path, definition);
        
        // 定義オブジェクトを返す
        return getPackageDefinition(definition);
    }   
    
    /**
     * 指定されたプロセス定義名に対応するパス名を返す。
     * なければ null を返す。
     * 
     * @param  engineName エンジン名
     * @param  pdname     プロセス定義名
     * @return パス名
     * @throws WorkflowException エンジンへの接続に失敗した場合
     *                            エンジンからの切断に失敗した場合
     * @throws NamingException   ネーミング例外が発生した場合
     */
    public String getDefinitionPath(String engineName, String pdname)
    throws WorkflowException, NamingException
    {
        PreferenceManager pm = PreferenceManager.getInstance();
        EngineInfo einfo = pm.getEngineInfo(engineName);
        
        if (einfo == null) {
            return null;
        }
        
		WorkflowServiceAccessor wsa = einfo.getWorkflowServiceAccessor();
        try {
            String sid = wsa.open();
            RepositoryLoader loader = 
                getRepositoryLoader(einfo);

            return loader.getDefinitionPath(sid, pdname);

        } finally {
            if (wsa != null) { wsa.close(); }
        }
    }
    
    /**
     * リモートのRepositoryLoaderを返す。
     * 
     * @param scope エンジンスコープ
     * @param url   ネーミングサービスURL
     * @return RepositoryLoader
     * @throws NamingException ネーミング例外が発生した場合
     */
    private RepositoryLoader getRepositoryLoader(EngineInfo einfo)
    throws NamingException
    {
        RepositoryLoader loader = einfo.getRepositoryLoader();
        return loader;
    }
    
    /**
     * 文字列からパッケージ定義を生成して返す。
     * 
     * @param contents 定義の内容を表す文字列
     * @return パッケージ定義オブジェクト
     * @throws XmlException PackageDocumentの生成に失敗した場合
     * @throws IOException  入出力例外が発生した場合
     */
    private PackageDefinition getPackageDefinition(String contents)
    throws XmlException, IOException 
    {
        InputStream in = null;
        try {
            in = new ByteArrayInputStream(contents.getBytes("utf-8"));
            XmlOptions opts = new XmlOptions();
            opts.setCharacterEncoding("utf-8");
            PackageDocument doc = PackageDocument.Factory.parse(in, opts);
            return new PackageDefinitionImpl(doc);
        } finally {
            if (in != null) {
                try { in.close(); } catch (IOException e) { /* Ignore */ }
            }
        }
    }
}
