/* $Id: ActivityDefinitionPropertyComponent.java,v 1.2 2007/12/10 09:34:42 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfadm.engineview;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.util.Arrays;
import java.util.List;
import java.util.Properties;
import java.util.Set;

import javax.swing.BorderFactory;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTable;
import javax.swing.JTextArea;
import javax.swing.ListSelectionModel;
import javax.swing.border.Border;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.AbstractTableModel;

import jp.co.argo21.nautica.commons.util.ResourceManager;
import jp.co.argo21.nautica.tool.wfadm.WorkflowAdminManager;
import jp.co.argo21.nautica.tool.wfadm.common.DialogComponent;
import jp.co.argo21.nautica.workflow.definition.TransitionDefinition;
import jp.co.argo21.nautica.workflow.definition.VariableDefinition;
import jp.co.argo21.nautica.workflow.expression.Expression;

/**
 * アクティビティ定義用プロパティ用画面コンテンツ
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public class ActivityDefinitionPropertyComponent
extends DialogComponent
{
	/** シリアルバージョンID */
	private static final long serialVersionUID = 1L;

	/** プロパティテーブルの列名 */
	private static String[] pColumnNames;

	/** 遷移テーブルの列名 */
	private static String[] tColumnNames;

	/** 遷移条件のタイプ名 */
	private static String[] conditionTypes;

	/** 枠線 */
	protected static final Border border = BorderFactory.createEtchedBorder();

	/** 遷移テーブル */
	private JTable transitionTable;

	/** 条件表示エリア */
	private JTextArea conditionArea;

	/** プロパティ情報 */
	private Properties properties;
	
	/** 遷移一覧 */
	private List<TransitionDefinition> transitions;
	
	/** プロセス変数一覧 */
	private List<VariableDefinition> variables;
	
	/**
	 * 定数の初期化
	 */
	static
	{
		ResourceManager rm = WorkflowAdminManager.getResourceManager();

		pColumnNames = new String[2];
		pColumnNames[0] = rm.getResource("PropertyComponent.columnName.0");
		pColumnNames[1] = rm.getResource("PropertyComponent.columnName.1");

		tColumnNames = new String[2];
		tColumnNames[0] = rm.getResource("TransitionTableModel.columnName.0");
		tColumnNames[1] = rm.getResource("TransitionTableModel.columnName.1");

		conditionTypes = new String[5];
		conditionTypes[0] = rm.getResource("TransitionTableModel.conditionType.0");
		conditionTypes[1] = rm.getResource("TransitionTableModel.conditionType.1");
		conditionTypes[2] = rm.getResource("TransitionTableModel.conditionType.2");
		conditionTypes[3] = rm.getResource("TransitionTableModel.conditionType.3");
		conditionTypes[4] = rm.getResource("TransitionTableModel.conditionType.4");
	}

	/**
	 * プロパティ用画面を生成する。
	 *
	 * @param prop プロパティ
	 * @param trs 遷移情報
	 * @param vars プロセス変数
	 */
	ActivityDefinitionPropertyComponent(Properties prop, List<TransitionDefinition> trs, List<VariableDefinition> vars)
	{
		ResourceManager rm = WorkflowAdminManager.getResourceManager();

		properties = prop;
		transitions = trs;
		variables = vars;
		
		JPanel pPanel = createPropertyPanel(rm);
		addTab(rm.getResource("ActivityDefinitionPropertyComponent.tab.0"), pPanel);
		JPanel tPanel = createTransitionInfoPanel(rm);
		addTab(rm.getResource("ActivityDefinitionPropertyComponent.tab.1"), tPanel);

		validate();
	}
	
	/**
	 * プロパティ一覧のパネルを生成する。
	 * 
	 * @param rm リソースマネージャ
	 * @return パネル
	 */
	private JPanel createPropertyPanel(ResourceManager rm)
	{
		JPanel panel = new JPanel();
		panel.setLayout(new BorderLayout());
		panel.setBorder(border);
		Dimension size = new Dimension(500, 150);
		panel.setSize(size);
		panel.setPreferredSize(size);

		JTable propertyTable = new JTable(new PropertyTableModel());
		propertyTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		JScrollPane sp = new JScrollPane();
		sp.setViewportView(propertyTable);

		panel.add(sp,  BorderLayout.CENTER);

		validate();
		
		return panel;
	}
	
	/**
	 * 遷移情報のパネルを生成する。
	 * 
	 * @param rm リソースマネージャ
	 * @return パネル
	 */
	private JPanel createTransitionInfoPanel(ResourceManager rm)
	{
		JPanel panel = new JPanel();
		panel.setLayout(new BorderLayout());
		panel.setBorder(border);
		Dimension size = new Dimension(500, 200);
		panel.setSize(size);
		panel.setPreferredSize(size);

		transitionTable = new JTable(new TransitionTableModel());
		transitionTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		ListSelectionModel tlsm = transitionTable.getSelectionModel();
		tlsm.addListSelectionListener(new TransitionSelectAction());
		JScrollPane sp1 = new JScrollPane();
		sp1.setViewportView(transitionTable);
		sp1.setPreferredSize(new Dimension(500, 100));
		
		conditionArea = new JTextArea();
		conditionArea.setLineWrap(true);
		conditionArea.setEditable(false);
		JScrollPane sp2 = new JScrollPane();
		sp2.setViewportView(conditionArea);
		
		JSplitPane split = new JSplitPane(JSplitPane.VERTICAL_SPLIT, sp1, sp2);
		split.setDividerLocation(120);
		
		panel.add(split,  BorderLayout.CENTER);

		validate();

		return panel;
	}

	/**
	 * インスペクタを初期化する。
	 * 
	 * @see jp.co.argo21.nautica.tool.wfadm.common.DialogComponent#init()
	 */
	public void init()
	{
	}

	/**
	 * 特に処理はない。
	 *
	 * @return 常にtrue
	 * @see jp.co.argo21.nautica.tool.wfadm.common.DialogComponent#commit()
	 */
	public boolean commit()
	{
		return true;
	}
	
	/**
	 * プロパティテーブル制御モデル
	 */
	class PropertyTableModel extends AbstractTableModel
	{
		/** シリアルバージョンID */
		private static final long serialVersionUID = 1L;

		/**
		 * プロパティテーブル制御モデルを生成する。
		 */
		PropertyTableModel()
		{
		}

		/**
		 * データソースオブジェクトが管理する列数を返す。
		 * JTableはこのメソッドを使って、初期化時に作成および表示する
		 * 必要のある列数を判断する。
		 *
		 * @return	モデルの列数
		 */
		public int getColumnCount()
		{
			return pColumnNames.length;
		}

		/**
		 * データソースオブジェクトが管理するレコード数を返す。
		 * JTableはこのメソッドを使って、作成および表示する必要のある行数を判断する。
		 *
		 * @return	モデルの行数
		 */
		public int getRowCount()
		{
			if (properties == null) return 0;

			return properties.size();
		}

		/**
		 * colにある列の名前を返す。
		 * このメソッドは、テーブルの列ヘッダ名を初期化するために使われる。
		 *
		 * @param	col		列のインデックス
		 * @return	列ヘッダ名
		 */
		public String getColumnName(int col)
		{
			return pColumnNames[col];
		}

		/**
		 * セルの値を返す。
		 *
		 * @param	row		値が参照される行
		 * @param	col		値が参照される列
		 * @return	指定されたセルにある値
		 */
		public Object getValueAt(int row, int col)
		{
			if (properties == null) return "";

			//キー項目をソートする。
			Set<?> keyset = properties.keySet();
			Object[] keys = keyset.toArray(); 
			Arrays.sort(keys);
			
			if (col == 0) {
				for (int i = 0; i < keys.length; i++) {
					if (i == row) return keys[i].toString();
				}
			} else if (col == 1) {
				for (int i = 0; i < keys.length; i++) {
					if (i == row) {
						return properties.get(keys[i]).toString();
					}
				}
			}
			return "";
		}

		/**
		 * 列のClassを返す。
		 *
		 * @param	col		列
		 * @return	モデルのオブジェクト値の共通の上位クラス
		 */
		public Class<?> getColumnClass(int col)
		{
			return String.class;
		}

		/**
		 * rowとcolにあるセルが編集可能な場合はtrueを返す。
		 * そうでない場合、セルでsetValueAt()を呼び出しても、そのセルの値は変更されない。
		 *
		 * @param	row		値が参照される行
		 * @param	col		値が参照される列
		 * @return	セルが編集可能な場合はtrue
		 */
		public boolean isCellEditable(int row, int col)
		{
			return false;
		}
	}

	/**
	 * 遷移が選択された場合の処理
	 */	
	class TransitionSelectAction implements ListSelectionListener
	{
		/**
		 * 選択された遷移情報を表示する。
		 * 
		 * @param e	イベント
		 * @see javax.swing.event.ListSelectionListener#valueChanged(javax.swing.event.ListSelectionEvent)
		 */
		public void valueChanged(ListSelectionEvent e)
		{
			if (e.getValueIsAdjusting()) return;

			ListSelectionModel lsm = (ListSelectionModel)e.getSource();
			if (lsm.isSelectionEmpty() == false) {
				int selected = lsm.getMinSelectionIndex();
				Expression exp = ((TransitionDefinition)transitions.get(selected)).getCondition();
				if (exp != null) {
					conditionArea.setText(exp.getExpressionString(variables));
				} else {
					conditionArea.setText("");
				}
			}
		}
	}
	
	/**
	 * 遷移テーブル制御モデル
	 */
	class TransitionTableModel extends AbstractTableModel
	{
		/** シリアルバージョンID */
		private static final long serialVersionUID = 1L;

		/**
		 * 遷移テーブル制御モデルを生成する。
		 */
		TransitionTableModel()
		{
		}

		/**
		 * データソースオブジェクトが管理する列数を返す。
		 * JTableはこのメソッドを使って、初期化時に作成および表示する
		 * 必要のある列数を判断する。
		 *
		 * @return	モデルの列数
		 */
		public int getColumnCount()
		{
			return tColumnNames.length;
		}

		/**
		 * データソースオブジェクトが管理するレコード数を返す。
		 * JTableはこのメソッドを使って、作成および表示する必要のある行数を判断する。
		 *
		 * @return	モデルの行数
		 */
		public int getRowCount()
		{
			return transitions.size();
		}

		/**
		 * colにある列の名前を返す。
		 * このメソッドは、テーブルの列ヘッダ名を初期化するために使われる。
		 *
		 * @param	col		列のインデックス
		 * @return	列ヘッダ名
		 */
		public String getColumnName(int col)
		{
			return tColumnNames[col];
		}

		/**
		 * セルの値を返す。
		 *
		 * @param	row		値が参照される行
		 * @param	col		値が参照される列
		 * @return	指定されたセルにある値
		 */
		public Object getValueAt(int row, int col)
		{
			TransitionDefinition tr = (TransitionDefinition)transitions.get(row);
			if (col == 0) {
				return tr.getDestination().getName();
			} else {
				int type = tr.getType();
				if (type == TransitionDefinition.OTHERWISE) {
					return conditionTypes[1];
				} else if (type == TransitionDefinition.EXCEPTION) {
					return conditionTypes[2];
				} else if (type == TransitionDefinition.DEFAULTEXCEPTION) {
					return conditionTypes[3];
				} else {
					if (tr.getCondition() == null) {
						return conditionTypes[4];
					} else {
						return conditionTypes[0];
					}
				}
			}
		}

		/**
		 * 列のClassを返す。
		 *
		 * @param	col		列
		 * @return	モデルのオブジェクト値の共通の上位クラス
		 */
		public Class<?> getColumnClass(int col)
		{
			return String.class;
		}

		/**
		 * rowとcolにあるセルが編集可能な場合はtrueを返す。
		 * そうでない場合、セルでsetValueAt()を呼び出しても、そのセルの値は変更されない。
		 *
		 * @param	row		値が参照される行
		 * @param	col		値が参照される列
		 * @return	セルが編集可能な場合はtrue
		 */
		public boolean isCellEditable(int row, int col)
		{
			return false;
		}
	}
}
