/* $Id: EngineViewController.java,v 1.2 2007/12/10 09:34:42 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfadm.engineview;

import java.util.Arrays;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;

import jp.co.argo21.nautica.commons.swing.ExceptionViewer;
import jp.co.argo21.nautica.tool.wfadm.WorkflowAdminManager;
import jp.co.argo21.nautica.tool.wfadm.definition.loader.DefinitionLoader;
import jp.co.argo21.nautica.tool.wfadm.pref.EngineInfo;
import jp.co.argo21.nautica.workflow.client.WorkflowServiceAccessor;
import jp.co.argo21.nautica.workflow.definition.PackageDefinition;
import jp.co.argo21.nautica.workflow.wfmc.DefinitionRepository;
import jp.co.argo21.nautica.workflow.wfmc.WorkflowEngineHandler;
import jp.co.argo21.nautica.workflow.wfmc.Activity;
import jp.co.argo21.nautica.workflow.wfmc.ActivityState;
import jp.co.argo21.nautica.workflow.wfmc.Attribute;
import jp.co.argo21.nautica.workflow.wfmc.Filter;
import jp.co.argo21.nautica.workflow.wfmc.ProcDef;
import jp.co.argo21.nautica.workflow.wfmc.Process;
import jp.co.argo21.nautica.workflow.wfmc.ProcessState;

/**
 * エンジンとクライアント間の関係を管理する。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public class EngineViewController
{
	/** エンジン情報 */
	private EngineInfo engineInfo;
	/** プロセス定義名-パスマッピング */
	private Map<String, String> pathMap = new Hashtable<String, String>();
	/** パス-プロセス定義マッピング */
	private Map<String, PackageDefinition> definitionMap = new Hashtable<String, PackageDefinition>();
	
	/**
	 * EngineManagerを生成する。
	 */
	public EngineViewController(EngineInfo info)
	{
		engineInfo = info;
	}

	/**
	 * エンジンへの接続を確認する。
	 *
	 * @return 接続確認が通った場合はtrue
	 */
	public boolean checkEngineConnection()
	{
		WorkflowServiceAccessor wsa = engineInfo.getWorkflowServiceAccessor();
		String session = null;
		try {
			session = wsa.open();
        	return true;
		} catch (Exception ex) {
			ExceptionViewer ev = WorkflowAdminManager.getExceptionViewer();
			ev.setThrowable(ex);
			return false;
		} finally {
			if (session != null) {
				try {
					wsa.close();
				} catch (Exception ex) {
					ExceptionViewer ev = WorkflowAdminManager.getExceptionViewer();
					ev.setThrowable(ex);
				}
			}
		}
	}
	
	/**
	 * フィルターで指定されたプロセス定義を取得する。
	 *
	 * @param filter フィルター。null設定可能。
	 * @return プロセス定義情報
	 * @throws Exception 任意の例外
	 */
	ProcDef[] getProcessDefinition(Filter filter) throws Exception
	{
		WorkflowServiceAccessor wsa = engineInfo.getWorkflowServiceAccessor();
		String session = null;
		try {
			session = wsa.open();
			DefinitionRepository dr = wsa.getDefinitionRepository();
			ProcDef[] defs = dr.getProcessDefinitions(session, filter);
			return defs;
		} catch (Exception ex) {
			throw ex;
		} finally {
			if (session != null) {
				try {
					wsa.close();
				} catch (Exception ex) {
					ExceptionViewer ev = WorkflowAdminManager.getExceptionViewer();
					ev.setThrowable(ex);
				}
			}
		}
	}
	
	/**
	 * フィルターで指定されたプロセスを取得する。
	 *
	 * @param filter フィルター。null設定可能。
	 * @return プロセス情報
	 * @throws Exception 任意の例外
	 */
	Process[] getProcess(Filter filter) throws Exception
	{
		WorkflowServiceAccessor wsa = engineInfo.getWorkflowServiceAccessor();
		String session = null;
		
		try {
			session = wsa.open();
			WorkflowEngineHandler weh = wsa.getWorkflowEngineHandler();
			Process[] procs = weh.getProcesses(session, filter);
			return procs;
		} catch (Exception ex) {
			throw ex;
		} finally {
			if (session != null) {
				try {
					wsa.close();
				} catch (Exception ex) {
					ExceptionViewer ev = WorkflowAdminManager.getExceptionViewer();
					ev.setThrowable(ex);
				}
			}
		}
	}
	
	/**
	 * プロセスIDで指定されたプロセスを取得する。
	 *
	 * @param pid プロセスID。
	 * @return プロセス情報
	 * @throws Exception 任意の例外
	 */
	Process getProcess(String pid) throws Exception
	{
		WorkflowServiceAccessor wsa = engineInfo.getWorkflowServiceAccessor();
		String session = null;
		
		try {
			session = wsa.open();
			WorkflowEngineHandler weh = wsa.getWorkflowEngineHandler();
			Process proc = weh.getProcess(session, pid);
			return proc;
		} catch (Exception ex) {
			throw ex;
		} finally {
			if (session != null) {
				try {
					wsa.close();
				} catch (Exception ex) {
					ExceptionViewer ev = WorkflowAdminManager.getExceptionViewer();
					ev.setThrowable(ex);
				}
			}
		}
	}
	
	/**
	 * プロセスとフィルターで指定されたアクティビティを取得する。
	 *
	 * @param proc プロセス
	 * @param filter フィルター。null設定可能。
	 * @return アクティビティ情報
	 * @throws Exception 任意の例外
	 */
	Activity[] getActivities(Process proc, Filter filter) throws Exception
	{
		WorkflowServiceAccessor wsa = engineInfo.getWorkflowServiceAccessor();
		String session = null;

		try {
			String pid = proc.getID();
			session = wsa.open();
			WorkflowEngineHandler weh = wsa.getWorkflowEngineHandler();
			Activity[] acts = weh.getActivities(session, pid, filter);
			return acts;
		} catch (Exception ex) {
			throw ex;
		} finally {
			if (session != null) {
				try {
					wsa.close();
				} catch (Exception ex) {
					ExceptionViewer ev = WorkflowAdminManager.getExceptionViewer();
					ev.setThrowable(ex);
				}
			}
		}
	}
	
	/**
	 * 保持しているエンジン情報を返す。
	 *
	 * @return エンジン情報
	 */
	EngineInfo getEngineInfo()
	{
		return engineInfo;
	}
	
	/**
	 * 引数に指定されたプロセス定義を含むパッケージ定義を取得する。
	 *
	 * @param pdname プロセス定義名
	 * @return パッケージ定義情報
	 * @throws Exception 任意の例外
	 */
	PackageDefinition getPackgeDefinition(String pdname)
	throws Exception
	{
		String ename = engineInfo.getName();
		String path = getDefinitionPath(ename, pdname);
		PackageDefinition def = getDefinition(ename, path);
		return def;
	}
	
	/**
	 * プロセスの状態を変更する。
	 *
	 * @param pid プロセスID
	 * @param state プロセス状態
	 * @throws Exception 任意の例外
	 */
	void changeProcessState(String pid, ProcessState state)
	throws Exception
	{
		WorkflowServiceAccessor wsa = engineInfo.getWorkflowServiceAccessor();
		String session = null;

		try {
			session = wsa.open();
			WorkflowEngineHandler weh = wsa.getWorkflowEngineHandler();
			weh.changeProcessState(session, pid, state);
		} catch (Exception ex) {
			throw ex;
		} finally {
			if (session != null) {
				try {
					wsa.close();
				} catch (Exception ex) {
					ExceptionViewer ev = WorkflowAdminManager.getExceptionViewer();
					ev.setThrowable(ex);
				}
			}
		}
	}
	
	/**
	 * アクティビティの状態を変更する。
	 *
	 * @param pid プロセスID
	 * @param aid アクティビティID
	 * @param state アクティビティ状態
	 * @throws Exception 任意の例外
	 */
	void changeActivityState(String pid, String aid, ActivityState state)
	throws Exception
	{
		WorkflowServiceAccessor wsa = engineInfo.getWorkflowServiceAccessor();
		String session = null;

		try {
			session = wsa.open();
			WorkflowEngineHandler weh = wsa.getWorkflowEngineHandler();
			weh.changeActivityState(session, pid, aid, state);
		} catch (Exception ex) {
			throw ex;
		} finally {
			if (session != null) {
				try {
					wsa.close();
				} catch (Exception ex) {
					ExceptionViewer ev = WorkflowAdminManager.getExceptionViewer();
					ev.setThrowable(ex);
				}
			}
		}
	}
	
	/**
	 * プロセスの状態を変更する。
	 *
	 * @param pid プロセスID
	 * @param filter フィルター。null指定可。
	 * @return プロセス変数一覧
	 * @throws Exception 任意の例外
	 */
	List<Attribute> getProcessAttributes(String pid, Filter filter)
	throws Exception
	{
		WorkflowServiceAccessor wsa = engineInfo.getWorkflowServiceAccessor();
		String session = null;

		try {
			session = wsa.open();
			WorkflowEngineHandler weh = wsa.getWorkflowEngineHandler();
			Attribute[] attrs = weh.getProcessAttributes(session, pid, filter);
			return Arrays.asList(attrs);
		} catch (Exception ex) {
			throw ex;
		} finally {
			if (session != null) {
				try {
					wsa.close();
				} catch (Exception ex) {
					ExceptionViewer ev = WorkflowAdminManager.getExceptionViewer();
					ev.setThrowable(ex);
				}
			}
		}
	}
	
	/**
	 * プロセスの状態を変更する。
	 *
	 * @param pid プロセスID
	 * @param attrs プロセス変数一覧
	 * @throws Exception 任意の例外
	 */
	void setProcessAttributes(String pid, List<Attribute> attrs)
	throws Exception
	{
		WorkflowServiceAccessor wsa = engineInfo.getWorkflowServiceAccessor();
		String session = null;
		if (attrs == null) return;

		try {
			session = wsa.open();
			WorkflowEngineHandler weh = wsa.getWorkflowEngineHandler();
			for (Attribute attr : attrs) {
				weh.assignProcessAttribute(session, pid, attr);
			}
		} catch (Exception ex) {
			throw ex;
		} finally {
			if (session != null) {
				try {
					wsa.close();
				} catch (Exception ex) {
					ExceptionViewer ev = WorkflowAdminManager.getExceptionViewer();
					ev.setThrowable(ex);
				}
			}
		}
	}
	
	/**
     * 指定されたプロセス定義名に対応するパス名を返す。
     * なければ null を返す。
     * 
     * @param eame エンジン名
     * @param pdname プロセス定義名
     * @return パス名
	 * @throws Exception 任意の例外
	 */
	private String getDefinitionPath(String ename, String pdname)
	throws Exception
	{
		String path = pathMap.get(pdname);
		
		if (path == null) {
			DefinitionLoader loader = DefinitionLoader.getInstance();
			path = loader.getDefinitionPath(ename, pdname);
		}
		return path;
	}

	/**
     * 定義オブジェクトを返す。
     * 定義が存在しない場合は null を返す。
     * 
     * @param ename エンジン名
     * @param path パス名
     * @return 定義オブジェクト
	 * @throws Exception 任意の例外
	 */
	private PackageDefinition getDefinition(String ename, String path)
	throws Exception
	{
		PackageDefinition def = definitionMap.get(path);
		
		if (def == null) {
			DefinitionLoader loader = DefinitionLoader.getInstance();
			def = loader.getDefinition(ename, path);
		}
		return def;
	}
}
