/* $Id: ProcessDefinitionNode.java,v 1.2 2007/12/10 09:34:42 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfadm.engineview;

import java.util.Enumeration;
import java.util.Properties;
import java.util.Vector;

import javax.swing.tree.TreeNode;

import jp.co.argo21.nautica.commons.util.ResourceManager;
import jp.co.argo21.nautica.workflow.filter.GroupFilter;
import jp.co.argo21.nautica.workflow.filter.NameFilter;
import jp.co.argo21.nautica.workflow.wfmc.Filter;
import jp.co.argo21.nautica.workflow.wfmc.ProcDef;
import jp.co.argo21.nautica.workflow.wfmc.Process;
import jp.co.argo21.nautica.workflow.wfmc.ProcessDefinitionState;

/**
 * エンジンビューのプロセスツリーのセカンドノードである。
 * 子ノードとしてプロセスノードを持つ。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public class ProcessDefinitionNode implements EngineTreeNode
{
	/** 検索キープリフィックス */
	public static final String PDID = "pdid:";
	

	/** 親ノード */
	private EngineNode parentNode;
	
	/** プロセス定義情報 */
	private ProcDef procDef;

	/** 子ノードの配列 */
	private Vector<ProcessNode> children;
	
	/** デフォルトのフィルター */
	private Filter defaultFilter;
	
	/** 現在設定中のフィルター */
	private Filter currentFilter;

	/**
	 * プロセス定義ノードを生成する。
	 */
	public ProcessDefinitionNode(EngineNode parent, ProcDef def)
	{
		parentNode = parent;
		procDef = def;
		children = new Vector<ProcessNode>();
		String fs = PDID + procDef.getID();
		defaultFilter = new NameFilter(fs, NameFilter.EQUAL);
		currentFilter = defaultFilter;
	}

	/**
	 * 子ノードをEnumerationで返す。
	 *
	 * @return 子ノード
	 * @see javax.swing.tree.TreeNode#children()
	 */
	public Enumeration<?> children()
	{
		return children.elements();
	}

	/**
	 * 子ノードを受け入れるかどうかを返す。
	 *
	 * @return 判定値
	 * @see javax.swing.tree.TreeNode#getAllowsChildren()
	 */
	public boolean getAllowsChildren()
	{
		return false;
	}

	/**
	 * インデクスで指定された子ノードを返す。
	 *
	 * @param childIndex インデクス
	 * @return 子ノード
	 * @see javax.swing.tree.TreeNode#getChildAt(int)
	 */
	public TreeNode getChildAt(int childIndex)
	{
		return (TreeNode)children.get(childIndex);
	}

	/**
	 * 子ノードの数を返す。
	 *
	 * @return 子ノード数
	 * @see javax.swing.tree.TreeNode#getChildCount()
	 */
	public int getChildCount()
	{
		return children.size();
	}

	/**
	 * 指定されたノードのインデクス値を返す。
	 *
	 * @param node ノード
	 * @return インデクス
	 * @see javax.swing.tree.TreeNode#getIndex(javax.swing.tree.TreeNode)
	 */
	public int getIndex(TreeNode node)
	{
		return children.indexOf(node);
	}
	
	/**
	 * 親ノードを返す。
	 *
	 * @return 親ノード
	 * @see javax.swing.tree.TreeNode#getParent()
	 */
	public TreeNode getParent()
	{
		return parentNode;
	}

	/**
	 * 末端ノードかどうかを返す。
	 *
	 * @return 末端ノードならtrue
	 * @see javax.swing.tree.TreeNode#isLeaf()
	 */
	public boolean isLeaf()
	{
		return false;
	}
	
	/**
	 * ノード名を返す。
	 *
	 * @return
	 * @see java.lang.Object#toString()
	 */
	public String toString()
	{
		return procDef.getID();
	}
	
	/**
	 * ノードの情報を更新する。
	 *
	 * @param filter
	 * @throws Exception
	 * @see jp.co.argo21.nautica.tool.wfadm.engineview.EngineTreeNode#reload(jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	public void reload(Filter filter) throws Exception
	{
		if (filter != null) {
			if (filter instanceof GroupFilter) {
				Filter[] sub = ((GroupFilter)filter).getChild();
				sub[0] = defaultFilter;
			}
			currentFilter = filter;
		} else {
			currentFilter = defaultFilter;
		}
		
		EngineViewFrame frame = getFrame();
		EngineViewController evc = frame.getEngineViewController();
		Process[] procs = evc.getProcess(currentFilter);
		children.clear();
		for (int i = 0; i < procs.length; i++) {
			ProcessNode child = new ProcessNode(this, procs[i]);
			children.add(child);
		}
	}
	
	/**
	 * ノードのプロパティ情報を返す。
	 *
	 * @param rm リソースマネージャ
	 * @return プロパティ
	 * @see jp.co.argo21.nautica.tool.wfadm.engineview.EngineTreeNode#getProperties()
	 */
	public Properties getProperties(ResourceManager rm)
	{
		Properties prop = new Properties();

		String name = procDef.getName();
		if (name == null) name = "";
		ProcessDefinitionState state = procDef.getState();
		String stateString = null;
		String stateInt = null;
		if (state == null) {
			stateString = "";
			stateInt = "";
		} else {
			stateString = state.toString();
			stateInt = "" + state.toInt();
		}
		prop.put(rm.getResource("ProcessDefinitionNode.key.0"), procDef.getID());
		prop.put(rm.getResource("ProcessDefinitionNode.key.1"), name);
		prop.put(rm.getResource("ProcessDefinitionNode.key.2"), stateString);
		prop.put(rm.getResource("ProcessDefinitionNode.key.3"), stateInt);

		return prop;
	}
	
	/**
	 * プロセス定義情報を返す。
	 *
	 * @return プロセス定義情報
	 */
	public ProcDef getProcDef()
	{
		return procDef;
	}

	/**
	 * エンジンビューフレームを返す。
	 *
	 * @return エンジンビューフレーム
	 */
	EngineViewFrame getFrame()
	{
		return parentNode.getFrame();
	}
}
