using System;
using System.Collections.Generic;
using System.Configuration;
using System.Data;
using System.Data.SqlClient;
using System.Diagnostics;
using System.Text;
using NDac.Datas.QueryGenerators;

namespace NDac.Datas
{
	/// <summary>
	/// データセットホルダーを表します。
	/// </summary>
	public partial class DataSetHolder
	{
		private string						_connectionString;
		private DataSet						_ds;
		private DataAdapterHolderCollection	_adapterHolders = new DataAdapterHolderCollection();
		private TraceSource					_trace = new TraceSource( "DataAccessLog" );

		/// <summary>
		/// コンストラクタ デフォルトコンストラクタを使用する場合は、必ずFillDefaultConnectionStringを実装して下さい。
		/// </summary>
		/// <param name="connectionString">接続文字列</param>
		public DataSetHolder()
			: this( DefaultConnectionString )
		{
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="connectionString">接続文字列</param>
		public DataSetHolder( string connectionString )
		{
			this._connectionString = connectionString;

			this._ds = new DataSet();

			this.Constructed();
		}

		/// <summary>
		/// デフォルトの接続文字列を表します。
		/// </summary>
		private static string DefaultConnectionString
		{
			get
			{
				ConnectionStringHolder connectionStringHolder = new ConnectionStringHolder();

				FillDefaultConnectionString( connectionStringHolder );

				return( connectionStringHolder.ConnectionString );
			}
		}

		/// <summary>
		/// インデクサ
		/// </summary>
		/// <param name="tableName">テーブル名</param>
		/// <returns>データテーブル</returns>
		public DataTable this[ string tableName ]
		{
			get
			{
				return( this._ds.Tables[ tableName ] );
			}
		}

		/// <summary>
		/// 保持しているデータを表します。
		/// </summary>
		public DataSet Data
		{
			get
			{
				return( this._ds );
			}
		}

		/// <summary>
		/// 接続文字列を表します。
		/// </summary>
		public string ConnectionString
		{
			get
			{
				return( this._connectionString );
			}
		}

		/// <summary>
		/// トレースソースを表します。
		/// </summary>
		public TraceSource Trace
		{
			get
			{
				return( this._trace );
			}
		}

		/// <summary>
		/// 充填されているデータを全てクリアします。
		/// </summary>
		public void Clear()
		{
			this._ds.Clear();

			this._ds.Tables.Clear();

			this._adapterHolders.Clear();
		}

		/// <summary>
		/// 充填済みのテーブルを削除します。
		/// </summary>
		/// <param name="tableName">テーブル名</param>
		public void Remove( string tableName )
		{
			this._ds.Tables.Remove( tableName );

			this._adapterHolders.Remove( tableName );
		}

		/// <summary>
		/// 充填済みテーブル名の一覧を取得します。
		/// </summary>
		/// <returns>充填済みテーブル名の一覧</returns>
		public IList< string > GetTableNames()
		{
			List< string > tableNames = new List< string >();

			foreach( DataTable table in this._ds.Tables )
			{
				tableNames.Add( table.TableName );
			}

			return( tableNames );
		}

		/// <summary>
		/// テーブルが充填済みか判定します。
		/// </summary>
		/// <param name="tableName">テーブル名</param>
		/// <returns>テーブルが充填済みの場合trueが返ります。</returns>
		public bool HasTable( string tableName )
		{
			return( this._ds.Tables.Contains( tableName ) );
		}

		/// <summary>
		/// データを充填します。
		/// </summary>
		/// <param name="tableName">充填するテーブル名</param>
		/// <param name="commandText">充填するクエリ</param>
		public void FillData( string tableName, string commandText )
		{
			this.FillData( tableName, new SqlCommand( commandText ) );
		}

		/// <summary>
		/// データを充填します。
		/// </summary>
		/// <param name="tableName">充填するテーブル名</param>
		/// <param name="selectCommand">充填するDBコマンド</param>
		public void FillData( string tableName, SqlCommand selectCommand )
		{
			if( this.HasTable( tableName ) )
			{
				throw( new MultipleLoadException( "データの重複ロードは実行できません。" ) );
			}
			else
			{
				using( SqlConnection connection = new SqlConnection( this._connectionString ) )
				{
					connection.Open();

					selectCommand.Connection = connection;

					this.ChangeCommandTimeout( selectCommand );

					this.Fill( tableName, selectCommand );
				}
			}
		}

		/// <summary>
		/// データを充填します。
		/// </summary>
		/// <param name="tableName">テーブル名</param>
		/// <param name="selectCommand">充填するDBコマンド</param>
		private void Fill( string tableName, SqlCommand selectCommand )
		{
			SqlDataAdapter adapter = new SqlDataAdapter( selectCommand );

			adapter.SetCommandBuilder( new SqlCommandBuilder() );

			Stopwatch watch = Stopwatch.StartNew();

			try
			{
				adapter.Fill( this._ds, tableName );

				adapter.FillSchema( this._ds.Tables[ tableName ], SchemaType.Mapped );
			}
			finally
			{
				watch.Stop();

				this.WriteTrace( selectCommand, watch.ElapsedMilliseconds );
			}

			this._adapterHolders.Add( tableName, new DataAdapterHolder( tableName, adapter ) );
		}

		/// <summary>
		/// データを追加充填します。（但し既に充填されているプライマリキー情報が同一のデータについては充填対象から外されます。）
		/// </summary>
		/// <param name="tableName">充填するテーブル名</param>
		/// <param name="commandText">充填するクエリ</param>
		public void MoreFillData( string tableName, string commandText )
		{
			this.MoreFillData( tableName, new SqlCommand( commandText ) );
		}

		/// <summary>
		/// データを追加充填します。（但し既に充填されているプライマリキー情報が同一のデータについては充填対象から外されます。）
		/// </summary>
		/// <param name="tableName">充填するテーブル名</param>
		/// <param name="selectCommand">充填するDBコマンド</param>
		public void MoreFillData( string tableName, SqlCommand selectCommand )
		{
			using( SqlConnection connection = new SqlConnection( this._connectionString ) )
			{
				connection.Open();

				selectCommand.Connection = connection;

				this.ChangeCommandTimeout( selectCommand );

				this.MoreFill( tableName, selectCommand );
			}
		}

		/// <summary>
		/// データを追加充填します。（但し既に充填されているプライマリキー情報が同一のデータについては充填対象から外されます。）
		/// </summary>
		/// <param name="tableName">充填するテーブル名</param>
		/// <param name="selectCommand">充填するDBコマンド</param>
		private void MoreFill( string tableName, SqlCommand selectCommand )
		{
			DataSet ds = new DataSet();

			SqlDataAdapter adapter = new SqlDataAdapter( selectCommand );

			adapter.SetCommandBuilder( new SqlCommandBuilder() );

			adapter.Fill( ds, tableName );

			adapter.FillSchema( ds.Tables[ tableName ], SchemaType.Mapped );

			foreach( DataRow row in ds.Tables[ tableName ].Rows )
			{
				List< object > keyValues = new List< object >();

				foreach( DataColumn column in ds.Tables[ tableName ].PrimaryKey )
				{
					keyValues.Add( row[ column ] );
				}

				if( !this._ds.Tables[ tableName ].Rows.Contains( keyValues.ToArray() ) )
				{
					this._ds.Tables[ tableName ].ImportRow( row );
				}
			}
		}

		/// <summary>
		/// データを更新します。
		/// </summary>
		public void Update()
		{
			using( SqlConnection connection = new SqlConnection( this._connectionString ) )
			{
				connection.Open();

				this.FillSchemaTables( connection, this._adapterHolders );

				using( SqlTransaction transaction = connection.BeginTransaction() )
				{
					foreach( DataAdapterHolder adapterHolder in this._adapterHolders.DeadlockAvoidanceDataAdapterHolders )
					{
						if( adapterHolder.IsMustDbCommand )
						{
							this.UpdateDbCommand( transaction, adapterHolder );
						}
						else
						{
							this.UpdateDataAdapter( transaction, adapterHolder );
						}
					}

					transaction.Commit();
				}
			}
		}

		/// <summary>
		/// テーブルスキーマを充填します。
		/// </summary>
		/// <param name="connection">接続オブジェクト</param>
		/// <param name="adapterHolders">データアダプタホルダー</param>
		private void FillSchemaTables( SqlConnection connection, DataAdapterHolderCollection adapterHolders )
		{
			foreach( DataAdapterHolder adapterHolder in adapterHolders.DataAdapterHolders )
			{
				if( adapterHolder.IsMustDbCommand )
				{
					adapterHolder.SchemaTable = this.GetSchemaTable( connection, adapterHolder.TableName );
				}
			}
		}

		/// <summary>
		/// テーブルスキーマを取得します。
		/// </summary>
		/// <param name="connection">接続オブジェクト</param>
		/// <param name="tableName">テーブル名</param>
		/// <returns>テーブルスキーマ</returns>
		private DataTable GetSchemaTable( SqlConnection connection, string tableName )
		{
			DataTable schemaTable = new DataTable();

			SqlDataAdapter adapter = new SqlDataAdapter( string.Format( "SELECT TOP 0 * FROM {0}", tableName ), connection );

			adapter.FillSchema( schemaTable, SchemaType.Mapped );

			return( schemaTable );
		}

		/// <summary>
		/// データアダプタを使用しデータを更新します。
		/// </summary>
		/// <param name="transaction">トランザクション</param>
		/// <param name="adapterHolder">データアダプタホルダー</param>
		private void UpdateDataAdapter( SqlTransaction transaction, DataAdapterHolder adapterHolder )
		{
			adapterHolder.DataAdapter.SelectCommand.Connection = transaction.Connection;

			adapterHolder.DataAdapter.SelectCommand.Transaction = transaction;

			adapterHolder.DataAdapter.SetCommandBuilder( new SqlCommandBuilder() );

			adapterHolder.DataAdapter.Update( this._ds, adapterHolder.TableName );
			//this.UpdateWithTraceLog( adapterHolder.DataAdapter, adapterHolder.TableName );
		}

		/// <summary>
		/// コマンドを使用しデータを更新します。
		/// </summary>
		/// <param name="transaction">トランザクション</param>
		/// <param name="adapterHolder">データアダプタホルダー</param>
		private void UpdateDbCommand( SqlTransaction transaction, DataAdapterHolder adapterHolder )
		{
			foreach( DataRow row in this._ds.Tables[ adapterHolder.TableName ].Rows )
			{
				if( QueryGeneratorManager.Instance.HasGenerator( row.RowState ) )
				{
					IQueryGenerator generator = QueryGeneratorManager.Instance[ row.RowState ];

					SqlCommand command = new SqlCommand( generator.Generate( row, adapterHolder.SchemaTable ), transaction.Connection );

					command.Transaction = transaction;

					command.ExecuteNonQuery();
				}
			}

			this._ds.Tables[ adapterHolder.TableName ].AcceptChanges();
		}

		/// <summary>
		/// トレースソースにログを記憶しながら、データを更新します。
		/// </summary>
		/// <param name="adapter">データアダプタ</param>
		/// <param name="tableName">テーブル名</param>
		private void UpdateWithTraceLog( SqlDataAdapter adapter, string tableName )
		{
			Stopwatch watch = null;

			SqlRowUpdatingEventHandler updateing = delegate( object sender, SqlRowUpdatingEventArgs e )
			{
				watch = Stopwatch.StartNew();
			};

			SqlRowUpdatedEventHandler updated = delegate( object sender, SqlRowUpdatedEventArgs e )
			{
				watch.Stop();

				this.WriteTrace( e.Command, watch.ElapsedMilliseconds );
			};

			adapter.RowUpdating	+= updateing; 
			adapter.RowUpdated	+= updated; 

			try
			{
				adapter.Update( this._ds, tableName );
			}
			catch
			{
				throw;
			}
			finally
			{
				adapter.RowUpdating	-= updateing; 
				adapter.RowUpdated	-= updated; 
			}
		}

		/// <summary>
		/// レコードが存在するか判定します。
		/// </summary>
		/// <param name="commandText">SELECT条件のクエリ</param>
		/// <returns>SELECT条件に合うレコードが存在する場合trueを返します。</returns>
		public bool HasRecord( string commandText )
		{
			return( this.HasRecord( new SqlCommand( commandText ) ) );
		}

		/// <summary>
		/// レコードが存在するか判定します。
		/// </summary>
		/// <param name="selectCommand">SELECT条件のDBコマンド</param>
		/// <returns>SELECT条件に合うレコードが存在する場合trueを返します。</returns>
		public bool HasRecord( SqlCommand selectCommand )
		{
			using( SqlConnection connection = new SqlConnection( this._connectionString ) )
			{
				connection.Open();

				selectCommand.Connection = connection;

				SqlDataAdapter adapter = new SqlDataAdapter( selectCommand );

				adapter.SetCommandBuilder( new SqlCommandBuilder() );

				Stopwatch watch = Stopwatch.StartNew();

				DataTable table = new DataTable();

				try
				{
					adapter.Fill( table );

					return( 0 < table.Rows.Count ? true : false );
				}
				finally
				{
					watch.Stop();

					this.WriteTrace( selectCommand, watch.ElapsedMilliseconds );
				}
			}
		}

		/// <summary>
		/// トレースソースにログメッセージを追記します。
		/// </summary>
		/// <param name="command">DBコマンド</param>
		/// <param name="implementationMilliseconds">クエリ実行時間(msec)</param>
		private void WriteTrace( SqlCommand command, long implementationMilliseconds )
		{
			string message = "実行クエリ: {0}, 実行パラメータ: {1}, 実行時間(msec): {2}";

			message = message
						.Replace( "{0}", command.CommandText )
						.Replace( "{1}", this.ToParametersText( command.Parameters ) )
						.Replace( "{2}", implementationMilliseconds.ToString() );

			this._trace.TraceEvent( TraceEventType.Information, 1, message );
		}

		/// <summary>
		/// パラメータリストをテキストに変換します。
		/// </summary>
		/// <param name="parameters">パラメータのコレクション</param>
		/// <returns>パラメータリストのテキスト</returns>
		private string ToParametersText( SqlParameterCollection parameters )
		{
			StringBuilder builder = new StringBuilder();

			foreach( SqlParameter parameter in parameters )
			{
				if( 0 < builder.Length )
				{
					builder.Append( "," );
				}

				builder.Append( string.Format( "Name = {0} Value = [{1}]", parameter.ParameterName, parameter.Value ) );
			}

			return( builder.ToString() );
		}

		/// <summary>
		/// コマンドのタイムアウト値を変更します。
		/// </summary>
		/// <param name="command">コマンド</param>
		partial void ChangeCommandTimeout( SqlCommand command );

		/// <summary>
		/// 接続文字列ホルダーにデフォルトの接続文字列を充填します。
		/// <para>
		/// </para>
		/// <para>
		/// 例えば以下の様な実装を行って下さい。
		/// </para>
		/// <para>
		/// static partial void FillDefaultConnectionString( ConnectionStringHolder connectionStringHolder )
		/// </para>
		/// <para>
		/// {
		/// </para>
		/// <para>
		///	connectionStringHolder.ConnectionString = ConfigurationManager.ConnectionStrings[ "ConnectionString" ].ConnectionString;
		/// </para>
		/// <para>
		/// }
		/// </para>
		/// </summary>
		/// <param name="connectionStringHolder">接続文字列ホルダー</param>
		static partial void FillDefaultConnectionString( ConnectionStringHolder connectionStringHolder );

		/// <summary>
		/// コンストラクタが呼び出された後にコールされます。
		/// </summary>
		partial void Constructed();

		/// <summary>
		/// 接続文字列ホルダーを表します。
		/// </summary>
		internal class ConnectionStringHolder
		{
			private string _connectionString = string.Empty;

			/// <summary>
			/// 接続文字列を表します。
			/// </summary>
			public string ConnectionString
			{
				get
				{
					return( this._connectionString );
				}
				set
				{
					this._connectionString = value;
				}
			}
		}
	}
}

namespace System.Data.Common
{
	/// <summary>
	/// DataAdapterの拡張メソッド定義クラスを表します。
	/// </summary>
	public static class DataAdapterExtension
	{
		/// <summary>
		/// コマンドビルダーをセットします。
		/// </summary>
		/// <param name="adapter">データアダプタ</param>
		/// <param name="builder">コマンドビルダー</param>
		public static void SetCommandBuilder( this DbDataAdapter adapter, DbCommandBuilder builder )
		{
			builder.DataAdapter = adapter;
		}
	}
}

namespace System.Data.SqlClient
{
	/// <summary>
	/// SqlDataAdapterの拡張メソッド定義クラスを表します。
	/// </summary>
	public static class SqlDataAdapterExtension
	{
		/// <summary>
		/// コマンドビルダーをセットします。
		/// </summary>
		/// <param name="adapter">データアダプタ</param>
		/// <param name="builder">コマンドビルダー</param>
		public static void SetCommandBuilder( this SqlDataAdapter adapter, SqlCommandBuilder builder )
		{
			builder.DataAdapter = adapter;
		}
	}
}
