using System;
using System.Data;
using System.Reflection;
using NDac.Datas;
using NDac.Keys;

namespace NDac.Gateways
{
	/// <summary>
	/// テーブルデータゲートウェイの抽象クラスを表します。
	/// </summary>
	public abstract partial class TableDataGateway : ITableDataGateway< IKey >
	{
		private string			_tableName;
		private DataSetHolder	_holder;

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="tableName">テーブル名</param>
		/// <param name="connectionString">接続文字列</param>
		public TableDataGateway( string tableName, string connectionString )
			: this( tableName, new DataSetHolder( connectionString ) )
		{
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="tableName">テーブル名</param>
		/// <param name="holder">データセットホルダー</param>
		public TableDataGateway( string tableName, DataSetHolder holder )
		{
			this._tableName = tableName;

			this._holder = holder;
		}

		/// <summary>
		/// キー情報に一致したレコードが存在するかを判定します。
		/// </summary>
		/// <param name="loadAllKey">全データ充填キー</param>
		public bool HasValue( LoadAllKey loadAllKey )
		{
			string query = string.Format( "SELECT TOP 1 * FROM {0}", this._tableName );

			return( this._holder.HasRecord( query ) );
		}

		/// <summary>
		/// キー情報に一致したデータを充填します。
		/// </summary>
		/// <param name="loadAllKey">全データ充填キー</param>
		public void LoadBy( LoadAllKey loadAllKey )
		{
			this.LoadAll();
		}

		/// <summary>
		/// キー情報に一致したデータを充填します。
		/// </summary>
		/// <param name="schemaKey">テーブルスキーマ取得キー</param>
		public void LoadBy( TableSchemaKey schemaKey )
		{
			this.LoadSchema();
		}

		#region ITableDataGateway<IKey> メンバ

		/// <summary>
		/// キー情報に一致したレコードが存在するかを判定します。
		/// </summary>
		/// <param name="key"></param>
		/// <returns>キー情報に一致したレコードが存在する場合trueを返します。</returns>
		bool ITableDataGateway< IKey >.HasValue( IKey key )
		{
			return( ( ( ITableDataGateway )this ).HasValue( key ) );
		}

		/// <summary>
		/// キー情報に一致したデータを充填します。
		/// </summary>
		/// <param name="key">キー</param>
		void ITableDataGateway< IKey >.LoadBy( IKey key )
		{
			( ( ITableDataGateway )this ).LoadBy( key );
		}

		/// <summary>
		/// キー情報に一致したデータを追加充填します。
		/// </summary>
		/// <param name="key">キー</param>
		void ITableDataGateway< IKey >.MoreLoadBy( IKey key )
		{
			( ( ITableDataGateway )this ).MoreLoadBy( key );
		}

		#endregion

		#region ITableDataGateway メンバ

		/// <summary>
		/// データテーブルを表します。
		/// </summary>
		public DataTable Table
		{
			get
			{
				return( this._holder.Data.Tables[ this._tableName ] );
			}
		}

		/// <summary>
		/// テーブル名を表します。
		/// </summary>
		public string TableName
		{
			get
			{
				return( this._tableName );
			}
		}

		/// <summary>
		/// データセットホルダーを表します。
		/// </summary>
		public DataSetHolder Holder
		{
			get
			{
				return( this._holder );
			}
		}

		/// <summary>
		/// キー情報に一致したレコードが存在するかを判定します。
		/// </summary>
		/// <param name="key"></param>
		/// <returns>キー情報に一致したレコードが存在する場合trueを返します。</returns>
		bool ITableDataGateway.HasValue( IKey key )
		{
			try
			{
				MethodInfo method = this.GetType().GetMethod( "HasValue", new Type[]{ key.GetType() } );

				if( method != null )
				{
					object returnValue = method.Invoke( this, new object[]{ key } );

					return( ( bool )returnValue );
				}
				else
				{
					throw( new MissingMethodException( string.Format( "{0}をパラメータとするHasValueメソッドは存在しません。", key.GetType().ToString() ) ) );
				}
			}
			catch
			{
				throw;
			}
		}

		/// <summary>
		/// WHERE条件でレコードが存在するか判定します。
		/// </summary>
		/// <param name="whereClause">WHERE節</param>
		/// <returns>WHERE条件に合うレコードが存在する場合trueを返します。</returns>
		public bool HasWhere( string whereClause )
		{
			string query = string.Format( "SELECT TOP 1 * FROM {0} WHERE {1}", this._tableName, whereClause );

			return( this._holder.HasRecord( query ) );
		}

		/// <summary>
		/// テーブルより全データを充填します。
		/// </summary>
		public void LoadAll()
		{
			string query = string.Format( "SELECT * FROM {0}", this._tableName );

			this._holder.FillData( this._tableName, query );
		}

		/// <summary>
		/// キー情報に一致したデータを充填します。
		/// </summary>
		/// <param name="key">キー</param>
		void ITableDataGateway.LoadBy( IKey key )
		{
			try
			{
				MethodInfo method = this.GetType().GetMethod( "LoadBy", new Type[]{ key.GetType() } );

				if( method != null )
				{
					object returnValue = method.Invoke( this, new object[]{ key } );
				}
				else
				{
					throw( new MissingMethodException( string.Format( "{0}をパラメータとするLoadByメソッドは存在しません。", key.GetType().ToString() ) ) );
				}
			}
			catch
			{
				throw;
			}
		}

		/// <summary>
		/// テーブルよりスキーマ情報を充填します。
		/// </summary>
		public void LoadSchema()
		{
			string query = string.Format( "SELECT TOP 0 * FROM {0}", this._tableName );

			this._holder.FillData( this._tableName, query );
		}

		/// <summary>
		/// WHERE条件でデータを充填します。
		/// </summary>
		/// <param name="whereClause">WHERE節</param>
		public void LoadWhere( string whereClause )
		{
			string query = string.Format( "SELECT * FROM {0} WHERE {1}", this._tableName, whereClause );

			this._holder.FillData( this._tableName, query );
		}

		/// <summary>
		/// キー情報に一致したデータを追加充填します。
		/// </summary>
		/// <param name="key">キー</param>
		void ITableDataGateway.MoreLoadBy( IKey key )
		{
			try
			{
				MethodInfo method = this.GetType().GetMethod( "MoreLoadBy", new Type[]{ key.GetType() } );

				if( method != null )
				{
					object returnValue = method.Invoke( this, new object[]{ key } );
				}
				else
				{
					throw( new MissingMethodException( string.Format( "{0}をパラメータとするMoreLoadByメソッドは存在しません。", key.GetType().ToString() ) ) );
				}
			}
			catch
			{
				throw;
			}
		}

		#endregion
	}

	/// <summary>
	/// テーブルデータゲートウェイの抽象クラスを表します。
	/// </summary>
	/// <typeparam name="TKey">キーの型</typeparam>
	public abstract partial class TableDataGateway< TKey > : TableDataGateway, ITableDataGateway< TKey >
		where TKey : class, IKey
	{
		/// コンストラクタ
		/// </summary>
		/// <param name="tableName">テーブル名</param>
		/// <param name="connectionString">接続文字列</param>
		public TableDataGateway( string tableName, string connectionString ) : base( tableName, connectionString )
		{
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="tableName">テーブル名</param>
		/// <param name="holder">データセットホルダー</param>
		public TableDataGateway( string tableName, DataSetHolder holder ) : base( tableName, holder )
		{
		}

		/// <summary>
		/// キー情報に一致したレコードが存在するかを判定します。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		/// <returns>キー情報に一致したレコードが存在する場合trueを返します。</returns>
		public abstract bool HasValue( TKey primaryKey );

		/// <summary>
		/// キー情報に一致したデータを充填します。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		public abstract void LoadBy( TKey primaryKey );

		/// <summary>
		/// キー情報に一致したデータを追加充填します。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		public abstract void MoreLoadBy( TKey primaryKey );

		#region ITableDataGateway<TKey> メンバ

		/// <summary>
		/// キー情報に一致したレコードが存在するかを判定します。
		/// </summary>
		/// <param name="key">キー</param>
		/// <returns>キー情報に一致したレコードが存在する場合trueを返します。</returns>
		bool ITableDataGateway< TKey >.HasValue( TKey key )
		{
			return( this.HasValue( key ) );
		}

		/// <summary>
		/// キー情報に一致したデータを充填します。
		/// </summary>
		/// <param name="key">キー</param>
		void ITableDataGateway< TKey >.LoadBy( TKey key )
		{
			this.LoadBy( key );
		}

		/// <summary>
		/// キー情報に一致したデータを追加充填します。
		/// </summary>
		/// <param name="key">キー</param>
		void ITableDataGateway< TKey >.MoreLoadBy( TKey key )
		{
			this.MoreLoadBy( key );
		}

		#endregion
	}
}
