/*
 * Copyright 2009-2010 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.automata.nfa.op;

import java.util.Collections;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Set;

import net.morilib.automata.NFA;
import net.morilib.automata.NFAEdges;
import net.morilib.automata.NFAState;
import net.morilib.automata.TextBound;
import net.morilib.range.Interval;
import net.morilib.range.Range;
import net.morilib.util.IntervalMap;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2014/01/19
 */
public class JoinNFA<T, A, B> implements NFA<T, A, B> {

	//
	private final NFAEdges<T> eedg = new EpsilonEdge<T>(null);
	private final NFAState bgst = new NFAState() {};
	private final NFAState edst = new NFAState() {};

	//
	private NFA<T, A, B> nfa1, nfa2;
	private A end;

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#isState(net.morilib.automata.NFAState)
	 */
	@Override
	public boolean isState(NFAState o) {
		return (o == bgst || o == edst ||
				nfa1.isState(o) || nfa2.isState(o));
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getStates(net.morilib.automata.NFAState, java.lang.Object)
	 */
	@Override
	public Set<NFAState> getStates(NFAState state, T alphabet) {
		Set<NFAState> r;

		r = new HashSet<NFAState>();
		if(state == bgst) {
			// do nothing
		} else if(state == edst) {
			// do nothing
		} else if(nfa1.isState(state)) {
			r.addAll(nfa1.getStates(state, alphabet));
			if(nfa1.isFinal(state))  r.add(edst);
		} else if(nfa2.isState(state)) {
			r.addAll(nfa2.getStates(state, alphabet));
			if(nfa2.isFinal(state))  r.add(edst);
		}
		return r;
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getStates(net.morilib.automata.NFAState, net.morilib.range.Range)
	 */
	@Override
	public Set<NFAState> getStates(NFAState state, Range rng) {
		Set<NFAState> r;

		r = new HashSet<NFAState>();
		if(state == bgst) {
			// do nothing
		} else if(state == edst) {
			// do nothing
		} else if(nfa1.isState(state)) {
			r.addAll(nfa1.getStates(state, rng));
		} else if(nfa2.isState(state)) {
			r.addAll(nfa2.getStates(state, rng));
		}
		return r;
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getStates(net.morilib.automata.NFAState, java.util.EnumSet)
	 */
	@Override
	public Set<NFAState> getStates(NFAState state,
			EnumSet<TextBound> bound) {
		Set<NFAState> r;

		r = new HashSet<NFAState>();
		if(state == bgst) {
			// do nothing
		} else if(state == edst) {
			// do nothing
		} else if(nfa1.isState(state)) {
			r.addAll(nfa1.getStates(state, bound));
		} else if(nfa2.isState(state)) {
			r.addAll(nfa2.getStates(state, bound));
		}
		return r;
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getStatesEpsilon(net.morilib.automata.NFAState)
	 */
	@Override
	public Set<NFAState> getStatesEpsilon(NFAState state) {
		Set<NFAState> r;

		r = new HashSet<NFAState>();
		if(state == bgst) {
			r.addAll(nfa1.getInitialStates());
			r.addAll(nfa2.getInitialStates());
		} else if(state == edst) {
			// do nothing
		} else if(nfa1.isState(state)) {
			r.addAll(nfa1.getStatesEpsilon(state));
			if(nfa1.isFinal(state))  r.add(edst);
		} else if(nfa2.isState(state)) {
			r.addAll(nfa2.getStatesEpsilon(state));
			if(nfa2.isFinal(state))  r.add(edst);
		}
		return r;
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getStatesBound(net.morilib.automata.NFAState, java.util.EnumSet)
	 */
	@Override
	public Set<NFAState> getStatesBound(NFAState state,
			EnumSet<TextBound> bound) {
		Set<NFAState> r;

		r = new HashSet<NFAState>();
		if(state == bgst) {
			// do nothing
		} else if(state == edst) {
			// do nothing
		} else if(nfa1.isState(state)) {
			r.addAll(nfa1.getStatesBound(state, bound));
		} else if(nfa2.isState(state)) {
			r.addAll(nfa2.getStatesBound(state, bound));
		}
		return r;
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getInitialStates()
	 */
	@Override
	public Set<NFAState> getInitialStates() {
		return Collections.singleton(bgst);
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#isInitialState(net.morilib.automata.NFAState)
	 */
	@Override
	public boolean isInitialState(NFAState o) {
		return o == bgst;
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#isFinal(net.morilib.automata.NFAState)
	 */
	@Override
	public boolean isFinal(NFAState state) {
		return state == edst;
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#isFinalAny(java.util.Set)
	 */
	@Override
	public boolean isFinalAny(Set<NFAState> states) {
		for(NFAState s : states) {
			if(isFinal(s))  return true;
		}
		return false;
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getEdges(net.morilib.automata.NFAState)
	 */
	@Override
	public NFAEdges<T> getEdges(NFAState state) {
		if(state == bgst) {
			return new EpsilonEdge<T>(getStatesEpsilon(state));
		} else if(state == edst) {
			return eedg;
		} else if(nfa1.isState(state)) {
			return nfa1.getEdges(state);
		} else if(nfa2.isState(state)) {
			return nfa2.getEdges(state);
		} else {
			return eedg;
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#nextAlphabets(net.morilib.automata.NFAState)
	 */
	@Override
	public Set<Interval> nextAlphabets(NFAState state) {
		if(state == bgst) {
			return Collections.emptySet();
		} else if(state == edst) {
			return Collections.emptySet();
		} else if(nfa1.isState(state)) {
			return nfa1.nextAlphabets(state);
		} else if(nfa2.isState(state)) {
			return nfa2.nextAlphabets(state);
		} else {
			return Collections.emptySet();
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#nextAlphabets(java.util.Set)
	 */
	@Override
	public Iterable<Interval> nextAlphabets(Set<NFAState> states) {
		IntervalMap<Void> m = new IntervalMap<Void>();

		for(NFAState s : states) {
			for(Interval v : nextAlphabets(s)) {
				m.put(v, null);
			}
		}
		return m.keySet();
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#nextDiscreteAlphabets(net.morilib.automata.NFAState)
	 */
	@Override
	public Set<T> nextDiscreteAlphabets(NFAState state) {
		if(state == bgst) {
			return Collections.emptySet();
		} else if(state == edst) {
			return Collections.emptySet();
		} else if(nfa1.isState(state)) {
			return nfa1.nextDiscreteAlphabets(state);
		} else if(nfa2.isState(state)) {
			return nfa2.nextDiscreteAlphabets(state);
		} else {
			return Collections.emptySet();
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#nextDiscreteAlphabets(java.util.Set)
	 */
	@Override
	public Iterable<T> nextDiscreteAlphabets(Set<NFAState> states) {
		Set<T> m = new HashSet<T>();

		for(NFAState s : states) {
			m.addAll(nextDiscreteAlphabets(s));
		}
		return m;
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getAcceptedStates()
	 */
	@Override
	public Set<NFAState> getAcceptedStates() {
		return Collections.singleton(edst);
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getMatchTag(net.morilib.automata.NFAState)
	 */
	@Override
	public Set<B> getMatchTag(NFAState state) {
		if(state == bgst) {
			return Collections.emptySet();
		} else if(state == edst) {
			return Collections.emptySet();
		} else if(nfa1.isState(state)) {
			return nfa1.getMatchTag(state);
		} else if(nfa2.isState(state)) {
			return nfa2.getMatchTag(state);
		} else {
			return Collections.emptySet();
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getMatchTagEnd(net.morilib.automata.NFAState)
	 */
	@Override
	public Set<B> getMatchTagEnd(NFAState state) {
		if(state == bgst) {
			return Collections.emptySet();
		} else if(state == edst) {
			return Collections.emptySet();
		} else if(nfa1.isState(state)) {
			return nfa1.getMatchTagEnd(state);
		} else if(nfa2.isState(state)) {
			return nfa2.getMatchTagEnd(state);
		} else {
			return Collections.emptySet();
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getAccept(net.morilib.automata.NFAState)
	 */
	@Override
	public Set<A> getAccept(NFAState state) {
		if(state == edst) {
			return Collections.singleton(end);
		} else {
			return Collections.emptySet();
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#isAccepted(net.morilib.automata.NFAState)
	 */
	@Override
	public boolean isAccepted(NFAState state) {
		return !getAccept(state).isEmpty();
	}

}
