/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.awk.nano.value;

import java.util.Collection;
import java.util.List;

import net.morilib.awk.nano.AwkReturnException;
import net.morilib.awk.nano.expr.AwkExpression;
import net.morilib.awk.nano.expr.AwkFunctionPrototype;
import net.morilib.awk.nano.io.AwkFiles;
import net.morilib.awk.nano.namespace.AwkFunctionNamespace;
import net.morilib.awk.nano.namespace.AwkNamespace;

/**
 * awkiumユーザ関数のクラスです。
 * 
 * @author MORIGUCHI, Yuichiro 2013/03
 */
public class AwkUserFunction extends AwkFunction {

	private AwkExpression function;
	private String[] argnames;

	/**
	 * ユーザ関数を生成します。
	 * 
	 * @param name 関数名
	 * @param env  含まれる環境
	 * @param function 関数の実装
	 * @param argnames 引数の名称
	 */
	public AwkUserFunction(String name, AwkNamespace env,
			AwkExpression function, Collection<String> argnames) {
		super(name, env);
		this.function = function.compileInternally();
		this.argnames = argnames.toArray(new String[0]);
	}

	/**
	 * ユーザ関数を関数プロトタイプから生成します。
	 * 
	 * @param f   関数プロトタイプ
	 * @param env 含まれる環境
	 */
	public AwkUserFunction(AwkFunctionPrototype f, AwkNamespace env) {
		this(f.getName(), env, f.getFunction(), f.getArgnames());
	}

	AwkValue _apply(AwkNamespace ns, AwkFiles o, List<AwkValue> args) {
		AwkNamespace e = ns;
		AwkValue v;

		for(int i = 0; i < argnames.length; i++) {
			v = i < args.size() ? args.get(i) : AwkUndefined.UNDEF;
			e.assignArgument(argnames[i], v);
		}

		try {
			return function.eval(e, o);
		} catch(AwkReturnException f) {
			return f.getValue();
		}
	}

	@Override
	public AwkValue apply(AwkNamespace ns, AwkFiles o,
			List<AwkValue> args) {
		return _apply(env == null ?
				new AwkFunctionNamespace(ns) : env, o, args);
	}

	@Override
	public AwkValue init(AwkNamespace ns, AwkFiles o,
			List<AwkValue> args) {
		return _apply(new AwkFunctionNamespace(ns), o, args);
	}

	@Override
	public String getName() {
		return name;
	}

	@Override
	public boolean isEmpty() {
		return false;
	}

}
