/*
 * Copyright 2009 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.lisp.nano;

import java.util.ArrayList;
import java.util.List;

import net.morilib.lisp.nano.CompiledCode.Builder;

/**
 * 
 *
 *
 * @author MORIGUCHI, Yuichiro 2011/05/24
 */
/*package*/ class LispCompiler {

	//
	private LispMessage message;

	//
	/*package*/ LispCompiler(LispMessage msg) {
		message = msg;
	}

	//
	private void compileArgsBind(
			Datum bcdr,
			Environment env,
			Builder builder,
			Cons symcall,
			List<Cons> symlist,
			CodeExecutor exec,
			IntStack memento) {
		Datum prms = symcall.getCdr();
		List<Datum> vals = new ArrayList<Datum>();

		while(true) {
			if(prms == Nil.NIL) {
				if(bcdr != Nil.NIL) {
					throw message.getError(
							"err.parameter.insufficient");
				}
				break;
			} else if(prms instanceof Atom) {
				compile(bcdr, env, builder, symcall, false, symlist,
						exec, memento);
				builder.addBind(prms);
				break;
			} else if(!(bcdr instanceof Cons)) {
				throw message.getError("err.parameter.insufficient");
			} else if(prms instanceof Cons) {
				Datum a1 = ((Cons)prms).getCar();
				Datum a2 = ((Cons)bcdr).getCar();

				compile(a2, env, builder, symcall, false, symlist,
						exec, memento);
				//builder.addBind(a1);
				vals.add(a1);

				prms = ((Cons)prms).getCdr();
				bcdr = ((Cons)bcdr).getCdr();
			} else {
				throw message.getError("err.type.invalid");
			}
		}

		for(int i = vals.size() - 1; i >= 0; i--) {
			builder.addBind(vals.get(i));
		}
	}

	//
	private void compileSexp(
			Datum bcar,
			Datum bcdr,
			Environment env,
			Builder builder,
			Cons symcall,
			boolean istail,
			List<Cons> symlist,
			CodeExecutor exec,
			IntStack memento,
			boolean toplv) {
		if(istail && bcar.equals(symcall.getCar())) {
			compileArgsBind(
					bcdr, env, builder, symcall, symlist,
					exec, memento);
			builder.addJmpTop();
		} else {
			boolean tlv;

			tlv = (toplv && bcar instanceof Cons &&
					((Cons)bcar).getCar() instanceof SymbolName &&
					((SymbolName)((Cons)bcar).getCar())
					.getName().equals(SynBegin.BEGIN.getName()));
			compile(bcar, env, builder, tlv, symcall, false, symlist,
					exec, memento);
			compileArgs(
					bcdr, env, builder, symcall, symlist,
					exec, memento);

			if(istail) {
				builder.addCallTail(symlist.size());
			} else {
				builder.addCall();
			}
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.LispCompiler#compileArgs(net.morilib.lisp.Datum, net.morilib.lisp.Environment, net.morilib.lisp.CompiledCode.Builder, net.morilib.lisp.Cons, java.util.List, net.morilib.lisp.CodeExecutor, net.morilib.lisp.IntStack, net.morilib.lisp.LispCompiler.MiscInfo)
	 */
	public void compileArgs(
			Datum bcdr,
			Environment env,
			Builder builder,
			Cons symcall,
			List<Cons> symlist,
			CodeExecutor exec,
			IntStack memento) {
		builder.addBeginList();
		while(true) {
			if(bcdr instanceof Nil) {
				builder.addEndList();
				break;
			} else if(bcdr instanceof Atom) {
				compile(bcdr, env, builder, symcall, false, symlist,
						exec, memento);
				builder.addEndListDot();
				break;
			} else if(bcdr instanceof Cons) {
				compile(((Cons)bcdr).getCar(), env, builder,
						symcall, false, symlist,
						exec, memento);
				builder.addAppendList();

				bcdr = ((Cons)bcdr).getCdr();
			} else {
				throw message.getError("err.type.invalid");
			}
		}
	}

	//
	private Datum getSym(Datum bcar, Environment env) {
		Datum ddd;

		ddd = env.findDatum((Symbol)bcar);
		return ddd;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.LispCompiler#compile(net.morilib.lisp.Datum, net.morilib.lisp.Environment, net.morilib.lisp.CompiledCode.Builder, boolean, net.morilib.lisp.Cons, boolean, java.util.List, net.morilib.lisp.CodeExecutor, net.morilib.lisp.IntStack, net.morilib.lisp.LispCompiler.MiscInfo)
	 */
	public void compile(
			Datum body,          // S-expression to be compiled
			Environment env,     // environment to be compiled
			Builder builder,     // builder of byte code
			boolean toplevel,    // is in top level
			Cons symcall,        // list of (<procedure name> <args>)
			boolean istail,      // is in tail
			List<Cons> symlist,  // list of symcall
			CodeExecutor exec,
			IntStack memento) {
		if(body instanceof SymbolName) {
			builder.addReferSymbol(body);
		} else if(body instanceof Cons) {
			Datum bcar = ((Cons)body).getCar();
			Datum bcdr = ((Cons)body).getCdr();

			if(bcar instanceof SymbolName) {
				Datum ddd = getSym(bcar, env);

				if(ddd instanceof Syntax) {
					// syntax
					Syntax syn = (Syntax)ddd;
					Datum dz = bcdr;

					syn.compile(dz, env, this, builder, toplevel,
							symcall, istail, message, symlist,
							exec, memento);
				} else {
					compileSexp(
							bcar, bcdr, env, builder,
							symcall, istail, symlist,
							exec, memento, toplevel);
				}
			} else {
				compileSexp(
						bcar, bcdr, env, builder,
						symcall, istail, symlist,
						exec, memento, toplevel);
			}
		} else {
			builder.addPush(body);
		}
	}

	/**
	 * 
	 * @param body
	 * @param env
	 * @param builder
	 * @param callsym
	 * @param istail
	 * @param symlist
	 * @param exec
	 * @param memento
	 * @param syncased
	 */
	public void compile(
			Datum body,
			Environment env,
			CompiledCode.Builder builder,
			Cons callsym,
			boolean istail,
			List<Cons> symlist,
			CodeExecutor exec,
			IntStack memento) {
		compile(body,
				env,
				builder,
				false,
				callsym,
				istail,
				symlist,
				exec,
				memento);
	}

}
