/*
 * Copyright 2009 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.lisp.nano;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

/**
 * 
 *
 *
 * @author MORIGUCHI, Yuichiro 2009
 */
public final class LispMessage {

	//
	private static final String CLSNAME =
		"net/morilib/lisp/nano/init/messages";

	//
	private static Map<Locale, LispMessage> instances =
		new HashMap<Locale, LispMessage>();

	/**
	 * 
	 */
	public static LispMessage ROOT_MESSAGE = getInstance(Locale.US);

	//
	private ResourceBundle messages;
	private List<ResourceBundle> pluggable =
		new ArrayList<ResourceBundle>();
	private Locale locale;

	//
	private LispMessage(Locale lc) {
		messages = ResourceBundle.getBundle(CLSNAME, lc);
		locale   = lc;
	}

	/**
	 * 
	 * @param lc
	 * @return
	 */
	public static LispMessage getInstance(Locale locale) {
		LispMessage res;
		Locale lc = (locale == null) ? Locale.getDefault() : locale;

		res = instances.get(lc);
		if(res == null) {
			synchronized(LispMessage.class) {
				res = new LispMessage(lc);
				instances.put(lc, res);
			}
		}
		return res;
	}

	/**
	 * 
	 * @return
	 */
	public static LispMessage getInstance() {
		return getInstance(Locale.getDefault());
	}

	//
	private String get(ResourceBundle rb, String prop) {
		try {
			return messages.getString(prop);
		} catch(MissingResourceException e) {
			return ROOT_MESSAGE.messages.getString(prop);
		}
	}

	/**
	 * 
	 * @param prop
	 * @return
	 */
	public String get(String prop) {
		try {
			return get(messages, prop);
		} catch(MissingResourceException e) {
			for(ResourceBundle rb : pluggable) {
				try {
					return get(rb, prop);
				} catch(MissingResourceException e2) {
					// ignore
				}
			}
			return (messages.getString("err.unknown") +
					"(" + e.getKey() + ")");
		}
	}

	/**
	 * 
	 * @param prop
	 * @param msg2
	 * @return
	 */
	public String get(String prop, String msg2) {
		return get(prop) + ":" + msg2;
	}

	/**
	 * 
	 * @param prop
	 * @param d
	 * @return
	 */
	public String get(String prop, Datum d) {
		return get(prop) + ":" + LispUtils.toWrite(d);
	}

	/**
	 * 
	 * @param prop
	 * @return
	 */
	public LispException getError(String prop) {
		return new LispException(prop, get(prop));
	}

	/**
	 * 
	 * @param prop
	 * @param msg2
	 * @return
	 */
	public LispException getError(String prop, String msg2) {
		return new LispException(prop, get(prop, msg2));
	}

	/**
	 * 
	 * @param prop
	 * @param d
	 * @return
	 */
	public LispException getError(String prop, Datum d) {
		return new LispException(prop, get(prop, d));
	}

	/**
	 * 
	 * @param prop
	 * @return
	 */
	public ReadException getReadError(String prop) {
		return new ReadException(prop, get(prop));
	}

	/**
	 * 
	 * @param prop
	 * @param d
	 * @return
	 */
	public ReadException getReadError(String prop, Datum d) {
		return new ReadException(prop, get(prop, d));
	}

	/**
	 * 
	 * @param prop
	 */
	public void warn(String prop) {
		System.err.print(get("warn.repl.err"));
		System.err.print(" ");
		System.err.println(get(prop));
	}

	/**
	 * @return the locale
	 */
	public Locale getLocale() {
		return locale;
	}

}
