// This software is a part of NOODLYBOX.
// This software is distributed under the terms of the new BSD License.
// Copyright (c) 2009, molelord
// All rights reserved.

#include <cassert>      // assert()
#include <map>          // map

#include "cMpu.h"       // Mpu
#include "inlineutil.h" // negative_logic()

// ユーザプログラムの初期化処理
void userstartup(nbox::MpuIf *mpu);

namespace nbox {
    // 静的メンバ変数の実体
    const uint32_t Mpu::RESET_X = 1<<0;

    Mpu::Mpu(int mpu_id) : MpuIf(), mpu_id(mpu_id)
    {
        // 信号はすべて'x'に初期化
        this->a.aval     = 0xffffffff;
        this->a.bval     = 0xffffffff;
        this->dout.aval  = 0xffffffff;
        this->dout.bval  = 0xffffffff;
        this->ctrlo.aval = 0xffffffff;
        this->ctrlo.bval = 0xffffffff;

        addInstance(mpu_id, this);
    }

    Mpu::~Mpu()
    {
        removeInstance();
    }

    int Mpu::getId() const
    {
        return this->mpu_id;
    }

    // リセットならば信号を'x'でない状態にして、trueを返す
    // リセットでないならfalseを返す
    bool Mpu::reset(const s_vpi_vecval* ctrli)
    {
        bool active = negative_logic(ctrli->aval & RESET_X);
        if (active) {
            this->a.aval     = 0x00000000;
            this->a.bval     = 0x00000000;
            this->dout.aval  = 0x00000000;
            this->dout.bval  = 0x00000000;

            // ctrloに含まれる信号はほとんどが負論理のはずなので、
            // とりあえず'1'にする。
            this->ctrlo.aval = 0xffffffff;
            this->ctrlo.bval = 0x00000000;

            return true;
        }
        return false;
    }

    // TODO ユーザプログラムの初期化処理を呼び出す
    // この呼び出しがあるために、ユーザプログラムがリンク対象に含まれて
    // いない場合はリンクエラーになるので、将来的にNOODLYBOXをdllとして
    // 提供するようなことがあれば再考する必要がある。
    void Mpu::startup()
    {
        ::userstartup(this);

        // プログラム終了後少しだけシミュレーションが続行していたほうが、
        // 波形が見やすくなるため
        nop(1); 
     }

    bool Mpu::readSuccess() const
    {
        return read_success;
    }

    // 静的メンバ変数のように使われる、ファイルスコープの変数
    static std::map<int, Mpu *> mpuMap; 

    // mpu_idに対応したインスタンスを返す
    Mpu *Mpu::getInstance(int mpu_id)
    {
        std::map<int, Mpu *>::iterator p(mpuMap.find(mpu_id));
        assert(p != mpuMap.end());

        return p->second;
    }

    void Mpu::addInstance(int mpu_id, Mpu *target)
    {
        // 既に同じmpu_idが登録されていたら異常終了させる
        assert(mpuMap.count(mpu_id) == 0);

        mpuMap.insert(std::make_pair(mpu_id, target));
    }

    void Mpu::removeInstance()
    {
        mpuMap.erase(this->mpu_id);
    }

} // namespace nbox
