/* Copyright 2014 Akira Ohta (akohta001@gmail.com)
    This file is part of ntch.

    The ntch is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    The ntch is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with ntch.  If not, see <http://www.gnu.org/licenses/>.
    
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <wchar.h>
#include <assert.h>
#include <iconv.h>
#include <sys/types.h>
#include <regex.h>


#include "usr/ng_word_t.h"
#include "usr/usr_db_t.h"
#include "utils/text.h"
#include "utils/file.h"
#include "utils/nt_conv_char.h" 

#define NT_NG_WORD_CHK_SUM (1678428)

#define NT_NG_ROOT_MUTEX_KEY (L"ng")
#define NT_NG_WORD_MUTEX_KEY (L"ng_word")
#define NT_NG_NAME_MUTEX_KEY (L"ng_name")
#define NT_NG_ID_MUTEX_KEY (L"ng_id")


typedef struct tag_nt_ng_word_t *nt_ng_word_tp;
typedef struct tag_nt_ng_word_t {
	nt_ng_word_handle_t handle;
	int ref_count;
	wchar_t *key;
	nt_link_tp ng_words;
	nt_link_tp ng_names;
	nt_link_tp ng_ids;
	nt_link_tp ng_word_patterns;
	nt_link_tp ng_name_patterns;
	nt_link_tp ng_id_patterns;
	nt_link_tp ng_word_reg_patterns;
	nt_link_tp ng_name_reg_patterns;
	nt_link_tp ng_id_reg_patterns;
} nt_ng_word_t;

static nt_mutex_handle local_ng_word_get_mutex(
		const wchar_t *key, nt_ng_word_handle handle);
static nt_ng_word_handle nt_ng_word_alloc();
static BOOL ng_word_add(
		nt_cloud_handle h_cloud, const char *cloud_file_name,
		nt_ng_word_handle handle, 
		const char *ng_word,
		nt_link_tp *org_linkpp,
		nt_link_tp *text_linkpp,
		nt_link_tp *regx_linkpp);
static wchar_t *parse_plain_text(const wchar_t *source);
static regex_t *get_regx_pattern(const char *source);
static BOOL word_match_local(nt_ng_word_tp ng_wordp,
		const wchar_t *source, 
		nt_link_tp patterns,
		nt_link_tp reg_patterns,
		wchar_t *match_buf, size_t match_buf_len);


static wchar_t *parse_plain_text(const wchar_t *source){
	if(!source || source[0] == L'\0')
		return NULL;
	
	if(source[0] == L'\\' &&
			source[0] == L'/'){
		return nt_w_str_clone(source+1);
	}else{
		return nt_w_str_clone(source);
	}
}

static regex_t *get_regx_pattern(const char *source)
{
	int len, cflags;
	char *cptr;
	char buf[256];
	regex_t *regp;
	
	if(!source || '\0' == source[0])
		return NULL;
	
	if('/' != source[0])
		return NULL;
	
	cptr = strrchr (source, '/');
	if(!cptr || cptr == source)
		return NULL;
	cflags = REG_NEWLINE | REG_EXTENDED;
	if(cptr[1] == 'i')
		cflags |= REG_ICASE;
	
	len = cptr - source - 1;
	if(len >= sizeof(buf)-1)
		return NULL;
	strncpy(buf, source+1, len);
	buf[len] = '\0';
	
	regp = malloc(sizeof(regex_t));
	if(!regp){
		return NULL;
	}
	if(0 != regcomp(regp, buf, cflags)){
		free(regp);
		return NULL;
	}
	return regp;
}

nt_link_tp nt_ng_word_get_words(nt_ng_word_handle handle)
{
	nt_ng_word_tp ng_wordp;
	assert(handle);
	assert(handle->chk_sum == NT_NG_WORD_CHK_SUM);
	ng_wordp = (nt_ng_word_tp)handle;
	assert(ng_wordp->ref_count > 0);
	return ng_wordp->ng_words;
}

nt_link_tp nt_ng_word_get_names(nt_ng_word_handle handle)
{
	nt_ng_word_tp ng_wordp;
	assert(handle);
	assert(handle->chk_sum == NT_NG_WORD_CHK_SUM);
	ng_wordp = (nt_ng_word_tp)handle;
	assert(ng_wordp->ref_count > 0);
	return ng_wordp->ng_names;
}

nt_link_tp nt_ng_word_get_ids(nt_ng_word_handle handle)
{
	nt_ng_word_tp ng_wordp;
	assert(handle);
	assert(handle->chk_sum == NT_NG_WORD_CHK_SUM);
	ng_wordp = (nt_ng_word_tp)handle;
	assert(ng_wordp->ref_count > 0);
	return ng_wordp->ng_ids;
}

static void reg_free(void *ptr){
	regex_t *regp;
	regp = (regex_t*)ptr;
	regfree(regp);
	free(regp);
}

BOOL nt_ng_word_set_words(nt_ng_word_handle handle, 
		nt_link_tp text_list, nt_cloud_handle h_cloud)
{
	nt_ng_word_tp ng_wordp;
	nt_link_tp linkp;
	wchar_t *cptr;
	char buf[256];
	
	assert(handle);
	assert(handle->chk_sum == NT_NG_WORD_CHK_SUM);
	ng_wordp = (nt_ng_word_tp)handle;
	assert(ng_wordp->ref_count > 0);
	if(ng_wordp->ng_words)
		nt_all_link_free(ng_wordp->ng_words, free);
	if(ng_wordp->ng_word_patterns)
		nt_all_link_free(ng_wordp->ng_word_patterns, free);
	if(ng_wordp->ng_word_reg_patterns)
		nt_all_link_free(ng_wordp->ng_word_reg_patterns, reg_free);
	ng_wordp->ng_words = NULL;
	ng_wordp->ng_word_patterns = NULL;
	ng_wordp->ng_word_reg_patterns = NULL;
	if(text_list){
		linkp = text_list;
		do{
			cptr = (wchar_t*)linkp->data;
			if((size_t)-1 != wcstombs(buf, cptr, sizeof(buf))){
				nt_ng_word_add_ng_word(NULL, &ng_wordp->handle, buf);
			}
			linkp = linkp->next;
		}while(linkp != text_list);
		nt_all_link_free(text_list, free);
		if(h_cloud)
			nt_cloud_upload_file(h_cloud, 
					"ngwd.txt", ng_wordp->ng_words);
	}
	return TRUE;
}

BOOL nt_ng_word_set_names(nt_ng_word_handle handle, 
		nt_link_tp text_list, nt_cloud_handle h_cloud)
{
	nt_ng_word_tp ng_wordp;
	nt_link_tp linkp;
	wchar_t *cptr;
	char buf[256];
	
	assert(handle);
	assert(handle->chk_sum == NT_NG_WORD_CHK_SUM);
	ng_wordp = (nt_ng_word_tp)handle;
	assert(ng_wordp->ref_count > 0);
	if(ng_wordp->ng_names)
		nt_all_link_free(ng_wordp->ng_names, free);
	if(ng_wordp->ng_name_patterns)
		nt_all_link_free(ng_wordp->ng_name_patterns, free);
	if(ng_wordp->ng_name_reg_patterns)
		nt_all_link_free(ng_wordp->ng_name_reg_patterns, reg_free);
	ng_wordp->ng_names = NULL;
	ng_wordp->ng_name_patterns = NULL;
	ng_wordp->ng_name_reg_patterns = NULL;
	if(text_list){
		linkp = text_list;
		do{
			cptr = (wchar_t*)linkp->data;
			if((size_t)-1 != wcstombs(buf, cptr, sizeof(buf))){
				nt_ng_word_add_ng_name(NULL, &ng_wordp->handle, buf);
			}
			linkp = linkp->next;
		}while(linkp != text_list);
		nt_all_link_free(text_list, free);
		if(h_cloud)
			nt_cloud_upload_file(h_cloud, 
					"ngnm.txt", ng_wordp->ng_names);
	}
	return TRUE;
}

BOOL nt_ng_word_set_ids(nt_ng_word_handle handle, 
		nt_link_tp text_list, nt_cloud_handle h_cloud)
{
	nt_ng_word_tp ng_wordp;
	nt_link_tp linkp;
	wchar_t *cptr;
	char buf[256];
	
	assert(handle);
	assert(handle->chk_sum == NT_NG_WORD_CHK_SUM);
	ng_wordp = (nt_ng_word_tp)handle;
	assert(ng_wordp->ref_count > 0);
	if(ng_wordp->ng_ids)
		nt_all_link_free(ng_wordp->ng_ids, free);
	if(ng_wordp->ng_id_patterns)
		nt_all_link_free(ng_wordp->ng_id_patterns, free);
	if(ng_wordp->ng_id_reg_patterns)
		nt_all_link_free(ng_wordp->ng_id_reg_patterns, reg_free);
	ng_wordp->ng_ids = NULL;
	ng_wordp->ng_id_patterns = NULL;
	ng_wordp->ng_id_reg_patterns = NULL;
	if(text_list){
		linkp = text_list;
		do{
			cptr = (wchar_t*)linkp->data;
			if((size_t)-1 != wcstombs(buf, cptr, sizeof(buf))){
				nt_ng_word_add_ng_id(NULL, &ng_wordp->handle, buf);
			}
			linkp = linkp->next;
		}while(linkp != text_list);
		nt_all_link_free(text_list, free);
		if(h_cloud)
			nt_cloud_upload_file(h_cloud, 
					"ngid.txt", ng_wordp->ng_ids);
	}
	return TRUE;
}

nt_ng_word_handle nt_ng_word_load(
		nt_cloud_handle h_cloud,
		const char *ng_word_file_name,
		const char *ng_name_file_name,
		const char *ng_id_file_name
		)
{
	nt_ng_word_tp ng_wordp;
	nt_link_tp text_linkp, linkp;
	wchar_t *cptr;
	char buf[256];
	
	ng_wordp = (nt_ng_word_tp)nt_ng_word_alloc();
	if(!ng_wordp)
		return NULL;
	text_linkp = NULL;
	if(h_cloud)
		text_linkp = nt_cloud_download_file(h_cloud, "ngwd.txt");
	if(!text_linkp)
		text_linkp = nt_read_text_file(ng_word_file_name);
	if(text_linkp){
		linkp = text_linkp;
		do{
			cptr = (wchar_t*)linkp->data;
			if((size_t)-1 != wcstombs(buf, cptr, sizeof(buf))){
				nt_ng_word_add_ng_word(NULL, &ng_wordp->handle, buf);
			}
			linkp = linkp->next;
		}while(linkp != text_linkp);
		nt_all_link_free(text_linkp, free);
	}
	text_linkp = NULL;
	if(h_cloud)
		text_linkp = nt_cloud_download_file(h_cloud, "ngnm.txt");
	if(!text_linkp)
		text_linkp = nt_read_text_file(ng_name_file_name);
	if(text_linkp){
		linkp = text_linkp;
		do{
			cptr = (wchar_t*)linkp->data;
			if((size_t)-1 != wcstombs(buf, cptr, sizeof(buf))){
				nt_ng_word_add_ng_name(NULL, &ng_wordp->handle, buf);
			}
			linkp = linkp->next;
		}while(linkp != text_linkp);
		nt_all_link_free(text_linkp, free);
	}
	text_linkp = NULL;
	if(h_cloud)
		text_linkp = nt_cloud_download_file(h_cloud, "ngid.txt");
	if(!text_linkp)
		text_linkp = nt_read_text_file(ng_id_file_name);
	if(text_linkp){
		linkp = text_linkp;
		do{
			cptr = (wchar_t*)linkp->data;
			if((size_t)-1 != wcstombs(buf, cptr, sizeof(buf))){
				nt_ng_word_add_ng_id(NULL, &ng_wordp->handle, buf);
			}
			linkp = linkp->next;
		}while(linkp != text_linkp);
		nt_all_link_free(text_linkp, free);
	}
	
	return &ng_wordp->handle;
}

BOOL nt_ng_word_save(nt_ng_word_handle handle,
		const char *ng_word_file_name,
		const char *ng_name_file_name,
		const char *ng_id_file_name
		)
{
	nt_ng_word_tp ng_wordp;
	nt_mutex_handle h_mutex;
	
	assert(handle);
	assert(handle->chk_sum == NT_NG_WORD_CHK_SUM);
	ng_wordp = (nt_ng_word_tp)handle;
	assert(ng_wordp->ref_count > 0);
	
	h_mutex = nt_ng_get_mutex(handle);
	if(h_mutex)
		nt_mutex_lock(h_mutex);
	
	if(ng_wordp->ng_words)
		nt_write_text_file(ng_word_file_name, ng_wordp->ng_words);
	if(ng_wordp->ng_names)
		nt_write_text_file(ng_name_file_name, ng_wordp->ng_names);
	if(ng_wordp->ng_ids)
		nt_write_text_file(ng_id_file_name, ng_wordp->ng_ids);
	
	if(h_mutex)
		nt_mutex_unlock(h_mutex);
	return TRUE;
}

BOOL nt_ng_word_add_ng_word(
		nt_cloud_handle h_cloud,
		nt_ng_word_handle handle, const char *ng_word)
{
	nt_ng_word_tp ng_wordp;
	nt_mutex_handle h_mutex;
	BOOL bret;
	assert(handle);
	assert(handle->chk_sum == NT_NG_WORD_CHK_SUM);
	ng_wordp = (nt_ng_word_tp)handle;
	assert(ng_wordp->ref_count > 0);
	h_mutex = nt_ng_word_get_mutex(handle);
	if(h_mutex)
		nt_mutex_lock(h_mutex);
	bret = ng_word_add(
			h_cloud, "ngwd.txt",
			handle, ng_word,
			&ng_wordp->ng_words, 
			&ng_wordp->ng_word_patterns, 
			&ng_wordp->ng_word_reg_patterns);
	if(h_mutex)
		nt_mutex_unlock(h_mutex);
	return bret;
}

BOOL nt_ng_word_add_ng_name(
		nt_cloud_handle h_cloud,
		nt_ng_word_handle handle, const char *ng_name)
{
	nt_ng_word_tp ng_wordp;
	nt_mutex_handle h_mutex;
	BOOL bret;
	assert(handle);
	assert(handle->chk_sum == NT_NG_WORD_CHK_SUM);
	ng_wordp = (nt_ng_word_tp)handle;
	assert(ng_wordp->ref_count > 0);
	h_mutex = nt_ng_name_get_mutex(handle);
	if(h_mutex)
		nt_mutex_lock(h_mutex);
	bret = ng_word_add(
			h_cloud, "ngnm.txt",
			handle, ng_name,
			&ng_wordp->ng_names,
			&ng_wordp->ng_name_patterns,
			&ng_wordp->ng_name_reg_patterns);
	if(h_mutex)
		nt_mutex_unlock(h_mutex);
	return bret;
}

BOOL nt_ng_word_add_ng_id(
		nt_cloud_handle h_cloud,
		nt_ng_word_handle handle, const char *ng_id)
{
	nt_ng_word_tp ng_wordp;
	nt_mutex_handle h_mutex;
	BOOL bret;
	assert(handle);
	assert(handle->chk_sum == NT_NG_WORD_CHK_SUM);
	ng_wordp = (nt_ng_word_tp)handle;
	assert(ng_wordp->ref_count > 0);
	h_mutex = nt_ng_id_get_mutex(handle);
	if(h_mutex)
		nt_mutex_lock(h_mutex);
	bret = ng_word_add(
			h_cloud, "ngid.txt",
			handle, ng_id,
			&ng_wordp->ng_ids,
			&ng_wordp->ng_id_patterns,
			&ng_wordp->ng_id_reg_patterns);
	if(h_mutex)
		nt_mutex_unlock(h_mutex);
	return bret;
}

static BOOL ng_word_add(
		nt_cloud_handle h_cloud, const char *cloud_file_name, 
		nt_ng_word_handle handle, 
		const char *ng_word,
		nt_link_tp *org_linkpp,
		nt_link_tp *text_linkpp,
		nt_link_tp *regx_linkpp)
{
	int len;
	wchar_t wbuf[1024];
	wchar_t *cptr;
	nt_link_tp linkp;
	regex_t *regp;
	
	if(!ng_word || (0 == (len = strlen(ng_word))))
		return FALSE;
	
	if(-1 == mbstowcs(wbuf, ng_word, sizeof(wbuf)))
		return FALSE;
	
	if(*org_linkpp){
		cptr = (wchar_t*)nt_link_find(
				*org_linkpp, (void*)wbuf, nt_link_wcscmp_fnc);
		if(cptr)
			return TRUE;
	}
	
	if(NULL == (cptr = nt_w_str_clone(wbuf)))
		return FALSE;
	linkp = nt_link_add_data(*org_linkpp, cptr);
	if(!linkp){
		free(cptr);
		return FALSE;
	}
	if(!(*org_linkpp))
		*org_linkpp = linkp;
	
	if(h_cloud){
		cptr = nt_w_str_clone(wbuf);
		if(cptr){
			linkp = nt_link_add_data(NULL, (void*)cptr);
			if(linkp){
				nt_cloud_insert_lines_into_file_async(h_cloud, cloud_file_name, linkp);
			}
		}
	}
	
	if(NULL != (regp = get_regx_pattern(ng_word))){
		linkp = nt_link_add_data(*regx_linkpp, regp);
		if(!linkp){
			regfree(regp);
			free(regp);
			return FALSE;
		}
		if(!(*regx_linkpp))
			*regx_linkpp = linkp;
		return TRUE;
	}
	if(NULL != (cptr = parse_plain_text(wbuf))){
		linkp = nt_link_add_data(*text_linkpp, cptr);
		if(!linkp){
			free(cptr);
			return FALSE;
		}
		if(!(*text_linkpp))
			*text_linkpp = linkp;
		return TRUE;
	}
	
	return FALSE;
}

int ng_word_match(nt_ng_word_handle handle,
		int item_type, const wchar_t *source, 
		wchar_t *match_buf, size_t match_buf_len)
{
	nt_ng_word_tp ng_wordp;
	
	//assert(match);
	assert(handle);
	assert(handle->chk_sum == NT_NG_WORD_CHK_SUM);
	ng_wordp = (nt_ng_word_tp)handle;
	assert(ng_wordp->ref_count > 0);
	if(item_type & NG_ITEM_MSG){
		if(word_match_local(ng_wordp, source,
				ng_wordp->ng_word_patterns,
				ng_wordp->ng_word_reg_patterns,
				match_buf, match_buf_len)){
			return NG_ITEM_MSG;
		}
	}else if(item_type & NG_ITEM_NAME){
		if(word_match_local(ng_wordp, source,
				ng_wordp->ng_name_patterns,
				ng_wordp->ng_name_reg_patterns,
				match_buf, match_buf_len)){
			return NG_ITEM_NAME;
		}
	}else if(item_type & NG_ITEM_ID){
		if(word_match_local(ng_wordp, source,
				ng_wordp->ng_id_patterns,
				ng_wordp->ng_id_reg_patterns,
				match_buf, match_buf_len)){
			return NG_ITEM_ID;
		}
	}
	return NG_ITEM_NONE;
}

static BOOL word_match_local(nt_ng_word_tp ng_wordp,
		const wchar_t *source, 
		nt_link_tp patterns,
		nt_link_tp reg_patterns,
		wchar_t *match_buf, size_t match_buf_len)
{
	nt_link_tp linkp;
	wchar_t *cptr1, *cptr2;
	regex_t *regp;
	regmatch_t match[2];
	char *ptr;
	int len;
	
	if(patterns){
		linkp = patterns;
		do{
			cptr1 = (wchar_t*)linkp->data;
			if(NULL != (cptr2 = wcsstr(source, cptr1))){
				if(match_buf_len > wcslen(cptr1))
					wcscpy(match_buf, cptr1);
				else
					match_buf[0] = L'\0';
				return TRUE;
			}
			linkp = linkp->next;
		}while(linkp != patterns);
	}
	if(reg_patterns){
		len = wcslen(source);
		if(len == 0)
			goto EXIT_NG_ITEM_MSG;
		ptr = malloc((len+1)*3);/*max utf-8 char size*/
		if(!ptr)
			goto EXIT_NG_ITEM_MSG;
		if((size_t)-1 == wcstombs(ptr, source, (len+1)*3)){
			free(ptr);
			goto EXIT_NG_ITEM_MSG;
		}
		linkp = reg_patterns;
		do{
			regp = (regex_t*)linkp->data;
			if(0 == regexec(regp, ptr, 
					sizeof(match)/sizeof(regmatch_t),
					match, 0)){
				if(match[0].rm_so > -1){
					len = match[0].rm_eo - match[0].rm_so;
					if(match_buf_len > len){
						ptr[match[0].rm_eo] = '\0';
						if((size_t)-1 == mbstowcs(
								match_buf, 
								ptr + match[0].rm_so, 
								match_buf_len)){
							match_buf[0] = L'\0';
						}
					}else{
						match_buf[0] = L'\0';
					}
				}else{
					match_buf[0] = L'\0';
				}
				free(ptr);
				return TRUE;
			}
			linkp = linkp->next;
		}while(linkp != reg_patterns);
		free(ptr);
	}
EXIT_NG_ITEM_MSG:
	return FALSE;
}


static nt_ng_word_handle nt_ng_word_alloc()
{
	 nt_ng_word_tp ng_wordp =
		malloc(sizeof(nt_ng_word_t));
	if(!ng_wordp)
		return NULL;
	
	ng_wordp->handle.chk_sum = NT_NG_WORD_CHK_SUM;
	ng_wordp->ng_words = NULL;
	ng_wordp->ng_names = NULL;
	ng_wordp->ng_word_patterns = NULL;
	ng_wordp->ng_name_patterns = NULL;
	ng_wordp->ng_id_patterns = NULL;
	ng_wordp->ng_word_reg_patterns = NULL;
	ng_wordp->ng_name_reg_patterns = NULL;
	ng_wordp->ng_id_reg_patterns = NULL;

	ng_wordp->ng_ids = NULL;
	ng_wordp->ref_count = 1;
	ng_wordp->key = NT_NG_ROOT_MUTEX_KEY;
	return (nt_ng_word_handle)&ng_wordp->handle;
}


nt_mutex_handle nt_ng_get_mutex(nt_ng_word_handle handle)
{
	nt_mutex_handle h_mutex;
	nt_ng_word_tp ng_wordp;
	
	assert(handle);
	assert(handle->chk_sum == NT_NG_WORD_CHK_SUM);
	ng_wordp = (nt_ng_word_tp)handle;
	assert(ng_wordp->ref_count > 0);
	
	nt_ng_word_add_ref(handle);
	
	h_mutex = nt_mutex_get_one_time_handle(ng_wordp->key);
	
	nt_ng_word_release_ref(handle);
	if(!h_mutex){
		return NULL;
	}
	return h_mutex;
}

static nt_mutex_handle local_ng_word_get_mutex(
		const wchar_t *key, nt_ng_word_handle handle)
{
	nt_mutex_handle h_mutex;
	nt_ng_word_tp ng_wordp;
	
	assert(handle);
	assert(handle->chk_sum == NT_NG_WORD_CHK_SUM);
	ng_wordp = (nt_ng_word_tp)handle;
	assert(ng_wordp->ref_count > 0);
	
	nt_ng_word_add_ref(handle);
	
	h_mutex = nt_mutex_get_one_time_handle(ng_wordp->key);
	
	nt_ng_word_release_ref(handle);
	if(!h_mutex){
		return NULL;
	}
	if(!nt_mutex_add_moniker(h_mutex, key)){
		return NULL;
	}
	return h_mutex;
}

nt_mutex_handle nt_ng_word_get_mutex(nt_ng_word_handle handle)
{
	return local_ng_word_get_mutex(NT_NG_WORD_MUTEX_KEY,  handle);
}
nt_mutex_handle nt_ng_name_get_mutex(nt_ng_word_handle handle)
{
	return local_ng_word_get_mutex(NT_NG_NAME_MUTEX_KEY,  handle);
}
nt_mutex_handle nt_ng_id_get_mutex(nt_ng_word_handle handle)
{
	return local_ng_word_get_mutex(NT_NG_ID_MUTEX_KEY,  handle);
}


int nt_ng_word_add_ref(nt_ng_word_handle handle)
{
	nt_ng_word_tp ng_wordp;
	assert(handle);
	assert(handle->chk_sum == NT_NG_WORD_CHK_SUM);
	ng_wordp = (nt_ng_word_tp)handle;
	assert(ng_wordp->ref_count > 0);
	return ++ng_wordp->ref_count;
}


int nt_ng_word_release_ref(nt_ng_word_handle handle)
{
	nt_ng_word_tp ng_wordp;
	assert(handle);
	assert(handle->chk_sum == NT_NG_WORD_CHK_SUM);
	ng_wordp = (nt_ng_word_tp)handle;
	assert(ng_wordp->ref_count > 0);
	ng_wordp->ref_count--;
	if(0 != ng_wordp->ref_count)
		return ng_wordp->ref_count;

	
	if(ng_wordp->ng_words)
		nt_all_link_free(ng_wordp->ng_words, free);
	if(ng_wordp->ng_word_patterns)
		nt_all_link_free(ng_wordp->ng_word_patterns, free);
	if(ng_wordp->ng_word_reg_patterns)
		nt_all_link_free(ng_wordp->ng_word_reg_patterns, reg_free);
	if(ng_wordp->ng_names)
		nt_all_link_free(ng_wordp->ng_names, free);
	if(ng_wordp->ng_name_patterns)
		nt_all_link_free(ng_wordp->ng_name_patterns, free);
	if(ng_wordp->ng_name_reg_patterns)
		nt_all_link_free(ng_wordp->ng_name_reg_patterns, reg_free);
	if(ng_wordp->ng_ids)
		nt_all_link_free(ng_wordp->ng_ids, free);
	if(ng_wordp->ng_id_patterns)
		nt_all_link_free(ng_wordp->ng_id_patterns, free);
	if(ng_wordp->ng_id_reg_patterns)
		nt_all_link_free(ng_wordp->ng_id_reg_patterns, reg_free);
	free(ng_wordp);
	return 0;
}

BOOL nt_ng_word_upload_cloud(nt_cloud_handle h_cloud, nt_ng_word_handle h_ng_word)
{
	nt_ng_word_tp ng_wordp;
	nt_link_tp empty_linkp;
	assert(h_ng_word);
	assert(h_ng_word->chk_sum == NT_NG_WORD_CHK_SUM);
	ng_wordp = (nt_ng_word_tp)h_ng_word;
	assert(ng_wordp->ref_count > 0);
	
	empty_linkp = nt_link_add_data(NULL, L"");
	
	if(ng_wordp->ng_words){
		nt_cloud_upload_file(h_cloud, "ngwd.txt", ng_wordp->ng_words);
	}else if(empty_linkp){
		nt_cloud_upload_file(h_cloud, "ngwd.txt", empty_linkp);
	}
	
	if(ng_wordp->ng_names){
		nt_cloud_upload_file(h_cloud, "ngnm.txt", ng_wordp->ng_names);
	}else if(empty_linkp){
		nt_cloud_upload_file(h_cloud, "ngnm.txt", empty_linkp);
	}
	if(ng_wordp->ng_ids){
		nt_cloud_upload_file(h_cloud, "ngid.txt", ng_wordp->ng_ids);
	}else if(empty_linkp){
		nt_cloud_upload_file(h_cloud, "ngid.txt", empty_linkp);
	}
	
	if(empty_linkp)
		nt_all_link_free(empty_linkp, NULL);
	return TRUE;
}

