/*
 * Copyright (c) 2003-2004 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: image_ui.c,v 1.5.2.15 2004/11/09 22:42:28 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"
#include "ochusha_async_buffer.h"
#include "ochusha_network_broker.h"
#include "utils.h"

#include "worker.h"

#include "ochusha_ui.h"
#include "image_ui.h"

#include <glib.h>
#include <gtk/gtk.h>

#include <fcntl.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <utime.h>

/* #undef DEBUG_CACHE		0 */
/* #undef DEBUG_CACHE_MOST 	0 */


/*
 * åϢ
 */

static off_t current_cache_size = 0;
static int current_number_of_cache_files = 0;

static GHashTable *temporary_cache_table = NULL;
static GSList *cache_entry_list = NULL;

typedef struct _CacheEntry CacheEntry;

static CacheEntry *cache_entry_new(char *filename, time_t atime, off_t size);
static void cache_entry_free(CacheEntry *entry);

static void cache_image(OchushaApplication *application, const char *url,
			OchushaAsyncBuffer *buffer);
static void touch_cache(OchushaApplication *application, const char *url);
static void invalidate_cache(OchushaApplication *application, const char *url);

static void update_cache_info(OchushaApplication *application);


/*
 * åȴϢ
 */

typedef struct _ImageInfo ImageInfo;
struct _ImageInfo
{
  char *url;
  int width;
  int height;
  OchushaAsyncBuffer *buffer;
  GdkPixbufLoader *loader;
  GdkPixbufAnimation *animation;
  GdkPixbuf *pixbuf;
  GSList *widget_list;
  guint timeout_id;
};


static GHashTable *image_info_table = NULL;
static GHashTable *a_bone_url_table = NULL;
static GHashTable *a_bone_server_table = NULL;


#if 0
#define IMAGE_CACHE_TIMEOUT	180000	/* 180 */
#else
#define IMAGE_CACHE_TIMEOUT	5000	/* 5 */
#endif


static void
image_info_free(ImageInfo *info)
{
  /* gdk_threads_enter()ĶƤӽФ */
  g_free(info->url);
  if (info->loader != NULL)
    g_object_unref(info->loader);
  if (info->animation != NULL)
    g_object_unref(info->animation);
  if (info->pixbuf != NULL)
    g_object_unref(info->pixbuf);
  if (info->timeout_id != 0)
    g_source_remove(info->timeout_id);
  g_free(info);
}


/* ϥåơ֥뤫info
 * image_info_free(info)褦timeoutؿ
 */
static gboolean
image_info_timeout_cb(ImageInfo *info)
{
  GSList *info_list;
  gboolean result = FALSE;

  gdk_threads_enter();

  if (info->loader != NULL || info->widget_list != NULL)
    {
      result = TRUE;
      goto retry;
    }

  info_list = (GSList *)g_hash_table_lookup(image_info_table, info->url);
  info_list = g_slist_remove(info_list, info);
  if (info_list == NULL)
    g_hash_table_remove(image_info_table, info->url);
  else
    {
      ImageInfo *another_info = (ImageInfo *)info_list->data;
      g_hash_table_insert(image_info_table, another_info->url, info_list);
    }

  image_info_free(info);

 retry:
  gdk_threads_leave();

  return result;
}


static void
image_destroy_cb(GtkWidget *image, ImageInfo *info)
{
  /* gdk_threads_enter()ĶƤӽФϤ */
  info->widget_list = g_slist_remove(info->widget_list, image);

  if (info->widget_list == NULL && info->loader == NULL
      && (info->animation != NULL || info->pixbuf != NULL))
    {
      /* 180øinfoơ֥뤫õ */
      info->timeout_id = g_timeout_add(IMAGE_CACHE_TIMEOUT,
				       (GSourceFunc)image_info_timeout_cb,
				       info);
    }
}


static ImageInfo *
image_info_new(const char *url, GdkPixbufLoader *loader, GtkWidget *image)
{
  ImageInfo *info = g_new0(ImageInfo, 1);

  info->url = g_strdup(url);

  if (loader != NULL)
    g_object_ref(loader);

  info->loader = loader;

  info->widget_list = g_slist_append(NULL, image);
  g_signal_connect(G_OBJECT(image), "destroy",
		   G_CALLBACK(image_destroy_cb), info);
  return info;
}


static ImageInfo *
lookup_image_info(const char *url, int width, int height)
{
  /* gdk_threads_enter()ĶƤӽФ */
  GSList *info_list = g_hash_table_lookup(image_info_table, url);

  while (info_list != NULL)
    {
      ImageInfo *info = (ImageInfo *)info_list->data;
      if (info->width == width && info->height == height)
	return info;
      info_list = info_list->next;
    }
  return NULL;
}


void
initialize_image_ui(OchushaApplication *application)
{
  int fd;

  update_cache_info(application);

  image_info_table = g_hash_table_new(g_str_hash, g_str_equal);

  a_bone_url_table = g_hash_table_new_full(g_str_hash, g_str_equal,
					   (GDestroyNotify)g_free, NULL);
  a_bone_server_table = g_hash_table_new_full(g_str_hash, g_str_equal,
					      (GDestroyNotify)g_free, NULL);

  /* ܡURLꥹȤե뤫ɤ */
  fd = ochusha_config_open_file(&application->config, OCHUSHA_A_BONE_IMAGE_TXT,
				NULL, O_RDONLY);
  if (fd >= 0)
    {
      FILE *file = fdopen(fd, "r");
      if (file != NULL)
	{
	  char url[PATH_MAX];
	  url[PATH_MAX - 1] = '\0';
	  while (!feof(file))
	    {
	      char *tmp_str = fgets(url, PATH_MAX - 1, file);
	      if (tmp_str != NULL)
		{
		  char *key;
		  tmp_str = strpbrk(url, "\r\n");
		  if (tmp_str != NULL)
		    *tmp_str = '\0';
		  key = g_strdup(url);
		  g_hash_table_insert(a_bone_url_table, key, key);
		}
	    }
	  fclose(file);
	}
      else
	close(fd);
    }


  /* ܡ󥵡ФΥꥹȤե뤫ɤ */
  fd = ochusha_config_open_file(&application->config,
				OCHUSHA_A_BONE_SERVER_TXT, NULL, O_RDONLY);
  if (fd >= 0)
    {
      FILE *file = fdopen(fd, "r");
      if (file != NULL)
	{
	  char server[PATH_MAX];
	  server[PATH_MAX - 1] = '\0';
	  while (!feof(file))
	    {
	      char *tmp_str = fgets(server, PATH_MAX - 1, file);
	      if (tmp_str != NULL)
		{
		  char *key;
		  tmp_str = strpbrk(server, "\r\n");
		  if (tmp_str != NULL)
		    *tmp_str = '\0';
		  key = g_strdup(server);
		  g_hash_table_insert(a_bone_server_table, key, key);
		}
	    }
	  fclose(file);
	}
      else
	close(fd);
    }
}


static gboolean
output_a_bone_key(gpointer key, gpointer value, gpointer user_data)
{
  FILE *file = (FILE *)user_data;

  fprintf(file, "%s\n", (const char *)key);

  return TRUE;
}


void
finalize_image_ui(OchushaApplication *application)
{
  int fd = ochusha_config_open_file(&application->config,
				    OCHUSHA_A_BONE_IMAGE_TXT, NULL,
				    O_WRONLY | O_TRUNC | O_CREAT);
  if (fd >= 0)
    {
      FILE *file = fdopen(fd, "w");
      /* ܡURLꥹȤե˽񤭽Ф */
      if (file != NULL)
	{
	  g_hash_table_foreach(a_bone_url_table,
			       (GHFunc)output_a_bone_key, file);
	  fclose(file);
	}
      else
	close(fd);
    }

  fd = ochusha_config_open_file(&application->config,
				OCHUSHA_A_BONE_SERVER_TXT, NULL,
				O_WRONLY | O_TRUNC | O_CREAT);
  if (fd >= 0)
    {
      FILE *file = fdopen(fd, "w");
      /* ܡURLꥹȤե˽񤭽Ф */
      if (file != NULL)
	{
	  g_hash_table_foreach(a_bone_server_table,
			       (GHFunc)output_a_bone_key, file);
	  fclose(file);
	}
      else
	close(fd);
    }
}


static void
a_bone_image_shown(GtkImage *image, gpointer unused)
{
  g_return_if_fail(GTK_IS_IMAGE(image));
  gtk_image_set_from_stock(image, GTK_STOCK_DIALOG_WARNING,
			   GTK_ICON_SIZE_DIALOG);
}


static void
a_bone_image_in_table(ImageInfo *info, gpointer unused)
{
  g_slist_foreach(info->widget_list, (GFunc)a_bone_image_shown, NULL);
}


void
ochusha_a_bone_image(OchushaApplication *application, const char *url)
{
  /* gdk_threads_enter()ĶƤ֤ */
  GSList *info_list;
  char *key;

  if (url == NULL || *url == '\0')
    return;

  key = g_strdup(url);

  g_hash_table_insert(a_bone_url_table, key, key);

  info_list = g_hash_table_lookup(image_info_table, key);
  g_slist_foreach(info_list, (GFunc)a_bone_image_in_table, NULL);

  ochusha_config_cache_unlink_file(&application->config, url);
  ochusha_config_image_cache_unlink_file(&application->config, url);
  invalidate_cache(application, url);
}


void
ochusha_a_bone_image_server(OchushaApplication *application, const char *url)
{
  /* gdk_threads_enter()ĶƤ֤ */
  char *server;

  if (url == NULL || *url == '\0')
    return;

  server = ochusha_utils_url_extract_http_server(url);
  if (server == NULL)
    return;

  g_hash_table_insert(a_bone_server_table, server, server);
}


static void
set_pixbuf(ImageInfo *image_info)
{
  /* gdk_threads_enter()ƤĶ餷ƤӽФʤߤΤ */
  GdkPixbufLoader *loader = image_info->loader;
  GdkPixbufAnimation *animation;

  if (image_info->animation != NULL)
    {
      GSList *widget_list = image_info->widget_list;
      while (widget_list != NULL)
	{
	  GtkImage *image = (GtkImage *)widget_list->data;
	  gtk_image_set_from_animation(image, image_info->animation);
	  gtk_widget_queue_resize(GTK_WIDGET(image));
	  widget_list = widget_list->next;
	}
      return;
    }

  if (image_info->pixbuf != NULL)
    {
      GSList *widget_list = image_info->widget_list;
      while (widget_list != NULL)
	{
	  GtkImage *image = (GtkImage *)widget_list->data;
	  gtk_image_set_from_pixbuf(image, image_info->pixbuf);
	  gtk_widget_queue_resize(GTK_WIDGET(image));
	  widget_list = widget_list->next;
	}
      return;
    }

  if (g_hash_table_lookup(a_bone_url_table, image_info->url) != NULL)
    return;

  if (loader == NULL)
    return;

  animation = gdk_pixbuf_loader_get_animation(loader);
  if (animation != NULL)
    {
      GSList *widget_list;

      image_info->animation = animation;
      g_object_ref(animation);

      widget_list = image_info->widget_list;
      while (widget_list != NULL)
	{
	  GtkImage *image = (GtkImage *)widget_list->data;
	  gtk_image_set_from_animation(image, animation);
	  gtk_widget_queue_resize(GTK_WIDGET(image));
	  widget_list = widget_list->next;
	}
    }
  else
    {
      GdkPixbuf *pixbuf = gdk_pixbuf_loader_get_pixbuf(loader);
      if (pixbuf != NULL)
	{
	  GSList *widget_list;
	  image_info->pixbuf = pixbuf;
	  g_object_ref(pixbuf);

	  widget_list = image_info->widget_list;
	  while (widget_list != NULL)
	    {
	      GtkImage *image = (GtkImage *)widget_list->data;
	      gtk_image_set_from_pixbuf(image, pixbuf);
	      gtk_widget_queue_resize(GTK_WIDGET(image));
	      widget_list = widget_list->next;
	    }
	}
      else
	{
	  GSList *widget_list = image_info->widget_list;
	  while (widget_list != NULL)
	    {
	      GtkImage *image = (GtkImage *)widget_list->data;
	      gtk_image_set_from_stock(image, GTK_STOCK_MISSING_IMAGE,
				       GTK_ICON_SIZE_DIALOG);
	      gtk_widget_queue_resize(GTK_WIDGET(image));
	      widget_list = widget_list->next;
	    }
	}
    }

  image_info->loader = NULL;
  g_object_unref(loader);

  if (image_info->widget_list == NULL
      && (image_info->animation != NULL || image_info->pixbuf != NULL))
    {
      /* 180øinfoơ֥뤫õ */
      image_info->timeout_id
	= g_timeout_add(IMAGE_CACHE_TIMEOUT,
			(GSourceFunc)image_info_timeout_cb, image_info);
    }
}


static void
size_prepared_cb(GdkPixbufLoader *loader, int width, int height,
		 ImageInfo *image_info)
{
  double width_request;
  double height_request;
  double width_scale;
  double height_scale;

  if (g_hash_table_lookup(a_bone_url_table, image_info->url) != NULL)
    return;

  width_request = image_info->width;
  height_request = image_info->height;

  if (width <= width_request && height <= height_request)
    return;	/* Ϥʤ */

  width_scale = width_request / width;
  height_scale = height_request / height;
  if (width_scale > height_scale)
    {
      width *= height_scale;
      height *= height_scale;
    }
  else
    {
      width *= width_scale;
      height *= width_scale;
    }
  if (width < 0 || height < 0)
    return;

  gdk_pixbuf_loader_set_size(loader, width, height);

  if (gdk_pixbuf_loader_get_animation(loader) != NULL
      || gdk_pixbuf_loader_get_pixbuf(loader) != NULL)
    set_pixbuf(image_info);
}


static void
area_prepared_cb(GdkPixbufLoader *loader, ImageInfo *image_info)
{
  set_pixbuf(image_info);
}


static void
area_updated_cb(GdkPixbufLoader *loader, int x, int y, int width, int height,
		ImageInfo *image_info)
{
  set_pixbuf(image_info);
}


static void
download_image(WorkerThread *employee, gpointer args)
{
  ImageInfo *image_info = (ImageInfo *)args;
  GdkPixbufLoader *loader = image_info->loader;
  OchushaAsyncBuffer *buffer = image_info->buffer;
  gboolean image_shown = FALSE;

  if (image_info->width >= 0 || image_info->height >= 0)
    g_signal_connect(G_OBJECT(loader), "size_prepared",
		     G_CALLBACK(size_prepared_cb), image_info);

  g_signal_connect(G_OBJECT(loader), "area_prepared",
		   G_CALLBACK(area_prepared_cb), image_info);

  g_signal_connect(G_OBJECT(loader), "area_updated",
		   G_CALLBACK(area_updated_cb), image_info);

  if (ochusha_async_buffer_active_ref(buffer))
    {
      size_t offset = 0;

      ochusha_async_buffer_lock(buffer);
      {
	while (TRUE)
	  {
	    size_t length = buffer->length;
	    if (length > 0 && length > offset)
	      {
		size_t updated_len = length - offset;
		gdk_threads_enter();
		image_shown
		  |= gdk_pixbuf_loader_write(loader,
					     ((const guchar *)buffer->buffer
					      + offset), updated_len, NULL);
		gdk_threads_leave();
	      }
	    if (buffer->fixed)
	      goto terminated;
	    offset = length;
	    do
	      {
		if (!ochusha_async_buffer_wait(buffer))
		  goto terminated;
	      } while (length >= buffer->length && !buffer->fixed);
	  }
      }
    terminated:
      ochusha_async_buffer_unlock(buffer);
      ochusha_async_buffer_active_unref(buffer);
    }

  gdk_threads_enter();

  if (!image_shown)
    set_pixbuf(image_info);

  gdk_pixbuf_loader_close(loader, NULL);

  image_info->loader = NULL;
  image_info->buffer = NULL;

  gdk_threads_leave();

  if (image_shown)
    {
      /* ХåեƤå¸ */
      OchushaApplication *application
	= g_object_get_data(G_OBJECT(buffer), "application");
      if (application != NULL)
	cache_image(application, image_info->url, buffer);
    }

  OCHU_OBJECT_UNREF(buffer);

  g_object_unref(loader);
}


GtkWidget *
ochusha_download_image(OchushaApplication *application, const char *url,
		       int width, int height)
{
  OchushaAsyncBuffer *buffer;
  GtkWidget *image;
  WorkerJob *job = NULL;
  GdkPixbufLoader *loader;
  char *scheme;
  char *server;

  ImageInfo *image_info;
  GSList *info_list;

  g_return_val_if_fail(url != NULL, NULL);

  if (*url == '\0')
    return NULL;

  if (g_hash_table_lookup(a_bone_url_table, url) != NULL)
    return NULL;

  server = ochusha_utils_url_extract_http_server(url);
  if (server == NULL)
    return NULL;
  if (g_hash_table_lookup(a_bone_server_table, server) != NULL)
    {
      G_FREE(server);
      return NULL;
    }
  G_FREE(server);

  scheme = ochusha_utils_url_extract_scheme(url);
  if (scheme == NULL)
    return NULL;
  if (strcmp(scheme, "http") != 0)
    {
      G_FREE(scheme);
      return NULL;
    }
  G_FREE(scheme);

  image_info = lookup_image_info(url, width, height);
  if (image_info != NULL)
    {
      if (image_info->animation != NULL)
	image = gtk_image_new_from_animation(image_info->animation);
      else if (image_info->pixbuf != NULL)
	image = gtk_image_new_from_pixbuf(image_info->pixbuf);
      else if (image_info->loader != NULL)
	image = gtk_image_new_from_stock(OCHUSHA_STOCK_MISSING_IMAGE,
					 GTK_ICON_SIZE_DIALOG);
      else
	image = gtk_image_new_from_stock(GTK_STOCK_MISSING_IMAGE,
					 GTK_ICON_SIZE_DIALOG);
      if (image_info->timeout_id != 0)
	{
	  g_source_remove(image_info->timeout_id);
	  image_info->timeout_id = 0;
	}

      image_info->widget_list = g_slist_append(image_info->widget_list, image);
      g_signal_connect(G_OBJECT(image), "destroy",
		       G_CALLBACK(image_destroy_cb), image_info);

      touch_cache(application, url);

      return image;
    }

  info_list = g_hash_table_lookup(image_info_table, url);
  buffer = NULL;
  while (info_list != NULL)
    {
      image_info = (ImageInfo *)info_list->data;
      if (image_info->loader == NULL
	  && image_info->animation == NULL && image_info->pixbuf == NULL)
	{
	  /* Ǥ뤳ȤȽƤ */
	  image = gtk_image_new_from_stock(GTK_STOCK_MISSING_IMAGE,
					   GTK_ICON_SIZE_DIALOG);
	  image_info = image_info_new(url, NULL, image);
	  image_info->width = width;
	  image_info->height = height;

	  info_list = g_hash_table_lookup(image_info_table, url);
	  if (info_list != NULL)
	    info_list = g_slist_append(info_list, image_info);
	  else
	    g_hash_table_insert(image_info_table, image_info->url,
				g_slist_append(NULL, image_info));
	  return image;
	}

      if (image_info->buffer != NULL)
	{
	  buffer = image_info->buffer;
	  OCHU_OBJECT_REF(buffer);
	  break;
	}
      info_list = info_list->next;
    }

  if (buffer == NULL)
    {
      /* ߴΤ */
      buffer = ochusha_network_broker_read_from_url(application->broker,
					NULL, url, NULL,
					OCHUSHA_NETWORK_BROKER_CACHE_ONLY,
					TRUE,
					application->image_chunksize);
      if (buffer != NULL)
	touch_cache(application, url);
    }

  if (buffer == NULL)
    {
      buffer = ochusha_network_broker_read_from_url(application->broker,
					NULL, url, NULL,
					OCHUSHA_NETWORK_BROKER_IMAGE_CACHE,
					TRUE,
					application->image_chunksize);
      if (buffer != NULL)
	touch_cache(application, url);
    }

  if (buffer == NULL)
    {
      buffer = ochusha_network_broker_read_from_url(application->broker,
					NULL, url, NULL,
					OCHUSHA_NETWORK_BROKER_CACHE_IGNORE,
					TRUE,
					application->image_chunksize);
      if (buffer != NULL)
	g_object_set_data(G_OBJECT(buffer), "application", application);
    }

  if (buffer == NULL)
    return NULL;

  loader = gdk_pixbuf_loader_new();
  image = gtk_image_new_from_stock(OCHUSHA_STOCK_MISSING_IMAGE,
				   GTK_ICON_SIZE_DIALOG);
  image_info = image_info_new(url, loader, image);
  image_info->width = width;
  image_info->height = height;
  image_info->buffer = buffer;

  info_list = g_hash_table_lookup(image_info_table, url);
  if (info_list != NULL)
    info_list = g_slist_append(info_list, image_info);
  else
    g_hash_table_insert(image_info_table, image_info->url,
			g_slist_append(NULL, image_info));


  job = G_NEW0(WorkerJob, 1);
  job->canceled = FALSE;
  job->job = download_image;
  job->args = image_info;

  commit_modest_job(job);

  return image;
}


struct _CacheEntry
{
  char *filename;
  time_t atime;
  off_t size;
};


static CacheEntry *
cache_entry_new(char *filename, time_t atime, off_t size)
{
  CacheEntry *entry = g_new0(CacheEntry, 1);
  entry->filename = filename;	// ԡǤʤͭȤäȡ
  entry->atime = atime;
  entry->size = size;
  return entry;
}


static void
cache_entry_free(CacheEntry *entry)
{
  g_free(entry->filename);
  g_free(entry);
}


/*
 * ǸŤ˥ȤϤ
 */
static gint
cache_entry_compare(gconstpointer a, gconstpointer b)
{
  const CacheEntry *entry_a = a;
  const CacheEntry *entry_b = b;

  if (entry_a->atime < entry_b->atime)
    return -1;
  if (entry_a->atime == entry_b->atime)
    return 0;
  return 1;
}


#ifdef DEBUG_CACHE_MOST
static void
dump_cache_entry(CacheEntry *entry, gpointer unused)
{
  fprintf(stderr, "name: %s\n", entry->filename);
  fprintf(stderr, "atime=%ld, size=%lld\n", entry->atime, entry->size);
}
#endif


static void
cache_image(OchushaApplication *application, const char *url,
	    OchushaAsyncBuffer *buffer)
{
  /* gdk_threads_enter()ơʤĶƤ֤ */

  char *filename;
  int fd;
  ssize_t len;
  struct stat sb;
  CacheEntry *entry;

  if (application->maximum_number_of_cache_files == 0
      && application->capacity_of_image_cache == 0)
    {
      /* ̵»ˤϺǽ餫tenure */
      int fd = ochusha_config_image_cache_open_file(&application->config, url,
						O_WRONLY | O_TRUNC | O_CREAT);
      if (fd >= 0)
	{
	  ssize_t len = write(fd, (void *)buffer->buffer, buffer->length);
	  close(fd);
	  if (len != buffer->length)
	    ochusha_config_image_cache_unlink_file(&application->config, url);
	}
      return;
    }

  if (buffer->length > (application->capacity_of_image_cache * 1024 * 1024))
    return;	/* Ǥ */

  /*
   * MEMO: ƱݤʤΤǡ¸Ƥ饵Ĵ롣
   *       Τꡢե¸Ƥ椫ĴޤǤδ֤
   *       capacity_of_image_cacheۤ뤳ȤΤϡˤʤ
   */

  gdk_threads_enter();
  entry = g_hash_table_lookup(temporary_cache_table, url);
  gdk_threads_leave();

  if (entry != NULL)
    return;

  filename = ochusha_utils_url_encode_string(url);

  fd = ochusha_config_open_file(&application->config, filename, "image_cache",
				O_WRONLY | O_TRUNC | O_CREAT);
  if (fd < 0)
    {
#ifdef DEBUG_CACHE
      fprintf(stderr, "Couldn't open image cache file: %s\n", url);
#endif
      G_FREE(filename);
      return;	/* äꤢ롣*/
    }

  len = write(fd, (void *)buffer->buffer, buffer->length);
  if (len != buffer->length || fstat(fd, &sb) != 0)
    {
      close(fd);
      ochusha_config_unlink_file(&application->config,
				 filename, "image_cache");
      G_FREE(filename);
#ifdef DEBUG_CACHE
      fprintf(stderr, "Writing failed for image cache file: %s\n", url);
#endif
      return;	/* äꤢ롣*/
    }
  close(fd);

  entry = cache_entry_new(filename, sb.st_atime, sb.st_size);

  gdk_threads_enter();
  g_hash_table_insert(temporary_cache_table, G_STRDUP(url), entry);
  cache_entry_list = g_slist_insert_sorted(cache_entry_list, entry,
					   cache_entry_compare);
  current_cache_size += entry->size;
  current_number_of_cache_files++;

  ochusha_image_cache_ensure_limits(application);

  gdk_threads_leave();

#ifdef DEBUG_CACHE
  fprintf(stderr, "\nCurrent Cache Status:\n");
#ifdef DEBUG_CACHE_MOST
  g_slist_foreach(cache_entry_list, (GFunc)dump_cache_entry, NULL);
#endif
  fprintf(stderr, "total_cache_size=%lld, number_of_cache_files=%d\n",
	  current_cache_size, current_number_of_cache_files);
#endif
}


static void
touch_cache(OchushaApplication *application, const char *url)
{
  /* gdk_threads_enter()ʴĶƤӽФ */
  CacheEntry *entry = g_hash_table_lookup(temporary_cache_table, url);
  char *filename;
  struct stat sb;

  if (entry == NULL)
    return;

  filename = ochusha_config_find_file(&application->config, entry->filename,
				      "image_cache");
  if (filename == NULL)
    {
      /* åʤʤäƤޤ衩 */
      current_cache_size -= entry->size;
      current_number_of_cache_files--;
      cache_entry_list = g_slist_remove(cache_entry_list, entry);
      g_hash_table_remove(temporary_cache_table, url);
      return;
    }

  if (utime(filename, NULL) == 0 && stat(filename, &sb) == 0)
    {
      entry->atime = sb.st_atime;
      cache_entry_list = g_slist_remove(cache_entry_list, entry);
      cache_entry_list = g_slist_insert_sorted(cache_entry_list,
					       entry,
					       cache_entry_compare);
    }

  G_FREE(filename);

#ifdef DEBUG_CACHE
  fprintf(stderr, "\nCurrent Cache Status:\n");
#ifdef DEBUG_CACHE_MOST
  g_slist_foreach(cache_entry_list, (GFunc)dump_cache_entry, NULL);
#endif
  fprintf(stderr, "total_cache_size=%lld, number_of_cache_files=%d\n",
	  current_cache_size, current_number_of_cache_files);
#endif
}


static void
invalidate_cache(OchushaApplication *application, const char *url)
{
  /* gdk_threads_enter()ʴĶƤӽФ */
  CacheEntry *entry = g_hash_table_lookup(temporary_cache_table, url);

  if (entry == NULL)
    return;

  ochusha_config_unlink_file(&application->config, entry->filename,
			     "image_cache");
  current_cache_size -= entry->size;
  current_number_of_cache_files--;
  cache_entry_list = g_slist_remove(cache_entry_list, entry);
  g_hash_table_remove(temporary_cache_table, url);

#ifdef DEBUG_CACHE
  fprintf(stderr, "\nCurrent Cache Status:\n");
#ifdef DEBUG_CACHE_MOST
  g_slist_foreach(cache_entry_list, (GFunc)dump_cache_entry, NULL);
#endif
  fprintf(stderr, "total_cache_size=%lld, number_of_cache_files=%d\n",
	  current_cache_size, current_number_of_cache_files);
#endif
}


static void
update_cache_info(OchushaApplication *application)
{
  char *cache_dir;
  struct stat sb;
  GDir *dir_list;
  const gchar *filename;
  char cache_file_path[PATH_MAX];

  current_cache_size = 0;
  current_number_of_cache_files = 0;
  if (temporary_cache_table != NULL)
    g_hash_table_destroy(temporary_cache_table);
  if (cache_entry_list != NULL)
    g_slist_free(cache_entry_list);

  temporary_cache_table
    = g_hash_table_new_full(g_str_hash, g_str_equal, NULL,
			    (GDestroyNotify)cache_entry_free);
  cache_entry_list = NULL;

  cache_dir = ochusha_config_find_directory(&application->config,
					    "image_cache", NULL);
  if (cache_dir == NULL)
    return;

  dir_list = g_dir_open(cache_dir, 0, NULL);

  if (dir_list == NULL)
    {
      G_FREE(cache_dir);
      return;
    }

  while ((filename = g_dir_read_name(dir_list)) != NULL)
    {
      if (snprintf(cache_file_path, PATH_MAX,
		   "%s/%s", cache_dir, filename) < PATH_MAX)
	{
	  if (stat(cache_file_path, &sb) == 0 && !S_ISDIR(sb.st_mode))
	    {
	      CacheEntry *entry = cache_entry_new(G_STRDUP(filename),
						  sb.st_atime, sb.st_size);
	      char *url = ochusha_utils_url_decode_string(filename);
	      g_hash_table_insert(temporary_cache_table, url, entry);
	      cache_entry_list = g_slist_insert_sorted(cache_entry_list,
						       entry,
						       cache_entry_compare);
	      current_cache_size += sb.st_size;
	      current_number_of_cache_files++;
	    }
	}
    }
  G_FREE(cache_dir);

  g_dir_close(dir_list);

#ifdef DEBUG_CACHE
  fprintf(stderr, "\nCurrent Cache Status:\n");
#ifdef DEBUG_CACHE_MOST
  g_slist_foreach(cache_entry_list, (GFunc)dump_cache_entry, NULL);
#endif
  fprintf(stderr, "total_cache_size=%lld, number_of_cache_files=%d\n",
	  current_cache_size, current_number_of_cache_files);
#endif
}


void
ochusha_tenure_cache_image(OchushaApplication *application, const char *url)
{
  CacheEntry *entry;
  char cache_file_path[PATH_MAX];
  char tenured_file_path[PATH_MAX];

  g_return_if_fail(application != NULL && application->config.home != NULL);
  g_return_if_fail(url != NULL || *url != '\0');
  g_return_if_fail(strncmp(url, "http://", 7) == 0);

  entry = g_hash_table_lookup(temporary_cache_table, url);
  if (entry == NULL)
    return;

  if (snprintf(cache_file_path, PATH_MAX, "%s/image_cache/%s",
	       application->config.home, entry->filename) >= PATH_MAX)
    return;	/* äꤢ */

  if (snprintf(tenured_file_path, PATH_MAX, "%s/image/%s",
	       application->config.home, url + 7) >= PATH_MAX)
    return;	/* äꤢ */

  if (rename(cache_file_path, tenured_file_path) == 0)
    {
      cache_entry_list = g_slist_remove(cache_entry_list, entry);
      current_cache_size -= entry->size;
      current_number_of_cache_files--;
      g_hash_table_remove(temporary_cache_table, url);
    }
#ifdef DEBUG_CACHE
  fprintf(stderr, "\nCurrent Cache Status:\n");
#ifdef DEBUG_CACHE_MOST
  g_slist_foreach(cache_entry_list, (GFunc)dump_cache_entry, NULL);
#endif
  fprintf(stderr, "total_cache_size=%lld, number_of_cache_files=%d\n",
	  current_cache_size, current_number_of_cache_files);
#endif
}


void
ochusha_image_cache_ensure_limits(OchushaApplication *application)
{
  /* gdk_threads_enter()ʴĶƤӽФ */
  int max_num_files = application->maximum_number_of_cache_files;
  off_t cache_capacity;

  if (max_num_files > 0)
    {
      while (max_num_files < current_number_of_cache_files)
	{
	  /* ե */
	  CacheEntry *entry = cache_entry_list->data;
	  char *url = ochusha_utils_url_decode_string(entry->filename);
	  invalidate_cache(application, url);
	  G_FREE(url);
	}
    }

  cache_capacity = application->capacity_of_image_cache * 1024 * 1024;
  if (cache_capacity > 0)
    {
      while (cache_capacity < current_cache_size)
	{
	  /* å */
	  CacheEntry *entry = cache_entry_list->data;
	  char *url = ochusha_utils_url_decode_string(entry->filename);
	  invalidate_cache(application, url);
	  G_FREE(url);
	}
    }
}

