/*
 * Copyright (c) 2003-2004 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ochusha_async_buffer.c,v 1.34.2.5 2004/11/09 22:42:10 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"

#include "ochusha_async_buffer.h"
#include "marshal.h"

#include "monitor.h"
#include "worker.h"

#include <glib.h>

#include <errno.h>
#include <fcntl.h>

#include <pthread.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/types.h>
#include <sys/mman.h>

#ifdef HAVE_NANOSLEEP
#include <time.h>
#endif

#include <unistd.h>

#include <zlib.h>


#ifndef MAP_NOCORE
# define MAP_NOCORE	0
#endif


#define OCHUSHA_INITIAL_ASYNC_BUFFER_SIZE	4096


static void ochusha_async_buffer_class_init(OchushaAsyncBufferClass *klass);
static void ochusha_async_buffer_init(OchushaAsyncBuffer *buffer);
static void ochusha_async_buffer_finalize(GObject *object);


GType
ochusha_async_buffer_get_type(void)
{
  static GType async_buffer_type = 0;

  if (async_buffer_type == 0)
    {
      static const GTypeInfo async_buffer_info =
	{
	  sizeof(OchushaAsyncBufferClass),
	  NULL, /* base_init */
	  NULL, /* base_finalize */
	  (GClassInitFunc)ochusha_async_buffer_class_init,
	  NULL, /* class_finalize */
	  NULL, /* class_data */
	  sizeof(OchushaAsyncBuffer),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)ochusha_async_buffer_init,
	};

      async_buffer_type = g_type_register_static(G_TYPE_OBJECT,
						 "OchushaAsyncBuffer",
						 &async_buffer_info, 0);
    }

  return async_buffer_type;
}


static Monitor *get_buffer_monitor(void);

static GObjectClass *parent_class = NULL;
static Monitor *global_monitor = NULL;


/*
 * pthreadϢΥ֥ȤλȤΤƤƧߤʤΤǡѤ
 * ݤ롣
 */
static GSList *free_monitor_list = NULL;

/*
 * Active(number_of_active_users > 0)Ƥ
 * OchushaAsyncBufferݻ
 */
static GSList *active_buffer_list = NULL;


#define LOCK_BUFFER_LOCK(buffer)					\
  do									\
    {									\
      if (!ochusha_monitor_try_enter(buffer->monitor))			\
	{								\
	  ochusha_monitor_enter(global_monitor);			\
	  buffer->number_of_lock_waiters++;				\
	  ochusha_monitor_exit(global_monitor);				\
	  ochusha_monitor_enter(buffer->monitor);			\
	  ochusha_monitor_enter(global_monitor);			\
	  buffer->number_of_lock_waiters--;				\
	  ochusha_monitor_exit(global_monitor);				\
	}								\
    } while (0)

#define UNLOCK_BUFFER_LOCK(buffer)				\
  do { ochusha_monitor_exit(buffer->monitor); } while (0)

#define WAIT_BUFFER_CONDITION(buffer)				\
  do { ochusha_monitor_wait(buffer->monitor); } while (0)

#define SIGNAL_BUFFER_CONDITION(buffer)				\
  do { ochusha_monitor_notify(buffer->monitor); } while (0)

#define BROADCAST_BUFFER_CONDITION(buffer)			\
  do { ochusha_monitor_notify_all(buffer->monitor); } while (0)


enum {
  WAKEUP_NOW_SIGNAL,

  ACCESS_STARTED_SIGNAL,
  ACCESS_PROGRESSED_SIGNAL,
  ACCESS_FINISHED_SIGNAL,
  ACCESS_FAILED_SIGNAL,

  LAST_SIGNAL
};


static int async_buffer_signals[LAST_SIGNAL] = { 0, 0, 0, 0, 0 };


static void
blind_awake(OchushaAsyncBuffer *buffer)
{
  LOCK_BUFFER_LOCK(buffer);

  if (buffer->state == OCHUSHA_ASYNC_BUFFER_OK
      && buffer->number_of_active_users > 0)
    {
      BROADCAST_BUFFER_CONDITION(buffer);
    }

  UNLOCK_BUFFER_LOCK(buffer);

  g_object_unref(G_OBJECT(buffer));	/* ڥˤʤäƤϤ */
}


static GSList *to_be_signaled_buffer_list = NULL;


static void
iterate_buffers(OchushaAsyncBuffer *buffer, GSList **list)
{
  g_object_ref(G_OBJECT(buffer));	/* ڥˤʤäƤϤ */
  *list = g_slist_append(*list, buffer);
}


static void
force_awake_job(WorkerThread *employee, void *unused)
{
#ifdef HAVE_NANOSLEEP
  struct timespec interval = { 0, 100000000 };	/* == 100ms */
#endif

  while (TRUE)
    {
      ochusha_monitor_enter(global_monitor);
      to_be_signaled_buffer_list = NULL;
      g_slist_foreach(active_buffer_list,
		      (GFunc)iterate_buffers, &to_be_signaled_buffer_list);
      ochusha_monitor_exit(global_monitor);

      g_slist_foreach(to_be_signaled_buffer_list, (GFunc)blind_awake, NULL);

      if (employee->command == DIE_NOW)
	break;
#ifdef HAVE_NANOSLEEP
      nanosleep(&interval, NULL);
#else
      sleep(1);		/* Ĺʤ뤬ʤ */
#endif
    }
}


static void
ochusha_async_buffer_class_init(OchushaAsyncBufferClass *klass)
{
  GObjectClass *o_class = (GObjectClass *)klass;
  WorkerJob *job = G_NEW0(WorkerJob, 1);

  global_monitor = ochusha_monitor_new(NULL);

  parent_class = g_type_class_peek_parent(klass);

  async_buffer_signals[WAKEUP_NOW_SIGNAL] =
    g_signal_new("wakeup_now",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaAsyncBufferClass, wakeup_now),
		 NULL, NULL,
		 libochusha_marshal_VOID__VOID,
		 G_TYPE_NONE, 0);

  async_buffer_signals[ACCESS_STARTED_SIGNAL] =
    g_signal_new("access_started",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaAsyncBufferClass, access_started),
		 NULL, NULL,
		 libochusha_marshal_VOID__VOID,
		 G_TYPE_NONE, 0);
  async_buffer_signals[ACCESS_PROGRESSED_SIGNAL] =
    g_signal_new("access_progressed",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaAsyncBufferClass, access_progressed),
		 NULL, NULL,
		 libochusha_marshal_VOID__INT_INT,
		 G_TYPE_NONE, 2,
		 G_TYPE_INT,
		 G_TYPE_INT);
  async_buffer_signals[ACCESS_FINISHED_SIGNAL] =
    g_signal_new("access_finished",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaAsyncBufferClass, access_finished),
		 NULL, NULL,
		 libochusha_marshal_VOID__VOID,
		 G_TYPE_NONE, 0);
  async_buffer_signals[ACCESS_FAILED_SIGNAL] =
    g_signal_new("access_failed",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaAsyncBufferClass, access_failed),
		 NULL, NULL,
		 libochusha_marshal_VOID__INT_STRING,
		 G_TYPE_NONE, 2,
		 G_TYPE_INT,
		 G_TYPE_STRING);

  o_class->finalize = ochusha_async_buffer_finalize;

  klass->wakeup_now = NULL;

  klass->access_started = NULL;
  klass->access_progressed = NULL;
  klass->access_finished = NULL;
  klass->access_failed = NULL;

  job->canceled = FALSE;
  job->job = force_awake_job;
  job->args = NULL;

  commit_job(job);
}


static void
ochusha_async_buffer_init(OchushaAsyncBuffer *buffer)
{
  buffer->fixed = FALSE;
  buffer->number_of_lock_waiters = 0;

  ochusha_monitor_enter(global_monitor);
  buffer->monitor = get_buffer_monitor();
  ochusha_monitor_exit(global_monitor);
#if DEBUG_MEMORY_USAGE
  fprintf(stderr, "ochusha_async_buffer_init: OchushaAsyncBuffer(%p)\n",
	  buffer);
#endif
}


static void
ochusha_async_buffer_finalize(GObject *object)
{
  OchushaAsyncBuffer *buffer = OCHUSHA_ASYNC_BUFFER(object);
#if DEBUG_MEMORY_USAGE
  fprintf(stderr, "ochusha_async_buffer_finalize: OchushaAsyncBuffer(%p)\n",
	  object);
#endif
  if (buffer->destructor != NULL)
    {
      (*buffer->destructor)(buffer);
      buffer->destructor = NULL;
    }

  ochusha_monitor_enter(global_monitor);
  if (buffer->monitor != NULL)
    {
      free_monitor_list = g_slist_append(free_monitor_list, buffer->monitor);
      buffer->monitor = NULL;
    }
  if (g_slist_find(active_buffer_list, buffer) != NULL)
    {
      active_buffer_list = g_slist_remove(active_buffer_list, buffer);
      g_warning("OchushaAsyncBuffer(%p) is finalized in active state!.\n",
		buffer);
    }
  ochusha_monitor_exit(global_monitor);

  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


static Monitor *
get_buffer_monitor(void)
{
  /* global_monitorƱ */
  Monitor *monitor;
  GSList *recycled_monitor = free_monitor_list;
  if (recycled_monitor != NULL)
    {
      free_monitor_list
	= g_slist_remove_link(free_monitor_list, recycled_monitor);
      monitor = (Monitor *)recycled_monitor->data;
      g_slist_free_1(recycled_monitor);
      return monitor;
    }

  /* MonitorϽλޤǻȤ󤵤롣*/
  return ochusha_monitor_new(global_monitor);
}


OchushaAsyncBuffer *
ochusha_async_buffer_new(char *buffer, size_t length, DestructFunc *destructor)
{
  OchushaAsyncBuffer *buf
    = OCHUSHA_ASYNC_BUFFER(g_object_new(OCHUSHA_TYPE_ASYNC_BUFFER, NULL));

  if (buffer == NULL && destructor == NULL)
    {
      if (length != 0)
	buffer = (char *)G_MALLOC(length);
      destructor = ochusha_async_buffer_free_when_finished;
      buf->length = 0;
    }
  else
    buf->length = length;
  
  buf->fixed = FALSE;
  buf->buffer = buffer;

  buf->buffer_length = length;	/* XXX: Ǥ餦٤ġġ*/

  buf->destructor = destructor;

#if DEBUG_MEMORY_USAGE
  fprintf(stderr, "ochusha_async_buffer_new(): OchushaAsyncBuffer(%p)\n", buf);
#endif

  return buf;
}


OchushaAsyncBuffer *
ochusha_async_buffer_new_with_file(int fd)
{
  OchushaAsyncBuffer *buf
    = OCHUSHA_ASYNC_BUFFER(g_object_new(OCHUSHA_TYPE_ASYNC_BUFFER, NULL));
  gzFile gzfile;
  int chunk_size;

  buf->fixed = FALSE;

  buf->destructor = ochusha_async_buffer_free_when_finished;

  buf->length = 0;
  buf->buffer = 0;
  buf->buffer_length = 0;

  gzfile = gzdopen(fd, "rb");
  if (gzfile == NULL)
    return buf;

  chunk_size = OCHUSHA_INITIAL_ASYNC_BUFFER_SIZE;

  while (!gzeof(gzfile))
    {
      char *buffer;
      int result;
      if (!ochusha_async_buffer_ensure_free_space(buf, chunk_size))
	abort();	/* why? */

      buffer = (char *)buf->buffer + buf->length;
      chunk_size = buf->buffer_length - buf->length;
      result = gzread(gzfile, buffer, chunk_size);
      if (result == -1)
	{
	  fprintf(stderr, "gzread cause some error\n");
	  break;
	}

      buf->length += result;
    }

  gzclose(gzfile);

#if DEBUG_MEMORY_USAGE
  fprintf(stderr, "ochusha_async_buffer_new(): OchushaAsyncBuffer(%p)\n", buf);
#endif

  return buf;
}


static void
munmap_when_finished(OchushaAsyncBuffer *buffer)
{
  munmap((void *)buffer->buffer, buffer->length);
#if DEBUG_MMAP
  fprintf(stderr, "mmapped buffer(at %p) is unmapped.\n",
	  buffer->buffer);
#endif
}


OchushaAsyncBuffer *
ochusha_async_buffer_new_with_file_mmap(int fd)
{
  char *buffer;
  OchushaAsyncBuffer *buf
    = OCHUSHA_ASYNC_BUFFER(g_object_new(OCHUSHA_TYPE_ASYNC_BUFFER, NULL));
  off_t len;

  len = lseek(fd, 0, SEEK_END);
  buffer = mmap(NULL, len, PROT_READ, MAP_NOCORE | MAP_PRIVATE, fd, 0);
  if (buffer == MAP_FAILED)
    {
      fprintf(stderr, "mmap failed due to: %s (%d)\n", strerror(errno), errno);
      buf->buffer = NULL;
      buf->length = 0;
      buf->buffer_length = 0;
      buf->destructor = NULL;
    }
  else
    {
      buf->buffer = buffer;
      buf->length = len;
      buf->buffer_length = len;
      buf->destructor = munmap_when_finished;
    }
  close(fd);
  
  buf->fixed = FALSE;

#if DEBUG_MEMORY_USAGE
  fprintf(stderr, "ochusha_async_buffer_new_with_file_mmap(): OchushaAsyncBuffer(%p)\n", buf);
#endif

  return buf;
}


gboolean
ochusha_async_buffer_reset(OchushaAsyncBuffer *buffer)
{
  g_return_val_if_fail(OCHUSHA_IS_ASYNC_BUFFER(buffer), FALSE);
  /* g_return_val_if_fail(buffer->number_of_active_users == 0, FALSE); */
  LOCK_BUFFER_LOCK(buffer);

  buffer->fixed = FALSE;
  buffer->state = OCHUSHA_ASYNC_BUFFER_OK;
  buffer->number_of_suspended_users = 0;
  buffer->length = 0;

  UNLOCK_BUFFER_LOCK(buffer);
  return TRUE;
}


/* ƱߴϢ */
/*
 * MEMO: OchushaAsyncBufferȤäƤƤΥåɤ⤷Υå
 *       Фᥤ󥹥åɤƱŪ˳ߤ򤫤ȻפäΤǡ
 *       ʲAPIѰդۤɰΤΤǤʤġġ
 *       pthreadUNIXʥ뤬ꤷƻȤФȤ⤢Τ
 *       pthreadUNIXʥȹ碌ϵʤΤ򤱤롣
 *
 * active user: OchushaAsyncBufferproducer⤷consumeråɡ
 *              ХåեݻƤΥƥȤϴޤޤʤ
 *              active userϥХåե˿ˡ
 *		ochusha_async_buffer_active_ref()ƤӽФ⤦Хåե
 *		ʤȤochusha_async_buffer_active_unref()
 *		ӽФȤˤ롣
 *
 *              δؿg_object_{ref,unref}Ƥǡ
 *              number_of_active_usersĴ롣
 *
 *              active userϻsuspend/resume/terminate׵᤬Ƥ
 *              ʤɤǧ뤳Ȥ롣
 */
gboolean
ochusha_async_buffer_active_ref(OchushaAsyncBuffer *buffer)
{
  gboolean result;

#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr, "OchushaAsyncBuffer(%p) is actively refered\n", buffer);
#endif
  g_object_ref(G_OBJECT(buffer));	/* Ĵ٤Ƥʤ */

  LOCK_BUFFER_LOCK(buffer);

  buffer->number_of_active_users++;
  while (buffer->state == OCHUSHA_ASYNC_BUFFER_SUSPENDED)
    {
      buffer->number_of_suspended_users++;

      ochusha_monitor_enter(global_monitor);
      ochusha_monitor_notify(global_monitor);
      ochusha_monitor_exit(global_monitor);

      WAIT_BUFFER_CONDITION(buffer);
    }

  result = buffer->state != OCHUSHA_ASYNC_BUFFER_TERMINATED;
  if (!result)
    {
      buffer->number_of_active_users--;

      ochusha_monitor_enter(global_monitor);
      ochusha_monitor_notify(global_monitor);
      ochusha_monitor_exit(global_monitor);

      g_object_unref(G_OBJECT(buffer));	/* Ĵ٤Ƥʤ */
    }

  if (buffer->number_of_active_users == 1)
    {
      ochusha_monitor_enter(global_monitor);
      g_assert(g_slist_find(active_buffer_list, buffer) == NULL);
      active_buffer_list = g_slist_append(active_buffer_list, buffer);
      ochusha_monitor_exit(global_monitor);
    }

  UNLOCK_BUFFER_LOCK(buffer);

  return result;
}


void
ochusha_async_buffer_active_unref(OchushaAsyncBuffer *buffer)
{
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr,
	  "OchushaAsyncBuffer(%p) actively refered is unrefered\n",
	  buffer);
#endif

  LOCK_BUFFER_LOCK(buffer);

  buffer->number_of_active_users--;

  if (buffer->state != OCHUSHA_ASYNC_BUFFER_OK)
    {
      ochusha_monitor_enter(global_monitor);
      ochusha_monitor_notify(global_monitor);
      ochusha_monitor_exit(global_monitor);
    }

  if (buffer->number_of_active_users == 0)
    {
      ochusha_monitor_enter(global_monitor);
      g_assert(g_slist_find(active_buffer_list, buffer) != NULL);
      active_buffer_list = g_slist_remove(active_buffer_list, buffer);
      ochusha_monitor_exit(global_monitor);
    }

  UNLOCK_BUFFER_LOCK(buffer);

  g_object_unref(G_OBJECT(buffer));	/* Ĵ٤Ƥʤ */
}


void
ochusha_async_buffer_suspend_all(void)
{
  GSList *to_be_suspended_list = NULL;
  if (global_monitor == NULL)
    return;
  ochusha_monitor_enter(global_monitor);
  g_slist_foreach(active_buffer_list,
		  (GFunc)iterate_buffers, &to_be_suspended_list);
  ochusha_monitor_exit(global_monitor);

  g_slist_foreach(to_be_suspended_list,
		  (GFunc)ochusha_async_buffer_suspend, NULL);
  g_slist_foreach(to_be_suspended_list, (GFunc)g_object_unref, NULL);
  g_slist_free(to_be_suspended_list);
}


void
ochusha_async_buffer_resume_all(void)
{
  GSList *to_be_resumed_list = NULL;
  if (global_monitor == NULL)
    return;
  ochusha_monitor_enter(global_monitor);
  g_slist_foreach(active_buffer_list,
		  (GFunc)iterate_buffers, &to_be_resumed_list);
  ochusha_monitor_exit(global_monitor);

  g_slist_foreach(to_be_resumed_list,
		  (GFunc)ochusha_async_buffer_resume, NULL);
  g_slist_foreach(to_be_resumed_list, (GFunc)g_object_unref, NULL);
  g_slist_free(to_be_resumed_list);
}


void
ochusha_async_buffer_terminate_all(void)
{
  GSList *to_be_terminated_list = NULL;
  if (global_monitor == NULL)
    return;
  ochusha_monitor_enter(global_monitor);
  g_slist_foreach(active_buffer_list,
		  (GFunc)iterate_buffers, &to_be_terminated_list);
  ochusha_monitor_exit(global_monitor);

  g_slist_foreach(to_be_terminated_list,
		  (GFunc)ochusha_async_buffer_terminate, NULL);
  g_slist_foreach(to_be_terminated_list, (GFunc)g_object_unref, NULL);
  g_slist_free(to_be_terminated_list);
}


void
ochusha_async_buffer_suspend(OchushaAsyncBuffer *buffer)
{
  gboolean dead_buffer;

  ochusha_monitor_enter(global_monitor);
  dead_buffer = g_slist_find(active_buffer_list, buffer) == NULL;
  ochusha_monitor_exit(global_monitor);
  if (dead_buffer)
    return;

  LOCK_BUFFER_LOCK(buffer);
  buffer->number_of_suspended_users = 0;
  buffer->state = OCHUSHA_ASYNC_BUFFER_SUSPENDED;

  g_signal_emit(G_OBJECT(buffer),
		async_buffer_signals[WAKEUP_NOW_SIGNAL],
		0);

  BROADCAST_BUFFER_CONDITION(buffer);

  while (buffer->number_of_suspended_users < buffer->number_of_active_users)
    {
      UNLOCK_BUFFER_LOCK(buffer);
      if (!ochusha_monitor_timedwait(global_monitor, 500))
	{
#if 0
	  fprintf(stderr, "Some threads not responding...\n");
#endif
	  return;
	}
      LOCK_BUFFER_LOCK(buffer);
    }
  UNLOCK_BUFFER_LOCK(buffer);
}


void
ochusha_async_buffer_resume(OchushaAsyncBuffer *buffer)
{
  gboolean dead_buffer;

  ochusha_monitor_enter(global_monitor);
  dead_buffer = g_slist_find(active_buffer_list, buffer) == NULL;
  ochusha_monitor_exit(global_monitor);
  if (dead_buffer)
    return;

  LOCK_BUFFER_LOCK(buffer);
  buffer->state = OCHUSHA_ASYNC_BUFFER_OK;
  BROADCAST_BUFFER_CONDITION(buffer);
  UNLOCK_BUFFER_LOCK(buffer);
}


void
ochusha_async_buffer_terminate(OchushaAsyncBuffer *buffer)
{
  gboolean dead_buffer;

  ochusha_monitor_enter(global_monitor);
  dead_buffer = g_slist_find(active_buffer_list, buffer) == NULL;
  ochusha_monitor_exit(global_monitor);
  if (dead_buffer)
    return;

  LOCK_BUFFER_LOCK(buffer);
  buffer->state = OCHUSHA_ASYNC_BUFFER_TERMINATED;

  g_signal_emit(G_OBJECT(buffer),
		async_buffer_signals[WAKEUP_NOW_SIGNAL],
		0);

  BROADCAST_BUFFER_CONDITION(buffer);

  while (buffer->number_of_active_users > 0)
    {
      UNLOCK_BUFFER_LOCK(buffer);
      if (!ochusha_monitor_timedwait(global_monitor, 500))
	{
#if 0
	  fprintf(stderr, "Some threads not responding...\n");
#endif
	  return;
	}
      LOCK_BUFFER_LOCK(buffer);
    }
  UNLOCK_BUFFER_LOCK(buffer);
}


/*
 * ХåեTERMINATED֤ξFALSE֤
 */
gboolean
ochusha_async_buffer_update_length(OchushaAsyncBuffer *buffer, size_t length)
{
  if (buffer->state == OCHUSHA_ASYNC_BUFFER_TERMINATED)
    return FALSE;	/* XXX: 䤷 */
  LOCK_BUFFER_LOCK(buffer);
  buffer->length = length;
  UNLOCK_BUFFER_LOCK(buffer);

  return ochusha_async_buffer_broadcast(buffer);
}


/*
 * ХåեTERMINATED֤ξFALSE֤
 */
gboolean
ochusha_async_buffer_fix(OchushaAsyncBuffer *buffer)
{
  if (buffer->state == OCHUSHA_ASYNC_BUFFER_TERMINATED)
    return FALSE;	/* XXX: 䤷 */
  LOCK_BUFFER_LOCK(buffer);
  buffer->fixed = TRUE;
  UNLOCK_BUFFER_LOCK(buffer);

  return ochusha_async_buffer_broadcast(buffer);
}


/*
 * Out of memory⤷ϥХåեTERMINATED֤ξFALSE֤
 */
gboolean
ochusha_async_buffer_resize(OchushaAsyncBuffer *buffer, size_t length)
{
  if (buffer->state == OCHUSHA_ASYNC_BUFFER_TERMINATED)
    return FALSE;	/* XXX: 䤷 */
  LOCK_BUFFER_LOCK(buffer);

  if (buffer->fixed)
    {
      fprintf(stderr, "Invalid use of ochusha_async_buffer: Fixed buffer isn't resizable.\n");
      abort();
    }

  buffer->buffer = (char *)G_REALLOC((void *)buffer->buffer, length);
  buffer->buffer_length = length;

  UNLOCK_BUFFER_LOCK(buffer);

  return ochusha_async_buffer_broadcast(buffer);
}


/*
 * ochusha_async_buffer_ensure_free_space:
 *
 * ХåեlengthʾĹĤȤݾڤ롣
 *
 * ХåեTERMINATED֤ξFALSE֤
 */
gboolean
ochusha_async_buffer_ensure_free_space(OchushaAsyncBuffer *buffer,
				       size_t length)
{
  gboolean result = TRUE;

  LOCK_BUFFER_LOCK(buffer);
  if (buffer->fixed)
    {
      fprintf(stderr,
	      "ochusha_async_buffer_ensure_free_space(): buffer is fixed.\n");
      abort();
    }

  while (buffer->state == OCHUSHA_ASYNC_BUFFER_SUSPENDED)
    {
      buffer->number_of_suspended_users++;

      ochusha_monitor_enter(global_monitor);
      ochusha_monitor_notify(global_monitor);
      ochusha_monitor_exit(global_monitor);

      WAIT_BUFFER_CONDITION(buffer);
      /* λbuffer->state != OCHUSHA_ASYNC_BUFFER_SUSPENDEDʤ
       * ʥsuspend򤯤˥ᥤ󥹥åɤäƤΡ
       *
       * buffer->state == OCHUSHA_ASYNC_BUFFER_SUSPENDEDʤ顢waitƤ֤
       * ⤦ٿSUSPENDED֤ˤʤäȤȡ
       */
    }

  while (buffer->buffer_length - buffer->length < length)
    {
      size_t new_buf_len
	= (buffer->buffer_length > 0)
	? (buffer->buffer_length * 2) : OCHUSHA_INITIAL_ASYNC_BUFFER_SIZE;
      buffer->buffer = (char *)G_REALLOC((void *)buffer->buffer, new_buf_len);
      buffer->buffer_length = new_buf_len;
    }
  result = (buffer->state != OCHUSHA_ASYNC_BUFFER_TERMINATED);
  UNLOCK_BUFFER_LOCK(buffer);

  return result;
}


gboolean
ochusha_async_buffer_append_data(OchushaAsyncBuffer *buffer, const char *data,
				 size_t length)
{
  gboolean result = TRUE;

  if (ochusha_async_buffer_ensure_free_space(buffer, length))
    {
      memcpy((char *)buffer->buffer + buffer->length, data, length);
      buffer->length += length;
    }
  else
    result = FALSE;

  return ochusha_async_buffer_broadcast(buffer) && result;
}


#define MAXIMUM_CHUNK_SIZE	65536


gboolean
ochusha_async_buffer_read_file(OchushaAsyncBuffer *buffer, int fd)
{
  gzFile gzfile;
  int chunk_size;
  int result;

  g_return_val_if_fail(fd >= 0, FALSE);

  gzfile = gzdopen(fd, "rb");
  g_return_val_if_fail(gzfile != NULL, FALSE);

  chunk_size = OCHUSHA_INITIAL_ASYNC_BUFFER_SIZE;
  result = 0;

  while (!gzeof(gzfile))
    {
      char *buf;
      if (!ochusha_async_buffer_ensure_free_space(buffer, chunk_size))
	{
	  result = -1;
	  break;
	}

      buf = (char *)buffer->buffer + buffer->length;
      chunk_size = MIN(buffer->buffer_length - buffer->length,
		       MAXIMUM_CHUNK_SIZE);
      result = gzread(gzfile, buf, chunk_size);
      if (result == -1)
	{
	  fprintf(stderr, "gzread cause some error\n");
	  break;
	}

      if (!ochusha_async_buffer_update_length(buffer, buffer->length + result))
	{
	  result = -1;
	  break;
	}
    }

  gzclose(gzfile);

  return result >= 0;
}


void
ochusha_async_buffer_lock(OchushaAsyncBuffer *buffer)
{
  LOCK_BUFFER_LOCK(buffer);
}


void
ochusha_async_buffer_unlock(OchushaAsyncBuffer *buffer)
{
  UNLOCK_BUFFER_LOCK(buffer);
}


gboolean
ochusha_async_buffer_check_active(OchushaAsyncBuffer *buffer)
{
  gboolean result;

  LOCK_BUFFER_LOCK(buffer);

  while (buffer->state == OCHUSHA_ASYNC_BUFFER_SUSPENDED)
    {
      buffer->number_of_suspended_users++;

      ochusha_monitor_enter(global_monitor);
      ochusha_monitor_notify(global_monitor);
      ochusha_monitor_exit(global_monitor);

      WAIT_BUFFER_CONDITION(buffer);	/* ԤäƤ륷ʥ
					 * ᥤ󥹥åɤΤΡ
					 */
    }
  /* ХåեmutexunlockǤstateѤäƤǽΤ
   * λξ֤ǺοȤο롣Ĥޤꡢλ
   * AYNC_BUFFER_TERMINATEDǤʤä顢ĴޤǤϾư
   * ޤλǤOCHUSHA_ASYNC_BUFFER_OKʤΤǡbuffercondvarԤäƤ
   * ΤconsumerΤߤǤ뤳ȤݾڤƤ롣
   */
  result = buffer->state != OCHUSHA_ASYNC_BUFFER_TERMINATED;

  UNLOCK_BUFFER_LOCK(buffer);

  return result;
}


gboolean
ochusha_async_buffer_signal(OchushaAsyncBuffer *buffer)
{
  gboolean result;

  LOCK_BUFFER_LOCK(buffer);

  while (buffer->state == OCHUSHA_ASYNC_BUFFER_SUSPENDED)
    {
      buffer->number_of_suspended_users++;

      ochusha_monitor_enter(global_monitor);
      ochusha_monitor_notify(global_monitor);
      ochusha_monitor_exit(global_monitor);

      WAIT_BUFFER_CONDITION(buffer);	/* ԤäƤ륷ʥ
					 * ᥤ󥹥åɤΤΡ
					 */
    }
  /* ХåեmutexunlockǤstateѤäƤǽΤ
   * λξ֤ǺοȤο롣Ĥޤꡢλ
   * AYNC_BUFFER_TERMINATEDǤʤä顢ĴޤǤϾư
   * ޤλǤOCHUSHA_ASYNC_BUFFER_OKʤΤǡbuffercondvarԤäƤ
   * ΤconsumerΤߤǤ뤳ȤݾڤƤ롣
   */
  result = buffer->state != OCHUSHA_ASYNC_BUFFER_TERMINATED;
  if (result)
    SIGNAL_BUFFER_CONDITION(buffer);	/* λǡbufferwaitƤ
					 * åɤԤäƤ륷ʥ
					 * ᥤ󥹥åɤΤΤǤϤʤ
					 */
  UNLOCK_BUFFER_LOCK(buffer);

  return result;
}


gboolean
ochusha_async_buffer_broadcast(OchushaAsyncBuffer *buffer)
{
  gboolean result;

  LOCK_BUFFER_LOCK(buffer);

  while (buffer->state == OCHUSHA_ASYNC_BUFFER_SUSPENDED)
    {
      buffer->number_of_suspended_users++;

      ochusha_monitor_enter(global_monitor);
      ochusha_monitor_notify(global_monitor);
      ochusha_monitor_exit(global_monitor);

      WAIT_BUFFER_CONDITION(buffer);	/* ԤäƤ륷ʥ
					 * ᥤ󥹥åɤΤΡ
					 */
    }
  /* ХåեmutexunlockǤstateѤäƤǽΤ
   * λξ֤ǺοȤο롣Ĥޤꡢλ
   * AYNC_BUFFER_TERMINATEDǤʤä顢ĴޤǤϾư
   */
  result = buffer->state != OCHUSHA_ASYNC_BUFFER_TERMINATED;
  if (result)
      BROADCAST_BUFFER_CONDITION(buffer);/* λǡbufferwaitƤ
					  * åɤԤäƤ륷ʥ
					  * ᥤ󥹥åɤΤΤǤϤʤ
					  */
  UNLOCK_BUFFER_LOCK(buffer);

  return result;
}


gboolean
ochusha_async_buffer_wait(OchushaAsyncBuffer *buffer)
{
  while (buffer->state == OCHUSHA_ASYNC_BUFFER_SUSPENDED)
    {
      buffer->number_of_suspended_users++;

      ochusha_monitor_enter(global_monitor);
      ochusha_monitor_notify(global_monitor);
      ochusha_monitor_exit(global_monitor);

      WAIT_BUFFER_CONDITION(buffer);
      /* λbuffer->state != OCHUSHA_ASYNC_BUFFER_SUSPENDEDʤ
       * ʥsuspend򤯤˥ᥤ󥹥åɤäƤΡ
       */
    }

  if (buffer->state != OCHUSHA_ASYNC_BUFFER_TERMINATED)
    WAIT_BUFFER_CONDITION(buffer);

  /* λOCHUSHA_ASYNC_BUFFER_SUSPENDEDʾ⤢뤬
   * ξsuspendϼȤˤ롣
   */

  return buffer->state != OCHUSHA_ASYNC_BUFFER_TERMINATED;
}


/*
 * ochusha_async_buffer_is_busy:
 *
 * bufferΥåcontentionäƤȤˤTRUE֤
 */
gboolean
ochusha_async_buffer_is_busy(OchushaAsyncBuffer *buffer)
{
  g_return_val_if_fail(OCHUSHA_IS_ASYNC_BUFFER(buffer), FALSE);

  return buffer->state != OCHUSHA_ASYNC_BUFFER_OK
    || buffer->number_of_lock_waiters > 0;
}


void
ochusha_async_buffer_free_when_finished(OchushaAsyncBuffer *buffer)
{
  if (buffer->buffer != NULL)
    G_FREE((void *)buffer->buffer);
}


void
ochusha_async_buffer_emit_access_started(OchushaAsyncBuffer *buffer)
{
  g_return_if_fail(OCHUSHA_IS_ASYNC_BUFFER(buffer));
  g_signal_emit(G_OBJECT(buffer),
		async_buffer_signals[ACCESS_STARTED_SIGNAL], 0);
}


void
ochusha_async_buffer_emit_access_progressed(OchushaAsyncBuffer *buffer,
					    int bytes_read, int bytes_total)
{
  g_return_if_fail(OCHUSHA_IS_ASYNC_BUFFER(buffer));
  g_signal_emit(G_OBJECT(buffer),
		async_buffer_signals[ACCESS_PROGRESSED_SIGNAL], 0,
		bytes_read, bytes_total);
}


void
ochusha_async_buffer_emit_access_finished(OchushaAsyncBuffer *buffer)
{
  g_return_if_fail(OCHUSHA_IS_ASYNC_BUFFER(buffer));
  g_signal_emit(G_OBJECT(buffer),
		async_buffer_signals[ACCESS_FINISHED_SIGNAL], 0);
}


void
ochusha_async_buffer_emit_access_failed(OchushaAsyncBuffer *buffer,
					int reason_code, const gchar *reason)
{
  g_return_if_fail(OCHUSHA_IS_ASYNC_BUFFER(buffer));
  g_signal_emit(G_OBJECT(buffer),
		async_buffer_signals[ACCESS_FAILED_SIGNAL], 0,
		reason_code, reason);
}
