/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id$
 */

#include "ochusha.h"

#include "htmlutils.h"

#include <glib.h>
#include <libxml/SAX.h>
#include <libxml/parser.h>

#include <errno.h>

#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>


#define DEBUG_SAX_HANDLER	0
#define DEBUG_SAX_HANDLER_NEW	0


typedef struct _SAXContext SAXContext;

static void write_board(gpointer data, gpointer user_data);
static void write_category(gpointer data, gpointer user_data);
static void cleanup_sax_context(SAXContext *context);
static void startElementHandler(void *context, const xmlChar *name,
				const xmlChar **atts);
static gboolean hash_table_cleanup_func(gpointer key, gpointer value,
					gpointer unused);
static void hash_table_cleanup(GHashTable *hash_table);
static gpointer hash_table_free_key(GHashTable *hash_table, gpointer key);
static void endElementHandler(void *context, const xmlChar *name);
static void charactersHandler(void *context, const xmlChar *ch, int len);
static void nopHandler(void *context);
static xmlEntityPtr getEntityHandler(void *context, const xmlChar *name);
static void input_board_attributes(BulletinBoard *board,
				   GHashTable *hash_table,
				   ReadBoardElementFunc *optional,
				   gpointer user_data);


/*
 * ochusha_prepare_home
 *
 * ochushaΤΥ桼ۡǥ쥯ȥ(ɸǤ$HOME/.ochusha)
 * ¸ߤå¸ߤʤФΥǥ쥯ȥ롣
 *
 * ǽŪ$HOME/.ochushaǥ쥯ȥ꤬¸ߤ֤ˤʤХǥ쥯ȥؤ
 * PATH̾򡢤ʤNULLconf->home˳Ǽ롣
 */
void
ochusha_prepare_home(OchushaConfig *conf)
{
  int len;
  char *ochusha_home = getenv("OCHUSHA_HOME");

  if (ochusha_home == NULL)
    {
      char buffer[PATH_MAX];
      char *home = getenv("HOME");
      if (home == NULL)
	goto no_home_exit;

      len = snprintf(buffer, PATH_MAX, "%s/.ochusha", home);
      if (len >= PATH_MAX)
	goto no_home_exit;

      ochusha_home = strdup(buffer);
    }
  else
    ochusha_home = strdup(ochusha_home);

  if (mkdir_p(ochusha_home))
    {
      conf->home = ochusha_home;
      return;
    }

  fprintf(stderr, "Couldn't create directory %s: %s\n", ochusha_home,
	  strerror(errno));
  free(ochusha_home);

 no_home_exit:
  conf->home = NULL;
  return;
}


/*
 * ochusha_find_file
 *
 * ochushahomeǥ쥯ȥФpath̾filenameĥե뤬
 * ¸ߤ뤫ɤ򸡺¸ߤˤϥѥ̾ɽʸ֤
 * callerϥѥ̾ʸפˤʤäfree()̳餦
 */
char *
ochusha_find_file(OchushaConfig *conf, const char *filename)
{
  int len;
  char pathname[PATH_MAX];

  if (conf->home == NULL)
    return NULL;

  len = snprintf(pathname, PATH_MAX, "%s/%s", conf->home, filename);
  if (len >= PATH_MAX)
    return NULL;

  /* Ƥߤ¸˽ */
  len = open(pathname, O_RDONLY);
  if (len >= 0)
    {
      close(len);
      return strdup(pathname);
    }

  return NULL;
}


/*
 * boardlist.xmlϢ
 */
#define OUTPUT_CATEGORY_ATTRIBUTE_BOOLEAN(file, category, attribute)	\
  fprintf(file,								\
	  "      <attribute name=\"" #attribute	"\">\n"			\
	  "        <boolean val=\"%s\"/>\n"				\
	  "      </attribute>\n",					\
	  (category)->attribute ? "true" : "false")

#define OUTPUT_CATEGORY_ATTRIBUTE_INT(file, category, attribute)	\
  fprintf(file,								\
	  "      <attribute name=\"" #attribute	"\">\n"			\
	  "        <int val=\"%d\"/>\n"					\
	  "      </attribute>\n",					\
	  (category)->attribute)

#define OUTPUT_CATEGORY_ATTRIBUTE_STRING(file, category, attribute)	\
  fprintf(file,								\
	  "      <attribute name=\"" #attribute	"\">\n"			\
	  "        <string>%s</string>\n"				\
	  "      </attribute>\n", (category)->attribute)

#define OUTPUT_BOARD_ATTRIBUTE_INT(file, board, attribute)		\
  fprintf(file,								\
	  "        <attribute name=\"" #attribute	"\">\n"		\
	  "          <int val=\"%d\"/>\n"				\
	  "        </attribute>\n", (board)->attribute)

#define OUTPUT_BOARD_ATTRIBUTE_STRING(file, board, attribute)		\
  fprintf(file,								\
	  "        <attribute name=\"" #attribute	"\">\n"		\
	  "          <string>%s</string>\n"				\
	  "        </attribute>\n", (board)->attribute)

#define OUTPUT_BOARD_ATTRIBUTE_IF_NONNULL(file, board, attribute)	\
  if ((board)->attribute != NULL)					\
    fprintf(file,							\
	    "        <attribute name=\"" #attribute "\">\n"		\
	    "          <string>%s</string>\n"				\
	    "        </attribute>\n", (board)->attribute)


typedef struct _WriteBoardlistArgs
{
  FILE *boardlist_xml;
  WriteCategoryElementFunc *cat_optional;
  WriteBoardElementFunc *board_optional;
  gpointer user_data;
} WriteBoardlistArgs;


static void
write_board(gpointer data, gpointer user_data)
{
  WriteBoardlistArgs *args = (WriteBoardlistArgs *)user_data;
  BulletinBoard *board = (BulletinBoard *)data;
  FILE *file = args->boardlist_xml;

  fprintf(file, "      <board>\n");

  OUTPUT_BOARD_ATTRIBUTE_STRING(file, board, name);
  OUTPUT_BOARD_ATTRIBUTE_STRING(file, board, base_url);

  OUTPUT_BOARD_ATTRIBUTE_IF_NONNULL(file, board, title_picture);
  OUTPUT_BOARD_ATTRIBUTE_IF_NONNULL(file, board, title_color);
  OUTPUT_BOARD_ATTRIBUTE_IF_NONNULL(file, board, title_link);
  OUTPUT_BOARD_ATTRIBUTE_IF_NONNULL(file, board, bg_color);
  OUTPUT_BOARD_ATTRIBUTE_IF_NONNULL(file, board, bg_picture);
  OUTPUT_BOARD_ATTRIBUTE_IF_NONNULL(file, board, noname_name);
  OUTPUT_BOARD_ATTRIBUTE_IF_NONNULL(file, board, makethread_color);
  OUTPUT_BOARD_ATTRIBUTE_IF_NONNULL(file, board, menu_color);
  OUTPUT_BOARD_ATTRIBUTE_IF_NONNULL(file, board, thread_color);
  OUTPUT_BOARD_ATTRIBUTE_IF_NONNULL(file, board, text_color);
  OUTPUT_BOARD_ATTRIBUTE_IF_NONNULL(file, board, name_color);
  OUTPUT_BOARD_ATTRIBUTE_IF_NONNULL(file, board, link_color);
  OUTPUT_BOARD_ATTRIBUTE_IF_NONNULL(file, board, alink_color);
  OUTPUT_BOARD_ATTRIBUTE_IF_NONNULL(file, board, vlink_color);
  OUTPUT_BOARD_ATTRIBUTE_IF_NONNULL(file, board, subject_color);

  if (args->board_optional != NULL)
    (*args->board_optional)(file, board, args->user_data);
  else
    fprintf(stderr, "No optional board attributes?\n");

  fprintf(file, "      </board>\n");
}


static void
write_category(gpointer data, gpointer user_data)
{
  WriteBoardlistArgs *args = (WriteBoardlistArgs *)user_data;
  BoardCategory *category = (BoardCategory *)data;
  FILE *file = args->boardlist_xml;

  fprintf(file, "    <category>\n");

  OUTPUT_CATEGORY_ATTRIBUTE_STRING(file, category, name);

  if (args->cat_optional != NULL)
    (*args->cat_optional)(file, category, args->user_data);
  else
    fprintf(stderr, "No optional attributes for category?\n");

  g_slist_foreach(category->board_list, write_board, args);

  fprintf(file, "    </category>\n");
}


gboolean
ochusha_write_boardlist_xml(OchushaConfig *conf, BBSTable *table,
			    WriteCategoryElementFunc *cat_optional,
			    WriteBoardElementFunc *board_optional,
			    gpointer user_data)
{
  FILE *boardlist_xml;
  int len;
  char pathname[PATH_MAX];
  WriteBoardlistArgs args = { NULL, cat_optional, board_optional, user_data };
  GSList *category_list = table->category_list;

  if (conf->home == NULL)
    {
      fprintf(stderr, "No home!\n");
      return FALSE;
    }

  len = snprintf(pathname, PATH_MAX, "%s/%s",
		 conf->home, OCHUSHA_BOARDLIST_XML);
  if (len >= PATH_MAX)
    {
      fprintf(stderr, "Too long path name.\n");
      return FALSE;
    }

  boardlist_xml = fopen(pathname, "w");
  if (boardlist_xml == NULL)
    {
      fprintf(stderr, "Couldn't open \"%s\" to write.\n", pathname);
      return FALSE;
    }

  args.boardlist_xml = boardlist_xml;

  fprintf(boardlist_xml, "<?xml version=\"1.0\"?>\n");
  fprintf(boardlist_xml, "<ochusha>\n");
  fprintf(boardlist_xml, "  <boardlist>\n");

  g_slist_foreach(category_list, write_category, &args);

  fprintf(boardlist_xml, "  </boardlist>\n");
  fprintf(boardlist_xml, "</ochusha>\n");
  return fclose(boardlist_xml) == 0;
}


typedef enum
{
  SAX_INITIAL,
  SAX_OCHUSHA,
  SAX_PREFERENCE,
  SAX_PREFERENCE_ATTRIBUTE,
  SAX_PREFERENCE_ATTRIBUTE_BOOLEAN,
  SAX_PREFERENCE_ATTRIBUTE_INT,
  SAX_PREFERENCE_ATTRIBUTE_STRING,
  SAX_BOARDLIST,
  SAX_CATEGORY,
  SAX_CATEGORY_ATTRIBUTE,
  SAX_CATEGORY_ATTRIBUTE_BOOLEAN,
  SAX_CATEGORY_ATTRIBUTE_INT,
  SAX_CATEGORY_ATTRIBUTE_STRING,
  SAX_CATEGORY_BOARD,
  SAX_CATEGORY_BOARD_ATTRIBUTE,
  SAX_CATEGORY_BOARD_ATTRIBUTE_BOOLEAN,
  SAX_CATEGORY_BOARD_ATTRIBUTE_INT,
  SAX_CATEGORY_BOARD_ATTRIBUTE_STRING,
  SAX_THREADLIST,
  SAX_THREAD,
  SAX_THREAD_ATTRIBUTE,
  SAX_THREAD_ATTRIBUTE_BOOLEAN,
  SAX_THREAD_ATTRIBUTE_INT,
  SAX_THREAD_ATTRIBUTE_STRING,
  SAX_ACCEPTED,
  SAX_ERROR
} SAXState;


struct _SAXContext
{
  SAXState state;

  char *current_attr_name;
  char *current_attr_val;

  BBSTable *bbs_table;

  GSList *board_list;

  BulletinBoard *board;

  GHashTable *pref_attributes;

  GHashTable *category_attributes;
  ReadCategoryElementFunc *category_attributes_cb;
  GHashTable *board_attributes;
  ReadBoardElementFunc *board_attributes_cb;

  GHashTable *thread_attributes;
  ReadThreadElementFunc *thread_attributes_cb;

  gpointer user_data;
};


static void
board_list_free_helper(gpointer data, gpointer unused)
{
  ochusha_bulletin_board_free((BulletinBoard *)data);
}


static void
board_list_free(GSList *board_list)
{
  g_slist_foreach(board_list, board_list_free_helper, NULL);
  g_slist_free(board_list);
}


static void
cleanup_sax_context(SAXContext *context)
{
  if (context->current_attr_name != NULL)
    {
      free(context->current_attr_name);
      context->current_attr_name = NULL;
    }

  if (context->current_attr_val != NULL)
    {
      free(context->current_attr_val);
      context->current_attr_val = NULL;
    }

  if (context->bbs_table != NULL)
    {
      ochusha_bbs_table_free(context->bbs_table);
      context->bbs_table = NULL;
    }

  if (context->board_list != NULL)
    {
      board_list_free(context->board_list);
      context->board_list = NULL;
    }

  if (context->board != NULL)
    {
      context->board = NULL;
    }

  if (context->pref_attributes != NULL)
    {
      hash_table_cleanup(context->pref_attributes);
      g_hash_table_destroy(context->pref_attributes);
      context->pref_attributes = NULL;
    }

  if (context->category_attributes != NULL)
    {
      hash_table_cleanup(context->category_attributes);
      g_hash_table_destroy(context->category_attributes);
      context->category_attributes = NULL;
    }

  if (context->board_attributes != NULL)
    {
      hash_table_cleanup(context->board_attributes);
      g_hash_table_destroy(context->board_attributes);
      context->board_attributes = NULL;
    }

  if (context->thread_attributes != NULL)
    {
      hash_table_cleanup(context->thread_attributes);
      g_hash_table_destroy(context->thread_attributes);
      context->thread_attributes = NULL;
    }
}


static void
startElementHandler(void *context, const xmlChar *name, const xmlChar **atts)
{
  SAXContext *sax_context = (SAXContext *)context;

  switch (sax_context->state)
    {
    case SAX_INITIAL:
      if (strcmp(name, "ochusha") == 0)
	{ sax_context->state = SAX_OCHUSHA; return; }
      break;	/* 顼 */

    case SAX_OCHUSHA:
      if (strcmp(name, "boardlist") == 0)
	{
	  sax_context->category_attributes
	    = g_hash_table_new(g_str_hash, g_str_equal);
	  sax_context->board_attributes
	    = g_hash_table_new(g_str_hash, g_str_equal);
	  sax_context->state = SAX_BOARDLIST;
	  return;
	}
      else if (strcmp(name, "preference") == 0)
	{
	  sax_context->pref_attributes
	    = g_hash_table_new(g_str_hash, g_str_equal);
	  sax_context->state = SAX_PREFERENCE;
	  return;
	}
      else if (strcmp(name, "threadlist") == 0)
	{
	  sax_context->thread_attributes
	    = g_hash_table_new(g_str_hash, g_str_equal);
	  sax_context->state = SAX_THREADLIST;
	  return;
	}
      break;	/* 顼 */

    case SAX_PREFERENCE:
    case SAX_CATEGORY:
    case SAX_CATEGORY_BOARD:
    case SAX_THREAD:
      if (strcmp(name, "attribute") == 0
	  && atts != NULL && strcmp(atts[0], "name") == 0)
	{
	  switch (sax_context->state)
	    {
	    case SAX_PREFERENCE:
	      sax_context->state = SAX_PREFERENCE_ATTRIBUTE;
	      break;
	    case SAX_CATEGORY:
	      sax_context->state = SAX_CATEGORY_ATTRIBUTE;
	      break;
	    case SAX_CATEGORY_BOARD:
	      sax_context->state = SAX_CATEGORY_BOARD_ATTRIBUTE;
	      break;
	    case SAX_THREAD:
	      sax_context->state = SAX_THREAD_ATTRIBUTE;
	      break;
	    default:
	      fprintf(stderr, "Wrong implementation found.\n");
	      abort();
	    }
	  if (sax_context->current_attr_val != NULL)
	    {
#if DEBUG_SAX_HANDLER
	      fprintf(stderr, "unexpected attribute found: %s=%s\n",
		      sax_context->current_attr_name,
		      sax_context->current_attr_val);
#endif
	      free(sax_context->current_attr_name);
	      free(sax_context->current_attr_val);
	      sax_context->current_attr_name = NULL;
	      sax_context->current_attr_val = NULL;
	      break;	/* 顼 */
	    }
	  sax_context->current_attr_name = strdup(atts[1]);
	  return;
	}
      else if (sax_context->state == SAX_CATEGORY
	       && strcmp(name, "board") == 0)
	{ sax_context->state = SAX_CATEGORY_BOARD; return; }
      break;	/* 顼 */

    case SAX_PREFERENCE_ATTRIBUTE:
    case SAX_CATEGORY_ATTRIBUTE:
    case SAX_CATEGORY_BOARD_ATTRIBUTE:
    case SAX_THREAD_ATTRIBUTE:
      if (atts != NULL && strcmp(atts[0], "val") == 0)
	{
	  /* int/booleanβǽ */
	  if (strcmp(name, "int") == 0)
	    {
	      switch (sax_context->state)
		{
		case SAX_PREFERENCE_ATTRIBUTE:
		  sax_context->state = SAX_PREFERENCE_ATTRIBUTE_INT;
		  break;
		case SAX_CATEGORY_ATTRIBUTE:
		  sax_context->state = SAX_CATEGORY_ATTRIBUTE_INT;
		  break;
		case SAX_CATEGORY_BOARD_ATTRIBUTE:
		  sax_context->state = SAX_CATEGORY_BOARD_ATTRIBUTE_INT;
		  break;
		case SAX_THREAD_ATTRIBUTE:
		  sax_context->state = SAX_THREAD_ATTRIBUTE_INT;
		  break;
		default:
		  fprintf(stderr, "Wrong implementation found.\n");
		  abort();
		}
	    }
	  else if (strcmp(name, "boolean") == 0)
	    {
	      switch (sax_context->state)
		{
		case SAX_PREFERENCE_ATTRIBUTE:
		  sax_context->state = SAX_PREFERENCE_ATTRIBUTE_BOOLEAN;
		  break;
		case SAX_CATEGORY_ATTRIBUTE:
		  sax_context->state = SAX_CATEGORY_ATTRIBUTE_BOOLEAN;
		  break;
		case SAX_CATEGORY_BOARD_ATTRIBUTE:
		  sax_context->state = SAX_CATEGORY_BOARD_ATTRIBUTE_BOOLEAN;
		  break;
		case SAX_THREAD_ATTRIBUTE:
		  sax_context->state = SAX_THREAD_ATTRIBUTE_BOOLEAN;
		  break;
		default:
		  fprintf(stderr, "Wrong implementation found.\n");
		  abort();
		}
	    }
	  else
	    {
#if DEBUG_SAX_HANDLER
	      fprintf(stderr, "element unexpected in state(%d) found: %s\n",
		      sax_context->state, name);
#endif
	      break;	/* 顼 */
	    }

	  if (sax_context->current_attr_val != NULL)
	    {
	      /* 顼ǤɤΤġ̯ */
#if DEBUG_SAX_HANDLER
	      fprintf(stderr,
		      "attribute %s=\"%s\" is overwritten by \"%s\"!\n",
		      sax_context->current_attr_name,
		      sax_context->current_attr_val, atts[1]);
#endif
	      free(sax_context->current_attr_val);
	    }

	  sax_context->current_attr_val = strdup(atts[1]);
	  return;
	}
      else if (strcmp(name, "string") == 0)
	{
	  switch (sax_context->state)
	    {
	    case SAX_PREFERENCE_ATTRIBUTE:
	      sax_context->state = SAX_PREFERENCE_ATTRIBUTE_STRING;
	      return;
	    case SAX_CATEGORY_ATTRIBUTE:
	      sax_context->state = SAX_CATEGORY_ATTRIBUTE_STRING;
	      return;
	    case SAX_CATEGORY_BOARD_ATTRIBUTE:
	      sax_context->state = SAX_CATEGORY_BOARD_ATTRIBUTE_STRING;
	      return;
	    case SAX_THREAD_ATTRIBUTE:
	      sax_context->state = SAX_THREAD_ATTRIBUTE_STRING;
	      return;
	    default:
	      break;	/* 顼 */
	    }
#if DEBUG_SAX_HANDLER
	  fprintf(stderr, "string element is unexpected in state(%d).\n",
		  sax_context->state);
#endif
	}
      break;	/* 顼 */

    case SAX_BOARDLIST:
      if (strcmp(name, "category") == 0)
	{
	  if (sax_context->board_list != NULL)
	    {
#if DEBUG_SAX_HANDLER
	      fprintf(stderr, "free board_list found!\n");
#endif
	      board_list_free(sax_context->board_list);
	    }
	  sax_context->board_list = NULL;
	  sax_context->state = SAX_CATEGORY;
	  return;
	}
      break;	/* 顼 */

    case SAX_THREADLIST:
      if (strcmp(name, "thread") == 0)
	{ sax_context->state = SAX_THREAD; return; }
      break;	/* 顼 */

    case SAX_PREFERENCE_ATTRIBUTE_STRING:
    case SAX_PREFERENCE_ATTRIBUTE_BOOLEAN:
    case SAX_PREFERENCE_ATTRIBUTE_INT:
    case SAX_CATEGORY_ATTRIBUTE_STRING:
    case SAX_CATEGORY_ATTRIBUTE_BOOLEAN:
    case SAX_CATEGORY_BOARD_ATTRIBUTE_STRING:
    case SAX_CATEGORY_BOARD_ATTRIBUTE_INT:
    case SAX_THREAD_ATTRIBUTE_INT:
    case SAX_THREAD_ATTRIBUTE_STRING:
    case SAX_ACCEPTED:
    case SAX_ERROR:
      break;	/* 顼 */

    default:
      fprintf(stderr, "startHandler is called in unknown state: %d\n",
	      sax_context->state);
    }
  sax_context->state = SAX_ERROR;
}


static gboolean
hash_table_cleanup_func(gpointer key, gpointer value, gpointer unused)
{
  free(key);
  free(value);
  return TRUE;
}


static void
hash_table_cleanup(GHashTable *hash_table)
{
  g_hash_table_foreach_remove(hash_table, hash_table_cleanup_func, NULL);
}


static gpointer
hash_table_free_key(GHashTable *hash_table, gpointer key)
{
  gpointer orig_key = NULL;
  gpointer value = NULL;
  g_hash_table_lookup_extended(hash_table, key, &orig_key, &value);
  if (orig_key != NULL)
    {
      g_hash_table_remove(hash_table, orig_key);
      free(orig_key);
    }

  return value;
}


int
get_attribute_int(GHashTable *hash_table, char *attr_name)
{
  char *num_str = g_hash_table_lookup(hash_table, attr_name);
  int num;

  if (num_str == NULL)
    return 0;

  num = 0;
  sscanf(num_str, "%d", &num);
  return num;
}


gboolean
get_attribute_boolean(GHashTable *hash_table, char *attr_name)
{
  char *bool_str = g_hash_table_lookup(hash_table, attr_name);

  if (bool_str == NULL)
    return FALSE;

  return strcasecmp("true", bool_str) == 0;
}


char *
get_attribute_string(GHashTable *hash_table, char *attr_name)
{
  return hash_table_free_key(hash_table, attr_name);
}


#define INPUT_ATTRIBUTE_INT(object, hash_table, attr_name)		\
  do									\
    {									\
      char *num_str = g_hash_table_lookup((hash_table), #attr_name);	\
      if (num_str != NULL)						\
	{								\
	  int num = 0;							\
  	  sscanf(num_str, "%d", &num);					\
	  (object)->attr_name = num;					\
	}								\
      else								\
	(object)->attr_name = 0;					\
    } while (0)

#define INPUT_ATTRIBUTE_BOOLEAN(object, hash_table, attr_name)		\
  do									\
    {									\
      char *bool_str = g_hash_table_lookup((hash_table), #attr_name);	\
      if (bool_str != NULL)						\
	{								\
	  (object)->attr_name = (strcasecmp("true", bool_str) == 0);	\
	}								\
      else								\
	(object)->attr_name = FALSE;					\
    } while (0)

#define INPUT_ATTRIBUTE_STRING(object, hash_table, attr_name)		\
  (object)->attr_name = hash_table_free_key((hash_table), #attr_name)


static void
endElementHandler(void *context, const xmlChar *name)
{
  SAXContext *sax_context = (SAXContext *)context;

  switch (sax_context->state)
    {
    case SAX_OCHUSHA:
      if (strcmp(name, "ochusha") == 0)
	{ sax_context->state = SAX_ACCEPTED; return; }
      break;	/* 顼 */

    case SAX_PREFERENCE:
      if (strcmp(name, "preference") == 0)
	{ sax_context->state = SAX_OCHUSHA; return; }
      break;	/* 顼 */

    case SAX_PREFERENCE_ATTRIBUTE:
    case SAX_CATEGORY_ATTRIBUTE:
    case SAX_CATEGORY_BOARD_ATTRIBUTE:
    case SAX_THREAD_ATTRIBUTE:
      if (strcmp(name, "attribute") == 0)
	{
	  GHashTable *hash_table;
	  switch (sax_context->state)
	    {
	    case SAX_PREFERENCE_ATTRIBUTE:
	      hash_table = sax_context->pref_attributes;
	      sax_context->state = SAX_PREFERENCE;
	      break;
	    case SAX_CATEGORY_ATTRIBUTE:
	      hash_table = sax_context->category_attributes;
	      sax_context->state = SAX_CATEGORY;
	      break;
	    case SAX_CATEGORY_BOARD_ATTRIBUTE:
	      hash_table = sax_context->board_attributes;
	      sax_context->state = SAX_CATEGORY_BOARD;
	      break;
	    case SAX_THREAD_ATTRIBUTE:
	      hash_table = sax_context->thread_attributes;
	      sax_context->state = SAX_THREAD;
	      break;
	      break;
	    default:
	      fprintf(stderr, "Wrong implementation found.\n");
	      abort();
	    }
#if DEBUG_SAX_HANDLER_NEW
	  fprintf(stderr, "%s = \"%s\"\n", sax_context->current_attr_name,
		  sax_context->current_attr_val);
#endif
	  g_hash_table_insert(hash_table,
			      sax_context->current_attr_name,
			      sax_context->current_attr_val);
	  sax_context->current_attr_name = NULL;
	  sax_context->current_attr_val = NULL;
	  return;
	}
      break;	/* 顼 */

    case SAX_PREFERENCE_ATTRIBUTE_STRING:
    case SAX_CATEGORY_ATTRIBUTE_STRING:
    case SAX_CATEGORY_BOARD_ATTRIBUTE_STRING:
    case SAX_THREAD_ATTRIBUTE_STRING:
      if (strcmp(name, "string") == 0)
	{
	  switch (sax_context->state)
	    {
	    case SAX_PREFERENCE_ATTRIBUTE_STRING:
	      sax_context->state = SAX_PREFERENCE_ATTRIBUTE;
	      break;
	    case SAX_CATEGORY_ATTRIBUTE_STRING:
	      sax_context->state = SAX_CATEGORY_ATTRIBUTE;
	      break;
	    case SAX_CATEGORY_BOARD_ATTRIBUTE_STRING:
	      sax_context->state = SAX_CATEGORY_BOARD_ATTRIBUTE;
	      break;
	    case SAX_THREAD_ATTRIBUTE_STRING:
	      sax_context->state = SAX_THREAD_ATTRIBUTE;
	      break;
	    default:
	      fprintf(stderr, "Wrong implementation found.\n");
	      abort();
	    }
	  /* NOTE:
	   * stringattributeƤSAXѡԹ()ǡĤ
	   * ޤǤheapˤʤΤǤstrdup
	   * SAXѡԹȤΤϡΥץŪˤ1ʸǤ٤
	   * ƤSAXѡŪʣǤȤƸƤޤ礬ꡢ
	   * Ϣ뤹뤿ΰХåեɬפǤ뤫顣
	   */
	  sax_context->current_attr_val = strdup(sax_context->current_attr_val);
	  return;
	}
      break;	/* 顼 */

    case SAX_PREFERENCE_ATTRIBUTE_BOOLEAN:
      if (strcmp(name, "boolean") == 0)
	{ sax_context->state = SAX_PREFERENCE_ATTRIBUTE; return; }
      break;	/* 顼 */

    case SAX_PREFERENCE_ATTRIBUTE_INT:
      if (strcmp(name, "int") == 0)
	{ sax_context->state = SAX_PREFERENCE_ATTRIBUTE; return; }
      break;	/* 顼 */

    case SAX_BOARDLIST:
      if (strcmp(name, "boardlist") == 0)
	{
	  hash_table_cleanup(sax_context->category_attributes);
	  g_hash_table_destroy(sax_context->category_attributes);
	  sax_context->category_attributes = NULL;

	  hash_table_cleanup(sax_context->board_attributes);
	  g_hash_table_destroy(sax_context->board_attributes);
	  sax_context->board_attributes = NULL;
	  sax_context->state = SAX_OCHUSHA;
	  return;
	}
      break;	/* 顼 */

    case SAX_CATEGORY:
      if (strcmp(name, "category") == 0)
	{
	  gchar *category_name
	    = hash_table_free_key(sax_context->category_attributes, "name");
#if DEBUG_SAX_HANDLER
	  fprintf(stderr, "closing category\n");
#endif
	  if (sax_context->board_list != NULL)
	    {
	      BBSTable *bbs_table = sax_context->bbs_table;
	      BoardCategory *category
		= g_hash_table_lookup(bbs_table->category_table,
				      category_name);
	      if (category == NULL)
		{
		  category = ochusha_board_category_new(category_name);
		  if (category != NULL)
		    category_name = NULL;
		}

	      if (category != NULL)
		{
		  if (sax_context->category_attributes_cb != NULL)
		    (*sax_context->category_attributes_cb)(category, sax_context->category_attributes, sax_context->user_data);
		  else
		    fprintf(stderr, "No optional category attributes?\n");

		  category->board_list = g_slist_concat(category->board_list,
							sax_context->board_list);
		  sax_context->board_list = NULL;
		  ochusha_bbs_table_add_category(bbs_table, category);
#if DEBUG_SAX_HANDLER
		  fprintf(stderr, "adding category: %s\n", category_name);
#endif
		}
	      else
		{
		  board_list_free(sax_context->board_list);
		  sax_context->board_list = NULL;
		}
	    }
	  else
	    {
#if DEBUG_SAX_HANDLER
	      fprintf(stderr, "removing empty category: %s\n", category_name);
#endif
	      if (category_name != NULL)
		free(category_name);
	    }
	  hash_table_cleanup(sax_context->category_attributes);
	  sax_context->state = SAX_BOARDLIST;
	  return;
	}
      break;	/* 顼 */

    case SAX_CATEGORY_ATTRIBUTE_BOOLEAN:
      if (strcmp(name, "boolean") == 0)
	{ sax_context->state = SAX_CATEGORY_ATTRIBUTE; return; }
      break;	/* 顼 */

    case SAX_CATEGORY_ATTRIBUTE_INT:
      if (strcmp(name, "int") == 0)
	{ sax_context->state = SAX_CATEGORY_ATTRIBUTE; return; }
      break;	/* 顼 */

    case SAX_CATEGORY_BOARD:
      if (strcmp(name, "board") == 0)
	{
	  gchar *board_name
	    = hash_table_free_key(sax_context->board_attributes, "name");
	  char *board_url
	    = hash_table_free_key(sax_context->board_attributes, "base_url");
	  BBSTable *bbs_table = sax_context->bbs_table;
	  BulletinBoard *board;
	  if (bbs_table != NULL)
	    board = g_hash_table_lookup(bbs_table->board_table, board_url);
	  else
	    board = NULL;
#if DEBUG_SAX_HANDLER
	  fprintf(stderr, "closing category/board\n");
#endif
	  if (board == NULL && bbs_table != NULL)
	    {
	      board = ochusha_bulletin_board_new(board_name, board_url);
	      if (board != NULL)
		{
#if DEBUG_SAX_HANDLER
		  fprintf(stderr, "adding board: %s\n", board_name);
#endif
		  board_name = NULL;
		  board_url = NULL;
		  input_board_attributes(board, sax_context->board_attributes,
					 sax_context->board_attributes_cb,
					 sax_context->user_data);
		  ochusha_bbs_table_add_board(bbs_table, board);
		}
	    }

	  if (board != NULL)
	    sax_context->board_list = g_slist_append(sax_context->board_list,
						     board);

	  if (board_name != NULL)
	    free(board_name);
	  if (board_url != NULL)
	    free(board_url);

	  hash_table_cleanup(sax_context->board_attributes);
	  sax_context->state = SAX_CATEGORY;
	  return;
	}
      break;	/* 顼 */

    case SAX_CATEGORY_BOARD_ATTRIBUTE_BOOLEAN:
      if (strcmp(name, "boolean") == 0)
	{ sax_context->state = SAX_CATEGORY_BOARD_ATTRIBUTE; return; }
      break;	/* 顼 */

    case SAX_CATEGORY_BOARD_ATTRIBUTE_INT:
      if (strcmp(name, "int") == 0)
	{ sax_context->state = SAX_CATEGORY_BOARD_ATTRIBUTE; return; }
      break;	/* 顼 */

    case SAX_THREADLIST:
      if (strcmp(name, "threadlist") == 0)
	{
	  hash_table_cleanup(sax_context->thread_attributes);
	  g_hash_table_destroy(sax_context->thread_attributes);
	  sax_context->thread_attributes = NULL;

	  sax_context->state = SAX_OCHUSHA;
	  return;
	}
      break;	/* 顼 */

    case SAX_THREAD:
      if (strcmp(name, "thread") == 0)
	{
	  GHashTable *thread_attrs = sax_context->thread_attributes;

	  char *dat_filename = g_hash_table_lookup(thread_attrs,
						   "dat_filename");
	  if (dat_filename != NULL)
	    {
	      BBSThread *thread
		= g_hash_table_lookup(sax_context->board->thread_table,
				      dat_filename);

	      if (thread == NULL)
		{
		  BulletinBoard *board = sax_context->board;
		  gchar *title = get_attribute_string(thread_attrs, "title");
		  if (title != NULL)
		    {
		      dat_filename = get_attribute_string(thread_attrs,
							  "dat_filename");
		      thread = ochusha_bbs_thread_new(board,
						      dat_filename, title);
		      if (thread == NULL)
			{
			  free(title);
			  free(dat_filename);
			}
		      else
			{
			  board->thread_list
			    = g_slist_append(board->thread_list, thread);
			  g_hash_table_insert(board->thread_table,
					      dat_filename, thread);
			}
		    }
		}

	      if (thread != NULL)
		{
		  INPUT_ATTRIBUTE_INT(thread, thread_attrs,
				      number_of_responses_read);
		  INPUT_ATTRIBUTE_INT(thread, thread_attrs,
				      flags);
		  (*sax_context->thread_attributes_cb)(thread,
						       thread_attrs,
						       sax_context->user_data);
		}
	    }
	   /* ꡼󥢥å */
	  hash_table_cleanup(thread_attrs);
	  sax_context->state = SAX_THREADLIST;
	  return;
	}
      break;	/* 顼 */

    case SAX_THREAD_ATTRIBUTE_BOOLEAN:
      if (strcmp(name, "boolean") == 0)
	{ sax_context->state = SAX_THREAD_ATTRIBUTE; return; }
      break;	/* 顼 */

    case SAX_THREAD_ATTRIBUTE_INT:
      if (strcmp(name, "int") == 0)
	{ sax_context->state = SAX_THREAD_ATTRIBUTE; return; }
      break;	/* 顼 */

    case SAX_INITIAL:
    case SAX_ACCEPTED:
    case SAX_ERROR:
      break;	/* 顼 */

    default:
#if DEBUG_SAX_HANDLER
      fprintf(stderr, "endHandler called in unknown state: %d.\n",
	      sax_context->state);
#endif
      break;	/* 顼 */
    }
#if DEBUG_SAX_HANDLER
  fprintf(stderr, "Invalid document: </%s> appeared in state=%d\n",
	  name, sax_context->state);
#endif
  sax_context->state = SAX_ERROR;
}


#define STRING_BUFFER_LENGTH	1024
static void
charactersHandler(void *context, const xmlChar *ch, int len)
{
  SAXContext *sax_context = (SAXContext *)context;

  if (sax_context->state == SAX_PREFERENCE_ATTRIBUTE_STRING
      || sax_context->state == SAX_CATEGORY_ATTRIBUTE_STRING
      || sax_context->state == SAX_CATEGORY_BOARD_ATTRIBUTE_STRING
      || sax_context->state == SAX_THREAD_ATTRIBUTE_STRING)
    {
      static char buffer[STRING_BUFFER_LENGTH] = "";
      int value_len;
      if (sax_context->current_attr_val == NULL)
	buffer[0] = '\0';

      value_len = strlen(buffer);
      if (value_len + len >= STRING_BUFFER_LENGTH)
	len = STRING_BUFFER_LENGTH - value_len - 1;

      strncat(buffer, ch, len);
      sax_context->current_attr_val = buffer;
    }
}


static void
nopHandler(void *context)
{
}


static xmlEntityPtr
getEntityHandler(void *context, const xmlChar *name)
{
  return NULL;
}


static void
input_board_attributes(BulletinBoard *board, GHashTable *hash_table,
		       ReadBoardElementFunc *optional, gpointer user_data)
{
  INPUT_ATTRIBUTE_STRING(board, hash_table, title_picture);
  INPUT_ATTRIBUTE_STRING(board, hash_table, title_color);
  INPUT_ATTRIBUTE_STRING(board, hash_table, title_link);
  INPUT_ATTRIBUTE_STRING(board, hash_table, bg_color);
  INPUT_ATTRIBUTE_STRING(board, hash_table, bg_picture);
  INPUT_ATTRIBUTE_STRING(board, hash_table, noname_name);
  INPUT_ATTRIBUTE_STRING(board, hash_table, makethread_color);
  INPUT_ATTRIBUTE_STRING(board, hash_table, menu_color);
  INPUT_ATTRIBUTE_STRING(board, hash_table, thread_color);
  INPUT_ATTRIBUTE_STRING(board, hash_table, text_color);
  INPUT_ATTRIBUTE_STRING(board, hash_table, name_color);
  INPUT_ATTRIBUTE_STRING(board, hash_table, link_color);
  INPUT_ATTRIBUTE_STRING(board, hash_table, alink_color);
  INPUT_ATTRIBUTE_STRING(board, hash_table, vlink_color);
  INPUT_ATTRIBUTE_STRING(board, hash_table, subject_color);

  if (optional != NULL)
    (*optional)(board, hash_table, user_data);
  else
    fprintf(stderr, "No optional board attributes?\n");
}


BBSTable *
ochusha_read_boardlist_xml(OchushaConfig *conf,
			   ReadCategoryElementFunc *cat_optional,
			   ReadBoardElementFunc *board_optional,
			   gpointer user_data)
{
  SAXContext context =
    {
      SAX_INITIAL,
      NULL, NULL,
      ochusha_bbs_table_new(), NULL,
      NULL,
      NULL,
      NULL, cat_optional, NULL, board_optional,
      NULL, NULL,
      user_data
    };
  xmlSAXHandler sax_handler;
  BBSTable *bbs_table;
  char *pathname = ochusha_find_file(conf, OCHUSHA_BOARDLIST_XML);
  if (pathname == NULL)
    return NULL;

  xmlInitParser();

  sax_handler.initialized = 0;
  initxmlDefaultSAXHandler(&sax_handler, TRUE);

  sax_handler.getEntity = getEntityHandler;
  sax_handler.startDocument = nopHandler;
  sax_handler.endDocument = nopHandler;
  sax_handler.startElement = startElementHandler;
  sax_handler.endElement = endElementHandler;
  sax_handler.characters = charactersHandler;

  xmlSAXUserParseFile(&sax_handler, &context, pathname);

  xmlCleanupParser();

  bbs_table = context.bbs_table;
  context.bbs_table = NULL;

  cleanup_sax_context(&context);

  if (context.state == SAX_ACCEPTED)
    {
      free(pathname);
      return bbs_table;
    }

  fprintf(stderr, "%s is unacceptable as ochusha's boardlist.\n", pathname);

  free(pathname);
  ochusha_bbs_table_free(bbs_table);

  return NULL;
}


/*
 * config.xmlϢ
 */
#define OUTPUT_CONFIG_ATTRIBUTE_INT(file, conf, attribute)	\
  fprintf(file,							\
	  "    <attribute name=\"" #attribute	"\">\n"		\
	  "      <int val=\"%d\"/>\n"				\
	  "    </attribute>\n", (conf)->attribute)

#define OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, conf, attribute)	\
  fprintf(file,							\
	  "    <attribute name=\"" #attribute	"\">\n"		\
	  "      <boolean val=\"%s\"/>\n"			\
	  "    </attribute>\n", (conf)->attribute ? "true" : "false" )

#define OUTPUT_CONFIG_ATTRIBUTE_STRING(file, conf, attribute)	\
  fprintf(file,							\
	  "    <attribute name=\"" #attribute	"\">\n"		\
	  "      <string>%s</string>\n"				\
	  "    </attribute>\n", (conf)->attribute)


gboolean
ochusha_write_config_xml(OchushaConfig *conf,
			 WritePreferenceElementFunc *optional,
			 gpointer user_data)
{
  FILE *config_xml;
  int len;
  char pathname[PATH_MAX];

  if (conf->home == NULL)
    {
      fprintf(stderr, "No home!\n");
      return FALSE;
    }

  len = snprintf(pathname, PATH_MAX, "%s/%s", conf->home, OCHUSHA_CONFIG_XML);

  if (len >= PATH_MAX)
    {
      fprintf(stderr, "Too long path name.\n");
      return FALSE;
    }

  config_xml = fopen(pathname, "w");
  if (config_xml == NULL)
    {
      fprintf(stderr, "Couldn't open \"%s\" to write.\n", pathname);
      return FALSE;
    }

  fprintf(config_xml, "<?xml version=\"1.0\"?>\n");
  fprintf(config_xml, "<ochusha>\n");
  fprintf(config_xml, "  <preference>\n");

  OUTPUT_CONFIG_ATTRIBUTE_STRING(config_xml, conf, bbsmenu_url);
  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(config_xml, conf, offline);

  if (optional != NULL)
    (*optional)(config_xml, user_data);
  else
    fprintf(stderr, "No optional preference?\n");

  fprintf(config_xml, "  </preference>\n");
  fprintf(config_xml, "</ochusha>\n");
  return fclose(config_xml) == 0;
}


void
ochusha_read_config_xml(OchushaConfig *conf,
			ReadPreferenceElementFunc *optional,
			gpointer user_data)
{
  SAXContext context =
    {
      SAX_INITIAL,
      NULL, NULL,
      NULL, NULL,
      NULL,
      NULL,
      NULL, NULL, NULL, NULL,
      NULL, NULL,
      user_data,
    };
  xmlSAXHandler sax_handler;
  char *pathname = ochusha_find_file(conf, OCHUSHA_CONFIG_XML);
  if (pathname == NULL)
    {
      conf->bbsmenu_url = strdup(OCHUSHA_DEFAULT_BBSMENU_URL);
      conf->offline = FALSE;
      return;
    }

  xmlInitParser();

  sax_handler.initialized = 0;
  initxmlDefaultSAXHandler(&sax_handler, TRUE);

  sax_handler.getEntity = getEntityHandler;
  sax_handler.startDocument = nopHandler;
  sax_handler.endDocument = nopHandler;
  sax_handler.startElement = startElementHandler;
  sax_handler.endElement = endElementHandler;
  sax_handler.characters = charactersHandler;

  xmlSAXUserParseFile(&sax_handler, &context, pathname);

  xmlCleanupParser();

  if (context.state == SAX_ACCEPTED)
    {
      GHashTable *pref = context.pref_attributes;
      context.pref_attributes = NULL;

      INPUT_ATTRIBUTE_STRING(conf, pref, bbsmenu_url);
      INPUT_ATTRIBUTE_BOOLEAN(conf, pref, offline);

      if (optional != NULL)
	(*optional)(pref, user_data);
      else
	fprintf(stderr, "No optional preferences?\n");
    }
  else
    fprintf(stderr, "%s is unacceptable as an ochusha's config.xml.\n",
	    pathname);

  free(pathname);

  cleanup_sax_context(&context);

  return;
}


/*
 * threadlist.xmlϢ
 */

#define OUTPUT_THREAD_ATTRIBUTE_INT(file, thread, attribute)		\
  fprintf(file,								\
	  "      <attribute name=\"" #attribute	"\">\n"			\
	  "        <int val=\"%d\"/>\n"					\
	  "      </attribute>\n", (thread)->attribute)

#define OUTPUT_THREAD_ATTRIBUTE_BOOLEAN(file, thread, attribute)	\
  fprintf(file,								\
	  "      <attribute name=\"" #attribute	"\">\n"			\
	  "        <boolean val=\"%s\"/>\n"				\
	  "      </attribute>\n", (thread)->attribute ? "true" : "false")

#define OUTPUT_THREAD_ATTRIBUTE_STRING(file, thread, attribute)		\
  fprintf(file,								\
	  "      <attribute name=\"" #attribute	"\">\n"			\
	  "        <string>%s</string>\n"				\
	  "      </attribute>\n", (thread)->attribute)


typedef struct _WriteBBSThreadArgs
{
  FILE *threadlist_xml;
  WriteThreadElementFunc *optional;
  gpointer user_data;
} WriteBBSThreadArgs;


static void
write_bbs_thread(gpointer data, gpointer user_data)
{
  BBSThread *thread = (BBSThread *)data;
  WriteBBSThreadArgs *args = (WriteBBSThreadArgs *)user_data;
  FILE *file = args->threadlist_xml;
  char *title_orig;

#if 0
  if (thread->number_of_responses_read == 0)
    return;	/* 줿ȤΤʤåɤ̵ */
#endif

  title_orig = thread->title;
  thread->title = escape_for_sgml(title_orig);

  if (thread->title == NULL)
    {
      thread->title = title_orig;
      return;	/* Out of memory */
    }

  fprintf(file, "    <thread>\n");

  OUTPUT_THREAD_ATTRIBUTE_STRING(file, thread, title);
  OUTPUT_THREAD_ATTRIBUTE_STRING(file, thread, dat_filename);
  OUTPUT_THREAD_ATTRIBUTE_INT(file, thread, number_of_responses_read);
  OUTPUT_THREAD_ATTRIBUTE_INT(file, thread, flags);

  free(thread->title);
  thread->title = title_orig;

  if (args->optional != NULL)
    (*args->optional)(file, thread, args->user_data);
  else
    fprintf(stderr, "No optional fields?\n");

  fprintf(file, "    </thread>\n");
}


gboolean
ochusha_write_threadlist_xml(OchushaConfig *conf, BulletinBoard *board,
			     WriteThreadElementFunc *optional,
			     gpointer user_data)
{
  FILE *threadlist_xml;
  int len;
  char pathname[PATH_MAX];
  WriteBBSThreadArgs args = { NULL, optional, user_data };

  if (board->thread_list == NULL)
    return FALSE;	/* ƤʤĤ̵ */

  len = snprintf(pathname, PATH_MAX, "%s/cache/%s/%s%s", conf->home,
		 board->server, board->base_path, OCHUSHA_THREADLIST_XML);
  if (len >= PATH_MAX)
    {
      fprintf(stderr, "Too long pathname: %s\n", pathname);
      return FALSE;
    }

  threadlist_xml = fopen(pathname, "w");

  if (threadlist_xml == NULL)
    {
      fprintf(stderr, "Couldn't open \"%s\" to write.\n", pathname);
      return FALSE;
    }

  args.threadlist_xml = threadlist_xml;

  fprintf(threadlist_xml, "<?xml version=\"1.0\"?>\n");
  fprintf(threadlist_xml, "<ochusha>\n");
  fprintf(threadlist_xml, "  <threadlist>\n");

  g_slist_foreach(board->thread_list, write_bbs_thread, &args);

  fprintf(threadlist_xml, "  </threadlist>\n");
  fprintf(threadlist_xml, "</ochusha>\n");

  return fclose(threadlist_xml) == 0;
}


void
ochusha_read_threadlist_xml(OchushaConfig *conf, BulletinBoard *board,
			    ReadThreadElementFunc *optional,
			    gpointer user_data)
{
  SAXContext context =
    {
      SAX_INITIAL,
      NULL, NULL,
      NULL, NULL,
      board,
      NULL,
      NULL, NULL, NULL, NULL,
      NULL, optional,
      user_data
    };
  xmlSAXHandler sax_handler;
  int len;
  char pathname[PATH_MAX];

#if 0
  if (board->thread_list == NULL)
    return;	/* ƤʤĤ̵ */
#endif

  len = snprintf(pathname, PATH_MAX, "%s/cache/%s/%s%s", conf->home,
		 board->server, board->base_path, OCHUSHA_THREADLIST_XML);
  if (len >= PATH_MAX)
    return;	/* ɤ᤽ˤʤ̵ۤ */

  if (open(pathname, O_RDONLY) == -1)
    return;	/* ºݤɤʤ̵ۤ */

  xmlInitParser();

  sax_handler.initialized = 0;
  initxmlDefaultSAXHandler(&sax_handler, TRUE);

  sax_handler.getEntity = getEntityHandler;
  sax_handler.startDocument = nopHandler;
  sax_handler.endDocument = nopHandler;
  sax_handler.startElement = startElementHandler;
  sax_handler.endElement = endElementHandler;
  sax_handler.characters = charactersHandler;

  xmlSAXUserParseFile(&sax_handler, &context, pathname);

  xmlCleanupParser();

  context.board = NULL;

#if DEBUG_SAX_HANDLER_NEW
  if (context.state == SAX_ACCEPTED)
    fprintf(stderr, "threadlist.xml accepted.\n");
  else
    fprintf(stderr, "Couldn't parse threadlist.xml\n");
  fflush(stderr);
#endif

  cleanup_sax_context(&context);
}
