/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: bbs_thread_ui.c,v 1.4 2003/05/11 18:17:54 fuyu Exp $
 */

#include "config.h"

#include "ochusha.h"
#include "ochusha_ui.h"

#include "ochusha_thread_2ch.h"

#include "bbs_thread_ui.h"
#include "bbs_thread_view.h"
#include "boardlist_ui.h"
#include "icon_label.h"

#include "htmlutils.h"
#include "worker.h"

#include <pthread.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


#define DEBUG_POPUP	0

static void link_popup_menu_callback(gpointer data, guint action,
				     GtkWidget *widget);
static void link_popup_menu_destructed_cb(gpointer data);
static gboolean link_mouse_over_cb(BBSThreadView *view, GdkEventMotion *event,
				   OchushaBBSThread *thread, const gchar *link,
				   OchushaApplication *application);
static gboolean link_mouse_out_cb(BBSThreadView *view, GdkEventMotion *event,
				  OchushaBBSThread *thread, const gchar *link,
				  OchushaApplication *application);
static gboolean link_mouse_press_cb(BBSThreadView *view, GdkEventButton *event,
				    OchushaBBSThread *thread,
				    const gchar *link,
				    OchushaApplication *application);
static gboolean link_mouse_release_cb(BBSThreadView *view,
				      GdkEventButton *event,
				      OchushaBBSThread *thread,
				      const gchar *link,
				      OchushaApplication *application);

static void show_popup(OchushaApplication *application,
		       OchushaBBSThread *thread,
		       const gchar *link, gint x_pos, gint y_pos);
static gboolean popup_delay_timeout(gpointer data);
static void show_popup_real(void);
static void hide_popup(void);

static void render_bbs_thread(WorkerThread *employee, gpointer args);
static gboolean start_thread_rendering(OchushaBBSThread *thread,
				       const gchar *title,
				       gpointer user_data);
static gboolean render_response(OchushaBBSThread *thread, gint number,
				const OchushaBBSResponse *response,
				gpointer user_data);
static gboolean end_thread_rendering(OchushaBBSThread *thread,
				     gboolean finished,
				     gpointer user_data);
static void start_element_cb(void *context, const char *name,
			     const char *const attrs[]);
static void end_element_cb(void *context, const char *name);
static void characters_cb(void *context, const char *ch, int len);

static pthread_mutex_t bbs_thread_ui_lock;

/* ơС˥ᥤȤʤɤɽ뤿context ID */
static guint link_field_id;

/* Ǹ˥ơСɽåФmessage ID */
static guint message_id;

/* PopupϢ */
static GtkWidget *popup_response_window;
static GtkContainer *popup_response_frame;


#define BBS_THREAD_UI_LOCK				\
  if (pthread_mutex_lock(&bbs_thread_ui_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't lock a mutex.\n");	\
      abort();						\
    }

#define BBS_THREAD_UI_UNLOCK				\
  if (pthread_mutex_unlock(&bbs_thread_ui_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't unlock a mutex.\n");	\
      abort();						\
    }


typedef struct _RendererContext
{
  ElementHandler handler;
  BBSThreadView *view;
  OchushaBBSThread *thread;
  OchushaApplication *application;
  gint last_read_number;

  gchar *link;
  gchar *link_text_buffer;
  guint link_text_buffer_size;
  guint link_text_length;

  GHashTable *tag_table;

  GtkTextTag *thread_title_tag;
  GtkTextTag *response_name_tag;
  GtkTextTag *response_paragraph_tag;
  GtkTextTag *bold_tag;
} RendererContext;


#define DEFAULT_BUFFER_SIZE	4096

static GHashTable *tag_table = NULL;
gchar popup_renderer_default_text_buffer[DEFAULT_BUFFER_SIZE];
RendererContext popup_renderer_context =
{
  {
    0,
    start_element_cb,
    end_element_cb,
    NULL,
    characters_cb
  },
  NULL,
  NULL,
  NULL,
  -1,	/* ְ㤨ƥޡդʤ褦 */
  
  NULL,
  popup_renderer_default_text_buffer,
  DEFAULT_BUFFER_SIZE,
  0,

  NULL,
  NULL,
  NULL,
  NULL,
};


static int stack_depth = 0;
static gchar *popup_url = NULL;
static guint popup_delay_id = 0;
static gint popup_x_pos = 0;
static gint popup_y_pos = 0;


enum {
  LINK_POPUP_MENU_OPEN_LINK = 1,
  LINK_POPUP_MENU_OPEN_LINK_IN_TAB,
  LINK_POPUP_MENU_OPEN_LINK_WITH_WEB_BROWSER,
  LINK_POPUP_MENU_SAVE_LINK_AS,
  LINK_POPUP_MENU_COPY_LINK_LOCATION,
};


static GtkItemFactory *link_popup_menu_item_factory = NULL;


void
initialize_thread_ui(OchushaApplication *application)
{
  GtkWidget *frame;
  GtkWidget *dummy_widget;
  BBSThreadView *dummy_view;

  /* ݥåץåץ˥塼ν */
  GtkItemFactoryEntry menu_items[] =
    {
      {
	_("/_Open Link"),				/* menu path */
	NULL,						/* accelerator */
	link_popup_menu_callback,			/* callback_func */
	LINK_POPUP_MENU_OPEN_LINK,			/* act */
	NULL						/* type */
      },
      {
	_("/Open Link in New _Tab"),			/* menu path */
	NULL,						/* accelerator */
	link_popup_menu_callback,			/* callback_func */
	LINK_POPUP_MENU_OPEN_LINK_IN_TAB,		/* act */
	NULL						/* type */
      },
      {
	_("/Open Link with _Web Browser"),		/* menu path */
	NULL,						/* accelerator */
	link_popup_menu_callback,			/* callback_func */
	LINK_POPUP_MENU_OPEN_LINK_WITH_WEB_BROWSER,	/* act */
	NULL						/* type */
      },
      {
	_("/-------"),
	NULL,
	NULL,
	0,
	"<Separator>"
      },
      {
	_("/_Save Link As..."),				/* menu path */
	NULL,						/* accelerator */
	link_popup_menu_callback,			/* callback_func */
	LINK_POPUP_MENU_SAVE_LINK_AS,			/* act */
	NULL						/* type */
      },
      {
	_("/_Copy Link Location"),			/* menu path */
	NULL,						/* accelerator */
	link_popup_menu_callback,			/* callback_func */
	LINK_POPUP_MENU_COPY_LINK_LOCATION,		/* act */
	NULL						/* type */
      },
    };
  gint nmenu_items = sizeof(menu_items) / sizeof(menu_items[0]);
  link_popup_menu_item_factory = gtk_item_factory_new(GTK_TYPE_MENU,
						      "<ThreadLinkPopup>",
						      NULL);
  gtk_item_factory_create_items(link_popup_menu_item_factory,
				nmenu_items, menu_items, application);

  /* ̤εǽ˥塼Ĥ֤ */
  dummy_widget = gtk_item_factory_get_widget_by_action(
					link_popup_menu_item_factory,
					LINK_POPUP_MENU_SAVE_LINK_AS);
  if (dummy_widget != NULL)
    gtk_widget_set_sensitive(dummy_widget, FALSE);

#if DEBUG_GUI_MOST
  fprintf(stderr, "OchushaApplication=0x%x\n", (int)application);
#endif
  g_return_if_fail(tag_table == NULL);

  if (pthread_mutex_init(&bbs_thread_ui_lock, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a mutex.\n");
      abort();
    }

  link_field_id = gtk_statusbar_get_context_id(application->statusbar,
					       "response-link-field");
  popup_response_window = gtk_window_new(GTK_WINDOW_POPUP);
#if 0
  gtk_window_set_decorated(GTK_WINDOW(popup_response_window), TRUE);
#endif
  gtk_window_set_resizable(GTK_WINDOW(popup_response_window), FALSE);

  frame = gtk_frame_new(NULL);
  gtk_widget_show(frame);
  popup_response_frame = GTK_CONTAINER(frame);
  gtk_container_add(GTK_CONTAINER(popup_response_window), frame);
  
  dummy_widget = bbs_thread_view_new(NULL);
  dummy_view = BBS_THREAD_VIEW(dummy_widget);
#if 0
  bbs_thread_view_hide_scrollbar(dummy_view);
#endif
  popup_renderer_context.handler.converter = iconv_open("UTF-8//IGNORE",
							"CP932");
#if 0
  popup_renderer_context.view = popup_response_view;
#endif
  popup_renderer_context.application = application;

  popup_renderer_context.thread_title_tag
    = bbs_thread_view_get_tag_by_name(dummy_view, "thread_title");
  popup_renderer_context.response_name_tag
    = bbs_thread_view_get_tag_by_name(dummy_view, "response_name");
  popup_renderer_context.response_paragraph_tag
    = bbs_thread_view_get_tag_by_name(dummy_view, "response_paragraph");
  popup_renderer_context.bold_tag
    = bbs_thread_view_get_tag_by_name(dummy_view, "bold");

  /* Τ */
  gtk_object_sink(GTK_OBJECT(dummy_view));

  if (popup_renderer_context.handler.converter == (iconv_t)-1)
    {
      fprintf(stderr,
	      "Out of memory: I couldn't imagine this can run any more.\n");
      exit(0);
    }

  {
#if 0
    GtkTextTag *tag;
    GtkTextTagTable *default_tag_table = bbs_thread_view_class_get_default_tag_table(BBS_THREAD_VIEW_GET_CLASS(job_args->view));
#endif
    tag_table = g_hash_table_new(g_str_hash, g_str_equal);

    /* TODO: äȡ */
    g_hash_table_insert(tag_table, "b", popup_renderer_context.bold_tag);
  }
  popup_renderer_context.tag_table = tag_table;
}


BBSThreadGUIInfo *
ensure_bbs_thread_info(OchushaBBSThread *thread)
{
  BBSThreadGUIInfo *info;
  BBS_THREAD_UI_LOCK
    {
      info = (BBSThreadGUIInfo *)thread->user_data;
      if (info == NULL)
	{
	  info = (BBSThreadGUIInfo *)calloc(1, sizeof(BBSThreadGUIInfo));
	  thread->user_data = info;
	}
    }
  BBS_THREAD_UI_UNLOCK;
  return info;
}


typedef struct _BBSThreadJobArgs
{
  OchushaApplication *application;
  OchushaBBSThread *thread;
  BBSThreadView *view;
  IconLabel *tab_label;
  OchushaAsyncBuffer *buffer;
  gint start;
  gint number_of_responses;
  gboolean refresh;
} BBSThreadJobArgs;


GtkWidget *
open_bbs_thread(OchushaApplication *application, OchushaBBSThread *thread,
		IconLabel *tab_label)
{
  GtkWidget *thread_view;
  BBSThreadGUIInfo *info = ensure_bbs_thread_info(thread);

  if (info == NULL)
    {
      fprintf(stderr, "Out of memory.\n");
      return NULL;
    }

  BBS_THREAD_UI_LOCK
    {
      WorkerJob *job;
      BBSThreadJobArgs *job_args;
      OchushaAsyncBuffer *buffer;

      thread_view = bbs_thread_view_new(thread);
      gtk_widget_show(thread_view);

      job = (WorkerJob *)calloc(1, sizeof(WorkerJob));
      job_args = (BBSThreadJobArgs *)calloc(1, sizeof(BBSThreadJobArgs));
      if (job == NULL || job_args == NULL)
	{
	  /* Out of memoryǥåɤɽ򤢤 */
	  fprintf(stderr, "Couldn't start show thread worker.\n");
	  gtk_object_sink(GTK_OBJECT(thread_view));

	  if (job != NULL)
	    free(job);
	  if (job_args != NULL)
	    free(job_args);

	  BBS_THREAD_UI_UNLOCK;
	  return NULL;
	}

      buffer = ochusha_bbs_thread_get_responses_source(thread,
						&application->conf,
						info->last_modified,
						OCHUSHA_CACHE_TRY_UPDATE,
						&icon_indicator_funcs,
						tab_label);

      if (buffer == NULL)
	{
	  fprintf(stderr, "Couldn't access network!\n");
	  free(job_args);
	  free(job);
	  gtk_object_sink(GTK_OBJECT(thread_view));

	  BBS_THREAD_UI_UNLOCK;
	  return NULL;
	}

      info->dat_buffer = buffer;
      info->rendering_done = FALSE;

      job_args->application = application;
      job_args->thread = thread;
      job_args->view = BBS_THREAD_VIEW(thread_view);
      job_args->buffer = buffer;
      job_args->start = 0;
      job_args->number_of_responses = -1;
      job_args->refresh = FALSE;
      job_args->tab_label = tab_label;

      job->canceled = FALSE;
      job->job = render_bbs_thread;
      job->args = job_args;

      g_signal_connect(G_OBJECT(thread_view), "link_mouse_over",
		       G_CALLBACK(link_mouse_over_cb), application);
      g_signal_connect(G_OBJECT(thread_view), "link_mouse_out",
		       G_CALLBACK(link_mouse_out_cb), application);
      g_signal_connect(G_OBJECT(thread_view), "link_mouse_press",
		       G_CALLBACK(link_mouse_press_cb), application);
      g_signal_connect(G_OBJECT(thread_view), "link_mouse_release",
		       G_CALLBACK(link_mouse_release_cb), application);

      /* öref_count䤹 */
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "open_bbs_thread: ref AsyncBuffer(0x%x)\n", (int)buffer);
      fprintf(stderr, "* before ref: buffer->ref_count=%d\n",
	      G_OBJECT(buffer)->ref_count);
#endif
      g_object_ref(G_OBJECT(buffer));
      g_object_ref(G_OBJECT(tab_label));
      g_object_ref(G_OBJECT(thread_view));
      commit_job(job);
    }
  BBS_THREAD_UI_UNLOCK;

  return thread_view;
}


static void
link_popup_menu_callback(gpointer data, guint action, GtkWidget *widget)
{
  OchushaApplication *application = data;
  const char *url = gtk_item_factory_popup_data(link_popup_menu_item_factory);
#if DEBUG_GUI_MOST
  fprintf(stderr, "link_popup_menu_callback: application=0x%x, url=\"%s\"(@0x%x), action=%d\n",
	  (int)application, url, (int)url, action);
#endif
  switch (action)
    {
    case LINK_POPUP_MENU_OPEN_LINK:
      ochusha_open_url(application, url, FALSE, FALSE);
      break;

    case LINK_POPUP_MENU_OPEN_LINK_IN_TAB:
      ochusha_open_url(application, url, TRUE, FALSE);
      break;
      
    case LINK_POPUP_MENU_OPEN_LINK_WITH_WEB_BROWSER:
      ochusha_open_url(application, url, FALSE, TRUE);
      break;

    case LINK_POPUP_MENU_COPY_LINK_LOCATION:
      ochusha_clipboard_set_text(application, url);
      break;

    case LINK_POPUP_MENU_SAVE_LINK_AS:
      fprintf(stderr, "Not implemented yet.\n");
      break;
    }
}


static void
link_popup_menu_destructed_cb(gpointer data)
{
#if DEBUG_GUI_MOST
  fprintf(stderr, "link_popup_menu_destructed_cb: url=\"%s\"\n", (char *)data);
#endif
  free(data);
}


static gboolean
link_mouse_over_cb(BBSThreadView *view, GdkEventMotion *event,
		   OchushaBBSThread *thread, const gchar *link,
		   OchushaApplication *application)
{
  GtkStatusbar *statusbar = application->statusbar;
#if DEBUG_GUI_MOST
  fprintf(stderr, "link_mouse_over_cb: link=\"%s\"\n", link);
#endif
  if (message_id != 0)
    {
      gtk_statusbar_remove(statusbar, link_field_id, message_id);
      stack_depth--;
    }

  if (application->enable_popup_response)
    show_popup(application, thread, link, event->x_root, event->y_root);

  stack_depth++;
  if (stack_depth == 1)
    message_id = gtk_statusbar_push(statusbar, link_field_id, link);

  return FALSE;
}


static gboolean
link_mouse_out_cb(BBSThreadView *view, GdkEventMotion *event,
		  OchushaBBSThread *thread, const gchar *link,
		  OchushaApplication *application)
{
  GtkStatusbar *statusbar = application->statusbar;
#if DEBUG_GUI_MOST
  fprintf(stderr, "link_mouse_out_cb: link=\"%s\"\n", link);
#endif

  if (message_id != 0)
    {
      gtk_statusbar_remove(statusbar, link_field_id, message_id);
      message_id = 0;
    }
  stack_depth--;

  hide_popup();

  return FALSE;
}


static gboolean
link_mouse_press_cb(BBSThreadView *view, GdkEventButton *event,
		    OchushaBBSThread *thread, const gchar *link,
		    OchushaApplication *application)
{
#if DEBUG_GUI_MOST
  fprintf(stderr, "link_mouse_press_cb: x=%f, y=%f, link=\"%s\"(@0x%x)\n",
	  event->x_root, event->y_root, link, (int)link);
#endif

  if (link == NULL)
    return FALSE;

  if (event->button == 3)
    gtk_item_factory_popup_with_data(link_popup_menu_item_factory,
				     g_strdup(link),
				     link_popup_menu_destructed_cb,
				     event->x_root, event->y_root,
				     event->button,
				     gtk_get_current_event_time());
  else
    ochusha_open_url(application, link, event->button == 2, FALSE);

  return TRUE;
}


static gboolean
link_mouse_release_cb(BBSThreadView *view, GdkEventButton *event,
		      OchushaBBSThread *thread, const gchar *link,
		      OchushaApplication *application)
{
#if DEBUG_GUI_MOST
  fprintf(stderr, "link_mouse_release_cb: link=\"%s\"\n", link);
#endif
  return FALSE;
}


#define DEFAULT_SUBJECT_COLOR	"red"
#define DEFAULT_NAME_COLOR	"green"
#define DEFAULT_LINK_COLOR	"blue"


static void
render_bbs_thread(WorkerThread *employee, gpointer args)
{
  BBSThreadJobArgs *job_args = (BBSThreadJobArgs *)args;
  OchushaApplication *application = job_args->application;
  OchushaBBSThread *thread = job_args->thread;
  BBSThreadGUIInfo *info = (BBSThreadGUIInfo *)thread->user_data;
  OchushaAsyncBuffer *buffer = job_args->buffer;
  gchar default_text_buffer[DEFAULT_BUFFER_SIZE];
  RendererContext context =
    {
      {
	iconv_open("UTF-8//IGNORE", "CP932"),
	start_element_cb,
	end_element_cb,
	NULL,
	characters_cb
      },
      job_args->view,
      thread,
      application,
      thread->number_of_responses_read,

      NULL,
      default_text_buffer,
      DEFAULT_BUFFER_SIZE,
      0,

      tag_table,

      bbs_thread_view_get_tag_by_name(job_args->view, "thread_title"),
      bbs_thread_view_get_tag_by_name(job_args->view, "response_name"),
      bbs_thread_view_get_tag_by_name(job_args->view, "response_paragraph"),
      bbs_thread_view_get_tag_by_name(job_args->view, "bold"),
  };

  if (info->offsets == NULL)
    info->offsets = (gint *)calloc(MAX_RESPONSE, sizeof(gint));
  else 
    memset(info->offsets, 0, MAX_RESPONSE * sizeof(gint));

  if (context.handler.converter == (iconv_t)-1)
    {
      fprintf(stderr, "Out of memory\n");
      BBS_THREAD_UI_LOCK
	{
	  info->rendering_done = TRUE;
	}
      BBS_THREAD_UI_UNLOCK;
      return;
    }

#if DEBUG_GUI_MOST
  {
    gchar *title = convert_string(utf8_to_native, thread->title, -1);
    fprintf(stderr, "render_bbs_thread: %s\n", title);
    free(title);
  }
#endif

  if (job_args->start > 0 && !job_args->refresh)
    {
#if DEBUG_GUI
      fprintf(stderr, "XXX: Should this show thread title?");
#endif
    }

  if (job_args->refresh)
    {
      /* ɥ쥹˰*/
      if (info->last_read != NULL)
	bbs_thread_view_delete_mark(job_args->view, info->last_read);
      info->last_read = bbs_thread_view_create_mark(job_args->view);
      info->next_mark = info->last_read;
#if DEBUG_GUI
      fprintf(stderr, "create mark at the tail response.\n");
#endif
    }

  ochusha_bbs_thread_parse_responses(thread, buffer,
				     job_args->start,
				     job_args->number_of_responses,
				     start_thread_rendering,
				     render_response,
				     end_thread_rendering,
				     &context);

  update_threadlist_entry(job_args->application, thread);

#if DEBUG_WIDGET_MOST
  fprintf(stderr, "render_bbs_thread<done>: view=0x%x, view->ref_count=%d\n", (int)job_args->view, G_OBJECT(job_args->view)->ref_count);
#endif

  if (job_args->tab_label != NULL)
    g_object_unref(G_OBJECT(job_args->tab_label));
#if DEBUG_WIDGET
  fprintf(stderr, "render_bbs_thread: unref BBSThreadView(0x%x) ref_count=%d\n",
	  (int)job_args->view, G_OBJECT(job_args->view)->ref_count);
#endif
  g_object_unref(G_OBJECT(job_args->view));
  iconv_close(context.handler.converter);
#if DEBUG_ASYNC_BUFFER_MOST
  fprintf(stderr, "render_bbs_thread: unref AsyncBuffer(0x%x)\n", (int)buffer);
  fprintf(stderr, "* before unref: buffer->ref_count=%d\n",
	  G_OBJECT(buffer)->ref_count);
#endif
  g_object_unref(G_OBJECT(buffer));

  BBS_THREAD_UI_LOCK
    {
      OchushaNetworkStatus *status = (OchushaNetworkStatus *)buffer->user_data;
      if (status != NULL)
	{
	  if (status->last_modified != NULL)
	    {
	      if (info->last_modified != NULL)
		free(info->last_modified);
	      info->last_modified = g_strdup(status->last_modified);
	    }
	  else
	    if (status->state == OCHUSHA_ACCESS_FAILED)
	      {
		if (info->last_modified != NULL)
		  free(info->last_modified);
		info->last_modified = NULL;
	      }
	}

      info->rendering_done = TRUE;
    }
  BBS_THREAD_UI_UNLOCK;

  if (context.link_text_buffer != default_text_buffer)
    free(context.link_text_buffer);
  free(args);
}


static gboolean
start_thread_rendering(OchushaBBSThread *thread, const gchar *title,
		       gpointer user_data)
{
  RendererContext *context = (RendererContext *)user_data;
  BBSThreadView *view = context->view;

#if DEBUG_GUI_MOST
  fprintf(stderr, "start_rendering\n");
  fprintf(stderr, "thread_title_tag=0x%x\n", (int)context->thread_title_tag);
  fprintf(stderr, "response_name_tag=0x%x\n", (int)context->response_name_tag);
  fprintf(stderr, "response_paragraph_tag=0x%x\n", (int)context->response_paragraph_tag);
  fprintf(stderr, "bold_tag=0x%x\n", (int)context->bold_tag);
#endif

  gdk_threads_enter();

  if (context->last_read_number == 0)
    {
      /* ɤॹξ祿ȥߤˤΤǤ˥ޡդ롣*/
      BBSThreadGUIInfo *info = (BBSThreadGUIInfo *)context->thread->user_data;
      info->last_read = bbs_thread_view_create_mark(view);
      info->next_mark = info->last_read;
      bbs_thread_view_scroll_to_mark(view, info->last_read);
#if DEBUG_GUI_MOST
      fprintf(stderr, "scroll to mark<1>: last_read_number=0, last_read=0x%x\n",
	      (int)info->last_read);
#endif
    }

  bbs_thread_view_push_tag(view, context->thread_title_tag);

  parse_text(&context->handler, context, title, -1);
  bbs_thread_view_append_text(view, "\n", 1);

  bbs_thread_view_pop_tags(view, context->thread_title_tag);
  gdk_threads_leave();

  return TRUE;	/* go ahead! */
}


#define TEXT_BUFFER_LEN		4096
#define LARGE_COLON		"\357\274\232"

static gboolean
render_response(OchushaBBSThread *thread, gint number,
		const OchushaBBSResponse *response,
		gpointer user_data)
{
  RendererContext *context = (RendererContext *)user_data;
  BBSThreadView *view = context->view;
  BBSThreadGUIInfo *info = (BBSThreadGUIInfo *)thread->user_data;
  char buffer[13];
  const char *cur_pos;
#if DEBUG_GUI_MOST
  fprintf(stderr, "rendering response #%d, last_read_number=%d\n",
	  number, context->last_read_number);
#endif

  if (context->last_read_number >= 0 && info->offsets != NULL)
    {
      info->offsets[number] = bbs_thread_view_get_current_offset(view);
    }

  if (number == context->last_read_number)
    {
      /* ɥ쥹*/
      info->last_read = bbs_thread_view_create_mark(view);
      info->next_mark = info->last_read;
      gdk_threads_enter();
      bbs_thread_view_scroll_to_mark(view, info->last_read);
      gdk_threads_leave();
#if DEBUG_GUI_MOST
      fprintf(stderr, "scroll to mark<2>: number=%d, last_read=0x%x\n",
	      number, (int)info->last_read);
#endif
    }

  if (thread->number_of_responses_read < number)
    {
      thread->number_of_responses_read = number;
      update_threadlist_entry(context->application, thread);
    }

  snprintf(buffer, 13, "\n%d", number);

  gdk_threads_enter();
  bbs_thread_view_append_text(view, buffer, -1);
  bbs_thread_view_append_text(view, LARGE_COLON, 3);

  bbs_thread_view_push_tag(view, context->response_name_tag);
  if (response->mailto == NULL)
    {
      bbs_thread_view_push_tag(view, context->bold_tag);
      parse_text(&context->handler, context, response->name, -1);
      bbs_thread_view_pop_tags(view, context->bold_tag);
    }
  else
    {
      gchar *name = simple_string_canon(response->name, -1,
					context->handler.converter);
      gchar *mailto = convert_string(context->handler.converter,
				     response->mailto, -1);
      bbs_thread_view_append_text_as_link(view, name, -1, mailto);
      free(mailto);
      free(name);
    }
  bbs_thread_view_pop_tags(view, context->response_name_tag);

  bbs_thread_view_append_text(view, LARGE_COLON, 3);
  parse_text(&context->handler, context, response->date_id, -1);

  bbs_thread_view_append_text(view, "\n", 1);


  bbs_thread_view_push_tag(view, context->response_paragraph_tag);
  cur_pos = response->content;
  while (cur_pos != NULL)
    {
      cur_pos = parse_text(&context->handler, context, cur_pos, -1);
      if (cur_pos != NULL)
	{
	  if (*cur_pos == '&')
	    {
	      parse_text(&context->handler, context, "&amp;", 5);
	      cur_pos++;
	    }
	  else if (*cur_pos == '<')
	    {
	      parse_text(&context->handler, context, "&lt;", 4);
	      cur_pos++;
	    }
	  else
	    break;	/*  */
	}
    }

  bbs_thread_view_pop_tags(view, context->response_paragraph_tag);

  bbs_thread_view_append_text(view, "\n", 1);

  gdk_threads_leave();

  return TRUE;
}


static gboolean
end_thread_rendering(OchushaBBSThread *thread, gboolean finished,
		     gpointer user_data)
{
  RendererContext *context = (RendererContext *)user_data;
  BBSThreadView *view = context->view;
  BBSThreadGUIInfo *info = (BBSThreadGUIInfo *)context->thread->user_data;

  if (!finished)
    {
      /*
       * MEMO: å夬äƤ(ܡ󤵤줿쥹)
       *       å夵Ƥ̤ͤ꾯ʤäʤɡ
       *       DATեΤμľȤ롣
       */
#if DEBUG_GUI_MOST
      fprintf(stderr, "end_thread_rendering: clear current thread view\n");
#endif
      gdk_threads_enter();
      bbs_thread_view_clear(view);
      gdk_threads_leave();
    }
  else if (info->last_read == NULL)
    {
      /*
       * MEMO: ͭʤ
       *       äޤǤϿ쥹Ƭ˥ޡդƤΤǡ
       *       쥹ʤȥޡʤˤʤäƤޤΤǤ
       *       ХåեκǸ˥ޡĤƤ
       */
      info->last_read = bbs_thread_view_create_mark(view);
      info->next_mark = info->last_read;
      gdk_threads_enter();
      bbs_thread_view_scroll_to_mark(view, info->last_read);
      gdk_threads_leave();
#if DEBUG_GUI_MOST
      fprintf(stderr, "scroll to mark<3>: last_read=0x%x\n",
	      (int)info->last_read);
#endif
#if DEBUG_GUI
      fprintf(stderr, "end_thread_rendering: What's happen?\n");
#endif
    }

#if DEBUG_GUI_MOST
  fprintf(stderr, "end_rendering\n");
#endif

  return TRUE;
}


static void
process_link(RendererContext *context)
{
  if (context->link == NULL)
    return;

  bbs_thread_view_append_text_as_link(context->view,
				      context->link_text_buffer,
				      context->link_text_length,
				      context->link);
  free(context->link);
  context->link = NULL;
}


static void
start_element_cb(void *user_data, const char *name, const char *const attrs[])
{
  RendererContext *context = (RendererContext *)user_data;
  BBSThreadView *view = context->view;
  int i;

#if DEBUG_GUI_MOST
  fprintf(stderr, "<%s", name);
  i = 0;
  while (attrs[i * 2] != NULL)
    {
      fprintf(stderr, " %s=%s", attrs[i * 2], attrs[i * 2 + 1]);
      i++;
    }
  fprintf(stderr, ">\n");
#endif
  bbs_thread_view_push_tag(view,
			   g_hash_table_lookup(context->tag_table, name));

  if (strcasecmp(name, "br") == 0)
    {
      bbs_thread_view_pop_tags(view, context->response_paragraph_tag);
      bbs_thread_view_append_text(view, "\n", 1);
      bbs_thread_view_push_tag(view, context->response_paragraph_tag);
    }
  else if (strcasecmp(name, "a") == 0)
    {
      gchar *link = NULL;
      if (context->link != NULL)
	{
#if DEBUG_GUI
	  fprintf(stderr, "previous link hasn't been closed.\n");
#endif
	  process_link(context);
	}
      context->link_text_buffer[0] = '\0';
      context->link_text_length = 0;

      i = 0;
      while (attrs[i * 2] != NULL)
	if (strcasecmp(attrs[i * 2], "href") == 0)
	  {
	    link = convert_string(context->handler.converter,
					 attrs[i * 2 + 1], -1);
#if DEBUG_GUI_MOST
	    fprintf(stderr, "href=\"%s\"\n", link);
#endif
	    break;
	  }

      if (link == NULL)
	link = g_strdup(_("*empty link*"));

      context->link = link;
    }
}


static void
end_element_cb(void *user_data, const char *name)
{
  RendererContext *context = (RendererContext *)user_data;
  BBSThreadView *view = context->view;

#if DEBUG_GUI_MOST
  fprintf(stderr, "</%s>\n", name);
#endif
  bbs_thread_view_pop_tags(view,
			   g_hash_table_lookup(context->tag_table, name));

  if (strcasecmp(name, "a") == 0 && context->link != NULL)
    process_link(context);
}


static void
characters_cb(void *user_data, const char *ch, int len)
{
  gchar *buffer;
  int buffer_size;
  RendererContext *context = (RendererContext *)user_data;

  if (context->link == NULL)
    {
      bbs_thread_view_append_text(context->view, ch, len);
      return;
    }

  buffer = context->link_text_buffer;
  buffer_size = context->link_text_buffer_size;

  while (buffer_size <= (context->link_text_length + len))
    {
      fprintf(stderr, "buffer_size=%d, requirement=%d\n",
	      buffer_size, (context->link_text_length + len));
      buffer_size *= 2;
      if (context->link_text_buffer_size == DEFAULT_BUFFER_SIZE)
	buffer = (gchar *)malloc(buffer_size);
      else
	buffer = (gchar *)realloc(buffer, buffer_size);

      if (buffer != NULL
	  && context->link_text_buffer_size == DEFAULT_BUFFER_SIZE)
	memcpy(buffer, context->link_text_buffer, context->link_text_length);

      if (buffer != NULL)
	{
	  context->link_text_buffer = buffer;
	  context->link_text_buffer_size = buffer_size;
	}
      else
	{
	  buffer = context->link_text_buffer;
	  buffer_size = context->link_text_buffer_size;
#if DEBUG_GUI
	  fprintf(stderr, "Out of memory.\n");
#endif
	  break;
	}
    }

  if (buffer_size > context->link_text_length + len)
    {
      memcpy(buffer + context->link_text_length, ch, len);
      context->link_text_length += len;
    }
  else
    {
      int left = buffer_size - context->link_text_length - 1;
      memcpy(buffer + context->link_text_length, ch, left);
      context->link_text_length = buffer_size - 1;
    }

  buffer[context->link_text_length] = '\0';
}


static gboolean
prepare_buffer_for_popup(OchushaApplication *application,
			 OchushaBBSThread *thread,
			 const gchar *link,
			 unsigned int *response_from,
			 unsigned int *response_to,
			 OchushaAsyncBuffer **buffer)
{
  unsigned int from = 0;
  unsigned int to = 0;
  BBSThreadGUIInfo *info = (BBSThreadGUIInfo *)thread->user_data;

  if (application->maximum_number_of_popup_responses < 1)
    return FALSE;

  if (!ochusha_bbs_thread_check_url(thread, link, &from, &to))
    return FALSE;

#if DEBUG_GUI_MOST
  fprintf(stderr, "prepare_buffer_for_popup(): from=%d, to=%d\n", from, to);
#endif

  if ((from == 0 && to == 0) || from > thread->number_of_responses_read)
    return FALSE;

  if (to > thread->number_of_responses_read)
    to = thread->number_of_responses_read;

  if (to > 0 && from > to)
    {
      unsigned int tmp = to;
      to = from;
      from = tmp;
    }

  *response_from = from;
  *response_to = to;

  /*
   * ΤȤɽƤ륹ɬdat_buffer˥ХåեäƤ뤬
   * ⤷Ŭʥߥ󥰤ǲ٤⤷ʤΤǡΤ(
   * 夫Τɤ߹)ɤĤƤ
   */
  if (info->dat_buffer != NULL)
    {
      *buffer = info->dat_buffer;
      return TRUE;
    }

  *buffer = ochusha_bbs_thread_get_responses_source(thread, &application->conf,
						    info->last_modified,
						    OCHUSHA_CACHE_ONLY,
						    NULL, NULL);
  info->dat_buffer = *buffer;

  return (*buffer != NULL);
}


static gboolean need_popup_adjustment = FALSE;


static gboolean
popup_expose_event_cb(GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
  GtkRequisition requisition = { 0, 0 };
  gtk_widget_size_request(widget, &requisition);

#if DEBUG_GUI_MOST
  fprintf(stderr, "popup_expose_event_cb(widget=0x%x), w=%d, h=%d\n",
	  (int)widget, requisition.width, requisition.height);
#endif
  if (requisition.width > 0 && need_popup_adjustment)
    {
#if DEBUG_GUI_MOST
      fprintf(stderr, "Do adjust here.\n");
#endif
      BBS_THREAD_UI_LOCK
	{
	  need_popup_adjustment = FALSE;
	}
      BBS_THREAD_UI_UNLOCK;

      if (popup_renderer_context.application->enable_smart_popup_placement)
	{
	  GdkScreen *screen = gtk_widget_get_screen(popup_response_window);

	  popup_y_pos -= requisition.height;
	  popup_y_pos = popup_y_pos > 0 ? popup_y_pos : 0;

	  if (screen != NULL)
	    {
	      gint screen_width = gdk_screen_get_width(screen);
	      gint widget_width = requisition.width + THREAD_VIEW_EXTRA_WIDTH;
	      if (screen_width > 0
		  && (popup_x_pos + widget_width) > screen_width)
		{
		  popup_x_pos = screen_width - widget_width;
		  if (popup_x_pos < 0)
		    popup_x_pos = 0;
		}
	    }
	}

      gtk_widget_set_size_request(widget,
				  requisition.width
				  + THREAD_VIEW_EXTRA_WIDTH,
				  requisition.height);
      gtk_window_resize(GTK_WINDOW(popup_response_window),
			requisition.width + THREAD_VIEW_EXTRA_WIDTH,
			requisition.height);

      if (popup_renderer_context.application->enable_smart_popup_placement)
	gtk_window_move(GTK_WINDOW(popup_response_window),
			popup_x_pos, popup_y_pos);
    }

  return FALSE;
}


#if DEBUG_WIDGET_MOST
/*
 * XXX: Ȥʪˤʤ󥷥ʥġġ
 */
static gboolean
popup_map_event_cb(GtkWidget *widget, GdkEvent *event, gpointer data)
{
  GtkRequisition requisition = { 0, 0 };
  gtk_widget_size_request(widget, &requisition);
  fprintf(stderr, "popup_map_event_cb(widget=0x%x), w=%d, h=%d\n",
	  (int)widget, requisition.width, requisition.height);
  return FALSE;
}


static void
popup_realize_cb(GtkWidget *widget, gpointer data)
{
  GtkRequisition requisition = { 0, 0 };
  gtk_widget_size_request(widget, &requisition);
  fprintf(stderr, "popup_realize_cb(widget=0x%x), w=%d, h=%d\n",
	  (int)widget, requisition.width, requisition.height);
  return;
}
#endif


static void
render_popup_response(OchushaBBSThread *thread, OchushaAsyncBuffer *buffer,
		      gint response_from, gint num_responses)
{
  GtkWidget *view = bbs_thread_view_new(thread);
  bbs_thread_view_set_wrap_mode(BBS_THREAD_VIEW(view), GTK_WRAP_NONE);

  g_signal_connect_after(G_OBJECT(view), "expose_event",
			 G_CALLBACK(popup_expose_event_cb), NULL);
#if DEBUG_WIDGET_MOST
  g_signal_connect(G_OBJECT(view), "map_event",
		   G_CALLBACK(popup_map_event_cb), NULL);
  g_signal_connect_after(G_OBJECT(view), "realize",
			 G_CALLBACK(popup_realize_cb), NULL);
#endif

  gtk_widget_show_all(view);

  if (popup_renderer_context.view != NULL)
    gtk_container_remove(popup_response_frame,
			 GTK_WIDGET(popup_renderer_context.view));
  gtk_container_add(popup_response_frame, view);
  gtk_widget_show(GTK_WIDGET(popup_response_frame));

  popup_renderer_context.view = BBS_THREAD_VIEW(view);
  bbs_thread_view_hide_scrollbar(popup_renderer_context.view);
  popup_renderer_context.link_text_length = 0;
  popup_renderer_context.thread = thread;

#if DEBUG_GUI_MOST
  if (num_responses > 1)
    fprintf(stderr, "from=%d, num=%d\n", response_from, num_responses);
#endif

  gdk_threads_leave();
  ochusha_bbs_thread_parse_responses(thread, buffer,
				     response_from - 1, num_responses,
				     NULL, render_response, NULL,
				     &popup_renderer_context);
  gdk_threads_enter();
}


static void
show_popup(OchushaApplication *application, OchushaBBSThread *thread,
	   const gchar *link, gint x_pos, gint y_pos)
{
  OchushaAsyncBuffer *buffer;
  gint response_from = 0;
  gint response_to = 0;
  gint num_responses;

#if DEBUG_GUI_MOST
  fprintf(stderr, "show_popup: link=\"%s\", x_pos=%d, y_pos=%d\n",
	  link, x_pos, y_pos);
#endif
  if (!prepare_buffer_for_popup(application, thread, link,
				&response_from, &response_to, &buffer))
    {
#if DEBUG_POPUP
      fprintf(stderr, "show_popup(): from=%d to=%d\n",
	      response_from, response_to);
      fprintf(stderr, "Popup cannot be prepared...why?\n");
#endif
      return;
    }

  if (response_from == 0)
    response_from
      = response_to - application->maximum_number_of_popup_responses;
  if (response_to == 0)
    response_to = response_from;

  num_responses = response_to - response_from + 1;
  if (num_responses > application->maximum_number_of_popup_responses)
    num_responses = application->maximum_number_of_popup_responses;

#if DEBUG_POPUP
      fprintf(stderr, "show_popup(): from=%d num_responses=%d\n",
	      response_from, num_responses);
#endif
  render_popup_response(thread, buffer, response_from, num_responses);

  popup_x_pos = x_pos;
  popup_y_pos = y_pos;

  if (application->popup_response_delay == 0)
    show_popup_real();
  else
    {
      BBS_THREAD_UI_LOCK
	{
	  if (popup_delay_id != 0)
	    g_source_remove(popup_delay_id);

	  popup_url = g_strdup(link);
	  popup_delay_id = g_timeout_add(application->popup_response_delay,
					 popup_delay_timeout, popup_url);
	}
      BBS_THREAD_UI_UNLOCK;
#if DEBUG_GUI_MOST
      fprintf(stderr, "timeout_add, id=%d, 0x%x\n",
	      popup_delay_id, (int)popup_url);
#endif
    }
}


static gboolean
popup_delay_timeout(gpointer data)
{
  gboolean result = TRUE;

#if DEBUG_GUI_MOST
  fprintf(stderr, "popup_delay_timeout(0x%x)\n", (int)data);
#endif

  gdk_threads_enter();
  if (data == popup_url)
    {
#if DEBUG_GUI_MOST
      fprintf(stderr, "id=%d\n", popup_delay_id);
#endif
      BBS_THREAD_UI_LOCK
	{
	  if (popup_delay_id != 0)
	    popup_delay_id = 0;

	  g_free(data);
	  popup_url = NULL;
	}
      BBS_THREAD_UI_UNLOCK;

      show_popup_real();
      result = FALSE;
    }
  gdk_threads_leave();

  return result;
}


static void
show_popup_real(void)
{
  if (popup_renderer_context.view != NULL)
    {
      gint w, h;
      gboolean use_two_steps_hack = FALSE;
      GtkRequisition requisition = { 0, 0 };
      GdkScreen *screen = gtk_widget_get_screen(popup_response_window);
      gint screen_width = screen != NULL ? gdk_screen_get_width(screen) : 0;

      gtk_widget_size_request(GTK_WIDGET(popup_renderer_context.view),
			      &requisition);
      gtk_widget_get_size_request(GTK_WIDGET(popup_renderer_context.view),
				  &w, &h);
#if DEBUG_GUI_MOST
      fprintf(stderr, "-----------------------------------------------\n");
      fprintf(stderr, "show_popup_real(): widget=0x%x\n",
	      (int)popup_renderer_context.view);
      fprintf(stderr, "requisition(width=%d, height=%d), w=%d, h=%d\n",
	      requisition.width, requisition.height, w, h);
#endif

      if (requisition.width == 0)
	{
	  BBS_THREAD_UI_LOCK
	    {
#if DEBUG_GUI_MOST
	      fprintf(stderr, "Use two step hack.\n");
#endif
	      need_popup_adjustment = TRUE;
	    }
	  BBS_THREAD_UI_UNLOCK;

	  use_two_steps_hack = TRUE;
	}
      else
	{
	  gtk_widget_set_size_request(GTK_WIDGET(popup_renderer_context.view),
				      requisition.width
				      + THREAD_VIEW_EXTRA_WIDTH,
				      requisition.height);

	  gtk_window_resize(GTK_WINDOW(popup_response_window),
			    requisition.width + THREAD_VIEW_EXTRA_WIDTH,
			    requisition.height);
	  BBS_THREAD_UI_LOCK
	    {
	      need_popup_adjustment = FALSE;
	    }
	  BBS_THREAD_UI_UNLOCK;
	}

      popup_y_pos -= (requisition.height + 10);
      popup_y_pos = popup_y_pos > 0 ? popup_y_pos : 0;

      if (popup_renderer_context.application->enable_smart_popup_placement
	  && screen_width > 0
	  && (popup_x_pos + requisition.width + THREAD_VIEW_EXTRA_WIDTH) > screen_width)
	{
	  popup_x_pos =
	    screen_width - (requisition.width + THREAD_VIEW_EXTRA_WIDTH);
	  if (popup_x_pos < 0)
	    popup_x_pos = 0;
	}

      gtk_window_move(GTK_WINDOW(popup_response_window),
		      popup_x_pos, popup_y_pos);
      if (use_two_steps_hack)
	popup_y_pos += requisition.height;

      gtk_widget_show_all(popup_response_window);
    }
}


static void
hide_popup(void)
{
#if DEBUG_GUI_MOST
  fprintf(stderr, "hide_popup\n");
  fprintf(stderr, "id=%d, popup_url=0x%x\n", popup_delay_id, (int)popup_url);
#endif

  BBS_THREAD_UI_LOCK
    {
      if (popup_url != NULL)
	{
	  g_free(popup_url);
	  popup_url = NULL;

	  if (popup_delay_id != 0)
	    {
	      g_source_remove(popup_delay_id);
	      popup_delay_id = 0;
	    }
	}
    }
  BBS_THREAD_UI_UNLOCK;

  gtk_widget_hide_all(popup_response_window);
  if (popup_renderer_context.view != NULL)
    {
      gtk_container_remove(popup_response_frame,
			   GTK_WIDGET(popup_renderer_context.view));
      popup_renderer_context.view = NULL;
    }
}


void
refresh_thread(OchushaApplication *application, GtkWidget *widget,
	       OchushaBBSThread *thread, IconLabel *tab_label)
{
  BBSThreadGUIInfo *info = (BBSThreadGUIInfo *)thread->user_data;
  BBSThreadView *view = BBS_THREAD_VIEW(widget);
  WorkerJob *job = NULL;
  BBSThreadJobArgs *job_args = NULL;
#if DEBUG_GUI_MOST
  fprintf(stderr, "refresh_thread\n");
#endif

  BBS_THREAD_UI_LOCK
    {
      OchushaAsyncBuffer *buffer;
      OchushaAsyncBuffer *old_buffer;

      if (!info->rendering_done)
	{
#if DEBUG_GUI_MOST
	  fprintf(stderr, "refresh_thread: refresh has already been requested.\n");
#endif
	  goto finish_refresh;	/* 󥰤򤷤Ƥ */
	}
      job = (WorkerJob *)calloc(1, sizeof(WorkerJob));
      job_args = (BBSThreadJobArgs *)calloc(1, sizeof(BBSThreadJobArgs));
      if (job == NULL || job_args == NULL)
	{
	  /* Out of memoryrefresh򤢤 */
	  fprintf(stderr, "Couldn't start refresh thread worker.\n");
	  goto error_exit;
	}

      old_buffer = info->dat_buffer;
      buffer = ochusha_bbs_thread_get_responses_source(thread,
						&application->conf,
						info->last_modified,
						OCHUSHA_CACHE_TRY_UPDATE,
						&icon_indicator_funcs,
						tab_label);

      if (buffer == NULL)
	{
	  fprintf(stderr, "Couldn't access network!\n");
	  goto error_exit;
	}

      info->dat_buffer = buffer;
      info->rendering_done = FALSE;

      job_args->application = application;
      job_args->thread = thread;
      job_args->view = view;
      job_args->buffer = buffer;
      job_args->start = thread->number_of_responses_read;
      job_args->number_of_responses = -1;
      job_args->refresh = TRUE;
      job_args->tab_label = tab_label;

      job->canceled = FALSE;
      job->job = render_bbs_thread;
      job->args = job_args;

#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "refresh_thread: ref AsyncBuffer(0x%x)\n", (int)buffer);
      fprintf(stderr, "* before unref: buffer->ref_count=%d\n",
	      G_OBJECT(buffer)->ref_count);
#endif
      g_object_ref(G_OBJECT(buffer));
      g_object_ref(G_OBJECT(tab_label));
      g_object_ref(G_OBJECT(view));
      commit_job(job);

      if (old_buffer != NULL)
	{
#if DEBUG_ASYNC_BUFFER_MOST
	  fprintf(stderr, "refresh_thread: unref AsyncBuffer(0x%x)\n", (int)old_buffer);
	  fprintf(stderr, "* before unref: old_buffer->ref_count=%d\n",
		  G_OBJECT(old_buffer)->ref_count);
#endif
	  g_object_unref(G_OBJECT(old_buffer));
	}
    }
 finish_refresh:
  BBS_THREAD_UI_UNLOCK;
  return;

 error_exit:
  BBS_THREAD_UI_UNLOCK;
  if (job != NULL)
    free(job);
  if (job_args != NULL)
    free(job_args);
}


void
go_to_the_first_response(GtkWidget *widget, OchushaBBSThread *thread)
{
  BBSThreadView *view;
  g_return_if_fail(widget != NULL);
  view = BBS_THREAD_VIEW(widget);
  bbs_thread_view_scroll_to_start(view);
}


void
go_to_the_last_response(GtkWidget *widget, OchushaBBSThread *thread)
{
  BBSThreadView *view;
  g_return_if_fail(widget != NULL);
  view = BBS_THREAD_VIEW(widget);
  bbs_thread_view_scroll_to_end(view);
}


void
jump_to_new_comer_response(GtkWidget *widget, OchushaBBSThread *thread)
{
  BBSThreadView *view;
  g_return_if_fail(widget != NULL && thread != NULL
		   && thread->user_data != NULL);
  view = BBS_THREAD_VIEW(widget);
  bbs_thread_view_scroll_to_mark(view,
			((BBSThreadGUIInfo *)thread->user_data)->last_read);
}
