/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: paned_notebook.c,v 1.1.1.1 2003/05/10 16:34:26 fuyu Exp $
 */

#include "ochusha_private.h"
#include "ochusha_ui.h"

#include "paned_notebook.h"

#include "marshal.h"

#include <glib.h>
#include <gtk/gtk.h>

#include <stdio.h>
#include <stdlib.h>

#define ENABLE_ARROW_BUTTONS	1


static void paned_notebook_class_init(PanedNotebookClass *klass);
static void paned_notebook_set_property(GObject *object, guint prop_id,
					const GValue *value,
					GParamSpec *pspec);
static void paned_notebook_get_property(GObject *object, guint prop_id,
					GValue *value, GParamSpec *pspec);
static void item_view_required_cb(PanedNotebook *paned_notebook, gpointer item,
				  GtkWidget **item_view,
				  const gchar **item_title,
				  GtkWidget **tab_label);
static void paned_notebook_init(PanedNotebook *paned_notebook);
static void paned_notebook_finalize(GObject *object);
static void paned_notebook_destroy(GtkObject *object);
static void paned_notebook_size_request(GtkWidget *widget,
					GtkRequisition *requisition);
static void paned_notebook_size_allocate (GtkWidget *widget,
					  GtkAllocation *allocation);

typedef struct _ItemViewRelation ItemViewRelation;
static ItemViewRelation *item_view_relation_new(gpointer item,
						GtkWidget *item_view,
						GtkWidget *menu_item,
						GtkWidget *tab_label);
static gint compare_items(gconstpointer list_data, gconstpointer user_data);
static ItemViewRelation *find_item_view_relation(PanedNotebook *paned_notebook,
						 gconstpointer key);
static void foreach_item(gpointer data, gpointer user_data);
static void foreach_item_view(gpointer data, gpointer user_data);

static void close_nth_page(PanedNotebook *paned_notebook, gint page);

static void switch_page_cb(ModNotebook *notebook, ModNotebookPage *page,
			   guint page_num, PanedNotebook *paned_notebook);
static gboolean button_press_event_cb(ModNotebook *notebook,
				      GdkEventButton *event,
				      PanedNotebook *paned_notebook);
#if 0
static gboolean button_release_event_cb(ModNotebook *notebook,
					GdkEventButton *event,
					PanedNotebook *paned_notebook);
#endif
static void menu_changed_cb(GtkOptionMenu *option_menu,
			    PanedNotebook *paned_notebook);
static void menu_item_activate_cb(GtkMenuItem *menu_item,
				  PanedNotebook *paned_notebook);
static void foreach_item(gpointer data, gpointer user_data);
static void foreach_item_view(gpointer data, gpointer user_data);
static void close_item_view_button_cb(gpointer unused,
				      PanedNotebook *paned_notebook);
static void go_forward_button_cb(gpointer unused,
				 PanedNotebook *paned_notebook);
static void go_back_button_cb(gpointer unused, PanedNotebook *paned_notebook);

GType
paned_notebook_get_type(void)
{
  static GType pn_type = 0;

  if (pn_type == 0)
    {
      static const GTypeInfo pn_info =
	{
	  sizeof(PanedNotebookClass),
	  NULL, /* base_init */
	  NULL, /* base_finalize */
	  (GClassInitFunc)paned_notebook_class_init,
	  NULL, /* class_finalize */
	  NULL, /* class_data */
	  sizeof(PanedNotebook),
	  0,    /* n_preallocs */
	  (GInstanceInitFunc)paned_notebook_init,
	};

      pn_type = g_type_register_static(GTK_TYPE_PANED,
				       "PanedNotebook", &pn_info, 0);
    }

  return pn_type;
}


GType
paned_style_get_type(void)
{
  static GType s_type = 0;

  if (s_type == 0)
    {
      static const GEnumValue values[] =
	{
	  { HPANED_STYLE, "HPANED_STYLE", "hpaned" },
	  { VPANED_STYLE, "VPANED_STYLE", "vpaned" },
	  { 0, NULL, NULL }
	};

      s_type = g_enum_register_static("PanedStyle", values);
    }

  return s_type;
}


struct _ItemViewRelation
{
  gpointer item;
  GtkWidget *item_view;
  GtkWidget *menu_item;
  GtkWidget *tab_label;
};


enum {
  ITEM_VIEW_REQUIRED_SIGNAL,
  ITEM_VIEW_BEING_CLOSED_SIGNAL,
  ITEM_VIEW_CLOSED_SIGNAL,
  PAGE_SWITCHED_SIGNAL,
  LAST_SIGNAL
};


static GtkPanedClass *parent_class = NULL;
static gint paned_notebook_signals[LAST_SIGNAL] = { 0, 0, 0, 0 };


enum {
  PROP_0,
  PROP_PANED_STYLE,
  PROP_TAB_POLICY,
  PROP_ALWAYS_SHOW_CONTENTS_PANE,
};


static void
paned_notebook_class_init(PanedNotebookClass *klass)
{
  GObjectClass *o_class = G_OBJECT_CLASS(klass);
  GtkObjectClass *object_class = GTK_OBJECT_CLASS(klass);
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(klass);

  parent_class = g_type_class_peek_parent(klass);

  /* GObject signals */
  o_class->finalize = paned_notebook_finalize;
  o_class->set_property = paned_notebook_set_property;
  o_class->get_property = paned_notebook_get_property;

  g_object_class_install_property(o_class,
				  PROP_PANED_STYLE,
				  g_param_spec_enum("paned_style",
						       _("PanedStyle"),
						       _("What type of paned style to be used"),
						    PANED_STYLE_TYPE,
						    HPANED_STYLE,
						    G_PARAM_READWRITE));
  g_object_class_install_property(o_class,
				  PROP_TAB_POLICY,
				  g_param_spec_enum("tab_policy",
						    _("TabPolicy"),
						    _("When bookmark(tab)s should be shown"),
						    GTK_TYPE_POLICY_TYPE,
						    GTK_POLICY_AUTOMATIC,
						    G_PARAM_READWRITE));
  g_object_class_install_property(o_class,
				  PROP_ALWAYS_SHOW_CONTENTS_PANE,
				  g_param_spec_boolean("always_show_contents_pane",
						    _("AlwaysShowContentsPane"),
						    _("Whether contents pane should always be shown"),
						       FALSE,
						       G_PARAM_READWRITE));

  /* GtkObject signals */
  object_class->destroy = paned_notebook_destroy;

  /* GtkWidget signals */
  widget_class->size_request = paned_notebook_size_request;
  widget_class->size_allocate = paned_notebook_size_allocate;

  paned_notebook_signals[ITEM_VIEW_REQUIRED_SIGNAL] =
    g_signal_new("item_view_required",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(PanedNotebookClass, item_view_required),
		 NULL, NULL,
		 ochusha_marshal_VOID__POINTER_POINTER_POINTER_POINTER,
		 G_TYPE_NONE, 4,
		 G_TYPE_POINTER,
		 G_TYPE_POINTER,
		 G_TYPE_POINTER,
		 G_TYPE_POINTER);
  paned_notebook_signals[ITEM_VIEW_BEING_CLOSED_SIGNAL] =
    g_signal_new("item_view_being_closed",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(PanedNotebookClass, item_view_being_closed),
		 NULL, NULL,
		 ochusha_marshal_VOID__POINTER_POINTER,
		 G_TYPE_NONE, 2,
		 G_TYPE_POINTER,
		 GTK_TYPE_WIDGET);
  paned_notebook_signals[ITEM_VIEW_CLOSED_SIGNAL] =
    g_signal_new("item_view_closed",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(PanedNotebookClass, item_view_closed),
		 NULL, NULL,
		 ochusha_marshal_VOID__POINTER_POINTER,
		 G_TYPE_NONE, 2,
		 G_TYPE_POINTER,
		 GTK_TYPE_WIDGET);
  paned_notebook_signals[PAGE_SWITCHED_SIGNAL] =
    g_signal_new("page_switched",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(PanedNotebookClass, page_switched),
		 NULL, NULL,
		 ochusha_marshal_VOID__OBJECT_OBJECT,
		 G_TYPE_NONE, 2,
		 GTK_TYPE_WIDGET,
		 GTK_TYPE_WIDGET);

  klass->item_view_required = item_view_required_cb;
  klass->item_view_being_closed = NULL;
  klass->item_view_closed = NULL;
}


static void
paned_notebook_set_property(GObject *object, guint prop_id,
			    const GValue *value, GParamSpec *pspec)
{
  PanedNotebook *paned_notebook;

  g_return_if_fail(IS_PANED_NOTEBOOK(object));

  paned_notebook = PANED_NOTEBOOK(object);

  switch (prop_id)
    {
    case PROP_PANED_STYLE:
      paned_notebook_set_paned_style(paned_notebook, g_value_get_enum(value));
#if DEBUG_WIDGET_MOST
      fprintf(stderr, "setting paned_style: %d\n", g_value_get_enum(value));
#endif
      break;

    case PROP_TAB_POLICY:
      paned_notebook_set_tab_policy(paned_notebook, g_value_get_enum(value));
#if DEBUG_WIDGET_MOST
      fprintf(stderr, "setting tab_policy: %d\n", g_value_get_enum(value));
#endif
      break;

    case PROP_ALWAYS_SHOW_CONTENTS_PANE:
      paned_notebook_set_always_show_contents_pane(paned_notebook,
						   g_value_get_boolean(value));
#if DEBUG_WIDGET_MOST
      fprintf(stderr, "setting always_show: %d\n", g_value_get_boolean(value));
#endif
      break;

    default:
      break;
    }
}


static void
paned_notebook_get_property(GObject *object, guint prop_id,
			    GValue *value, GParamSpec *pspec)
{
  PanedNotebook *paned_notebook;

  g_return_if_fail(IS_PANED_NOTEBOOK(object));

  paned_notebook = PANED_NOTEBOOK(object);

  switch (prop_id)
    {
    case PROP_PANED_STYLE:
      g_value_set_enum(value, paned_notebook->paned_style);
      break;

    case PROP_TAB_POLICY:
      g_value_set_enum(value, paned_notebook->tab_policy);
      break;

    case PROP_ALWAYS_SHOW_CONTENTS_PANE:
      g_value_set_boolean(value, paned_notebook->always_show_contents_pane);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
      break;
    }
}


static void
item_view_required_cb(PanedNotebook *paned_notebook, gpointer item,
		      GtkWidget **item_view, const gchar **title,
		      GtkWidget **tab_label)
{
  /* TODO: 顼Ф */
#if DEBUG_WIDGET
  if (*item_view == NULL || *title == NULL)
    fprintf(stderr, "Cannot open item.\n");
#endif
}


static void
paned_notebook_init(PanedNotebook *paned_notebook)
{
  GtkPaned *paned;

  g_return_if_fail(GTK_IS_PANED(paned_notebook));

  paned_notebook->selector = NULL;
  paned_notebook->contents_pane = gtk_vbox_new(FALSE, 0);

  paned_notebook->toolbar_box = GTK_BOX(gtk_hbox_new(FALSE, 0));

  paned_notebook->toolbar = GTK_TOOLBAR(gtk_toolbar_new());
  gtk_toolbar_set_style(paned_notebook->toolbar, GTK_TOOLBAR_ICONS);
  gtk_toolbar_set_icon_size(paned_notebook->toolbar,
			    GTK_ICON_SIZE_SMALL_TOOLBAR);
  gtk_box_pack_start(paned_notebook->toolbar_box,
		     GTK_WIDGET(paned_notebook->toolbar),
		     TRUE, TRUE, 0);

  paned_notebook->right_toolbar = GTK_TOOLBAR(gtk_toolbar_new());
  gtk_toolbar_set_style(paned_notebook->right_toolbar, GTK_TOOLBAR_ICONS);
  gtk_toolbar_set_icon_size(paned_notebook->right_toolbar,
			    GTK_ICON_SIZE_SMALL_TOOLBAR);
  gtk_box_pack_end(paned_notebook->toolbar_box,
		   GTK_WIDGET(paned_notebook->right_toolbar),
		   FALSE, TRUE, 0);

  gtk_box_pack_start(GTK_BOX(paned_notebook->contents_pane),
		     GTK_WIDGET(paned_notebook->toolbar_box),
		     FALSE, TRUE, 0);

  paned_notebook->contents_menu = GTK_OPTION_MENU(gtk_option_menu_new());
  paned_notebook->contents_list = GTK_MENU(gtk_menu_new());
  gtk_option_menu_set_menu(paned_notebook->contents_menu,
			   GTK_WIDGET(paned_notebook->contents_list));

  g_signal_connect(G_OBJECT(paned_notebook->contents_menu),
		   "changed",
		   G_CALLBACK(menu_changed_cb), paned_notebook);

  /* MEMO: tooltipΥåpropertyǽˤ٤ */
  gtk_toolbar_insert_stock(paned_notebook->right_toolbar, GTK_STOCK_CLOSE,
			   _("Close current page"),
			   "close_current_page",
			   GTK_SIGNAL_FUNC(close_item_view_button_cb),
			   paned_notebook,
			   0);

  gtk_toolbar_insert_stock(paned_notebook->toolbar, GTK_STOCK_GO_FORWARD,
			   _("Go next page"),
			   "go_next_page",
			   GTK_SIGNAL_FUNC(go_forward_button_cb),
			   paned_notebook,
			   0);
  gtk_toolbar_insert_stock(paned_notebook->toolbar, GTK_STOCK_GO_BACK,
			   _("Go previous page"),
			   "go_previous_page",
			   GTK_SIGNAL_FUNC(go_back_button_cb),
			   paned_notebook,
			   0);
  gtk_toolbar_append_widget(paned_notebook->toolbar,
			    GTK_WIDGET(paned_notebook->contents_menu),
			    _("Select the page to be shown"),
			    "item_selection_menu");

  paned_notebook->item_contents = MOD_NOTEBOOK(mod_notebook_new());
  mod_notebook_set_scrollable(paned_notebook->item_contents, TRUE);
#if 0
  mod_notebook_set_tab_shrinkable(paned_notebook->item_contents, TRUE);
  mod_notebook_set_minimum_tab_label_size(paned_notebook->item_contents, 64);
#endif
  mod_notebook_set_show_tabs(paned_notebook->item_contents, FALSE);

  g_signal_connect(G_OBJECT(paned_notebook->item_contents),
		   "switch_page",
		   G_CALLBACK(switch_page_cb), paned_notebook);
  g_signal_connect(G_OBJECT(paned_notebook->item_contents),
		   "button_press_event",
		   G_CALLBACK(button_press_event_cb), paned_notebook);
#if 0
  g_signal_connect(G_OBJECT(paned_notebook->item_contents),
		   "button_release_event",
		   G_CALLBACK(button_release_event_cb), paned_notebook);
#endif
  gtk_box_pack_start(GTK_BOX(paned_notebook->contents_pane),
		     GTK_WIDGET(paned_notebook->item_contents),
		     TRUE, TRUE, 0);

  paned_notebook->items_shown = NULL;
  paned_notebook->number_of_items_shown = 0;
  paned_notebook->current_item = NULL;
  paned_notebook->current_item_view = NULL;
  paned_notebook->active_menu_item = NULL;

  paned = GTK_PANED(paned_notebook);
  paned->cursor_type = GDK_SB_H_DOUBLE_ARROW;
  paned->orientation = GTK_ORIENTATION_VERTICAL;

  paned_notebook->paned_style = HPANED_STYLE;
  paned_notebook->tab_policy = GTK_POLICY_AUTOMATIC;
  paned_notebook->always_show_contents_pane = FALSE;

  g_object_ref(G_OBJECT(paned_notebook->contents_pane));
  g_signal_connect(G_OBJECT(paned_notebook->contents_pane),
		   "destroy",
		   G_CALLBACK(gtk_widget_destroyed),
		   &paned_notebook->contents_pane);
  gtk_object_sink(GTK_OBJECT(paned_notebook->contents_pane));
}


static void
paned_notebook_finalize(GObject *object)
{
#if 0
  PanedNotebook *paned_notebook = (PanedNotebook *)object;
#endif
  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


static void
fake_close_signal(ItemViewRelation *relation, PanedNotebook *paned_notebook)
{
  g_signal_emit(G_OBJECT(paned_notebook),
		paned_notebook_signals[ITEM_VIEW_BEING_CLOSED_SIGNAL],
		0,
		relation->item,
		relation->item_view);
  g_signal_emit(G_OBJECT(paned_notebook),
		paned_notebook_signals[ITEM_VIEW_CLOSED_SIGNAL],
		0,
		relation->item,
		relation->item_view);
}


static void
paned_notebook_destroy(GtkObject *object)
{
  PanedNotebook *paned_notebook = (PanedNotebook *)object;

  /* item_view_being_closedʥФޤΤǴ֤˹礦Τ
   * ModNotebookγƥڡΥåȤƤΤ
   */
  if (paned_notebook->items_shown != NULL)
    {
      g_slist_foreach(paned_notebook->items_shown,
		      (GFunc)fake_close_signal, paned_notebook);
      g_slist_foreach(paned_notebook->items_shown,
		      (GFunc)free, NULL);
      g_slist_free(paned_notebook->items_shown);
      paned_notebook->items_shown = NULL;
    }

  /*
   * MEMO: ɥޥ͡ˤäƥץꥱ󤬽λ줿
   *       ˸¤ꡢcontents_pane˴Ƥ
   *       ȤȽΤǡad-hocɤ⡢֥Ȥ˿
   *       ʤ褦ˤ롣
   */
  if (paned_notebook->contents_pane != NULL)
    g_object_unref(G_OBJECT(paned_notebook->contents_pane));
#if DEBUG_WIDGET
  else
    fprintf(stderr, "paned_notebook.c: contents_pane seems to have already been destroyed.\n");
#endif

  if (GTK_OBJECT_CLASS(parent_class)->destroy)
    (*GTK_OBJECT_CLASS(parent_class)->destroy)(object);
}


void
paned_notebook_set_paned_style(PanedNotebook *paned_notebook,
			       PanedStyleType paned_style)
{
  g_return_if_fail(IS_PANED_NOTEBOOK(paned_notebook));

#if DEBUG_WIDGET_MOST
  fprintf(stderr, "paned_notebook_set_paned_style: %d\n", paned_style);
#endif

  if (paned_notebook->paned_style != paned_style)
    {
      GtkPaned *paned = GTK_PANED(paned_notebook);
      paned_notebook->paned_style = paned_style;

      if (paned_notebook->paned_style == HPANED_STYLE)
	{
	  paned->cursor_type = GDK_SB_H_DOUBLE_ARROW;
	  paned->orientation = GTK_ORIENTATION_VERTICAL;
	}
      else
	{
	  paned->cursor_type = GDK_SB_V_DOUBLE_ARROW;
	  paned->orientation = GTK_ORIENTATION_HORIZONTAL;
	}

      if (GTK_WIDGET_VISIBLE(paned_notebook))
	gtk_widget_queue_resize(GTK_WIDGET(paned_notebook));
    }

  g_object_notify(G_OBJECT(paned_notebook), "paned_style");
}


PanedStyleType
paned_notebook_get_paned_style(PanedNotebook *paned_notebook)
{
  return paned_notebook->paned_style;
}


void
paned_notebook_set_tab_policy(PanedNotebook *paned_notebook,
			      GtkPolicyType tab_policy)
{
  g_return_if_fail(tab_policy >= GTK_POLICY_ALWAYS
		   && tab_policy <= GTK_POLICY_NEVER);

#if DEBUG_WIDGET
  fprintf(stderr, "set_tab_policy: %d\n", tab_policy);
#endif

  paned_notebook->tab_policy = tab_policy;
  switch (tab_policy)
    {
    case GTK_POLICY_ALWAYS:
      mod_notebook_set_show_tabs(paned_notebook->item_contents, TRUE);
      break;

    case GTK_POLICY_AUTOMATIC:
      if (paned_notebook->number_of_items_shown > 1)
	mod_notebook_set_show_tabs(paned_notebook->item_contents, TRUE);
      break;

    case GTK_POLICY_NEVER:
      mod_notebook_set_show_tabs(paned_notebook->item_contents, FALSE);
      break;
    }

  gtk_widget_queue_resize(GTK_WIDGET(paned_notebook));
}


void
paned_notebook_set_tab_shrinkable(PanedNotebook *paned_notebook,
				  gboolean shrinkable)
{
  g_return_if_fail(IS_PANED_NOTEBOOK(paned_notebook));

  mod_notebook_set_tab_shrinkable(paned_notebook->item_contents, shrinkable);
}


gboolean
paned_notebook_get_tab_shrinkable(PanedNotebook *paned_notebook)
{
  g_return_val_if_fail(IS_PANED_NOTEBOOK(paned_notebook), FALSE);

  return mod_notebook_get_tab_shrinkable(paned_notebook->item_contents);
}


void
paned_notebook_set_minimum_tab_label_size(PanedNotebook *paned_notebook,
					  guint size)
{
  g_return_if_fail(IS_PANED_NOTEBOOK(paned_notebook));

  mod_notebook_set_minimum_tab_label_size(paned_notebook->item_contents, size);
}


guint
paned_notebook_get_minimum_tab_label_size(PanedNotebook *paned_notebook)
{
  g_return_val_if_fail(IS_PANED_NOTEBOOK(paned_notebook), 0);

  return mod_notebook_get_minimum_tab_label_size(paned_notebook->item_contents);
}


void
paned_notebook_set_tooltips(PanedNotebook *paned_notebook, gboolean enable)
{
  g_return_if_fail(IS_PANED_NOTEBOOK(paned_notebook));

  if (enable)
    mod_notebook_tooltips_enable(paned_notebook->item_contents);
  else
    mod_notebook_tooltips_disable(paned_notebook->item_contents);
}


void
paned_notebook_set_always_show_contents_pane(PanedNotebook *paned_notebook,
					     gboolean show_always)
{
  if (paned_notebook->always_show_contents_pane == show_always)
    return;

  paned_notebook->always_show_contents_pane = show_always;

  if (paned_notebook->number_of_items_shown > 0)
    return;

  if (show_always)
    {
      gtk_widget_show_all(GTK_WIDGET(paned_notebook->contents_pane));
      gtk_paned_add2(GTK_PANED(&paned_notebook->container),
		     GTK_WIDGET(paned_notebook->contents_pane));
    }
  else
    {
      gtk_container_remove(GTK_CONTAINER(&paned_notebook->container),
			   GTK_WIDGET(paned_notebook->contents_pane));
    }
}


gboolean
paned_notebook_get_always_show_contents_pane(PanedNotebook *paned_notebook)
{
  return paned_notebook->always_show_contents_pane;
}


GtkPolicyType
paned_notebook_get_tab_policy(PanedNotebook *paned_notebook)
{
  return paned_notebook->tab_policy;
}


static void
paned_notebook_size_request(GtkWidget *widget, GtkRequisition *requisition)
{
  PanedNotebook *paned_notebook = PANED_NOTEBOOK(widget);
  GtkPaned *paned = GTK_PANED(widget);
  GtkRequisition child_requisition;

  requisition->width = 0;
  requisition->height = 0;

  if (paned->child1 && GTK_WIDGET_VISIBLE (paned->child1))
    {
      gtk_widget_size_request (paned->child1, &child_requisition);

      requisition->height = child_requisition.height;
      requisition->width = child_requisition.width;
    }

  if (paned->child2 && GTK_WIDGET_VISIBLE (paned->child2))
    {
      gtk_widget_size_request (paned->child2, &child_requisition);

      if (paned_notebook->paned_style == HPANED_STYLE)
	{
	  requisition->height = MAX(requisition->height,
				    child_requisition.height);
	  requisition->width += child_requisition.width;
	}
      else
	{
	  requisition->width = MAX (requisition->width,
				    child_requisition.width);
	  requisition->height += child_requisition.height;
	}
    }

  requisition->height += GTK_CONTAINER (paned)->border_width * 2;
  requisition->width += GTK_CONTAINER (paned)->border_width * 2;
  
  if (paned->child1 && GTK_WIDGET_VISIBLE (paned->child1) &&
      paned->child2 && GTK_WIDGET_VISIBLE (paned->child2))
    {
      gint handle_size;

      gtk_widget_style_get (widget, "handle_size", &handle_size, NULL);
      if (paned_notebook->paned_style == HPANED_STYLE)
	requisition->width += handle_size;
      else
	requisition->height += handle_size;
    }
}


static void
paned_notebook_size_allocate (GtkWidget *widget, GtkAllocation *allocation)
{
  PanedNotebook *paned_notebook = PANED_NOTEBOOK(widget);
  GtkPaned *paned = GTK_PANED (widget);
  gint border_width = GTK_CONTAINER (paned)->border_width;

  widget->allocation = *allocation;

  if (paned->child1 && GTK_WIDGET_VISIBLE (paned->child1) &&
      paned->child2 && GTK_WIDGET_VISIBLE (paned->child2))
    {
      GtkAllocation child1_allocation;
      GtkAllocation child2_allocation;
      GtkRequisition child1_requisition;
      GtkRequisition child2_requisition;
      gint handle_size;
      
      gtk_widget_style_get (widget, "handle_size", &handle_size, NULL);

      gtk_widget_get_child_requisition (paned->child1, &child1_requisition);
      gtk_widget_get_child_requisition (paned->child2, &child2_requisition);

      if (paned_notebook->paned_style == HPANED_STYLE)
	{
	  gtk_paned_compute_position (paned,
				      MAX (1, widget->allocation.width
					   - handle_size
					   - 2 * border_width),
				      child1_requisition.width,
				      child2_requisition.width);

	  paned->handle_pos.x = widget->allocation.x + paned->child1_size + border_width;
	  paned->handle_pos.y = widget->allocation.y + border_width;
	  paned->handle_pos.width = handle_size;
	  paned->handle_pos.height = MAX (1, widget->allocation.height - 2 * border_width);
	}
      else
	{
	  gtk_paned_compute_position (paned,
				      MAX (1, widget->allocation.height
					   - handle_size
					   - 2 * border_width),
				      child1_requisition.height,
				      child2_requisition.height);

	  paned->handle_pos.x = widget->allocation.x + border_width;
	  paned->handle_pos.y = widget->allocation.y + paned->child1_size + border_width;
	  paned->handle_pos.width = MAX (1, (gint) widget->allocation.width - 2 * border_width);
	  paned->handle_pos.height = handle_size;
	}
      
      
      if (GTK_WIDGET_REALIZED (widget))
	{
	  if (GTK_WIDGET_MAPPED (widget))
	    gdk_window_show (paned->handle);
	  if (paned_notebook->paned_style == HPANED_STYLE)
	    {
	      gdk_window_move_resize (paned->handle,
				      paned->handle_pos.x,
				      paned->handle_pos.y,
				      handle_size,
				      paned->handle_pos.height);
	    }
	  else
	    {
	      gdk_window_move_resize (paned->handle,
				      paned->handle_pos.x,
				      paned->handle_pos.y,
				      paned->handle_pos.width,
				      handle_size);
	    }
	}

      if (paned_notebook->paned_style == HPANED_STYLE)
	{
	  child1_allocation.height = child2_allocation.height = MAX (1, (gint) allocation->height - border_width * 2);
	  child1_allocation.width = paned->child1_size;
	  child1_allocation.x = widget->allocation.x + border_width;
	  child1_allocation.y = child2_allocation.y = widget->allocation.y + border_width;

	  child2_allocation.x = child1_allocation.x + child1_allocation.width +  paned->handle_pos.width;
	  child2_allocation.width = MAX (1, widget->allocation.x + widget->allocation.width - child2_allocation.x - border_width);

	}
      else
	{
	  child1_allocation.width = child2_allocation.width = MAX (1, (gint) allocation->width - border_width * 2);
	  child1_allocation.height = paned->child1_size;
	  child1_allocation.x = child2_allocation.x = widget->allocation.x + border_width;
	  child1_allocation.y = widget->allocation.y + border_width;

	  child2_allocation.y = child1_allocation.y + child1_allocation.height + paned->handle_pos.height;
	  child2_allocation.height = MAX (1, widget->allocation.y + widget->allocation.height - child2_allocation.y - border_width);
	}
      
      
      /* Now allocate the childen, making sure, when resizing not to
       * overlap the windows
       */
      if (paned_notebook->paned_style == HPANED_STYLE)
	{
	  if (GTK_WIDGET_MAPPED (widget) &&
	      paned->child1->allocation.width < child1_allocation.width)
	    {
	      gtk_widget_size_allocate (paned->child2, &child2_allocation);
	      gtk_widget_size_allocate (paned->child1, &child1_allocation);
	    }
	  else
	    {
	      gtk_widget_size_allocate (paned->child1, &child1_allocation);
	      gtk_widget_size_allocate (paned->child2, &child2_allocation);
	    }
	}
      else
	{
	  if (GTK_WIDGET_MAPPED (widget) &&
	      paned->child1->allocation.height < child1_allocation.height)
	    {
	      gtk_widget_size_allocate (paned->child2, &child2_allocation);
	      gtk_widget_size_allocate (paned->child1, &child1_allocation);
	    }
	  else
	    {
	      gtk_widget_size_allocate (paned->child1, &child1_allocation);
	      gtk_widget_size_allocate (paned->child2, &child2_allocation);
	    }
	}
    }
  else
    {
      GtkAllocation child_allocation;

      if (GTK_WIDGET_REALIZED (widget))      
	gdk_window_hide (paned->handle);
	  
      child_allocation.x = widget->allocation.x + border_width;
      child_allocation.y = widget->allocation.y + border_width;
      child_allocation.width = MAX (1, allocation->width - 2 * border_width);
      child_allocation.height = MAX (1, allocation->height - 2 * border_width);
      
      if (paned->child1 && GTK_WIDGET_VISIBLE (paned->child1))
	gtk_widget_size_allocate (paned->child1, &child_allocation);
      else if (paned->child2 && GTK_WIDGET_VISIBLE (paned->child2))
	gtk_widget_size_allocate (paned->child2, &child_allocation);
    }
}


GtkWidget *
paned_notebook_new(PanedStyleType paned_style)
{
  GtkWidget *widget;

  widget = GTK_WIDGET(g_object_new(PANED_NOTEBOOK_TYPE,
				   "paned_style", paned_style,
				   NULL));

  return widget;
}


GtkWidget *
paned_notebook_new_with_selector(PanedStyleType paned_style,
				 GtkWidget *selector)
{
  GtkWidget *widget = paned_notebook_new(paned_style);

  g_return_val_if_fail((widget != NULL), NULL);

  paned_notebook_set_selector(PANED_NOTEBOOK(widget), selector);

  return widget;
}


GtkWidget *
paned_notebook_get_selector(PanedNotebook *paned_notebook)
{
  return paned_notebook->selector;
}


static void
open_view_cb(GtkWidget *selector, gpointer item, gboolean in_tab,
	     PanedNotebook *paned_notebook)
{
  paned_notebook_open_item_view(paned_notebook, item, in_tab);
}


void
paned_notebook_set_selector(PanedNotebook *paned_notebook, GtkWidget *selector)
{
  if (paned_notebook->selector == selector)
    return;	/* ⤷ʤ */

  if (paned_notebook->selector != NULL)
    gtk_container_remove(GTK_CONTAINER(&paned_notebook->container),
			 paned_notebook->selector);

  if (selector != NULL)
    {
      g_signal_connect(G_OBJECT(selector), "open_view",
		       G_CALLBACK(open_view_cb), paned_notebook);
      gtk_paned_add1(GTK_PANED(&paned_notebook->container), selector);
    }

  paned_notebook->selector = selector;

  return;
}


static ItemViewRelation *
item_view_relation_new(gpointer item, GtkWidget *item_view,
		       GtkWidget *menu_item, GtkWidget *tab_label)
{
  ItemViewRelation *relation
    = (ItemViewRelation *)g_malloc(sizeof(ItemViewRelation));
  relation->item = item;
  relation->item_view = item_view;
  relation->menu_item = menu_item;
  relation->tab_label = tab_label;
  return relation;
}


static gint
compare_items(gconstpointer list_data, gconstpointer user_data)
{
  return (((ItemViewRelation *)list_data)->item != user_data
	  && ((ItemViewRelation *)list_data)->item_view != user_data
	  && ((ItemViewRelation *)list_data)->menu_item != user_data
	  && ((ItemViewRelation *)list_data)->tab_label != user_data);
}


static ItemViewRelation *
find_item_view_relation(PanedNotebook *paned_notebook, gconstpointer key)
{
  GSList *entry = g_slist_find_custom(paned_notebook->items_shown, key,
				      compare_items);
  return (ItemViewRelation *)(entry != NULL ? entry->data : NULL);
}


GtkToolbar *
paned_notebook_get_toolbar(PanedNotebook *paned_notebook)
{
  return paned_notebook->toolbar;
}


gpointer
paned_notebook_get_current_item(PanedNotebook *paned_notebook)
{
  return paned_notebook->current_item;
}


GtkWidget *
paned_notebook_get_current_item_view(PanedNotebook *paned_notebook)
{
  return paned_notebook->current_item_view;
}


gpointer
paned_notebook_get_item(PanedNotebook *paned_notebook, GtkWidget *item_view)
{
  ItemViewRelation *relation = find_item_view_relation(paned_notebook,
						       item_view);

  if (relation == NULL)
    return NULL;
  return relation->item;
}


GtkWidget *
paned_notebook_get_item_view(PanedNotebook *paned_notebook, gpointer item)
{
  ItemViewRelation *relation = find_item_view_relation(paned_notebook, item);

  if (relation == NULL)
    return NULL;
  return relation->item_view;
}


GtkWidget *
paned_notebook_get_tab_label(PanedNotebook *paned_notebook, gpointer item)
{
  ItemViewRelation *relation = find_item_view_relation(paned_notebook, item);

  if (relation == NULL)
    return NULL;
  return relation->tab_label;
}


typedef struct _ForeachItemArgs
{
  GFunc func;
  gpointer user_data;
} ForeachItemArgs;


static void
foreach_item(gpointer data, gpointer user_data)
{
  ItemViewRelation *relation = (ItemViewRelation *)data;
  ForeachItemArgs *args = (ForeachItemArgs *)user_data;
  (*args->func)(relation->item, args->user_data);
}


void
paned_notebook_foreach_item(PanedNotebook *paned_notebook, GFunc func,
			    gpointer user_data)
{
  ForeachItemArgs args =
    {
      func,
      user_data
    };
  g_slist_foreach(paned_notebook->items_shown, foreach_item, &args);
}


static void
foreach_item_view(gpointer data, gpointer user_data)
{
  ItemViewRelation *relation = (ItemViewRelation *)data;
  ForeachItemArgs *args = (ForeachItemArgs *)user_data;
  (*args->func)(relation->item_view, args->user_data);
}


void
paned_notebook_foreach_item_view(PanedNotebook *paned_notebook, GFunc func,
				 gpointer user_data)
{
  ForeachItemArgs args =
    {
      func,
      user_data
    };
  g_slist_foreach(paned_notebook->items_shown, foreach_item_view, &args);
}


/*
 * itemNotebookɽ롣
 *
 * itemɽǤ(ModNotebook˴ޤޤƤ)硢in_tabǤС
 * itemɽƤڡ򤷡Ǥв⤷ʤ롣
 *
 * itemɽǤʤ硢item_view_requiredʥȯԤ
 * ηitemɽ륦åȤͿ줿ˤϤΥ
 * ȤModNotebookɲä롣λin_tabξˤϸ򤵤Ƥ
 * ڡɽƤitemĤǤΰ֤˥åȤ
 * Υڡ򤹤롣in_tabξˤñɲä롣
 */
void
paned_notebook_open_item_view(PanedNotebook *paned_notebook, gpointer item,
			      gboolean in_tab)
{
  ItemViewRelation *relation = find_item_view_relation(paned_notebook, item);
  GtkWidget *item_view = relation != NULL ? relation->item_view : NULL;
  const gchar *item_title = NULL;
  GtkWidget *tab_label = NULL;
  GtkWidget *menu_item;
  gint page;

  if (item_view != NULL)
    {
#if DEBUG_WIDGET_MOST
      fprintf(stderr, "Item already shown.\n");
#endif
      if (in_tab)
	return;

      page = mod_notebook_page_num(paned_notebook->item_contents, item_view);
      if (page == -1)
	{
	  /* ɽƤϤΥåȤĤʤ */
	  fprintf(stderr, "Something wrong happen.\n");
	  return;
	}

      mod_notebook_set_current_page(paned_notebook->item_contents, page);

      return;
    }

  /* itemɽƤʤ */
  g_signal_emit(G_OBJECT(paned_notebook),
		paned_notebook_signals[ITEM_VIEW_REQUIRED_SIGNAL],
		0,
		item,
		&item_view,
		&item_title,
		&tab_label);

  /* TODO: item_title == NULLξŬ˽٤ */
  if (item_view == NULL || item_title == NULL)
    {
#if DEBUG_WIDGET
      fprintf(stderr, "item_view or item_title is NULL\n");
#endif
      return;	/* itemɽǽ */
    }

  if (tab_label == NULL)
    tab_label = gtk_label_new(item_title);

  menu_item = gtk_menu_item_new_with_label(item_title);
  paned_notebook->items_shown
    = g_slist_append(paned_notebook->items_shown,
		     item_view_relation_new(item, item_view, menu_item,
					    tab_label));

  g_signal_connect(G_OBJECT(menu_item), "activate",
		   G_CALLBACK(menu_item_activate_cb), paned_notebook);

  gtk_widget_show(menu_item);
  gtk_menu_shell_append(GTK_MENU_SHELL(paned_notebook->contents_list),
			menu_item);

  if (paned_notebook->number_of_items_shown == 0
      && !paned_notebook->always_show_contents_pane)
    {
      gtk_paned_add2(GTK_PANED(&paned_notebook->container),
		     GTK_WIDGET(paned_notebook->contents_pane));
      gtk_widget_show_all(GTK_WIDGET(paned_notebook->contents_pane));
    }

  if (paned_notebook->number_of_items_shown++ == 0 || in_tab)
    {
#if DEBUG_WIDGET_MOST
      fprintf(stderr, "showing a page(%s) in tab.\n", item_title);
#endif
      mod_notebook_append_page_menu(paned_notebook->item_contents, item_view,
				    tab_label, gtk_label_new(item_title));
      if (paned_notebook->number_of_items_shown == 2
	  && paned_notebook->tab_policy == GTK_POLICY_AUTOMATIC)
	{
	  gtk_widget_show_all(GTK_WIDGET(paned_notebook->item_contents));
	  mod_notebook_set_show_tabs(paned_notebook->item_contents, TRUE);
	  gtk_widget_queue_resize(GTK_WIDGET(paned_notebook));
	}
      return;
    }

#if DEBUG_WIDGET_MOST
  fprintf(stderr, "showing a page(%s) in current page.\n", item_title);
#endif
  /* 򤵤Ƥڡ֤*/
  page = mod_notebook_get_current_page(paned_notebook->item_contents);
  if (page == -1)
    {
      /* */
#if DEBUG_WIDGET
      fprintf(stderr, "Internal Error: inconsistent number_of_items_shown.\n");
#endif
      mod_notebook_append_page_menu(paned_notebook->item_contents, item_view,
				    tab_label, gtk_label_new(item_title));
      return;
    }

  close_nth_page(paned_notebook, page);
  mod_notebook_insert_page_menu(paned_notebook->item_contents, item_view,
				tab_label, gtk_label_new(item_title), page);
  mod_notebook_set_current_page(paned_notebook->item_contents, page);
}


/*
 * itemɽǤä餽ΥڡĤ롣
 */
void
paned_notebook_close_item_view(PanedNotebook *paned_notebook, gpointer item)
{
  ItemViewRelation *relation = find_item_view_relation(paned_notebook, item);
  gint page;

  if (relation == NULL)
    return;

  page = mod_notebook_page_num(paned_notebook->item_contents,
			       relation->item_view);
  if (page == -1)
    {
      /* ɽƤϤΥåȤĤʤ */
      fprintf(stderr, "Something wrong happen.\n");
      return;
    }

  close_nth_page(paned_notebook, page);
}


void
paned_notebook_close_current_item_view(PanedNotebook *paned_notebook)
{
  gint page = mod_notebook_get_current_page(paned_notebook->item_contents);
  if (page == -1)
    return;

  close_nth_page(paned_notebook, page);
}


/*
 * ꤷڡĤΥڡбitemˤĤ
 * item_view_closedʥȯԤ롣
 */
static void
close_nth_page(PanedNotebook *paned_notebook, gint page)
{
  ItemViewRelation *relation;
  gpointer item;
  GtkWidget *menu_item;
  GtkWidget *widget = mod_notebook_get_nth_page(paned_notebook->item_contents,
						page);
  g_return_if_fail(widget != NULL);

  relation = find_item_view_relation(paned_notebook, widget);
#if DEBUG_WIDGET
  g_return_if_fail(relation != NULL && relation->item_view == widget);
#else
  if (relation == NULL || relation->item_view != widget)
    return;
#endif

  item = relation->item;
  menu_item = relation->menu_item;
  paned_notebook->items_shown = g_slist_remove(paned_notebook->items_shown,
					       relation);
  g_free(relation);

  if (paned_notebook->current_item_view == widget)
    {
      paned_notebook->current_item = NULL;
      paned_notebook->current_item_view = NULL;
    }

  g_object_ref(G_OBJECT(widget));

  g_signal_emit(G_OBJECT(paned_notebook),
		paned_notebook_signals[ITEM_VIEW_BEING_CLOSED_SIGNAL],
		0,
		item,
		widget);

  mod_notebook_remove_page(paned_notebook->item_contents, page);

  paned_notebook->number_of_items_shown--;

  if (paned_notebook->number_of_items_shown == 0
      && !paned_notebook->always_show_contents_pane)
    gtk_container_remove(GTK_CONTAINER(&paned_notebook->container),
			 GTK_WIDGET(paned_notebook->contents_pane));

  if (paned_notebook->number_of_items_shown == 1
      && paned_notebook->tab_policy == GTK_POLICY_AUTOMATIC)
    {
      gtk_widget_show_all(GTK_WIDGET(paned_notebook->item_contents));
      mod_notebook_set_show_tabs(paned_notebook->item_contents, FALSE);
      gtk_widget_queue_resize(GTK_WIDGET(paned_notebook));
    }

  g_signal_emit(G_OBJECT(paned_notebook),
		paned_notebook_signals[ITEM_VIEW_CLOSED_SIGNAL],
		0,
		item,
		widget);

  g_object_unref(G_OBJECT(widget));

  gtk_container_remove(GTK_CONTAINER(paned_notebook->contents_list),
		       menu_item);
  gtk_option_menu_set_history(paned_notebook->contents_menu, 0);
}


static void
switch_page_cb(ModNotebook *notebook, ModNotebookPage *page, guint page_num,
	       PanedNotebook *paned_notebook)
{
  ItemViewRelation *relation;
  GtkWidget *new_current_page = mod_notebook_get_nth_page(notebook, page_num);
  GtkWidget *previous_page = paned_notebook->current_item_view;

#if DEBUG_WIDGET_MOST
  fprintf(stderr, "page #%d selected.\n", page_num);
#endif

  if (previous_page == new_current_page)
    return;

  relation = find_item_view_relation(paned_notebook, new_current_page);
  if (relation == NULL)
    {
#if DEBUG_WIDGET
      fprintf(stderr, "HELP! Relation missed.\n");
#endif
      return;
    }

  paned_notebook->current_item = relation->item;
  paned_notebook->current_item_view = new_current_page;

  gtk_menu_reorder_child(paned_notebook->contents_list,
			 relation->menu_item, 0);
  gtk_option_menu_set_history(paned_notebook->contents_menu, 0);

  g_signal_emit(G_OBJECT(paned_notebook),
		paned_notebook_signals[PAGE_SWITCHED_SIGNAL],
		0,
		previous_page,
		new_current_page);

  gtk_widget_queue_resize(new_current_page);
}


static gboolean
button_press_event_cb(ModNotebook *notebook, GdkEventButton *event,
		      PanedNotebook *paned_notebook)
{
  g_return_val_if_fail(notebook == paned_notebook->item_contents, FALSE);

  if (event->button == 2)
    {
      gint page = mod_notebook_get_mouse_event_page_num(notebook,
							(GdkEventAny*)event);
#if DEBUG_WIDGET_MOST
      fprintf(stderr, "PanedNotebook(button_press_event_cb): page=%d\n", page);
#endif
      if (page != -1)
	{
	  close_nth_page(paned_notebook, page);
	  return TRUE;
	}
    }

  return FALSE;
}


#if 0
static gboolean
button_release_event_cb(ModNotebook *notebook, GdkEventButton *event,
			PanedNotebook *paned_notebook)
{
  g_return_val_if_fail(notebook == paned_notebook->item_contents, FALSE);

  /* do something */

  return FALSE;
}
#endif


static void
menu_changed_cb(GtkOptionMenu *option_menu, PanedNotebook *paned_notebook)
{
  ItemViewRelation *relation;
  GtkMenuItem *menu_item = paned_notebook->active_menu_item;
  gint page;
#if DEBUG_WIDGET_MOST
  gint history = gtk_option_menu_get_history(option_menu);
  fprintf(stderr, "Option menu changed.  history=%d\n", history);
#endif

  if (menu_item == NULL)
    return;

  paned_notebook->active_menu_item = NULL;

  relation = find_item_view_relation(paned_notebook, menu_item);
  if (relation == NULL)
    return;

  page = mod_notebook_page_num(paned_notebook->item_contents,
			       relation->item_view);
  if (page != -1)
    mod_notebook_set_current_page(paned_notebook->item_contents, page);
}


static void
menu_item_activate_cb(GtkMenuItem *menu_item, PanedNotebook *paned_notebook)
{
  paned_notebook->active_menu_item = menu_item;
#if DEBUG_WIDGET_MOST
  fprintf(stderr, "Menu item activated.\n");
#endif
}


static void
close_item_view_button_cb(gpointer unused, PanedNotebook *paned_notebook)
{
  paned_notebook_close_current_item_view(paned_notebook);
}


#if ENABLE_ARROW_BUTTONS
static void
go_forward_button_cb(gpointer unused, PanedNotebook *paned_notebook)
{
  mod_notebook_next_page(paned_notebook->item_contents);
}


static void
go_back_button_cb(gpointer unused, PanedNotebook *paned_notebook)
{
  mod_notebook_prev_page(paned_notebook->item_contents);
}
#endif
