/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: threadlist_view.c,v 1.4 2003/05/11 23:04:36 fuyu Exp $
 */

#include "config.h"

#include "ochusha.h"
#include "ochusha_ui.h"

#include "ochusha_bbs_thread.h"

#include "threadlist_view.h"

#include "marshal.h"

#include <gdk/gdkkeysyms.h>
#include <glib.h>
#include <gtk/gtk.h>

#if DEBUG_WIDGET || DEBUG_WIDGET_MOST
#include <stdio.h>
#endif
#include <stdlib.h>
#include <string.h>


static void threadlist_view_class_init(ThreadlistViewClass *klass);
static void popup_menu_callback(gpointer data, guint action,
				GtkWidget *widget);
static void popup_menu_destructed_cb(gpointer data);
static void threadlist_view_init(ThreadlistView *view);

static void threadlist_view_finalize(GObject *object);
static void threadlist_view_destroy(GtkObject *object);

static gboolean threadlist_view_title_not_equals(GtkTreeModel *model,
						 gint column,
						 const gchar *key,
						 GtkTreeIter *iter,
						 gpointer search_data);

static gboolean threadlist_view_move_cursor(ThreadlistView *view,
					    GtkMovementStep step,
					    gint count);
#if 0
/*
 * TODO: Emacs^Säݤ񤤤¸ˤϡ롼Ǽ
 *       뤷ʤäݤʡ
 */
static gboolean threadlist_view_search_title(ThreadlistView *view,
					     int direction);
#endif

static void threadlist_view_do_on_selection(ThreadlistView *view,
					    gint what_to_do);
static void threadlist_view_do_on_thread_at_cursor(ThreadlistView *view,
						   gint what_to_do);

static gboolean tree_view_motion_notify_event_cb(GtkWidget *widget,
						 GdkEventMotion *event,
						 ThreadlistView *view);
static gboolean tree_view_leave_notify_event_cb(GtkWidget *widget,
						GdkEventButton *event,
						ThreadlistView *view);
static gboolean tree_view_button_press_event_cb(GtkWidget *widget,
						GdkEventButton *event,
						ThreadlistView *view);
static gboolean tree_view_button_release_event_cb(GtkWidget *widget,
						  GdkEventButton *event,
						  ThreadlistView *view);

static void decorate_threadlist_entry(GtkTreeViewColumn *tree_column,
				       GtkCellRenderer *renderer,
				       GtkTreeModel *tree_model,
				       GtkTreeIter *iter, gpointer unused);
static void active_set_cell_data_flags(GtkTreeViewColumn *tree_column,
				       GtkCellRenderer *renderer,
				       GtkTreeModel *tree_model,
				       GtkTreeIter *iter,
				       gpointer unused);
static void text_set_cell_data_rank_diff(GtkTreeViewColumn *tree_column,
					 GtkCellRenderer *renderer,
					 GtkTreeModel *tree_model,
					 GtkTreeIter *iter,
					 gpointer unused);


GType
threadlist_view_get_type(void)
{
  static GType tlv_type = 0;

  if (tlv_type == 0)
    {
      static const GTypeInfo tlv_info =
	{
	  sizeof(ThreadlistViewClass),
	  NULL, /* base_init */
	  NULL, /* base_finalize */
	  (GClassInitFunc)threadlist_view_class_init,
	  NULL, /* class_finalize */
	  NULL, /* class_data */
	  sizeof(ThreadlistView),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)threadlist_view_init,
	};

      tlv_type = g_type_register_static(GTK_TYPE_SCROLLED_WINDOW,
					"ThreadlistView", &tlv_info, 0);
    }

  return tlv_type;
}


enum {
  OPEN_VIEW_SIGNAL,

  THREAD_MOUSE_OVER_SIGNAL,
  THREAD_MOUSE_OUT_SIGNAL,

  TOGGLE_MARK_SIGNAL,
  TOGGLE_HIDE_SIGNAL,
  MARK_THREAD_SIGNAL,
  ADVANCE_VIEW_SIGNAL,
  BACK_VIEW_SIGNAL,

  MOVE_CURSOR_SIGNAL,

  DO_ON_SELECTION_SIGNAL,
  DO_ON_THREAD_AT_CURSOR_SIGNAL,

  LAST_SIGNAL
};


enum {
  MARK_THREAD,
  UNMARK_THREAD,
  TOGGLE_HIDDEN_STATE,
  ADVANCE_VIEW,
  BACK_VIEW,
};


static GtkScrolledWindowClass *parent_class = NULL;
static gint threadlist_view_signals[LAST_SIGNAL] =
  { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };


enum {
  TREE_VIEW_MOVE_CURSOR_SIGNAL,
  TREE_VIEW_UNSELECT_ALL_SIGNAL,
  TREE_VIEW_SELECT_CURSOR_ROW_SIGNAL,
  TREE_VIEW_START_INTERACTIVE_SEARCH_SIGNAL,
  TREE_VIEW_LAST_SIGNAL
};
static gint tree_view_signals[TREE_VIEW_LAST_SIGNAL] = { 0, 0, 0 };


enum {
  POPUP_MENU_OPEN_SELECTION = 1,
  POPUP_MENU_OPEN_SELECTION_IN_TAB,
  POPUP_MENU_MARK_SELECTION,
  POPUP_MENU_UNMARK_SELECTION,
  POPUP_MENU_TOGGLE_HIDE_SELECTION
};


static GtkItemFactory *popup_menu_item_factory = NULL;


#if 0	/* TODO: 롼Ǽ뤷ʤäݤ */
enum {
  THREADLIST_SEARCH_FORWARD,
  THREADLIST_SEARCH_BACKWARD,
  THREADLIST_SEARCH_REGEXP_FORWARD,
  THREADLIST_SEARCH_REGEXP_BACKWARD,
};
#endif


static void
threadlist_view_class_init(ThreadlistViewClass *klass)
{
  GObjectClass *o_class = (GObjectClass *)klass;
  GtkObjectClass *object_class = (GtkObjectClass *)klass;
  GtkBindingSet *binding_set;

  /* ݥåץåץ˥塼ν */
  GtkItemFactoryEntry menu_items[] =
    {
      {
	_("/_Open Selected Threads"),		/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback,			/* callback_func */
	POPUP_MENU_OPEN_SELECTION,		/* act */
	NULL					/* type */
      },
      {
	_("/Open Selected Threads In _Tab"),	/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback,			/* callback_func */
	POPUP_MENU_OPEN_SELECTION_IN_TAB,	/* act */
	NULL					/* type */
      },
      {
	_("/-------"),
	NULL,
	NULL,
	0,
	"<Separator>"
      },
      {
	_("/Mark Selected Threads(_*)"),	/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback,			/* callback_func */
	POPUP_MENU_MARK_SELECTION,		/* act */
	NULL					/* type */
      },
      {
	_("/_Unmark Selected Threads"),		/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback,			/* callback_func */
	POPUP_MENU_UNMARK_SELECTION,		/* act */
	NULL					/* type */
      },
      {
	_("/Toggle Hide Selected Threads(_D)"),	/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback,			/* callback_func */
	POPUP_MENU_TOGGLE_HIDE_SELECTION,	/* act */
	NULL					/* type */
      },
    };
  gint nmenu_items = sizeof(menu_items) / sizeof(menu_items[0]);

  popup_menu_item_factory = gtk_item_factory_new(GTK_TYPE_MENU,
						 "<ThreadlistViewPopup>",
						 NULL);
  gtk_item_factory_create_items(popup_menu_item_factory,
				nmenu_items, menu_items, NULL);

  parent_class = g_type_class_peek_parent(klass);
  binding_set = gtk_binding_set_by_class(klass);


  /* GObject signals */
  o_class->finalize = threadlist_view_finalize;

  /* GtkObject signals */
  object_class->destroy = threadlist_view_destroy;

  threadlist_view_signals[OPEN_VIEW_SIGNAL] =
    g_signal_new("open_view",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(ThreadlistViewClass, open_view),
		 NULL, NULL,
		 ochusha_marshal_VOID__POINTER_BOOLEAN,
		 G_TYPE_NONE, 2,
		 G_TYPE_POINTER,
		 G_TYPE_BOOLEAN);

  threadlist_view_signals[THREAD_MOUSE_OVER_SIGNAL] =
    g_signal_new("thread_mouse_over",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(ThreadlistViewClass, thread_mouse_over),
		 NULL, NULL,
		 ochusha_marshal_VOID__POINTER,
		 G_TYPE_NONE, 1,
		 G_TYPE_POINTER);
  threadlist_view_signals[THREAD_MOUSE_OUT_SIGNAL] =
    g_signal_new("thread_mouse_out",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(ThreadlistViewClass, thread_mouse_out),
		 NULL, NULL,
		 ochusha_marshal_VOID__POINTER,
		 G_TYPE_NONE, 1,
		 G_TYPE_POINTER);

  threadlist_view_signals[TOGGLE_MARK_SIGNAL] =
    g_signal_new("toggle_mark",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(ThreadlistViewClass, toggle_mark),
		 NULL, NULL,
		 ochusha_marshal_VOID__POINTER,
		 G_TYPE_NONE, 1,
		 G_TYPE_POINTER);
  threadlist_view_signals[TOGGLE_HIDE_SIGNAL] =
    g_signal_new("toggle_hide",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(ThreadlistViewClass, toggle_hide),
		 NULL, NULL,
		 ochusha_marshal_VOID__POINTER,
		 G_TYPE_NONE, 1,
		 G_TYPE_POINTER);
  threadlist_view_signals[MARK_THREAD_SIGNAL] =
    g_signal_new("mark_thread",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(ThreadlistViewClass, mark_thread),
		 NULL, NULL,
		 ochusha_marshal_VOID__POINTER_BOOLEAN,
		 G_TYPE_NONE, 2,
		 G_TYPE_POINTER,
		 G_TYPE_BOOLEAN);
  threadlist_view_signals[ADVANCE_VIEW_SIGNAL] =
    g_signal_new("advance_view",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(ThreadlistViewClass, advance_view),
		 NULL, NULL,
		 ochusha_marshal_BOOLEAN__POINTER,
		 G_TYPE_BOOLEAN, 1,
		 G_TYPE_POINTER);
  threadlist_view_signals[BACK_VIEW_SIGNAL] =
    g_signal_new("back_view",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(ThreadlistViewClass, back_view),
		 NULL, NULL,
		 ochusha_marshal_BOOLEAN__POINTER,
		 G_TYPE_BOOLEAN, 1,
		 G_TYPE_POINTER);

  threadlist_view_signals[MOVE_CURSOR_SIGNAL] =
    g_signal_new("move_cursor",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
		 G_STRUCT_OFFSET(ThreadlistViewClass, move_cursor),
		 NULL, NULL,
		 ochusha_marshal_BOOLEAN__ENUM_INT,
		 G_TYPE_NONE, 2,
		 GTK_TYPE_MOVEMENT_STEP,
		 G_TYPE_INT);

#if 0	/* TODO: 롼Ǽ뤷ʤäݤ */
  threadlist_view_signals[SEARCH_TITLE_SIGNAL] =
    g_signal_new("search_title",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
		 G_STRUCT_OFFSET(ThreadlistViewClass, search_title),
		 NULL, NULL,
		 ochusha_marshal_VOID__INT,
		 G_TYPE_NONE, 1,
		 G_TYPE_INT);
#endif

  threadlist_view_signals[DO_ON_SELECTION_SIGNAL] =
    g_signal_new("do_on_selection",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
		 G_STRUCT_OFFSET(ThreadlistViewClass, do_on_selection),
		 NULL, NULL,
		 ochusha_marshal_VOID__INT,
		 G_TYPE_NONE, 1,
		 G_TYPE_INT);
  threadlist_view_signals[DO_ON_THREAD_AT_CURSOR_SIGNAL] =
    g_signal_new("do_on_thread_at_cursor",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
		 G_STRUCT_OFFSET(ThreadlistViewClass, do_on_thread_at_cursor),
		 NULL, NULL,
		 ochusha_marshal_VOID__INT,
		 G_TYPE_NONE, 1,
		 G_TYPE_INT);

  tree_view_signals[TREE_VIEW_MOVE_CURSOR_SIGNAL] =
    g_signal_lookup("move_cursor", GTK_TYPE_TREE_VIEW);
  tree_view_signals[TREE_VIEW_UNSELECT_ALL_SIGNAL] =
    g_signal_lookup("unselect_all", GTK_TYPE_TREE_VIEW);
  tree_view_signals[TREE_VIEW_SELECT_CURSOR_ROW_SIGNAL] =
    g_signal_lookup("select_cursor_row", GTK_TYPE_TREE_VIEW);
  tree_view_signals[TREE_VIEW_START_INTERACTIVE_SEARCH_SIGNAL] =
    g_signal_lookup("start_interactive_search", GTK_TYPE_TREE_VIEW);

  /* Key bindings */
  gtk_binding_entry_add_signal(binding_set, GDK_o, 0, "do_on_selection", 1,
			       G_TYPE_INT, POPUP_MENU_OPEN_SELECTION);
  gtk_binding_entry_add_signal(binding_set, GDK_t, 0, "do_on_selection", 1,
			       G_TYPE_INT, POPUP_MENU_OPEN_SELECTION_IN_TAB);
  gtk_binding_entry_add_signal(binding_set, GDK_asterisk, 0,
			       "do_on_thread_at_cursor", 1,
			       G_TYPE_INT, MARK_THREAD);
  gtk_binding_entry_add_signal(binding_set, GDK_u, 0,
			       "do_on_thread_at_cursor", 1,
			       G_TYPE_INT, UNMARK_THREAD);
  gtk_binding_entry_add_signal(binding_set, GDK_d, 0,
			       "do_on_thread_at_cursor", 1,
			       G_TYPE_INT, TOGGLE_HIDDEN_STATE);
  gtk_binding_entry_add_signal(binding_set, GDK_space, 0,
			       "do_on_thread_at_cursor", 1,
			       G_TYPE_INT, ADVANCE_VIEW);
  gtk_binding_entry_add_signal(binding_set, GDK_BackSpace, 0,
			       "do_on_thread_at_cursor", 1,
			       G_TYPE_INT, BACK_VIEW);

  gtk_binding_entry_add_signal(binding_set, GDK_v, GDK_CONTROL_MASK,
			       "move_cursor", 2,
			       G_TYPE_ENUM, GTK_MOVEMENT_PAGES,
			       G_TYPE_INT, 1);
  gtk_binding_entry_add_signal(binding_set, GDK_v, GDK_MOD1_MASK,
			       "move_cursor", 2,
			       G_TYPE_ENUM, GTK_MOVEMENT_PAGES,
			       G_TYPE_INT, -1);
  gtk_binding_entry_add_signal(binding_set, GDK_v, GDK_MOD4_MASK,
			       "move_cursor", 2,
			       G_TYPE_ENUM, GTK_MOVEMENT_PAGES,
			       G_TYPE_INT, -1);

#if 0	/* TODO: 롼Ǽ뤷ʤäݤ */
  gtk_binding_entry_add_signal(binding_set, GDK_s, GDK_CONTROL_MASK,
			       "search_title", 1,
			       G_TYPE_ENUM, THREADLIST_SEARCH_FORWARD);
  gtk_binding_entry_add_signal(binding_set, GDK_r, GDK_CONTROL_MASK,
			       "search_title", 1,
			       G_TYPE_ENUM, THREADLIST_SEARCH_BACKWARD);
#endif

  klass->open_view = NULL;

  klass->thread_mouse_over = NULL;
  klass->thread_mouse_out = NULL;

  klass->toggle_mark = NULL;
  klass->toggle_hide = NULL;
  klass->mark_thread = NULL;
  klass->advance_view = NULL;
  klass->back_view = NULL;

  klass->move_cursor = threadlist_view_move_cursor;
#if 0	/* TODO: 롼Ǽ뤷ʤäݤ */
  klass->search_title = threadlist_view_search_title;
#endif

  klass->do_on_selection = threadlist_view_do_on_selection;
  klass->do_on_thread_at_cursor = threadlist_view_do_on_thread_at_cursor;
}


typedef enum
{
  THREADLIST_RANK = 0,
  THREADLIST_MARK,
  THREADLIST_TITLE,
  THREADLIST_N_RESPONSES,
  THREADLIST_N_RESPONSES_READ,
  THREADLIST_RANK_DIFFERENCE,
  THREADLIST_DATA,
  THREADLIST_N_COLUMNS
} OchushaThreadlistItem;


static void
threadlist_view_init(ThreadlistView *view)
{
  GtkWidget *tree_view;
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;

  GTK_WIDGET_SET_FLAGS(view, GTK_CAN_FOCUS | GTK_RECEIVES_DEFAULT);

  gtk_scrolled_window_set_policy(&view->container,
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

  tree_view = gtk_tree_view_new();
  view->view = GTK_TREE_VIEW(tree_view);

  gtk_scrolled_window_set_hadjustment(&view->container, GTK_ADJUSTMENT(gtk_adjustment_new(0.0, 0.0, 0.0, 0.0, 0.0, 0.0)));
  gtk_scrolled_window_set_vadjustment(&view->container, GTK_ADJUSTMENT(gtk_adjustment_new(0.0, 0.0, 0.0, 0.0, 0.0, 0.0)));

  /* MEMO: Τ褦GtkAdjustmentͿƤʤassertion˰óݤ:
   * Gtk-CRITICAL **: file gtkrange.c: line 438 (gtk_range_get_adjustment):
   * assertion 'GTK_IS_RANGE(range)' failed
   * 2Ф롣
   */
  gtk_container_add(GTK_CONTAINER(&view->container), tree_view);

  gtk_widget_show(tree_view);

  view->selection = gtk_tree_view_get_selection(view->view);
  gtk_tree_selection_set_mode(view->selection, GTK_SELECTION_MULTIPLE);

  g_signal_connect(G_OBJECT(view->view), "motion_notify_event",
		   G_CALLBACK(tree_view_motion_notify_event_cb), view);
  g_signal_connect(G_OBJECT(view->view), "leave_notify_event",
		   G_CALLBACK(tree_view_leave_notify_event_cb), view);
  g_signal_connect(G_OBJECT(view->view), "button_press_event",
		   G_CALLBACK(tree_view_button_press_event_cb), view);
  g_signal_connect(G_OBJECT(view->view), "button_release_event",
		   G_CALLBACK(tree_view_button_release_event_cb), view);

  view->model = NULL;

  view->number_of_threads = 0;

  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(_("Rank"), renderer,
						    "text", THREADLIST_RANK,
						    NULL);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  decorate_threadlist_entry,
					  NULL, NULL);
  gtk_tree_view_append_column(view->view, column);
  view->rank_column = column;

  renderer = gtk_cell_renderer_toggle_new();
  gtk_cell_renderer_toggle_set_radio(GTK_CELL_RENDERER_TOGGLE(renderer),
				     FALSE);
  column = gtk_tree_view_column_new_with_attributes(_("Mark"), renderer,
						    "active", THREADLIST_MARK,
						    NULL);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  active_set_cell_data_flags,
					  NULL, NULL);
  gtk_tree_view_append_column(view->view, column);
  view->mark_column = column;

  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(_("Thread Title"),
						    renderer, "text",
						    THREADLIST_TITLE, NULL);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  decorate_threadlist_entry,
					  NULL, NULL);
  gtk_tree_view_append_column(view->view, column);
  view->title_column = column;

  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(_("#Res"),
						    renderer, "text",
						    THREADLIST_N_RESPONSES,
						    NULL);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  decorate_threadlist_entry,
					  NULL, NULL);
  gtk_tree_view_append_column(view->view, column);
  view->n_res_column = column;

  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(_("#Read"),
						    renderer, "text",
						    THREADLIST_N_RESPONSES_READ,
						    NULL);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  decorate_threadlist_entry,
					  NULL, NULL);
  gtk_tree_view_append_column(view->view, column);
  view->n_read_column = column;

  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(_("Rank Difference"),
						    renderer, "text",
						    THREADLIST_RANK_DIFFERENCE,
						    NULL);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  text_set_cell_data_rank_diff,
					  NULL, NULL);
  gtk_tree_view_append_column(view->view, column);
  view->rank_diff_column = column;

  gtk_tree_view_set_search_column(view->view, THREADLIST_TITLE);
  gtk_tree_view_set_search_equal_func(view->view,
				      threadlist_view_title_not_equals,
				      NULL, NULL);
  gtk_tree_view_set_enable_search(view->view, TRUE);
}


static void
threadlist_view_finalize(GObject *object)
{
  ThreadlistView *view = (ThreadlistView *)object;

#if DEBUG_WIDGET_MOST
  fprintf(stderr, "threadlist_view_finalize: view->ref_count=%d\n",
	  object->ref_count);
  fprintf(stderr, "threadlist_view_finalize: model->ref_count=%d\n",
	  G_OBJECT(view->model)->ref_count);
#endif
  if (view->model != NULL)
    {
      GObject *model = G_OBJECT(view->model);
      view->model = NULL;
      g_object_unref(model);
    }

  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


static void
threadlist_view_destroy(GtkObject *object)
{
#if DEBUG_WIDGET_MOST
  fprintf(stderr, "threadlist_view_destroy: object->ref_count=%d\n",
	  G_OBJECT(object)->ref_count);
#endif

  if (GTK_OBJECT_CLASS(parent_class)->destroy)
    (*GTK_OBJECT_CLASS(parent_class)->destroy)(object);
}


static gboolean
threadlist_view_title_not_equals(GtkTreeModel *model, gint column,
				 const gchar *key, GtkTreeIter *iter,
				 gpointer search_data)
{
  const gchar *title;
  gchar *normalized_key;
  gchar *case_normalized_key;
  gchar *normalized_title;
  gchar *case_normalized_title;
  gboolean result;
#if DEBUG_SEARCH
  char *native_string;
#endif
  /* XXX */
#if DEBUG_SEARCH
  if (key != NULL)
    {
      native_string = convert_string(utf8_to_native, key, -1);
      fprintf(stderr, "threadlist_view_title_not_equals(): column=%d, key=\"%s\":",
	      column, key);
      free(native_string);
    }
#endif
  if (key == NULL)
    return FALSE;

  gtk_tree_model_get(model, iter, THREADLIST_TITLE, &title, -1);
#if DEBUG_SEARCH
  if (title != NULL)
    {
      native_string = convert_string(utf8_to_native, title, -1);
      fprintf(stderr, " title at iter=\"%s\"\n", native_string);
      free(native_string);
    }
  else
    fprintf(stderr, " title at iter=NULL\n");
#endif
  if (title == NULL)
    return FALSE;

  normalized_key = g_utf8_normalize(key, -1, G_NORMALIZE_ALL);
  case_normalized_key = g_utf8_casefold(normalized_key, -1);

  normalized_title = g_utf8_normalize(title, -1, G_NORMALIZE_ALL);
  case_normalized_title = g_utf8_casefold(normalized_title, -1);

  result = (strstr(case_normalized_title, case_normalized_key) == NULL);

  g_free(normalized_key);
  g_free(case_normalized_key);
  g_free(normalized_title);
  g_free(case_normalized_title);

  return result;
}


static gboolean
threadlist_view_move_cursor(ThreadlistView *view, GtkMovementStep step,
			    gint count)
{
  gboolean result;

  g_signal_emit(G_OBJECT(view->view),
		tree_view_signals[TREE_VIEW_MOVE_CURSOR_SIGNAL],
		0,
		step, count,
		&result);
  return result;
}


gboolean
threadlist_view_start_interactive_search(ThreadlistView *view)
{
  gboolean result;
  g_return_val_if_fail(IS_THREADLIST_VIEW(view) && view->view != NULL, FALSE);

  gtk_widget_grab_focus(GTK_WIDGET(view->view));

  g_signal_emit(G_OBJECT(view->view),
		tree_view_signals[TREE_VIEW_START_INTERACTIVE_SEARCH_SIGNAL],
		0,
		&result);

  return result;
}


#if 0	/* TODO: 롼Ǽ뤷ʤäݤ */
static gboolean
threadlist_view_search_title(ThreadlistView *view, int direction)
{
}
#endif


typedef struct _DoOnSelectionArgs
{
  ThreadlistView *view;
  gint what_to_do;
} DoOnSelectionArgs;


static void
do_on_selection_helper(GtkTreeModel *model, GtkTreePath *path,
		       GtkTreeIter *iter, DoOnSelectionArgs *args)
{
  OchushaBBSThread *thread;
  gint what_to_do = args->what_to_do;

  gtk_tree_model_get(model, iter,
		     THREADLIST_DATA, &thread, -1);
  if (thread == NULL)
    return;	/* ͭʤǰΤ*/

  switch (what_to_do)
    {
    case POPUP_MENU_OPEN_SELECTION:
      /* 2ܰʹߤ϶Ū˥֤ǳ */
      args->what_to_do = POPUP_MENU_OPEN_SELECTION_IN_TAB;

      /* FALL THROUGH */
    case POPUP_MENU_OPEN_SELECTION_IN_TAB:
      g_signal_emit(G_OBJECT(args->view),
		    threadlist_view_signals[OPEN_VIEW_SIGNAL],
		    0,
		    thread,
		    what_to_do == POPUP_MENU_OPEN_SELECTION_IN_TAB);
      break;

    case POPUP_MENU_MARK_SELECTION:
      g_signal_emit(G_OBJECT(args->view),
		    threadlist_view_signals[MARK_THREAD_SIGNAL],
		    0,
		    thread,
		    TRUE);
	  break;

    case POPUP_MENU_UNMARK_SELECTION:
      g_signal_emit(G_OBJECT(args->view),
		    threadlist_view_signals[MARK_THREAD_SIGNAL],
		    0,
		    thread,
		    FALSE);
	  break;

    case POPUP_MENU_TOGGLE_HIDE_SELECTION:
	  g_signal_emit(G_OBJECT(args->view),
			threadlist_view_signals[TOGGLE_HIDE_SIGNAL],
			0,
			thread);
	  break;
    }

}


static void
threadlist_view_do_on_selection(ThreadlistView *view, gint what_to_do)
{
  DoOnSelectionArgs args = { view, what_to_do };
  gtk_tree_selection_selected_foreach(view->selection,
			(GtkTreeSelectionForeachFunc)do_on_selection_helper,
			&args);
}


static void
threadlist_view_do_on_thread_at_cursor(ThreadlistView *view, gint what_to_do)
{
  GtkTreeIter iter;
  GtkTreePath *path = NULL;
  gpointer data = NULL;

  gtk_tree_view_get_cursor(view->view, &path, NULL);
  if (path == NULL)
    return;

  if (!gtk_tree_model_get_iter(view->model, &iter, path))
    goto done;

  gtk_tree_model_get(view->model, &iter, THREADLIST_DATA, &data, -1);

  if (data != NULL)
    {
      gboolean result = FALSE;
      g_signal_emit(G_OBJECT(view->view),
		    tree_view_signals[TREE_VIEW_UNSELECT_ALL_SIGNAL],
		    0,
		    &result);
      g_signal_emit(G_OBJECT(view->view),
		    tree_view_signals[TREE_VIEW_SELECT_CURSOR_ROW_SIGNAL],
		    0,
		    TRUE,
		    &result);

      switch (what_to_do)
	{
	case MARK_THREAD:
	  g_signal_emit(G_OBJECT(view),
			threadlist_view_signals[MARK_THREAD_SIGNAL],
			0,
			data,
			TRUE);
	  break;

	case UNMARK_THREAD:
	  g_signal_emit(G_OBJECT(view),
			threadlist_view_signals[MARK_THREAD_SIGNAL],
			0,
			data,
			FALSE);
	  break;

	case TOGGLE_HIDDEN_STATE:
	  g_signal_emit(G_OBJECT(view),
			threadlist_view_signals[TOGGLE_HIDE_SIGNAL],
			0,
			data);
	  break;

	case ADVANCE_VIEW:
	  g_signal_emit(G_OBJECT(view),
			threadlist_view_signals[ADVANCE_VIEW_SIGNAL],
			0,
			data,
			&result);
	  break;

	case BACK_VIEW:
	  g_signal_emit(G_OBJECT(view),
			threadlist_view_signals[BACK_VIEW_SIGNAL],
			0,
			data,
			&result);
	  break;
	}
    }

 done:
  gtk_tree_path_free(path);
}


static void
popup_menu_callback(gpointer data, guint action, GtkWidget *widget)
{
  ThreadlistView *view = gtk_item_factory_popup_data(popup_menu_item_factory);
#if DEBUG_GUI_MOST
  fprintf(stderr, "popup_menu_callback: view=0x%x, action=%d\n",
	  (int)view, action);
#endif
  threadlist_view_do_on_selection(view, action);
}


static void
popup_menu_destructed_cb(gpointer data)
{
#if DEBUG_GUI_MOST
  ThreadlistView *view = THREADLIST_VIEW(data);
  fprintf(stderr, "popup_menu_destructed_cb: data=0x%x\n", (int)view);
#endif
}


GtkWidget *
threadlist_view_new(void)
{
  return GTK_WIDGET(g_object_new(THREADLIST_VIEW_TYPE, NULL));
}


static gboolean
tree_view_motion_notify_event_cb(GtkWidget *widget, GdkEventMotion *event,
				 ThreadlistView *view)
{
  GtkTreeIter iter;
  GtkTreePath *path = NULL;
  GtkTreeViewColumn *column;
  gpointer data = NULL;

  if (!gtk_tree_view_get_path_at_pos(view->view, event->x, event->y, &path,
				     &column, NULL, NULL))
    goto done;

  if (!gtk_tree_model_get_iter(view->model, &iter, path))
    goto done;

  gtk_tree_model_get(view->model, &iter, THREADLIST_DATA, &data, -1);

  if (view->recently_pointed_thread != data)
    {
      OchushaBBSThread *thread = (OchushaBBSThread *)data;

      if (view->recently_pointed_thread != NULL)
	g_signal_emit(G_OBJECT(view),
		      threadlist_view_signals[THREAD_MOUSE_OUT_SIGNAL],
		      0,
		      view->recently_pointed_thread);

      view->recently_pointed_thread = thread;

      if (thread != NULL)
	g_signal_emit(G_OBJECT(view),
		      threadlist_view_signals[THREAD_MOUSE_OVER_SIGNAL],
		      0,
		      thread);
    }

 done:
  if (path != NULL)
    gtk_tree_path_free(path);

  return FALSE;
}


static gboolean
tree_view_leave_notify_event_cb(GtkWidget *widget, GdkEventButton *event,
				ThreadlistView *view)
{
  if (view->recently_pointed_thread != NULL)
    {
      g_signal_emit(G_OBJECT(view),
		    threadlist_view_signals[THREAD_MOUSE_OUT_SIGNAL],
		    0,
		    view->recently_pointed_thread);

      view->recently_pointed_thread = NULL;
    }

  return FALSE;
}


static gboolean
tree_view_button_press_event_cb(GtkWidget *widget, GdkEventButton *event,
				ThreadlistView *view)
{
  GtkTreeIter iter;
  GtkTreePath *path = NULL;
  GtkTreeViewColumn *column;
  gpointer data = NULL;
  gboolean result = FALSE;

  if (!gtk_tree_view_get_path_at_pos(view->view, event->x, event->y, &path,
				     &column, NULL, NULL))
    goto done;

  if (!gtk_tree_model_get_iter(view->model, &iter, path))
    goto done;

  gtk_tree_model_get(view->model, &iter, THREADLIST_DATA, &data, -1);

  if (data != NULL)
    {
      if (event->button == 3)
	{
	  gtk_item_factory_popup_with_data(popup_menu_item_factory,
					   view, popup_menu_destructed_cb,
					   event->x_root, event->y_root,
					   event->button,
					   gtk_get_current_event_time());
	  if (gtk_tree_selection_iter_is_selected(view->selection, &iter))
	    result = TRUE;
	}
      else if ((event->state & (GDK_CONTROL_MASK | GDK_SHIFT_MASK)) == 0)
	{
	  if (column == view->mark_column)
	    {
	      g_signal_emit(G_OBJECT(view),
			    threadlist_view_signals[TOGGLE_MARK_SIGNAL],
			    0,
			    data);
	    }
	  else
	    {
	      g_signal_emit(G_OBJECT(view),
			    threadlist_view_signals[OPEN_VIEW_SIGNAL],
			    0,
			    data,
			    event->button == 2);
	    }
	}
    }

 done:
  if (path != NULL)
    gtk_tree_path_free(path);

  return result;
}


static gboolean
tree_view_button_release_event_cb(GtkWidget *widget, GdkEventButton *event,
				  ThreadlistView *view)
{
  return FALSE;
}


#define RANK_UNSPECIFIED		-1
#define RANK_DAT_DROPPED		-2

#define RANK_NOT_CHANGED		0
#define RANK_NEW_COMER_MAGIC		0x7fffffff
#define RANK_SAME_RANK_DOWN_MAGIC	0x7ffffffe


static void
set_thread_values(ThreadlistView *view, GtkListStore *store,
		  int rank, OchushaBBSThread *thread)
{
  ThreadlistInfo *info = thread->user_data;

  if (rank != RANK_UNSPECIFIED)
    {
      gint rank_diff;
      if (rank == RANK_DAT_DROPPED)
	rank_diff = RANK_NOT_CHANGED;
      else
	rank_diff = (info->view_rank == 0
		     ? RANK_NEW_COMER_MAGIC : info->view_rank - rank);
      if (view->last_rank_diff != RANK_NEW_COMER_MAGIC
	  && view->last_rank_diff != RANK_NOT_CHANGED
	  && view->last_rank_diff == rank_diff
	  && rank_diff < 0)
	rank_diff = RANK_SAME_RANK_DOWN_MAGIC;
      else
	view->last_rank_diff = rank_diff;

      gtk_list_store_set(store, &info->iter,
		THREADLIST_RANK, rank,
		THREADLIST_MARK, info->view_flags,
		THREADLIST_TITLE, thread->title,
		THREADLIST_N_RESPONSES, thread->number_of_responses_on_server,
		THREADLIST_N_RESPONSES_READ, thread->number_of_responses_read,
		THREADLIST_RANK_DIFFERENCE, rank_diff,
		THREADLIST_DATA, thread,
		-1);
    }
  else
    {
      gtk_list_store_set(store, &info->iter,
		THREADLIST_TITLE, thread->title,
		THREADLIST_MARK, info->view_flags,
		THREADLIST_N_RESPONSES, thread->number_of_responses_on_server,
		THREADLIST_N_RESPONSES_READ, thread->number_of_responses_read,
		THREADLIST_DATA, thread,
		-1);
    }
}


static void
decorate_threadlist_entry(GtkTreeViewColumn *tree_column,
			  GtkCellRenderer *renderer, GtkTreeModel *tree_model,
			  GtkTreeIter *iter, gpointer unused)
{
  GValue flags = { 0, };
  GValue bg = { 0, };

  g_return_if_fail(gtk_list_store_iter_is_valid(GTK_LIST_STORE(tree_model),
						iter));
  gtk_tree_model_get_value(tree_model, iter, THREADLIST_MARK, &flags);
  g_value_init(&bg, G_TYPE_STRING);
  if (g_value_get_int(&flags) & BBS_THREAD_HIGHLIGHT)
    g_value_set_static_string(&bg, "PowderBlue");
  else
    g_value_set_static_string(&bg, NULL);

  g_object_set_property((GObject *)renderer, "cell_background", &bg);

  if (!GTK_IS_CELL_RENDERER_TEXT(renderer))
    return;

  if (g_value_get_int(&flags) & BBS_THREAD_HIDDEN)
    {
      GValue st = { 0, };
      g_value_init(&st, G_TYPE_BOOLEAN);
      g_value_set_boolean(&st, TRUE);
      g_object_set_property((GObject *)renderer, "strikethrough", &st);
      g_object_set_property((GObject *)renderer, "strikethrough_set", &st);
    }
  else
    {
      GValue st = { 0, };
      g_value_init(&st, G_TYPE_BOOLEAN);
      g_value_set_boolean(&st, FALSE);
      g_object_set_property((GObject *)renderer, "strikethrough", &st);
      g_object_set_property((GObject *)renderer, "strikethrough_set", &st);
    }
}


static void
active_set_cell_data_flags(GtkTreeViewColumn *tree_column,
			   GtkCellRenderer *renderer, GtkTreeModel *tree_model,
			   GtkTreeIter *iter, gpointer unused)
{
  GValue flags = { 0, };
  GtkCellRendererToggle *toggle_renderer = GTK_CELL_RENDERER_TOGGLE(renderer);
  g_return_if_fail(gtk_list_store_iter_is_valid(GTK_LIST_STORE(tree_model),
						iter));

  gtk_tree_model_get_value(tree_model, iter, THREADLIST_MARK, &flags);

  gtk_cell_renderer_toggle_set_active(toggle_renderer,
				g_value_get_int(&flags) & BBS_THREAD_FAVORITE);
  decorate_threadlist_entry(tree_column, renderer, tree_model, iter, unused);
}


static void
text_set_cell_data_rank_diff(GtkTreeViewColumn *tree_column,
			     GtkCellRenderer *renderer,
			     GtkTreeModel *tree_model, GtkTreeIter *iter,
			     gpointer unused)
{
  GValue src = { 0, };
  GValue value = { 0, };
  int diff;
  gchar buffer[512];

  g_return_if_fail(gtk_list_store_iter_is_valid(GTK_LIST_STORE(tree_model),
						iter));

  gtk_tree_model_get_value(tree_model, iter, THREADLIST_RANK_DIFFERENCE, &src);
  diff = g_value_get_int(&src);

  g_value_init(&value, G_TYPE_STRING);
  if (diff == RANK_NOT_CHANGED)
    {
      g_value_set_static_string(&value, "");
    }
  else if (diff == RANK_NEW_COMER_MAGIC)
    {
      /* оξ */
      g_value_set_static_string(&value, _("New Face!"));
    }
  else if (diff == RANK_SAME_RANK_DOWN_MAGIC)
    {
      /* ľΥȥƱ̥ξ */
      g_value_set_static_string(&value, _("Down"));
    }
  else if (diff > 0)
    {
      /* ̥åפξ */
      snprintf(buffer, 512, _("%d Rank Up"), diff);
      g_value_set_string(&value, buffer);
    }
  else
    {
      /* ̥ξ */
      snprintf(buffer, 512, _("%d Rank Down"), -diff);
      g_value_set_string(&value, buffer);
    }

  g_object_set_property((GObject *)renderer, "text", &value);
  decorate_threadlist_entry(tree_column, renderer, tree_model, iter, unused);
}


gboolean
threadlist_view_open(ThreadlistView *view, ThreadlistInfoNewFunc *factory)
{
  GtkListStore *store;

  g_return_val_if_fail(IS_THREADLIST_VIEW(view)
		       && !view->model_is_opened, FALSE);

  if (view->model != NULL)
    g_object_unref(G_OBJECT(view->model));

  store = gtk_list_store_new(THREADLIST_N_COLUMNS,
			     G_TYPE_INT,	/* rank */
			     G_TYPE_INT,	/* flags */
			     G_TYPE_STRING,	/* thread title */
			     G_TYPE_INT,	/* number of responses read */
			     G_TYPE_INT,	/* number of responses read */
			     G_TYPE_INT,	/* rank difference */
			     G_TYPE_POINTER);	/* data (OchushaBBSThread *) */

  gtk_tree_view_set_model(view->view, GTK_TREE_MODEL(store));
  view->model = GTK_TREE_MODEL(store);
  view->threadlist_info_new = factory;

  view->number_of_threads = 0;
  view->last_rank_diff = 0;

  return TRUE;
}


void
threadlist_view_close(ThreadlistView *view)
{
  g_return_if_fail(IS_THREADLIST_VIEW(view));
  view->model_is_opened = FALSE;
}


void
threadlist_view_append_thread(ThreadlistView *view, OchushaBBSThread *thread,
			      gint rank)
{
  ThreadlistInfo *info = thread->user_data;

  g_return_if_fail(IS_THREADLIST_VIEW(view));
  g_assert(view->model != NULL);

  if (info == NULL)
    {
      if (view->threadlist_info_new != NULL)
	info = (*view->threadlist_info_new)(thread);
      else
	info = (ThreadlistInfo *)calloc(1, sizeof(ThreadlistInfo));
      thread->user_data = info;
    }

  gtk_list_store_append(GTK_LIST_STORE(view->model), &info->iter);
  info->threadlist_view = view;
  set_thread_values(view, GTK_LIST_STORE(view->model), rank, thread);
}


void
threadlist_view_update_thread(ThreadlistView *view, OchushaBBSThread *thread)
{
  gpointer data = NULL;
  ThreadlistInfo *info;
  g_return_if_fail(IS_THREADLIST_VIEW(view) && thread != NULL
		   && thread->user_data != NULL);

  info = (ThreadlistInfo *)thread->user_data;

  if (info->threadlist_view != view)
    {
#if DEBUG_WIDGET
      fprintf(stderr, "info->threadlist_view=0x%x, view=0x%x\n",
	      (int)info->threadlist_view, (int)view);
#endif
      return;
    }

  if (!gtk_list_store_iter_is_valid(GTK_LIST_STORE(view->model), &info->iter))
    {
      /* 줿ե륿ξ֤ѤäơiterŤʤäȤ롣 */
      return;
    }

  gtk_tree_model_get(view->model, &info->iter,
		     THREADLIST_DATA, &data, -1);

#if DEBUG_WIDGET
  if (thread != data)
    fprintf(stderr, "inconsistent ThreadlistView?\n");
#endif
  g_return_if_fail(thread == data);

  set_thread_values(view, GTK_LIST_STORE(view->model), RANK_UNSPECIFIED,
		    thread);
}


void
threadlist_view_scroll_to_thread(ThreadlistView *view,
				 OchushaBBSThread *thread)
{
  gpointer data = NULL;
  GtkTreePath *path;
  ThreadlistInfo *info;
  g_return_if_fail(IS_THREADLIST_VIEW(view) && thread != NULL);

  info = (ThreadlistInfo *)thread->user_data;
  g_return_if_fail(info != NULL);

  if (info->threadlist_view != view
      || !gtk_list_store_iter_is_valid(GTK_LIST_STORE(view->model),
				       &info->iter))
    return;

  gtk_tree_model_get(view->model, &info->iter,
		     THREADLIST_DATA, &data, -1);
  g_return_if_fail(thread == data);

  path = gtk_tree_model_get_path(view->model, &info->iter);
  gtk_tree_view_scroll_to_cell(view->view, path, NULL, FALSE, 0.0f, 0.0f);
  gtk_tree_view_set_cursor_on_cell(view->view, path, NULL, NULL, FALSE);
  gtk_tree_path_free(path);
}

