/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ochusha_utils_2ch.c,v 1.1.1.1 2003/05/10 16:34:28 fuyu Exp $
 */

#include "ochusha.h"
#include "ochusha_private.h"

#include "ochusha_bbs_table.h"
#include "ochusha_board_2ch.h"
#include "ochusha_utils_2ch.h"

#include <glib.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


#define GUESS_2CH_BOARD_URL	1


OchushaAsyncBuffer *
ochusha_utils_2ch_get_bbsmenu(OchushaConfig *conf,
			      OchushaNetworkCallbacks *callbacks,
			      gpointer callback_data)
{
  return ochusha_read_from_url(conf, conf->bbsmenu_url, NULL,
			       OCHUSHA_CACHE_IGNORE, callbacks, callback_data);
}


static gboolean
is_2ch_board_url(char *url)
{
  if (strstr(url, ".html") != NULL)
    return FALSE;
  if (strstr(url, "del_2ch") != NULL)
    return FALSE;
  return TRUE;
}


static gboolean
extract_boards(OchushaBBSTable *table, OchushaBoardCategory *category,
	       iconv_t converter,
	       char *head, char *tail,
	       EachBoardCallback *each_board_callback,
	       BoardMovedCallback *board_moved_callback,
	       gpointer callback_data)
{
  /* ǽ(HTMLA)õ*/
  char *cur_pos = g_strstr_len(head, tail - head, "<A HREF=http");
  if (cur_pos == NULL)
    return FALSE;
  cur_pos += 8;	/* skip "<A HREF=" */

  while (cur_pos != NULL && cur_pos < tail)
    {
      char *tag_tail = memchr(cur_pos, '>', tail - cur_pos);
      char *url_tail;
      gchar *name = NULL;
      char *url = NULL;
      char *close_tag;
      OchushaBulletinBoard *board;
      OchushaBulletinBoard *board_by_name;

      if (tag_tail == NULL)
	return FALSE;	/* Ƥ롩 */

      url_tail = strpbrk(cur_pos, " \t\r\n>");
      /* tag_tail != NULLʤΤurl_tail != NULL*/

      close_tag = g_strstr_len(url_tail, tail - url_tail, "</A>");
      if (close_tag == NULL)
	return FALSE;	/* Ƥ롩 */

#if GUESS_2CH_BOARD_URL
      {
	/* 2chİʳؤΥ󥫡ˤTARGE°դƤ̣*/
	char *target = g_strstr_len(cur_pos, close_tag - cur_pos, "TARGET=");
	if (target != NULL)
	  goto search_next;
      }
#endif
      name = convert_string(converter, tag_tail + 1, close_tag - tag_tail - 1);
      url = g_strndup(cur_pos, url_tail - cur_pos);
#if GUESS_2CH_BOARD_URL
      if (!is_2ch_board_url(url))
	goto search_next;
#endif
      board = ochusha_bbs_table_lookup_board_by_url(table, url);
      board_by_name = ochusha_bbs_table_lookup_board_by_name(table, name);

      if (board == NULL)
	{
#if GUESS_2CH_BOARD_URL
	  const char *base_path;
#endif
	  if (name == NULL)
	    {
	      g_free(url);
	      return FALSE;	/* Out of memory */
	    }

	  board = ochusha_board_2ch_new(name, url);

#if GUESS_2CH_BOARD_URL
	  base_path = ochusha_bulletin_board_get_base_path(board);
	  /* ФΥ롼ľηǼĤϸʤġġ*/
	  if (base_path[0] == '\0'
	      || (base_path[0] == '/' && base_path[1] == '\0'))
	    {
	      g_object_unref(G_OBJECT(board));
	      board = NULL;
	      goto search_next;
	    }
#endif

	  if (board_by_name != NULL && board_moved_callback != NULL)
	    {
	      OchushaBulletinBoard *winner
		= (*board_moved_callback)(board_by_name, board, callback_data);
	      /* XXX:
	       *   winnerNULLʤ()boardɲá
	       *   winnerboard_by_nameʤ()boardΤƤƼء
	       *   winnerboardʤ(Ť)board_by_nameΤƤơ
	       *   ()boardɲäפˡƥ꤫board_by_name
	       *   
	       */
	      if (winner == board_by_name)
		{
		  /* boardϥơ֥ɲäʤ */
		  g_object_unref(G_OBJECT(board));
		  board = NULL;
		  goto search_next;
		}

	      if (winner != NULL)
		{
		  /* (winner == board)ȸʤ */
		  ochusha_board_category_remove_board(category, board_by_name);
		  ochusha_bbs_table_remove_board(table, board_by_name);
		}
	    }

	  if (board != NULL)
	    {
	      if (each_board_callback != NULL)
		{
		  gboolean result = (*each_board_callback)(board,
							   callback_data);
		  if (!result)
		    {
		      g_object_unref(G_OBJECT(board));
		      return FALSE;
		    }
		}
	      ochusha_bbs_table_add_board(table, board);
	    }
	}
      
      ochusha_board_category_add_board(category, board);

    search_next:
      if (url != NULL)
	g_free(url);
      if (name != NULL)
	g_free(name);
      /* (HTMLA)õ*/
      cur_pos = g_strstr_len(close_tag + 4, tail - close_tag, "<A HREF=http");
      if (cur_pos == NULL)
	return TRUE;
      cur_pos += 8;	/* skip "<A HREF=" */
    }
  return TRUE;
}


/*
 * ochusha_utils_2ch_analyze_bbsmenu
 *
 * Ϳ줿bufferbbsmenu.htmlǤȸʤƲϤ˴ޤޤ
 * ƥƥбOchushaBoardCategoryȳĤб
 * OchushaBulletinBoard(Υ֥饹Υ֥OchushaBoard2ch)ۤ
 * Ϳ줿OchushaBBSTable򹹿롣
 *
 * ޤ˥ХåؿͿ줿硢ƥƥꡢĤˤĤ
 * б륳ХåؿƤ֡
 * ХåؿFALSE֤硢ǲϤλ롣
 *
 * OchushaBBSTableURLOchushaBulletinBoardʤäˤؤ餺
 * ̾Ǥϰ褦ʾ硢İžεǻʤΤǤ˴ؤƤ⥳
 * Хåؿ(ͿƤ)Ƥ֡ХåؿϰȤƥݥ
 * Ϳ뿷ĤOchushaBulletinBoard⡢OchushaBBSTable˻Ĥ
 * ؤΥݥ󥿤֤ޤNULL֤ˤϿξĤ롣
 *
 * Ϥ˽λTRUE֤
 */
gboolean
ochusha_utils_2ch_analyze_bbsmenu(OchushaBBSTable *table,
				  OchushaAsyncBuffer *buffer,
				  EachCategoryCallback *category_callback,
				  EachBoardCallback *board_callback,
				  BoardMovedCallback *moved_callback,
				  gpointer callback_data)
{
  gboolean result = TRUE;
  iconv_t converter;

  g_return_val_if_fail(OCHUSHA_IS_BBS_TABLE(table) && buffer != NULL, FALSE);

  converter = iconv_open("UTF-8//IGNORE", "CP932");  /* ϡɥǥ */
  g_return_val_if_fail(converter != (iconv_t)-1, FALSE);

  if (!ochusha_async_buffer_active_ref(buffer, "ochusha_utils_2ch.c: ochusha_utils_2ch_analyze_bbsmenu"))
    {
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "buffer has been terminated.\n");
#endif
      iconv_close(converter);
      return FALSE;
    }

  ochusha_async_buffer_lock(buffer);
  {
    unsigned int offset = 0;

    while (result)
      {
	char *buffer_top = (char *)buffer->buffer;
	char *cur_pos = buffer_top + offset;
	unsigned int length = buffer->length;
	unsigned int rest_of_data = length - offset;

	while (rest_of_data > 0)
	  {
	    char *end_name_pos;
	    gchar *category_name;
	    char *end_category_pos;
	    OchushaBoardCategory *category;

	    cur_pos = g_strstr_len(cur_pos, rest_of_data, "<B>");

	    if (cur_pos == NULL)
	      break;	/* ǡ­ʤ⤦ƥ꤬ʤ */
	    cur_pos += 3;	/* skip "<B>" */
	    rest_of_data -= 3;

	    end_name_pos = g_strstr_len(cur_pos, rest_of_data, "</B>");

	    if (end_name_pos == NULL)
	      break;	/* ǡ­ʤ */

	    /* ߤΥƥϼΥƥľޤ */
	    end_category_pos = g_strstr_len(end_name_pos + 4,
					    rest_of_data - (end_name_pos
							    - cur_pos),
					    "<B>");
	    /* ǸΥƥ</BODY>ľޤ */
	    if (end_category_pos == NULL)
	      end_category_pos = g_strstr_len(end_name_pos + 4,
					      rest_of_data - (end_name_pos
							      - cur_pos),
					      "</BODY>");
	    if (end_category_pos == NULL)
	      break;

	    category_name = convert_string(converter, cur_pos,
					   end_name_pos - cur_pos);

	    category = ochusha_bbs_table_lookup_category(table, category_name);
	    if (category == NULL)
	      category = ochusha_board_category_new(category_name);
	    g_free(category_name);

	    cur_pos = end_name_pos + 4;	/* skip "</B>" */
	    rest_of_data -= 4;
	      
	    result = extract_boards(table, category, converter,
				    cur_pos, end_category_pos,
				    board_callback, moved_callback,
				    callback_data);

	    if (!result || category->board_list == NULL)
	      g_object_unref(G_OBJECT(category));
	    else
	      {
		ochusha_bbs_table_add_category(table, category);
		if (category_callback != NULL
		    && !(*category_callback)(category, callback_data))
		  {
		    result = FALSE;
		    break;
		  }
	      }

	    cur_pos = end_category_pos;
	    offset = cur_pos - buffer_top;
	    rest_of_data = (length - offset);
	  }

	if (buffer->fixed)
	  break;

	if (!ochusha_async_buffer_wait(buffer, "ochusha_utils_2ch.c: ochusha_utils_2ch_analyze_bbsmenu"))
	  {
	    /* bufferϤ޲٤callerˤޤ */
#if DEBUG_ASYNC_BUFFER_MOST
	    fprintf(stderr, "ochusha_utils_2ch_analyze_bbsmenu(): buffer has been terminated.\n");
#endif
	    result = FALSE;
	    break;
	  }

	if (((OchushaNetworkStatus *)buffer->user_data)->state
	    == OCHUSHA_CACHE_IS_DIRTY)
	  {
	    result = FALSE;
	    break;
	  }
      }
  }
  ochusha_async_buffer_unlock(buffer);

  ochusha_async_buffer_active_unref(buffer, "ochusha_utils_2ch.c: ochusha_utils_2ch_analyze_bbs_menu");

  iconv_close(converter);

  return result;
}


/*
 * url2chĤ⤷ϥؤURLȸʤƲϤ롣
 *
 * Ϸ̤Ȥơ
 * бĤɽOchushaBulletinBoardؤΥݥ󥿤*board_pˡ
 * ֹ*thread_number_p
 * ɽOchushaBBSThreadؤΥݥ󥿤*thread_pˡ
 * 쥹ֻ꤬硢ˡƬ*from_p*to_p˳Ǽ롣
 * ˲äơURLʸΥåɻʬʸΥԡؤΥݥ
 * *thread_part_p˳Ǽ롣
 *
 * caller*thread_partȤäfree뤳ȡ
 *
 * ǤˤĤƤ⡢Ϳ줿ݥ󥿤NULLξˤϷ̤ϳǼ
 * ʤ
 *
 * ϤˤTRUE֤URL2chĤ⤷ϥURL̵
 * ˤFALSE֤
 */
gboolean
ochusha_utils_2ch_parse_url(OchushaBBSTable *table, const char *url,
			    OchushaBulletinBoard **board_p,
			    unsigned int *thread_number_p,
			    OchushaBBSThread **thread_p,
			    unsigned int *from_p, unsigned int *to_p,
			    char **thread_part_p)
{
  char tmp_buf[PATH_MAX];
  char *server;
  char *path;
  char *base_path;
  char *tmp_pos = NULL;
  char tmp_char;
  OchushaBulletinBoard *board = NULL;
  OchushaBBSThread *thread = NULL;
  int thread_number = 0;
  int from = 0;
  int to = 0;
  int count;

  gboolean result = FALSE;

  g_return_val_if_fail(OCHUSHA_IS_BBS_TABLE(table) && url != NULL, FALSE);

  server = ochusha_utils_url_extract_http_server(url);
  path = ochusha_utils_url_extract_http_absolute_path(url);

  if (path == NULL || strncmp(path, "/test/read.cgi/", 15) != 0)
    goto done;

  base_path = path + 15;
  tmp_pos = strchr(base_path, '/');
  if (tmp_pos == NULL)
    goto done;

  tmp_char = tmp_pos[1];
  tmp_pos[1] = '\0';
  if (snprintf(tmp_buf, PATH_MAX, "http://%s/%s", server, base_path)
      >= PATH_MAX)
    goto done;

  board = ochusha_bbs_table_lookup_board_by_url(table, tmp_buf);
  if (board != NULL)
    result = TRUE;

  tmp_pos[1] = tmp_char;

  count = sscanf(tmp_pos + 1, "%u/%u-%u", &thread_number, &from, &to);

  if (thread_number < 0)
    {
      /* ʬäȤ */
      thread_number = 0;
      from = 0;
      to = 0;
      result = FALSE;
    }
  else
    {
      if (board != NULL)
	{
	  snprintf(tmp_buf, PATH_MAX, "%u.dat", thread_number);
	  thread = ochusha_bulletin_board_lookup_bbs_thread_by_id(board,
								  tmp_buf);
	}
      if (from < 0)
	{
	  to = -from;
	  from = 1;
	}
      else if (to < 0)
	{
	  to = -to;
	}
    }

 done:
  if (server != NULL)
    free(server);

  if (path != NULL)
    free(path);

  if (result)
    {
      if (board_p != NULL)
	*board_p = board;
      if (thread_number_p != NULL)
	*thread_number_p = thread_number;
      if (thread_p != NULL)
	*thread_p = thread;
      if (from_p != NULL)
	*from_p = from;
      if (to_p != NULL)
	*to_p = to;
      if (thread_part_p != NULL)
	*thread_part_p = g_strdup(tmp_pos + 1);
    }

  return result;
}
