/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: utils.c,v 1.4 2003/06/07 14:35:32 fuyu Exp $
 */

#define OCHUSHA_UTILS_IMPLEMENTATION

#include "config.h"

#include "ochusha_private.h"
#include "utils.h"

#include <errno.h>
#include <iconv.h>
#if HAVE_LANGINFO_CODESET
#include <langinfo.h>
#endif

#include <fcntl.h>
#include <limits.h>

#if TRACE_MEMORY_USAGE
#include <pthread.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

iconv_t utf8_to_native = 0;	/* ǥХå */

#define DEFAULT_BUFFER_SIZE 4096
#define MAX_HOSTNAME_LEN 256

#define DEBUG_UTILS	0


/*
 * initialize_common_convertes
 *
 * ɤȤencodingconverter롣
 *
 * MEMO: converterϥåɥդʤȤա
 */
void
initialize_common_converters(void)
{
  if (utf8_to_native == 0)
    {
#if HAVE_LANGINFO_CODESET
      char codeset[256];
      int len = snprintf(codeset, 256, "%s//IGNORE", nl_langinfo(CODESET));
      if (len < 256)
	{
	  utf8_to_native = iconv_open(codeset, "UTF-8");
	  if (utf8_to_native == (iconv_t)-1)
	    {
	      fprintf(stderr, "Couldn't iconv_open(\"%s\", \"UTF-8\")\n",
		      codeset);
	      utf8_to_native = 0;	/* try hardcoded one */
	    }
	}

      if (utf8_to_native == 0)
	utf8_to_native = iconv_open("EUC-JP//IGNORE", "UTF-8");
#else
      utf8_to_native = iconv_open("EUC-JP//IGNORE", "UTF-8");
#endif
      if (utf8_to_native == (iconv_t)-1)
	{
	  fprintf(stderr, "iconv_open() failed.\n");
	  exit(1);
	}
    }
}


/*
 * convert_string
 *
 * iconvȤäsrcʸencodingѴѴ줿ʸ֤
 * Ѵ줿ʸheapmalloc줿ΰ˳ǼΤǡ桼
 * פˤʤ꼡freeǤ롣
 * Ϳ줿length-1ξ硢srcNULüCʸȤƼ갷졢
 * strlen(src)η̤ʸĹȤƼ갷롣ʳξ硢
 * Ϳ줿lengthĹʬѴ롣
 */
char *
convert_string(iconv_t converter, const char *src, int length)
{
  char default_buffer[DEFAULT_BUFFER_SIZE];
  size_t buffer_size = DEFAULT_BUFFER_SIZE;
  char *buffer = default_buffer;
  const char *inbuf;
  char *outbuf;
  size_t outbytesleft;
  size_t inbytesleft;

  if (length < 0)
    length = strlen(src);

  while (1)
    {
      size_t size;

      inbuf = src;
      outbuf = buffer;
      outbytesleft = buffer_size;
      inbytesleft = length;

      size = iconv(converter, &inbuf, &inbytesleft, &outbuf, &outbytesleft);
      if (size == -1)
	{
	  if (errno == E2BIG)
	    {
	      /* allocate 2 times larger buffer in the heap. */
	      buffer_size *= 2;
	      if (buffer != default_buffer)
		buffer = (char *)G_REALLOC(buffer, buffer_size);
	      else
		buffer = (char *)G_MALLOC(buffer_size);
	    }
	  else
	    {
	      fprintf(stderr, "iconv failed(errno = %d).\n", errno);
	      fprintf(stderr, "character code = 0x%02x\n", (int)*inbuf);
	      if (buffer != default_buffer)
		G_FREE(buffer);
	      return NULL;
	    }
	}
      else
	break;	/* ｪλ */
    };

  if (outbytesleft == 0)
    {
      if (buffer != default_buffer)
	buffer = (char *)G_REALLOC(buffer, buffer_size + 1);
      else
	{
	  buffer = (char *)G_MALLOC(buffer_size + 1);
	  memcpy(buffer, src, buffer_size);
	}
      buffer[buffer_size] = '\0';
      return buffer;
    }

  *outbuf = '\0';

  if (buffer == default_buffer)
    return G_STRDUP(buffer);

  return (char *)G_REALLOC(buffer, outbuf - buffer + 1);
}


/*
 * wipe_string:
 *   string˴ޤޤnon-printableʸȥĤζʸ
 *   ԡ֤
 */
gchar *
wipe_string(const gchar *string)
{
  gchar *result_string = g_strchomp(G_STRDUP(string));
  gchar *head = result_string;

  while (*head != '\0')
    {
      if (*head > '\0' && *head <= '\37')
	{	/* ȯˤĤʬФ */
	  int len;
	  gchar *tmp_pos;
	  for (tmp_pos = g_utf8_next_char(head);
	       *tmp_pos > '\0' && *tmp_pos <= '\37';
	       tmp_pos = g_utf8_next_char(tmp_pos));
	  len = strlen(tmp_pos);
	  memmove(head, tmp_pos, len + 1);
	}
      else
	head = g_utf8_next_char(head);
    }
  return result_string;
}


/*
 * ochusha_utils_url_extract_http_server
 *
 * Ϳ줿ʸhttpФURLǤȤƥ̾ʬ֤
 * ˼ԤˤNULL֤
 */
char *
ochusha_utils_url_extract_http_server(const char *url)
{
  /*
   * ݤʤΤǡurlhttp://server/.*ʷǤȷǤġ
   */
  char *cur_pos = strstr(url, "http://");
  char *hostpart_tail;
  int length;
  char buffer[MAX_HOSTNAME_LEN];

  if (cur_pos == NULL)
    {
#if DEBUG_UTILS
      fprintf(stderr, "Not a http URL: %s\n", url);
#endif
      return NULL;
    }
  cur_pos += 7;	/* skip "http://" */

  hostpart_tail = strchr(cur_pos, '/');
  if (hostpart_tail == NULL)
    {
      if (*cur_pos != '\n')
	return G_STRDUP(cur_pos);
      else
	return NULL;
    }

  length = hostpart_tail - cur_pos;
  if (length <= 0 || length >= MAX_HOSTNAME_LEN)
    {
#if DEBUG_UTILS
      fprintf(stderr, "Couldn't parse URL: %s\n", url);
#endif
      return NULL;
    }

  memcpy(buffer, cur_pos, length);
  buffer[length] = '\0';
  return G_STRDUP(buffer);
}


/*
 * ochusha_utils_url_extract_http_absolute_path
 *
 * Ϳ줿ʸhttpФURLǤȤƲϤ
 * ɥȤΥФˤХѥ֤̾
 * ex) http://pc.2ch.net/unix/  /unix/
 * ˼ԤˤNULL֤
 */
char *
ochusha_utils_url_extract_http_absolute_path(const char *url)
{
  /*
   * ݤʤΤǡurlhttp://server/.*ʷǤȷǤġ
   */
  char *cur_pos = strstr(url, "http://");

  if (cur_pos == NULL)
    {
#if DEBUG_UTILS
      fprintf(stderr, "Couldn't parse URL: %s\n", url);
#endif
      return NULL;
    }
  cur_pos += 7; /* skip "http://" */
  cur_pos = strchr(cur_pos, '/');
  if (cur_pos == NULL)
    {
#if DEBUG_UTILS
      fprintf(stderr, "Couldn't get path parts: %s\n", url);
#endif
      return NULL;
    }

  return G_STRDUP(cur_pos);
}


char *
ochusha_utils_url_encode_string(const char *src)
{
  int src_len = strlen(src);
  char *buffer = (char *)G_MALLOC(src_len * 3 + 1);
  const unsigned char *cur_src = src;
  char *cur_dst = buffer;
  while (*cur_src != '\0')
    {
      sprintf(cur_dst, "%%%02X", *cur_src++);
      cur_dst += 3;
    }
  *cur_dst = '\0';

  return buffer;
}


char *
ochusha_utils_url_decode_string(const char *src)
{
  int src_len = strlen(src);
  char *buffer = (char *)G_MALLOC(src_len + 1);
  const char *cur_src = src;
  char *cur_dst = buffer;

  while (*cur_src != '\0')
    {
      if (*cur_src != '%')
	*cur_dst++ = *cur_src++;
      else
	{
	  unsigned int code;
	  int result = sscanf(cur_src, "%%%02x", &code);
	  if (result == 1)
	    {
	      *cur_dst++ = (char)code;
	      cur_src += 3;
	    }
	  else
	    *cur_dst++ = *cur_src++;
	}
    }
  *cur_dst = '\0';
  return buffer;
}


/*
 * mkdir_p
 *
 * mkdir -pޥɤΤ褦ˡpathǻꤵ줿ǥ쥯ȥŪ
 * 롣
 * ǥ쥯ȥ꤬¸ߤ뤫ǥ쥯ȥκˤ1
 * ʳξ0֤
 */
int
mkdir_p(const char *path)
{
  char buffer[PATH_MAX];
  char *cur_pos;

  if (path == NULL)
    return 0;

  strncpy(buffer, path, PATH_MAX);

  cur_pos = buffer;
  while (1)
    {
      int result;
      struct stat sb;
      char tmp_char;

      cur_pos = strchr(cur_pos, '/');
      if (cur_pos != NULL)
	{
	  tmp_char = *(++cur_pos);
	  *cur_pos = '\0';
	}
      else
	tmp_char = '\0';

      result = stat(buffer, &sb);

      if (result == -1)
	{
	  if (errno != ENOENT)
	    return 0;
	
	  result = mkdir(buffer, 0x1c0);
	  if (result != 0)
	    return 0;
	}
      else
	if (!S_ISDIR(sb.st_mode))
	  return 0;

      if (tmp_char == '\0')
	return 1;

      *cur_pos = tmp_char;
    }
}


#if TRACE_MEMORY_USAGE
/*
 * ꡼åϢ
 */
pthread_mutex_t memory_trace_lock;
GHashTable *memory_trace;


#define MEMORY_TRACE_LOCK				\
  if (pthread_mutex_lock(&memory_trace_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't lock a mutex.\n");	\
      abort();						\
    }

#define MEMORY_TRACE_UNLOCK				\
  if (pthread_mutex_unlock(&memory_trace_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't unlock a mutex.\n");	\
      abort();						\
    }


void
init_memory_trace(void)
{
  if (pthread_mutex_init(&memory_trace_lock, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a mutex.\n");
      abort();
    }

  memory_trace = g_hash_table_new_full(g_direct_hash, g_direct_equal,
				       NULL, (GDestroyNotify)g_free);
}


void
register_pointer(gpointer pointer, const char *where)
{
  gpointer previous;
  MEMORY_TRACE_LOCK
    {
      previous = g_hash_table_lookup(memory_trace, pointer);
      g_hash_table_insert(memory_trace, pointer, g_strdup(where));
    }
  MEMORY_TRACE_UNLOCK;

  if (previous != NULL)
    fprintf(stderr, "%p has already been registered: \"%s\" -> \"%s\"\n",
	    pointer, (char *)previous, where);
}


void
unregister_pointer(gpointer pointer, const char *where)
{
  gpointer usage;
  MEMORY_TRACE_LOCK
    {
      usage = g_hash_table_lookup(memory_trace, pointer);
      g_hash_table_remove(memory_trace, pointer);
    }
  MEMORY_TRACE_UNLOCK;

  if (usage == NULL && pointer != NULL)
    fprintf(stderr, "%p not registered: %s\n", pointer, where);
}


static void
print_pointer_info(gpointer key, gpointer value, gpointer user_data)
{
  fprintf(stderr, "%p: %s\n", key, (char *)value);
}


void
dump_pointers()
{
  MEMORY_TRACE_LOCK
    {
      g_hash_table_foreach(memory_trace, print_pointer_info, NULL);
    }
  MEMORY_TRACE_UNLOCK;
}
#endif
