/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: boardlist_view.c,v 1.16 2003/12/16 14:50:33 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"

#include "ochusha_ui.h"
#include "boardlist_view.h"
#include "marshal.h"

#include <gdk/gdkkeysyms.h>
#include <glib.h>
#include <gtk/gtk.h>

#if DEBUG_WIDGET || DEBUG_WIDGET_MOST
#include <stdio.h>
#endif
#include <stdlib.h>
#include <string.h>


static void boardlist_view_class_init(BoardlistViewClass *klass);
static void popup_menu_callback(gpointer data, guint action,
				GtkWidget *widget);
static void popup_menu_destructed_cb(gpointer data);
static void boardlist_view_init(BoardlistView *view);

static void boardlist_view_finalize(GObject *object);
static void boardlist_view_destroy(GtkObject *object);

static void boardlist_view_do_on_selection(BoardlistView *view,
					   gint what_to_do);
static void boardlist_view_do_on_item_at_cursor(BoardlistView *view,
						gint what_to_do);

static void tree_view_row_expanded_cb(GtkTreeView *tree_view,
				      GtkTreeIter *iter, GtkTreePath *path,
				      BoardlistView *view);
static void tree_view_row_collapsed_cb(GtkTreeView *tree_view,
				       GtkTreeIter *iter, GtkTreePath *path,
				       BoardlistView *view);
static gboolean tree_view_button_press_event_cb(GtkWidget *widget,
						GdkEventButton *event,
						BoardlistView *view);
static gboolean tree_view_button_release_event_cb(GtkWidget *widget,
						  GdkEventButton *event,
						  BoardlistView *view);
static void decorate_boardlist_entry(GtkTreeViewColumn *tree_column,
				     GtkCellRenderer *renderer,
				     GtkTreeModel *tree_model,
				     GtkTreeIter *iter, gpointer unused);
static void append_board(gpointer data, gpointer user_data);
static void append_category(gpointer data, gpointer user_data);


GType
boardlist_view_get_type(void)
{
  static GType blv_type = 0;

  if (blv_type == 0)
    {
      static const GTypeInfo blv_info =
	{
	  sizeof(BoardlistViewClass),
	  NULL, /* base_init */
	  NULL, /* base_finalize */
	  (GClassInitFunc)boardlist_view_class_init,
	  NULL, /* class_finalize */
	  NULL, /* class_data */
	  sizeof(BoardlistView),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)boardlist_view_init,
	};

      blv_type = g_type_register_static(GTK_TYPE_TREE_VIEW,
					"BoardlistView", &blv_info, 0);
    }

  return blv_type;
}


typedef struct _TreeNodeInfo
{
  GtkTreeIter iter;
} TreeNodeInfo;


enum {
  OPEN_THREADLIST_VIEW_SIGNAL,

  SELECT_CATEGORY_SIGNAL,
  CATEGORY_COLLAPSED_SIGNAL,
  CATEGORY_EXPANDED_SIGNAL,

  TOGGLE_HIDE_CATEGORY_SIGNAL,
  TOGGLE_HIDE_BOARD_SIGNAL,
  KILL_CATEGORY_SIGNAL,
  KILL_BOARD_SIGNAL,
  COPY_BOARD_URL_SIGNAL,

  ADVANCE_VIEW_SIGNAL,
  BACK_VIEW_SIGNAL,

  DO_ON_SELECTION_SIGNAL,
  DO_ON_ITEM_AT_CURSOR_SIGNAL,

  LAST_SIGNAL
};


enum {
  TOGGLE_HIDDEN_STATE,
  KILL_ITEM,
  ADVANCE_VIEW,
  BACK_VIEW,
};


enum {
  POPUP_MENU_OPEN_SELECTION = 1,
  POPUP_MENU_OPEN_SELECTION_IN_TAB,
  POPUP_MENU_OPEN_SELECTION_WITH_BROWSER,
  POPUP_MENU_TOGGLE_HIDE_SELECTION,
  POPUP_MENU_KILL_SELECTION,
  POPUP_MENU_COPY_BOARD_URL
};


static GtkTreeViewClass *parent_class = NULL;
static gint boardlist_view_signals[LAST_SIGNAL] =
  { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
static GtkItemFactory *popup_menu_item_factory = NULL;
static GQuark node_info_id;

static char *list_entry_color[4] = { NULL, NULL, NULL, NULL };


#if TRACE_MEMORY_USAGE
static void
trace_free(gpointer pointer)
{
  G_FREE(pointer);
}
#define TRACE_FREE	trace_free
#else
#define TRACE_FREE	g_free
#endif


static TreeNodeInfo *
ensure_node_info(GObject *object)
{
  TreeNodeInfo *info = g_object_get_qdata(object, node_info_id);
  if (info == NULL)
    {
      info = G_NEW0(TreeNodeInfo, 1);
      g_object_set_qdata_full(object, node_info_id,
			      info, (GDestroyNotify)TRACE_FREE);
    }

  return info;
}


static void
boardlist_view_class_init(BoardlistViewClass *klass)
{
  GObjectClass *o_class = (GObjectClass *)klass;
  GtkObjectClass *object_class = (GtkObjectClass *)klass;
  GtkWidgetClass *widget_class = (GtkWidgetClass *)klass;
  GtkBindingSet *binding_set;

  /* ݥåץåץ˥塼ν */
  GtkItemFactoryEntry menu_items[] =
    {
      {
	_("/_Open"),				/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback,			/* callback_func */
	POPUP_MENU_OPEN_SELECTION,		/* act */
	NULL					/* type */
      },
      {
	_("/Open Boards in _Tab"),		/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback,			/* callback_func */
	POPUP_MENU_OPEN_SELECTION_IN_TAB,	/* act */
	NULL					/* type */
      },
      {
	_("/Open Boards with Web Browser"),	/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback,			/* callback_func */
	POPUP_MENU_OPEN_SELECTION_WITH_BROWSER,	/* act */
	NULL					/* type */
      },
      {
	_("/-------"),
	NULL,
	NULL,
	0,
	"<Separator>"
      },
      {
	_("/Toggle Hide(_D)"),			/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback,			/* callback_func */
	POPUP_MENU_TOGGLE_HIDE_SELECTION,	/* act */
	NULL					/* type */
      },
      {
	_("/Kill(_K)"),				/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback,			/* callback_func */
	POPUP_MENU_KILL_SELECTION,		/* act */
	NULL					/* type */
      },
      {
	_("/-------"),
	NULL,
	NULL,
	0,
	"<Separator>"
      },
      {
	_("/Copy Board's URL"),			/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback,			/* callback_func */
	POPUP_MENU_COPY_BOARD_URL,		/* act */
	NULL					/* type */
      },
    };
  gint nmenu_items = sizeof(menu_items) / sizeof(menu_items[0]);

  popup_menu_item_factory = gtk_item_factory_new(GTK_TYPE_MENU,
						 "<BoardlistViewPopup>",
						 NULL);
  gtk_item_factory_create_items(popup_menu_item_factory,
				nmenu_items, menu_items, NULL);

  parent_class = g_type_class_peek_parent(klass);
  binding_set = gtk_binding_set_by_class(klass);

  /* GObject signals */
  o_class->finalize = boardlist_view_finalize;

  /* GtkObject signals */
  object_class->destroy = boardlist_view_destroy;

  gtk_widget_class_install_style_property(widget_class,
					  g_param_spec_string("fg_normal",
							      _("FG normal"),
							      _("Foreground color for ordinal list entry"),
							      NULL,
							      G_PARAM_READABLE));
  gtk_widget_class_install_style_property(widget_class,
					  g_param_spec_string("fg_hidden",
							      _("FG hidden"),
							      _("Foreground color for hidden list entry"),
							      "gray75",
							      G_PARAM_READABLE));
  gtk_widget_class_install_style_property(widget_class,
					  g_param_spec_string("bg_normal",
							      _("BG normal"),
							      _("Background color for ordinal list entry"),
							      NULL,
							      G_PARAM_READABLE));
  gtk_widget_class_install_style_property(widget_class,
					  g_param_spec_string("bg_hidden",
							      _("BG hidden"),
							      _("Background color for hidden list entry"),
							      NULL,
							      G_PARAM_READABLE));

  boardlist_view_signals[OPEN_THREADLIST_VIEW_SIGNAL] =
    g_signal_new("open_threadlist_view",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(BoardlistViewClass, open_threadlist_view),
		 NULL, NULL,
		 ochusha_marshal_VOID__OBJECT_BOOLEAN_BOOLEAN,
		 G_TYPE_NONE, 3,
		 OCHUSHA_TYPE_BULLETIN_BOARD,
		 G_TYPE_BOOLEAN,
		 G_TYPE_BOOLEAN);

  boardlist_view_signals[SELECT_CATEGORY_SIGNAL] =
    g_signal_new("select_category",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(BoardlistViewClass, select_category),
		 NULL, NULL,
		 ochusha_marshal_VOID__POINTER,
		 G_TYPE_NONE, 1,
		 G_TYPE_POINTER);
  boardlist_view_signals[CATEGORY_EXPANDED_SIGNAL] =
    g_signal_new("category_expanded",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(BoardlistViewClass, category_expanded),
		 NULL, NULL,
		 ochusha_marshal_VOID__OBJECT,
		 G_TYPE_NONE, 1,
		 OCHUSHA_TYPE_BOARD_CATEGORY);
  boardlist_view_signals[CATEGORY_COLLAPSED_SIGNAL] =
    g_signal_new("category_collapsed",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(BoardlistViewClass, category_collapsed),
		 NULL, NULL,
		 ochusha_marshal_VOID__OBJECT,
		 G_TYPE_NONE, 1,
		 OCHUSHA_TYPE_BOARD_CATEGORY);

  boardlist_view_signals[TOGGLE_HIDE_CATEGORY_SIGNAL] =
    g_signal_new("toggle_hide_category",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(BoardlistViewClass, toggle_hide_category),
		 NULL, NULL,
		 ochusha_marshal_VOID__OBJECT,
		 G_TYPE_NONE, 1,
		 OCHUSHA_TYPE_BOARD_CATEGORY);
  boardlist_view_signals[TOGGLE_HIDE_BOARD_SIGNAL] =
    g_signal_new("toggle_hide_board",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(BoardlistViewClass, toggle_hide_board),
		 NULL, NULL,
		 ochusha_marshal_VOID__OBJECT,
		 G_TYPE_NONE, 1,
		 OCHUSHA_TYPE_BULLETIN_BOARD);

  boardlist_view_signals[KILL_CATEGORY_SIGNAL] =
    g_signal_new("kill_category",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(BoardlistViewClass, kill_category),
		 NULL, NULL,
		 ochusha_marshal_VOID__OBJECT,
		 G_TYPE_NONE, 1,
		 OCHUSHA_TYPE_BOARD_CATEGORY);
  boardlist_view_signals[KILL_BOARD_SIGNAL] =
    g_signal_new("kill_board",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(BoardlistViewClass, kill_board),
		 NULL, NULL,
		 ochusha_marshal_VOID__OBJECT,
		 G_TYPE_NONE, 1,
		 OCHUSHA_TYPE_BULLETIN_BOARD);
  boardlist_view_signals[COPY_BOARD_URL_SIGNAL] =
    g_signal_new("copy_board_url",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(BoardlistViewClass, copy_board_url),
		 NULL, NULL,
		 ochusha_marshal_VOID__OBJECT,
		 G_TYPE_NONE, 1,
		 OCHUSHA_TYPE_BULLETIN_BOARD);

  boardlist_view_signals[ADVANCE_VIEW_SIGNAL] =
    g_signal_new("advance_view",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(BoardlistViewClass, advance_view),
		 NULL, NULL,
		 ochusha_marshal_BOOLEAN__OBJECT,
		 G_TYPE_BOOLEAN, 1,
		 OCHUSHA_TYPE_BULLETIN_BOARD);
  boardlist_view_signals[BACK_VIEW_SIGNAL] =
    g_signal_new("back_view",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(BoardlistViewClass, back_view),
		 NULL, NULL,
		 ochusha_marshal_BOOLEAN__OBJECT,
		 G_TYPE_BOOLEAN, 1,
		 OCHUSHA_TYPE_BULLETIN_BOARD);

  boardlist_view_signals[DO_ON_SELECTION_SIGNAL] =
    g_signal_new("do_on_selection",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
		 G_STRUCT_OFFSET(BoardlistViewClass, do_on_selection),
		 NULL, NULL,
		 ochusha_marshal_VOID__INT,
		 G_TYPE_NONE, 1,
		 G_TYPE_INT);
  boardlist_view_signals[DO_ON_ITEM_AT_CURSOR_SIGNAL] =
    g_signal_new("do_on_item_at_cursor",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
		 G_STRUCT_OFFSET(BoardlistViewClass, do_on_item_at_cursor),
		 NULL, NULL,
		 ochusha_marshal_VOID__INT,
		 G_TYPE_NONE, 1,
		 G_TYPE_INT);


  /* Key bindings */
  gtk_binding_entry_add_signal(binding_set, GDK_o, 0, "do_on_selection", 1,
			       G_TYPE_INT, POPUP_MENU_OPEN_SELECTION);
  gtk_binding_entry_add_signal(binding_set, GDK_t, 0, "do_on_selection", 1,
			       G_TYPE_INT, POPUP_MENU_OPEN_SELECTION_IN_TAB);
  gtk_binding_entry_add_signal(binding_set, GDK_d, 0, "do_on_item_at_cursor",
			       1, G_TYPE_INT, TOGGLE_HIDDEN_STATE);
  gtk_binding_entry_add_signal(binding_set, GDK_k, 0, "do_on_item_at_cursor",
			       1, G_TYPE_INT, KILL_ITEM);

  gtk_binding_entry_add_signal(binding_set, GDK_space, 0,
			       "do_on_item_at_cursor", 1,
			       G_TYPE_INT, ADVANCE_VIEW);
  gtk_binding_entry_add_signal(binding_set, GDK_BackSpace, 0,
			       "do_on_item_at_cursor", 1,
			       G_TYPE_INT, BACK_VIEW);

  gtk_binding_entry_add_signal(binding_set, GDK_Up, 0,
			       "move_cursor", 2,
			       G_TYPE_ENUM, GTK_MOVEMENT_DISPLAY_LINES,
			       G_TYPE_INT, -1);
  gtk_binding_entry_add_signal(binding_set, GDK_Down, 0,
			       "move_cursor", 2,
			       G_TYPE_ENUM, GTK_MOVEMENT_DISPLAY_LINES,
			       G_TYPE_INT, 1);
  gtk_binding_entry_add_signal(binding_set, GDK_p, GDK_CONTROL_MASK,
			       "move_cursor", 2,
			       G_TYPE_ENUM, GTK_MOVEMENT_DISPLAY_LINES,
			       G_TYPE_INT, -1);
  gtk_binding_entry_add_signal(binding_set, GDK_n, GDK_CONTROL_MASK,
			       "move_cursor", 2,
			       G_TYPE_ENUM, GTK_MOVEMENT_DISPLAY_LINES,
			       G_TYPE_INT, 1);
  gtk_binding_entry_add_signal(binding_set, GDK_v, GDK_CONTROL_MASK,
			       "move_cursor", 2,
			       G_TYPE_ENUM, GTK_MOVEMENT_PAGES,
			       G_TYPE_INT, 1);
  gtk_binding_entry_add_signal(binding_set, GDK_v, GDK_MOD1_MASK,
			       "move_cursor", 2,
			       G_TYPE_ENUM, GTK_MOVEMENT_PAGES,
			       G_TYPE_INT, -1);
  gtk_binding_entry_add_signal(binding_set, GDK_v, GDK_MOD4_MASK,
			       "move_cursor", 2,
			       G_TYPE_ENUM, GTK_MOVEMENT_PAGES,
			       G_TYPE_INT, -1);
			       

  klass->open_threadlist_view = NULL;

  klass->select_category = NULL;
  klass->category_expanded = NULL;
  klass->category_collapsed = NULL;

  klass->toggle_hide_category = NULL;
  klass->toggle_hide_board = NULL;
  klass->kill_category = NULL;
  klass->kill_board = NULL;
  klass->copy_board_url = NULL;

  klass->advance_view = NULL;
  klass->back_view = NULL;

  klass->do_on_selection = boardlist_view_do_on_selection;
  klass->do_on_item_at_cursor = boardlist_view_do_on_item_at_cursor;

  node_info_id = g_quark_from_static_string("BoardlistView::TreeNodeInfo");
}


enum
{
  BOARDLIST_CATEGORY_ROW,
  BOARDLIST_BOARD_ROW
};


enum
{
  BOARDLIST_LABEL = 0,
  BOARDLIST_ROW_TYPE,
  BOARDLIST_ROW_STATE,
  BOARDLIST_DATA,
  BOARDLIST_N_COLUMNS
};


static void
boardlist_view_init(BoardlistView *view)
{
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;

  GTK_WIDGET_SET_FLAGS(view, GTK_CAN_FOCUS | GTK_RECEIVES_DEFAULT);

  view->selection = gtk_tree_view_get_selection(&view->view);
  gtk_tree_selection_set_mode(view->selection, GTK_SELECTION_MULTIPLE);

  g_signal_connect(G_OBJECT(view), "row-expanded",
		   G_CALLBACK(tree_view_row_expanded_cb), view);
  g_signal_connect(G_OBJECT(view), "row-collapsed",
		   G_CALLBACK(tree_view_row_collapsed_cb), view);
  g_signal_connect(G_OBJECT(view), "button_press_event",
		   G_CALLBACK(tree_view_button_press_event_cb), view);
  g_signal_connect(G_OBJECT(view), "button_release_event",
		   G_CALLBACK(tree_view_button_release_event_cb), view);
  view->last_event_x = 0.0;
  view->last_event_y = 0.0;
  view->last_event_button = 0;

  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(_("Board List"), renderer,
						    "text", BOARDLIST_LABEL,
						    NULL);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  decorate_boardlist_entry,
					  view, NULL);
  gtk_tree_view_append_column(&view->view, column);
}


static void
boardlist_view_finalize(GObject *object)
{
  BoardlistView *view = (BoardlistView *)object;

#if DEBUG_WIDGET_MOST
  fprintf(stderr, "boardlist_view_finalize\n");
#endif
  g_object_unref(G_OBJECT(view->model));

  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


static void
boardlist_view_destroy(GtkObject *object)
{
#if DEBUG_WIDGET_MOST
  fprintf(stderr, "boardlist_view_destroy\n");
#endif

  if (GTK_OBJECT_CLASS(parent_class)->destroy)
    (*GTK_OBJECT_CLASS(parent_class)->destroy)(object);
}


gboolean
boardlist_view_get_default_open_in_tab(BoardlistView *view)
{
  g_return_val_if_fail(IS_BOARDLIST_VIEW(view), FALSE);
  return view->default_open_in_tab;
}


void
boardlist_view_set_default_open_in_tab(BoardlistView *view,
				       gboolean default_open_in_tab)
{
  g_return_if_fail(IS_BOARDLIST_VIEW(view));
  view->default_open_in_tab = default_open_in_tab;
}


void
boardlist_view_expand_category(BoardlistView *view,
			       OchushaBoardCategory *category)
{
  gboolean expanded;
  TreeNodeInfo *info = ensure_node_info(G_OBJECT(category));
  GtkTreePath *path;

  if (!gtk_tree_store_iter_is_valid(GTK_TREE_STORE(view->model),
				    &info->iter))
    return;

  path = gtk_tree_model_get_path(GTK_TREE_MODEL(view->model), &info->iter);
  expanded = gtk_tree_view_expand_row(&view->view, path, FALSE);

  gtk_tree_path_free(path);

  if (expanded)
    g_signal_emit(G_OBJECT(view),
		  boardlist_view_signals[CATEGORY_EXPANDED_SIGNAL],
		  0,
		  category);
}


void
boardlist_view_collapse_category(BoardlistView *view,
				 OchushaBoardCategory *category)
{
  gboolean collapsed;
  TreeNodeInfo *info = ensure_node_info(G_OBJECT(category));
  GtkTreePath *path;

  if (!gtk_tree_store_iter_is_valid(GTK_TREE_STORE(view->model),
				    &info->iter))
    return;

  path = gtk_tree_model_get_path(GTK_TREE_MODEL(view->model), &info->iter);
  collapsed = gtk_tree_view_collapse_row(&view->view, path);

  gtk_tree_path_free(path);

  if (collapsed)
    g_signal_emit(G_OBJECT(view),
		  boardlist_view_signals[CATEGORY_COLLAPSED_SIGNAL],
		  0,
		  category);
}


enum {
  BOARDLIST_ROW_STATE_HIDDEN = 1,
  BOARDLIST_ROW_STATE_KILLED = 1 << 1,
};


static guint
encode_row_state(gboolean hidden, gboolean killed)
{
  return (hidden ? BOARDLIST_ROW_STATE_HIDDEN : 0)
    | (killed ? BOARDLIST_ROW_STATE_KILLED : 0);
}


typedef struct _DoOnSelectionArgs
{
  BoardlistView *view;
  gint what_to_do;
} DoOnSelectionArgs;


static void
do_on_selection_helper(GtkTreeModel *model, GtkTreePath *path,
		       GtkTreeIter *iter, DoOnSelectionArgs *args)
{
  gpointer data;
  gint what_to_do = args->what_to_do;

  gtk_tree_model_get(model, iter,
		     BOARDLIST_DATA, &data, -1);
  g_return_if_fail(data != NULL);

  switch (what_to_do)
    {
    case POPUP_MENU_OPEN_SELECTION:
      /* 2ܰʹߤ϶Ū˥֤ǳ */
      args->what_to_do = POPUP_MENU_OPEN_SELECTION_IN_TAB;

      /* FALL THROUGH */
    case POPUP_MENU_OPEN_SELECTION_IN_TAB:
      if (OCHUSHA_IS_BOARD_CATEGORY(data))
	gtk_tree_view_expand_row(&args->view->view, path, TRUE);
      else
	g_signal_emit(G_OBJECT(args->view),
		      boardlist_view_signals[OPEN_THREADLIST_VIEW_SIGNAL],
		      0,
		      data,
		      what_to_do == POPUP_MENU_OPEN_SELECTION_IN_TAB,
		      FALSE);
      break;

    case POPUP_MENU_OPEN_SELECTION_WITH_BROWSER:
      if (OCHUSHA_IS_BULLETIN_BOARD(data))
	g_signal_emit(G_OBJECT(args->view),
		      boardlist_view_signals[OPEN_THREADLIST_VIEW_SIGNAL],
		      0,
		      data,
		      FALSE,
		      TRUE);
      break;

    case POPUP_MENU_TOGGLE_HIDE_SELECTION:
      if (OCHUSHA_IS_BOARD_CATEGORY(data))
	{
	  OchushaBoardCategory *category = OCHUSHA_BOARD_CATEGORY(data);
	  g_signal_emit(G_OBJECT(args->view),
			boardlist_view_signals[TOGGLE_HIDE_CATEGORY_SIGNAL],
			0,
			category);
	  gtk_tree_store_set(GTK_TREE_STORE(model), iter,
			     BOARDLIST_ROW_STATE,
			     encode_row_state(category->hidden,
					      category->killed),
			     -1);
	}
      else if (OCHUSHA_IS_BULLETIN_BOARD(data))
	{
	  OchushaBulletinBoard *board = OCHUSHA_BULLETIN_BOARD(data);
	  g_signal_emit(G_OBJECT(args->view),
			boardlist_view_signals[TOGGLE_HIDE_BOARD_SIGNAL],
			0,
			board);
	  gtk_tree_store_set(GTK_TREE_STORE(model), iter,
			     BOARDLIST_ROW_STATE,
			     encode_row_state(board->hidden, board->killed),
			     -1);
	}
      break;

    case POPUP_MENU_KILL_SELECTION:
      if (OCHUSHA_IS_BOARD_CATEGORY(data))
	{
	  OchushaBoardCategory *category = OCHUSHA_BOARD_CATEGORY(data);
	  g_signal_emit(G_OBJECT(args->view),
			boardlist_view_signals[KILL_CATEGORY_SIGNAL],
			0,
			category);
	  gtk_tree_store_set(GTK_TREE_STORE(model), iter,
			     BOARDLIST_ROW_STATE,
			     encode_row_state(category->hidden,
					      category->killed),
			     -1);
	}
      else if (OCHUSHA_IS_BULLETIN_BOARD(data))
	{
	  OchushaBulletinBoard *board = OCHUSHA_BULLETIN_BOARD(data);
	  g_signal_emit(G_OBJECT(args->view),
			boardlist_view_signals[KILL_BOARD_SIGNAL],
			0,
			board);
	  gtk_tree_store_set(GTK_TREE_STORE(model), iter,
			     BOARDLIST_ROW_STATE,
			     encode_row_state(board->hidden, board->killed),
			     -1);
	}
      break;

    case POPUP_MENU_COPY_BOARD_URL:
      if (OCHUSHA_IS_BULLETIN_BOARD(data))
	g_signal_emit(G_OBJECT(args->view),
		      boardlist_view_signals[COPY_BOARD_URL_SIGNAL],
		      0,
		      data);
      break;
    }
}


static void
boardlist_view_do_on_selection(BoardlistView *view, gint what_to_do)
{
  DoOnSelectionArgs args = { view, what_to_do };
#if DEBUG_WIDGET_MOST
  fprintf(stderr, "boardlist_view_do_on_selection: view=%p what_to_do=%d\n",
	  view, what_to_do);
#endif
  gtk_tree_selection_selected_foreach(view->selection,
			(GtkTreeSelectionForeachFunc)do_on_selection_helper,
			&args);
}


static void
boardlist_view_do_on_item_at_cursor(BoardlistView *view, gint what_to_do)
{
  GtkTreeIter iter;
  GtkTreePath *path = NULL;
  gpointer data = NULL;

  gtk_tree_view_get_cursor(&view->view, &path, NULL);
  if (path == NULL)
    return;

  if (!gtk_tree_model_get_iter(view->model, &iter, path))
    goto done;

  gtk_tree_model_get(view->model, &iter, BOARDLIST_DATA, &data, -1);

  if (data != NULL)
    {
      gboolean result = FALSE;
      gtk_tree_selection_unselect_all(view->selection);
      (*parent_class->select_cursor_row)(&view->view, FALSE);

      switch (what_to_do)
	{
	case TOGGLE_HIDDEN_STATE:
	  boardlist_view_do_on_selection(view,
					 POPUP_MENU_TOGGLE_HIDE_SELECTION);
	  break;

	case KILL_ITEM:
	  boardlist_view_do_on_selection(view,
					 POPUP_MENU_KILL_SELECTION);
	  break;

	case ADVANCE_VIEW:
	  if (OCHUSHA_IS_BOARD_CATEGORY(data))
	    {
	      if (gtk_tree_view_row_expanded(&view->view, path))
		boardlist_view_collapse_category(view,
						 OCHUSHA_BOARD_CATEGORY(data));
	      else
		boardlist_view_expand_category(view,
					       OCHUSHA_BOARD_CATEGORY(data));
	      break;
	    }
	  boardlist_view_do_on_selection(view, POPUP_MENU_OPEN_SELECTION);
	  break;

	case BACK_VIEW:
	  if (OCHUSHA_IS_BOARD_CATEGORY(data))
	    {
	      if (gtk_tree_view_row_expanded(&view->view, path))
		boardlist_view_collapse_category(view,
						 OCHUSHA_BOARD_CATEGORY(data));
	      else
		boardlist_view_expand_category(view,
					       OCHUSHA_BOARD_CATEGORY(data));
	      break;
	    }
	  g_signal_emit(G_OBJECT(view),
			boardlist_view_signals[BACK_VIEW_SIGNAL],
			0,
			data,
			&result);
	  break;
	}
    }

 done:
  gtk_tree_path_free(path);
}


static void
popup_menu_callback(gpointer data, guint action, GtkWidget *widget)
{
  BoardlistView *view = gtk_item_factory_popup_data(popup_menu_item_factory);
#if DEBUG_WIDGET_MOST
  fprintf(stderr,
	  "<boardlist_view.c> popup_menu_callback: view=%p, action=%d\n",
	  view, action);
#endif
  boardlist_view_do_on_selection(view, action);
}


static void
popup_menu_destructed_cb(gpointer data)
{
#if DEBUG_WIDGET_MOST
  BoardlistView *view = BOARDLIST_VIEW(data);
  fprintf(stderr, "<boardlist_view.c> popup_menu_destructed_cb: data=%p\n",
	  view);
#endif
}


GtkWidget *
boardlist_view_new(void)
{
  return GTK_WIDGET(g_object_new(BOARDLIST_VIEW_TYPE, NULL));
}


static gboolean
tree_view_button_press_event_cb(GtkWidget *widget, GdkEventButton *event,
				BoardlistView *view)
{
  GtkTreeIter iter;
  GtkTreePath *path = NULL;
  gint row_type;
  gpointer data = NULL;
  gboolean result = FALSE;

  view->last_event_x = event->x;
  view->last_event_y = event->y;
  view->last_event_button = event->button;

  if (!gtk_tree_view_get_path_at_pos(&view->view, event->x, event->y, &path,
				     NULL, NULL, NULL))
    goto done;

  if (!gtk_tree_model_get_iter(view->model, &iter, path))
    goto done;

  gtk_tree_model_get(view->model, &iter,
		     BOARDLIST_ROW_TYPE, &row_type,
		     BOARDLIST_DATA, &data,
		     -1);

  if (data == NULL)
    goto done;

  if (event->button == 3)
    {
      gtk_item_factory_popup_with_data(popup_menu_item_factory,
				       view, popup_menu_destructed_cb,
				       event->x_root, event->y_root,
				       event->button,
				       gtk_get_current_event_time());
      if (gtk_tree_selection_iter_is_selected(view->selection, &iter))
	result = TRUE;
      goto done;
    }

  if ((event->state & (GDK_CONTROL_MASK | GDK_SHIFT_MASK)) != 0)
    goto done;

  switch (row_type)
    {
    case BOARDLIST_CATEGORY_ROW:
      g_signal_emit(G_OBJECT(view),
		    boardlist_view_signals[SELECT_CATEGORY_SIGNAL],
		    0,
		    data);
      break;

    case BOARDLIST_BOARD_ROW:
      g_signal_emit(G_OBJECT(view),
		    boardlist_view_signals[OPEN_THREADLIST_VIEW_SIGNAL],
		    0,
		    data,
		    view->default_open_in_tab
		    ? event->button == 1 : event->button == 2,
		    FALSE);
      break;

    default:
      fprintf(stderr, "GtkTreeModel broken.\n");
    }

 done:
  if (path != NULL)
    gtk_tree_path_free(path);

  return result;
}


static gboolean
tree_view_button_release_event_cb(GtkWidget *widget, GdkEventButton *event,
				  BoardlistView *view)
{
  if (view != NULL)
    {
      view->last_event_x = event->x;
      view->last_event_y = event->y;
      view->last_event_button = event->button;
    }

  return FALSE;
}


static void
tree_view_row_expanded_cb(GtkTreeView *tree_view,
			  GtkTreeIter *iter, GtkTreePath *path,
			  BoardlistView *view)
{
  OchushaBoardCategory *category;
#if DEBUG_WIDGET_MOST
  gchar *name;
#endif

  g_return_if_fail(tree_view != NULL);
  g_return_if_fail(view != NULL);

#if DEBUG_WIDGET_MOST
  gtk_tree_model_get(view->model, iter,
		     BOARDLIST_LABEL, &name,
		     BOARDLIST_DATA, &category,
		     -1);
  fprintf(stderr, "row_expanded_cb: %s\n", name);
  G_FREE(name);
#else
  gtk_tree_model_get(view->model, iter,
		     BOARDLIST_DATA, &category,
		     -1);
#endif

  if (category != NULL)
    {
      g_signal_emit(G_OBJECT(view),
		    boardlist_view_signals[CATEGORY_EXPANDED_SIGNAL],
		    0,
		    category);
    }
}


static void
tree_view_row_collapsed_cb(GtkTreeView *tree_view,
			   GtkTreeIter *iter, GtkTreePath *path,
			   BoardlistView *view)
{
  OchushaBoardCategory *category;
#if DEBUG_WIDGET_MOST
  gchar *name;
#endif

  g_return_if_fail(tree_view != NULL);
  g_return_if_fail(view != NULL);

#if DEBUG_WIDGET_MOST
  gtk_tree_model_get(view->model, iter,
		     BOARDLIST_LABEL, &name,
		     BOARDLIST_DATA, &category,
		     -1);
  fprintf(stderr, "row_collapsed_cb: %s\n", name);
  G_FREE(name);
#else
  gtk_tree_model_get(view->model, iter,
		     BOARDLIST_DATA, &category,
		     -1);
#endif

  if (category != NULL)
    {
      g_signal_emit(G_OBJECT(view),
		    boardlist_view_signals[CATEGORY_COLLAPSED_SIGNAL],
		    0,
		    category);
    }
}


static const char *boardlist_view_get_color(BoardlistView *view,
					    guint color_code);

static const char *
find_replacement_color(BoardlistView *view, const char *color)
{
  const char *result_color;
  GdkColor *gdk_color;
  char name_buf[8];
  g_return_val_if_fail(color != NULL && *color == '<', NULL);

  if (strncmp(color + 1, "fg_", 3) == 0)
    {
      if (strcmp(color + 4, "normal>") == 0)
	result_color = boardlist_view_get_color(view, 0);
      else if (strcmp(color + 4, "hidden>") == 0)
	result_color = boardlist_view_get_color(view, 1);
      else
	return NULL;
      if (result_color != NULL)
	return result_color;

      gdk_color = &GTK_WIDGET(view)->style->text[GTK_STATE_NORMAL];
      snprintf(name_buf, 8, "#%02x%02x%02x",
	       gdk_color->red / 256, gdk_color->green / 256,
	       gdk_color->blue / 256);
      return g_strdup(name_buf);	/* Ĥϥ꡼뤱̵ */
    }
  else if (strncmp(color + 1, "bg_", 3) == 0)
    {
      if (strcmp(color + 4, "normal>") == 0)
	result_color = boardlist_view_get_color(view, 2);
      else if (strcmp(color + 4, "hidden>") == 0)
	result_color = boardlist_view_get_color(view, 3);
      else
	return NULL;
      if (result_color != NULL)
	return result_color;

      gdk_color = &GTK_WIDGET(view)->style->base[GTK_STATE_NORMAL];
      snprintf(name_buf, 8, "#%02x%02x%02x",
	       gdk_color->red / 256, gdk_color->green / 256,
	       gdk_color->blue / 256);
      return g_strdup(name_buf);	/* Ĥϥ꡼뤱̵ */
    }

  return NULL;
}


static const char *
boardlist_view_get_color(BoardlistView *view, guint color_code)
{
  const char *color;
  char *tmp_color;

  g_return_val_if_fail(color_code < 4, NULL);

  if (list_entry_color[color_code] != NULL)
    {
      color = list_entry_color[color_code];
      if (color != NULL && *color == '\0')
	return NULL;
      return color;
    }

  switch (color_code)
    {
    case 0:
      gtk_widget_style_get(GTK_WIDGET(view), "fg_normal", &color, NULL);
      break;
    case 1:
      gtk_widget_style_get(GTK_WIDGET(view), "fg_hidden", &color, NULL);
      break;
    case 2:
      gtk_widget_style_get(GTK_WIDGET(view), "bg_normal", &color, NULL);
      break;
    case 3:
      gtk_widget_style_get(GTK_WIDGET(view), "bg_hidden", &color, NULL);
      break;
    default:
      color = NULL;
    }

  if (color == NULL)
    return color;

  if (*color == '\0')
    return NULL;

  if (*color != '<')
    return color;

  list_entry_color[color_code] = G_STRDUP("");

  color = find_replacement_color(view, color);

  if (color == NULL)
    return NULL;

  tmp_color = G_STRDUP(color);
  G_FREE(list_entry_color[color_code]);

  list_entry_color[color_code] = tmp_color;
  if (*tmp_color == '\0')
    return NULL;
  return tmp_color;
}


static void
decorate_boardlist_entry(GtkTreeViewColumn *tree_column,
			 GtkCellRenderer *renderer, GtkTreeModel *tree_model,
			 GtkTreeIter *iter, gpointer view)
{
  guint state;
  GValue st = { 0, };
  GValue bg = { 0, };
  GValue fg = { 0, };
  const char *color;

  g_return_if_fail(gtk_tree_store_iter_is_valid(GTK_TREE_STORE(tree_model),
						iter));
  gtk_tree_model_get(tree_model, iter,
		     BOARDLIST_ROW_STATE, &state, -1);

  if (!GTK_IS_CELL_RENDERER_TEXT(renderer))
    return;

  if ((state & BOARDLIST_ROW_STATE_HIDDEN) != 0)
    {
      g_value_init(&st, G_TYPE_BOOLEAN);
      g_value_set_boolean(&st, TRUE);
      g_object_set_property((GObject *)renderer, "strikethrough", &st);
      g_object_set_property((GObject *)renderer, "strikethrough_set", &st);

      color = boardlist_view_get_color(view, 3);
      g_value_init(&bg, G_TYPE_STRING);
      g_value_set_static_string(&bg, color);
      g_object_set_property((GObject *)renderer, "cell_background", &bg);

      if ((state & BOARDLIST_ROW_STATE_KILLED) != 0)
	{
	  color = boardlist_view_get_color(view, 3);
	  g_value_init(&fg, G_TYPE_STRING);
	  g_value_set_static_string(&fg, color);
	  g_object_set_property((GObject *)renderer, "foreground", &fg);
	}
      else
	{
	  color = boardlist_view_get_color(view, 1);
	  g_value_init(&fg, G_TYPE_STRING);
	  g_value_set_static_string(&fg, color);
	  g_object_set_property((GObject *)renderer, "foreground", &fg);
	}
    }
  else
    {
      g_value_init(&st, G_TYPE_BOOLEAN);
      g_value_set_boolean(&st, FALSE);
      g_object_set_property((GObject *)renderer, "strikethrough", &st);
      g_object_set_property((GObject *)renderer, "strikethrough_set", &st);

      color = boardlist_view_get_color(view, 2);
      g_value_init(&bg, G_TYPE_STRING);
      g_value_set_static_string(&bg, color);
      g_object_set_property((GObject *)renderer, "cell_background", &bg);

      if ((state & BOARDLIST_ROW_STATE_KILLED) != 0)
	{
	  color = boardlist_view_get_color(view, 2);
	  g_value_init(&fg, G_TYPE_STRING);
	  g_value_set_static_string(&fg, color);
	  g_object_set_property((GObject *)renderer, "foreground", &fg);
	}
      else
	{
	  color = boardlist_view_get_color(view, 0);
	  g_value_init(&fg, G_TYPE_STRING);
	  g_value_set_static_string(&fg, color);
	  g_object_set_property((GObject *)renderer, "foreground", &fg);
	}
    }
}


typedef struct _AppendBoardArgs
{
  GtkTreeIter *category_iter;
  GtkTreeStore *store;
  gboolean really_hide_hidden_items;
} AppendBoardArgs;


static void
append_board(gpointer data, gpointer user_data)
{
  OchushaBulletinBoard *board = (OchushaBulletinBoard *)data;
  AppendBoardArgs *args = (AppendBoardArgs *)user_data;
  TreeNodeInfo *info = ensure_node_info(G_OBJECT(board));

  if (board->hidden && args->really_hide_hidden_items)
    return;

  gtk_tree_store_append(args->store, &info->iter, args->category_iter);
  gtk_tree_store_set(args->store, &info->iter,
		     BOARDLIST_LABEL, board->name,
		     BOARDLIST_ROW_TYPE, BOARDLIST_BOARD_ROW,
		     BOARDLIST_ROW_STATE, encode_row_state(board->hidden,
							   board->killed),
		     BOARDLIST_DATA, board,
		     -1);
}


static void
append_category(gpointer data, gpointer user_data)
{
  OchushaBoardCategory *category = (OchushaBoardCategory *)data;
  TreeNodeInfo *info = ensure_node_info(G_OBJECT(category));
  AppendBoardArgs *args = (AppendBoardArgs *)user_data;

  if (category->hidden && args->really_hide_hidden_items)
    return;
  args->category_iter = &info->iter;

  gtk_tree_store_append(args->store, args->category_iter, NULL);
  gtk_tree_store_set(args->store, args->category_iter,
		     BOARDLIST_LABEL, category->name,
		     BOARDLIST_ROW_TYPE, BOARDLIST_CATEGORY_ROW,
		     BOARDLIST_ROW_STATE, encode_row_state(category->hidden,
							   category->killed),
		     BOARDLIST_DATA, category,
		     -1);

  g_slist_foreach(category->board_list, append_board, args);
}


void
boardlist_view_open(BoardlistView *view, GSList *category_list,
		    gboolean really_hide_hidden_items)
{
  GtkTreeStore *store = gtk_tree_store_new(BOARDLIST_N_COLUMNS,
		/* tree view label */	   G_TYPE_STRING,
		/* row type */		   G_TYPE_INT,
		/* row state */		   G_TYPE_INT,
		/* data */		   G_TYPE_POINTER);
  		/* (OchushaBoardCategory *)  (OchushaBulletinBoard *) */
  AppendBoardArgs args = { NULL, store, really_hide_hidden_items };

  g_slist_foreach(category_list, append_category, &args);

  gtk_tree_view_set_model(&view->view, GTK_TREE_MODEL(store));

  if (view->model != NULL)
    g_object_unref(G_OBJECT(view->model));
  view->model = GTK_TREE_MODEL(store);
}


