/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ochusha_config.c,v 1.16 2003/12/19 21:13:47 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"

#include "utils.h"

#include <glib.h>
#include <libxml/SAX.h>
#include <libxml/parser.h>

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>


/*
 * ochusha_config_prepare_home
 *
 * ochushaΤΥ桼ۡǥ쥯ȥ(ɸǤ$HOME/.ochusha)
 * ¸ߤå¸ߤʤФΥǥ쥯ȥ롣
 *
 * ǽŪ$HOME/.ochushaǥ쥯ȥ꤬¸ߤ֤ˤʤХǥ쥯ȥؤ
 * PATH̾򡢤ʤNULLconf->home˳Ǽ롣
 */
void
ochusha_config_prepare_home(OchushaConfig *config)
{
  char *ochusha_home = getenv("OCHUSHA_HOME");

  if (ochusha_home == NULL)
    {
      char buffer[PATH_MAX];
      char *home = getenv("HOME");
      if (home == NULL)
	goto no_home_exit;

      if (snprintf(buffer, PATH_MAX, "%s/.ochusha", home) >= PATH_MAX)
	goto no_home_exit;

      ochusha_home = G_STRDUP(buffer);
    }
  else
    ochusha_home = G_STRDUP(ochusha_home);

  if (mkdir_p(ochusha_home))
    {
      config->home = ochusha_home;
      return;
    }

  fprintf(stderr, "Couldn't create directory %s: %s\n", ochusha_home,
	  strerror(errno));
  G_FREE(ochusha_home);

 no_home_exit:
  config->home = NULL;
  return;
}


/*
 * ochusha_config_find_file:
 *
 * ochushahomeǥ쥯ȥФpath̾filenameĥե뤬
 * ¸ߤ뤫ɤ򸡺¸ߤˤϥѥ̾ɽʸ֤
 * callerϥѥ̾ʸפˤʤäfree()̳餦
 */
char *
ochusha_config_find_file(const OchushaConfig *config, const char *filename)
{
  int stat_result;
  struct stat sb;
  char pathname[PATH_MAX];

  if (config->home == NULL)
    return NULL;

  if (snprintf(pathname, PATH_MAX, "%s/%s", config->home, filename)
      >= PATH_MAX)
    return NULL;

  stat_result = stat(pathname, &sb);
  if (stat_result == 0 && S_ISREG(sb.st_mode) && access(pathname, R_OK) == 0)
    return G_STRDUP(pathname);

  return NULL;
}


/*
 * ochusha_config_find_directory:
 *
 * ochushahomeǥ쥯ȥФpath̾filenameĥǥ쥯ȥ꤬
 * ¸ߤ뤫ɤ򸡺¸ߤˤϥѥ̾ɽʸ֤
 * callerϥѥ̾ʸפˤʤäfree()̳餦
 */
char *
ochusha_config_find_directory(const OchushaConfig *config,
			      const char *filename)
{
  int stat_result;
  struct stat sb;
  char pathname[PATH_MAX];

  if (config->home == NULL)
    return NULL;

  if (snprintf(pathname, PATH_MAX, "%s/%s", config->home, filename)
      >= PATH_MAX)
    return NULL;

  stat_result = stat(pathname, &sb);
  if (stat_result == 0 && S_ISDIR(sb.st_mode) && access(pathname, R_OK) == 0)
    return G_STRDUP(pathname);

  return NULL;
}


/*
 * ochusha_config_open_file
 *
 * ochushahomeǥ쥯ȥФpath̾filenameĥե
 * flagsͿ줿⡼ɤǳ
 */
int
ochusha_config_open_file(const OchushaConfig *config, const char *filename,
			 int flags)
{
  int result;
  int len;
  char pathname[PATH_MAX];

  if (config->home == NULL)
    return -1;

  len = snprintf(pathname, PATH_MAX, "%s/%s", config->home, filename);
  if (len >= PATH_MAX)
    return -1;

  if ((flags & O_CREAT) == 0)
    return open(pathname, flags);

  result = open(pathname, flags, 0x180);
  if (result >= 0)
    return result;

  /* 񤭹ߥ⡼ɤǳʤäˤϥǥ쥯ȥ򷡤äƤĩ */
  while (TRUE)
    {
      if (pathname[len] == '/')
	break;
      len--;
    }
  pathname[len] = '\0';

  if (!mkdir_p(pathname))
    return -1;

  pathname[len] = '/';

  return open(pathname, flags, 0x180);
}


/*
 * ochusha_config_cache_file_exist:
 *
 * ꤵ줿URLФ륭åե뤬뤫ɤĴ٤롣
 *
 * å夬¸ߤȤˤTRUE֤
 */
gboolean
ochusha_config_cache_file_exist(const OchushaConfig *config, const char *url)
{
  char *cur_pos;
  char pathname[PATH_MAX];
  int stat_result;
  struct stat sb;

  if (url == NULL)
    return FALSE;

  if (config->home == NULL)
    return FALSE;

  if (url[strlen(url) - 1] == '/')
    return FALSE;

  /*
   * ݤʤΤurlhttp://server/.*ǤȷǤ
   */
  cur_pos = strstr(url, "http://");
  if (cur_pos == NULL)
    return FALSE;

  if (snprintf(pathname, PATH_MAX, "%s/cache/%s", config->home, url + 7)
      >= PATH_MAX)
    return FALSE;

  stat_result = stat(pathname, &sb);
  if (stat_result == 0 && S_ISREG(sb.st_mode) && access(pathname, R_OK) == 0)
    return TRUE;

  return FALSE;
}


/*
 *
 * ꤵ줿URLФ륭åե뤬뤫ɤĴ٤롣
 *
 * å夬¸ߤȤˤϥåեΥѥ̾ɽʸ֤
 * callerϥѥ̾ʸפˤʤäfree()̳餦
 */
char *
ochusha_config_cache_find_file(const OchushaConfig *config, const char *url)
{
  char *cur_pos;
  char pathname[PATH_MAX];
  int stat_result;
  struct stat sb;

  if (url == NULL)
    return NULL;

  if (config->home == NULL)
    return NULL;

  if (url[strlen(url) - 1] == '/')
    return NULL;	/* index.htmlʤɤξάϵʤ */

  /*
   * ݤʤΤurlhttp://server/.*ǤȷǤ
   */
  cur_pos = strstr(url, "http://");
  if (cur_pos == NULL)
    return NULL;

  if (snprintf(pathname, PATH_MAX, "%s/cache/%s", config->home, url + 7)
      >= PATH_MAX)
    return NULL;

  stat_result = stat(pathname, &sb);
  if (stat_result == 0 && S_ISREG(sb.st_mode) && access(pathname, R_OK) == 0)
    return G_STRDUP(pathname);

  if (snprintf(pathname, PATH_MAX, "%s/cache/%s.gz", config->home, url + 7)
      >= PATH_MAX)
    return NULL;

  stat_result = stat(pathname, &sb);
  if (stat_result == 0 && S_ISREG(sb.st_mode) && access(pathname, R_OK) == 0)
    return G_STRDUP(pathname);

  return NULL;
}


/*
 * ochusha_config_cache_open_file:
 *
 * ꤵ줿URLФ륭åե򳫤
 * flagsopenƥॳͿflagsǤ롣
 */
int
ochusha_config_cache_open_file(const OchushaConfig *config, const char *url,
			       int flags)
{
  char *cur_pos;
  char pathname[PATH_MAX];
  int len;
  int result;

  if (url == NULL)
    return -1;

  if (config->home == NULL)
    return -1;

  if (url[strlen(url) - 1] == '/')
    return -1;

  /*
   * ݤʤΤurlhttp://server/.*ǤȷǤ
   */

  cur_pos = strstr(url, "http://");
  if (cur_pos == NULL)
    return -1;

  len = snprintf(pathname, PATH_MAX, "%s/cache/%s", config->home,
		 url + 7);	/* "http://"ʬά */

  if (len >= PATH_MAX)
    return -1;

  if ((flags & O_CREAT) == 0)
    return open(pathname, flags);

  result = open(pathname, flags, 0x180);
  if (result >= 0)
    return result;

  while (TRUE)
    {
      if (pathname[len] == '/')
	break;
      len--;
    }
  pathname[len] = '\0';

  if (!mkdir_p(pathname))
    return -1;

  pathname[len] = '/';

  return open(pathname, flags, 0x180);
}


/*
 * ochusha_config_cache_unlink_file:
 *
 * URLб륭åե뤬¸ߤunlink롣
 */
void
ochusha_config_cache_unlink_file(const OchushaConfig *config, const char *url)
{
  /*
   * ݤʤΤurlhttp://server/.*ǤȷǤ
   */
  char *cur_pos;
  char buffer[PATH_MAX];
  int len;
  char *pathname;

  if (url == NULL)
    return;

  if (config->home == NULL)
    return;

  if (url[strlen(url) - 1] == '/')
    return;

  cur_pos = strstr(url, "http://");
  if (cur_pos == NULL)
    return;

  len = snprintf(buffer, PATH_MAX, "cache/%s", url + 7);
  					       /* "http://"ʬά */

  if (len >= PATH_MAX)
    return;

  pathname = ochusha_config_find_file(config, buffer);
  if (pathname == NULL)
    return;

  unlink(pathname);

  G_FREE(pathname);
}


gboolean
ochusha_utils_get_attribute_boolean(GHashTable *hash_table,
				    const char *attr_name)
{
  char *bool_str = g_hash_table_lookup(hash_table, attr_name);

  if (bool_str == NULL)
    return FALSE;

  return g_ascii_strcasecmp("true", bool_str) == 0;
}


int
ochusha_utils_get_attribute_int(GHashTable *hash_table, const char *attr_name)
{
  char *num_str = g_hash_table_lookup(hash_table, attr_name);
  int num;

  if (num_str == NULL)
    return 0;

  num = 0;
  sscanf(num_str, "%d", &num);
  return num;
}


char *
ochusha_utils_get_attribute_string(GHashTable *hash_table,
				   const char *attr_name)
{
#if 0
  char *str = g_hash_table_lookup(hash_table, attr_name);
  char *result = str != NULL ? G_STRDUP(str) : NULL;
  fprintf(stderr, "result(%p)=\"%s\"\n", result, result);
  return result;
#else
  char *str = g_hash_table_lookup(hash_table, attr_name);
  return str != NULL ? G_STRDUP(str) : NULL;
#endif
}


typedef enum
{
  SAX_INITIAL,
  SAX_OCHUSHA,
  SAX_PREFERENCE,
  SAX_PREFERENCE_ATTRIBUTE,
  SAX_PREFERENCE_ATTRIBUTE_BOOLEAN,
  SAX_PREFERENCE_ATTRIBUTE_INT,
  SAX_PREFERENCE_ATTRIBUTE_STRING,
  SAX_ACCEPTED,
  SAX_ERROR
} SAXState;


typedef struct _SAXContext
{
  SAXState state;

  char *current_attr_name;
  char *current_attr_val;

  GHashTable *pref_attributes;
} SAXContext;


static void
cleanup_sax_context(SAXContext *context)
{
  if (context->current_attr_name != NULL)
    {
      G_FREE(context->current_attr_name);
      context->current_attr_name = NULL;
    }

  if (context->current_attr_val != NULL)
    {
      G_FREE(context->current_attr_val);
      context->current_attr_val = NULL;
    }

  if (context->pref_attributes != NULL)
    {
      g_hash_table_destroy(context->pref_attributes);
      context->pref_attributes = NULL;
    }
}


#if TRACE_MEMORY_USAGE
static void
trace_free(gpointer pointer)
{
  G_FREE(pointer);
}
#define TRACE_FREE	trace_free
#else
#define TRACE_FREE	g_free
#endif


static void
startElementHandler(void *context, const xmlChar *name, const xmlChar **atts)
{
  SAXContext *sax_context = (SAXContext *)context;

  switch (sax_context->state)
    {
    case SAX_INITIAL:
      if (strcmp(name, "ochusha") == 0)
	{ sax_context->state = SAX_OCHUSHA; return; }
      break;	/* 顼 */

    case SAX_OCHUSHA:
      if (strcmp(name, "preference") == 0)
	{
	  sax_context->pref_attributes
	    = g_hash_table_new_full(g_str_hash, g_str_equal,
				    TRACE_FREE, TRACE_FREE);
	  sax_context->state = SAX_PREFERENCE;
	  return;
	}
      break;	/* 顼 */

    case SAX_PREFERENCE:
      if (strcmp(name, "attribute") == 0
	  && atts != NULL && strcmp(atts[0], "name") == 0)
	{
	  sax_context->state = SAX_PREFERENCE_ATTRIBUTE;

	  if (sax_context->current_attr_val != NULL)
	    {
#if DEBUG_SAX_HANDLER
	      fprintf(stderr, "unexpected attribute found: %s=%s\n",
		      sax_context->current_attr_name,
		      sax_context->current_attr_val);
#endif
	      G_FREE(sax_context->current_attr_name);
	      G_FREE(sax_context->current_attr_val);
	      sax_context->current_attr_name = NULL;
	      sax_context->current_attr_val = NULL;
	      break;	/* 顼 */
	    }
	  sax_context->current_attr_name = G_STRDUP(atts[1]);
	  return;
	}
      break;	/* 顼 */

    case SAX_PREFERENCE_ATTRIBUTE:
      if (atts != NULL && strcmp(atts[0], "val") == 0)
	{
	  /* int/booleanβǽ */
	  if (strcmp(name, "int") == 0)
	    {
	      sax_context->state = SAX_PREFERENCE_ATTRIBUTE_INT;
	    }
	  else if (strcmp(name, "boolean") == 0)
	    {
	      sax_context->state = SAX_PREFERENCE_ATTRIBUTE_BOOLEAN;
	    }
	  else
	    {
#if DEBUG_SAX_HANDLER
	      fprintf(stderr, "element unexpected in state(%d) found: %s\n",
		      sax_context->state, name);
#endif
	      break;	/* 顼 */
	    }

	  if (sax_context->current_attr_val != NULL)
	    {
	      /* 顼ǤɤΤġ̯ */
#if DEBUG_SAX_HANDLER
	      fprintf(stderr,
		      "attribute %s=\"%s\" is overwritten by \"%s\"!\n",
		      sax_context->current_attr_name,
		      sax_context->current_attr_val, atts[1]);
#endif
	      G_FREE(sax_context->current_attr_val);
	    }

	  sax_context->current_attr_val = G_STRDUP(atts[1]);
	  return;
	}
      else if (strcmp(name, "string") == 0)
	{
	  sax_context->state = SAX_PREFERENCE_ATTRIBUTE_STRING;
	  return;
	}
      break;	/* 顼 */

    case SAX_PREFERENCE_ATTRIBUTE_STRING:
    case SAX_PREFERENCE_ATTRIBUTE_BOOLEAN:
    case SAX_PREFERENCE_ATTRIBUTE_INT:
    case SAX_ACCEPTED:
    case SAX_ERROR:
      break;	/* 顼 */

    default:
      fprintf(stderr, "startHandler is called in unknown state: %d\n",
	      sax_context->state);
    }
  sax_context->state = SAX_ERROR;
}


static void
endElementHandler(void *context, const xmlChar *name)
{
  SAXContext *sax_context = (SAXContext *)context;

  switch (sax_context->state)
    {
    case SAX_OCHUSHA:
      if (strcmp(name, "ochusha") == 0)
	{ sax_context->state = SAX_ACCEPTED; return; }
      break;	/* 顼 */

    case SAX_PREFERENCE:
      if (strcmp(name, "preference") == 0)
	{ sax_context->state = SAX_OCHUSHA; return; }
      break;	/* 顼 */

    case SAX_PREFERENCE_ATTRIBUTE:
      if (strcmp(name, "attribute") == 0)
	{
	  GHashTable *hash_table = sax_context->pref_attributes;
	  sax_context->state = SAX_PREFERENCE;
#if DEBUG_SAX_HANDLER_NEW
	  fprintf(stderr, "%s = \"%s\"\n", sax_context->current_attr_name,
		  sax_context->current_attr_val);
#endif
	  g_hash_table_insert(hash_table,
			      sax_context->current_attr_name,
			      sax_context->current_attr_val);
	  sax_context->current_attr_name = NULL;
	  sax_context->current_attr_val = NULL;
	  return;
	}
      break;	/* 顼 */

    case SAX_PREFERENCE_ATTRIBUTE_STRING:
      if (strcmp(name, "string") == 0)
	{
	  sax_context->state = SAX_PREFERENCE_ATTRIBUTE;

	  if (sax_context->current_attr_val == NULL)
	    sax_context->current_attr_val = G_STRDUP("");
	  return;
	}
      break;	/* 顼 */

    case SAX_PREFERENCE_ATTRIBUTE_BOOLEAN:
      if (strcmp(name, "boolean") == 0)
	{ sax_context->state = SAX_PREFERENCE_ATTRIBUTE; return; }
      break;	/* 顼 */

    case SAX_PREFERENCE_ATTRIBUTE_INT:
      if (strcmp(name, "int") == 0)
	{ sax_context->state = SAX_PREFERENCE_ATTRIBUTE; return; }
      break;	/* 顼 */

    case SAX_INITIAL:
    case SAX_ACCEPTED:
    case SAX_ERROR:
      break;	/* 顼 */

    default:
#if DEBUG_SAX_HANDLER
      fprintf(stderr, "endHandler called in unknown state: %d.\n",
	      sax_context->state);
#endif
      break;	/* 顼 */
    }
#if DEBUG_SAX_HANDLER
  fprintf(stderr, "Invalid document: </%s> appeared in state=%d\n",
	  name, sax_context->state);
#endif
  sax_context->state = SAX_ERROR;
}


static void
charactersHandler(void *context, const xmlChar *ch, int len)
{
  SAXContext *sax_context = (SAXContext *)context;

  if (sax_context->state == SAX_PREFERENCE_ATTRIBUTE_STRING)
    {
      if (sax_context->current_attr_val == NULL)
	sax_context->current_attr_val = G_STRNDUP(ch, len);
      else
	{
	  int old_len = strlen(sax_context->current_attr_val);
	  sax_context->current_attr_val
	    = G_REALLOC(sax_context->current_attr_val, old_len + len + 1);
	  strncat(sax_context->current_attr_val + old_len, ch, len);
	}
    }
}


static void
nopHandler(void *context)
{
}


static xmlEntityPtr
getEntityHandler(void *context, const xmlChar *name)
{
  return NULL;
}


/*
 * config.xmlϢ
 */
#define OUTPUT_CONFIG_ATTRIBUTE_INT(file, config, attribute)	\
  fprintf(file,							\
	  "    <attribute name=\"" #attribute	"\">\n"		\
	  "      <int val=\"%d\"/>\n"				\
	  "    </attribute>\n", (config)->attribute)

#define OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, config, attribute)\
  fprintf(file,							\
	  "    <attribute name=\"" #attribute	"\">\n"		\
	  "      <boolean val=\"%s\"/>\n"			\
	  "    </attribute>\n", (config)->attribute ? "true" : "false" )

#define OUTPUT_CONFIG_ATTRIBUTE_STRING(file, config, attribute)		\
  do {									\
    if ((config)->attribute != NULL)					\
      {									\
	gchar *text = g_markup_escape_text((config)->attribute, -1);	\
	fprintf(file,							\
		"    <attribute name=\"" #attribute	"\">\n"		\
		"      <string>%s</string>\n"				\
		"    </attribute>\n", text);				\
	g_free(text);							\
      }									\
  } while (0)


gboolean
ochusha_write_config_xml(OchushaConfig *config,
			 WritePreferenceElementFunc *optional,
			 gpointer user_data)
{
  FILE *config_xml;
  int len;
  char pathname[PATH_MAX];
  char *empty_string = "";

  if (config->home == NULL)
    {
      fprintf(stderr, "No home!\n");
      return FALSE;
    }

  len = snprintf(pathname, PATH_MAX, "%s/%s", config->home,
		 OCHUSHA_CONFIG_XML);

  if (len >= PATH_MAX)
    {
      fprintf(stderr, "Too long path name.\n");
      return FALSE;
    }

  config_xml = fopen(pathname, "w");
  if (config_xml == NULL)
    {
      fprintf(stderr, "Couldn't open \"%s\" to write.\n", pathname);
      return FALSE;
    }

  fprintf(config_xml, "<?xml version=\"1.0\"?>\n");
  fprintf(config_xml, "<ochusha>\n");
  fprintf(config_xml, "  <preference>\n");

  OUTPUT_CONFIG_ATTRIBUTE_STRING(config_xml, config, bbsmenu_url);
  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(config_xml, config, offline);

  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(config_xml, config, enable_proxy);
  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(config_xml, config,
				  enable_proxy_only_for_posting);

  if (config->proxy_url == NULL)
    config->proxy_url = empty_string;
  OUTPUT_CONFIG_ATTRIBUTE_STRING(config_xml, config, proxy_url);
  if (config->proxy_url == empty_string)
    config->proxy_url = NULL;

  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(config_xml, config, enable_proxy_auth);

  if (config->proxy_user == NULL)
    config->proxy_user = empty_string;
  OUTPUT_CONFIG_ATTRIBUTE_STRING(config_xml, config, proxy_user);
  if (config->proxy_user == empty_string)
    config->proxy_user = NULL;

  if (config->proxy_password == NULL)
    config->proxy_password = empty_string;
  OUTPUT_CONFIG_ATTRIBUTE_STRING(config_xml, config, proxy_password);
  if (config->proxy_password == empty_string)
    config->proxy_password = empty_string;

  if (optional != NULL)
    (*optional)(config_xml, user_data);
  else
    fprintf(stderr, "No optional preference?\n");

  fprintf(config_xml, "  </preference>\n");
  fprintf(config_xml, "</ochusha>\n");
  return fclose(config_xml) == 0;
}


void
ochusha_read_config_xml(OchushaConfig *config,
			ReadPreferenceElementFunc *optional,
			gpointer user_data)
{
  SAXContext context =
    {
      SAX_INITIAL,	/* state */
      NULL, NULL,	/* current_attr_name, current_attr_val */
      NULL		/* pref_attributes */
    };
  xmlSAXHandler sax_handler;
  char *pathname = ochusha_config_find_file(config, OCHUSHA_CONFIG_XML);
  if (pathname == NULL)
    {
      config->bbsmenu_url = G_STRDUP(OCHUSHA_DEFAULT_BBSMENU_URL);
      config->offline = FALSE;

      config->enable_proxy = FALSE;
      config->enable_proxy_only_for_posting = FALSE;
      config->enable_proxy_auth = FALSE;

      config->proxy_url = G_STRDUP("");
      config->proxy_user = G_STRDUP("");
      config->proxy_password = G_STRDUP("");

      return;
    }

  memset(&sax_handler, 0, sizeof(xmlSAXHandler));

#if LIBXML_VERSION >= 20600
  xmlSAX2InitDefaultSAXHandler(&sax_handler, TRUE);
#else
  initxmlDefaultSAXHandler(&sax_handler, TRUE);
#endif

  sax_handler.getEntity = getEntityHandler;
  sax_handler.startDocument = nopHandler;
  sax_handler.endDocument = nopHandler;
  sax_handler.startElement = startElementHandler;
  sax_handler.endElement = endElementHandler;
#if LIBXML_VERSION >= 20600
  sax_handler.startElementNs = NULL;
  sax_handler.endElementNs = NULL;
#endif
  sax_handler.characters = charactersHandler;

  xmlSAXUserParseFile(&sax_handler, &context, pathname);

  /* xmlCleanupParser(); */

  if (context.state == SAX_ACCEPTED)
    {
      GHashTable *pref = context.pref_attributes;

      config->bbsmenu_url
	= ochusha_utils_get_attribute_string(pref, "bbsmenu_url");
      config->offline = ochusha_utils_get_attribute_boolean(pref, "offline");

      config->enable_proxy
	= ochusha_utils_get_attribute_boolean(pref, "enable_proxy");

      config->enable_proxy_only_for_posting
	= ochusha_utils_get_attribute_boolean(pref,
					      "enable_proxy_only_for_posting");
      if (config->enable_proxy_only_for_posting)
	config->enable_proxy = FALSE;

      config->proxy_url
	= ochusha_utils_get_attribute_string(pref, "proxy_url");
      if (config->proxy_url == NULL)
	config->proxy_url = G_STRDUP("");

      config->enable_proxy_auth
	= ochusha_utils_get_attribute_boolean(pref, "enable_proxy_auth");

      config->proxy_user
	= ochusha_utils_get_attribute_string(pref, "proxy_user");
      if (config->proxy_user == NULL)
	config->proxy_user = G_STRDUP("");

      config->proxy_password
	= ochusha_utils_get_attribute_string(pref, "proxy_password");
      if (config->proxy_password == NULL)
	config->proxy_password = G_STRDUP("");

      if (optional != NULL)
	(*optional)(pref, user_data);
      else
	fprintf(stderr, "No optional preferences?\n");
    }
  else
    fprintf(stderr, "%s is unacceptable as an ochusha's config.xml.\n",
	    pathname);

  G_FREE(pathname);

  cleanup_sax_context(&context);

  return;
}
