/*
 * Copyright (c) 2003-2004 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: image_ui.c,v 1.5.2.7 2004/05/01 18:49:05 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"
#include "ochusha_async_buffer.h"
#include "ochusha_network_broker.h"
#include "utils.h"

#include "worker.h"

#include "ochusha_ui.h"
#include "image_ui.h"

#include <glib.h>
#include <gtk/gtk.h>

#include <fcntl.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


typedef struct _ImageInfo ImageInfo;
struct _ImageInfo
{
  char *url;
  int width;
  int height;
  OchushaAsyncBuffer *buffer;
  GdkPixbufLoader *loader;
  GdkPixbufAnimation *animation;
  GdkPixbuf *pixbuf;
  GSList *widget_list;
  guint timeout_id;
  guint idle_id;
};


static GHashTable *image_info_table = NULL;
static GHashTable *a_bone_url_table = NULL;
static GHashTable *a_bone_server_table = NULL;


#if 0
#define IMAGE_CACHE_TIMEOUT	180000	/* 180 */
#else
#define IMAGE_CACHE_TIMEOUT	5000	/* 5 */
#endif

static void
image_info_free(ImageInfo *info)
{
  /* gdk_threads_enter()ĶƤӽФ */
  g_free(info->url);
  if (info->loader != NULL)
    g_object_unref(info->loader);
  if (info->animation != NULL)
    g_object_unref(info->animation);
  if (info->pixbuf != NULL)
    g_object_unref(info->pixbuf);
  if (info->timeout_id != 0)
    g_source_remove(info->timeout_id);
  g_free(info);
}


/* ϥåơ֥뤫info
 * image_info_free(info)褦timeoutؿ
 */
static gboolean
image_info_timeout_cb(ImageInfo *info)
{
  GSList *info_list;
  gboolean result = FALSE;

  OCHU_THREADS_ENTER();

  if (info->loader != NULL || info->widget_list != NULL || info->idle_id != 0)
    {
      result = TRUE;
      goto retry;
    }

  info_list = (GSList *)g_hash_table_lookup(image_info_table, info->url);
  info_list = g_slist_remove(info_list, info);
  if (info_list == NULL)
    g_hash_table_remove(image_info_table, info->url);
  else
    {
      ImageInfo *another_info = (ImageInfo *)info_list->data;
      g_hash_table_insert(image_info_table, another_info->url, info_list);
    }

  image_info_free(info);

 retry:
  OCHU_THREADS_LEAVE();

  return result;
}


static void
image_destroy_cb(GtkWidget *image, ImageInfo *info)
{
  /* gdk_threads_enter()ĶƤӽФϤ */
  info->widget_list = g_slist_remove(info->widget_list, image);

  if (info->widget_list == NULL && info->loader == NULL
      && (info->animation != NULL || info->pixbuf != NULL))
    {
      /* 180øinfoơ֥뤫õ */
      info->timeout_id = g_timeout_add(IMAGE_CACHE_TIMEOUT,
				       (GSourceFunc)image_info_timeout_cb,
				       info);
    }
}


static ImageInfo *
image_info_new(const char *url, GdkPixbufLoader *loader, GtkWidget *image)
{
  ImageInfo *info = g_new0(ImageInfo, 1);

  info->url = g_strdup(url);

  if (loader != NULL)
    g_object_ref(loader);

  info->loader = loader;

  info->widget_list = g_slist_append(NULL, image);
  g_signal_connect(G_OBJECT(image), "destroy",
		   G_CALLBACK(image_destroy_cb), info);
  return info;
}


static ImageInfo *
lookup_image_info(const char *url, int width, int height)
{
  /* gdk_threads_enter()ĶƤӽФ */
  GSList *info_list = g_hash_table_lookup(image_info_table, url);

  while (info_list != NULL)
    {
      ImageInfo *info = (ImageInfo *)info_list->data;
      if (info->width == width && info->height == height)
	return info;
      info_list = info_list->next;
    }
  return NULL;
}


void
initialize_image_ui(OchushaApplication *application)
{
  int fd;

  image_info_table = g_hash_table_new_full(g_str_hash, g_str_equal,
					   NULL, NULL);
  a_bone_url_table = g_hash_table_new_full(g_str_hash, g_str_equal,
					   (GDestroyNotify)g_free, NULL);
  a_bone_server_table = g_hash_table_new_full(g_str_hash, g_str_equal,
					      (GDestroyNotify)g_free, NULL);

  /* ܡURLꥹȤե뤫ɤ */
  fd = ochusha_config_open_file(&application->config, OCHUSHA_A_BONE_IMAGE_TXT,
				NULL, O_RDONLY);
  if (fd >= 0)
    {
      FILE *file = fdopen(fd, "r");
      if (file != NULL)
	{
	  char url[PATH_MAX];
	  url[PATH_MAX - 1] = '\0';
	  while (!feof(file))
	    {
	      char *tmp_str = fgets(url, PATH_MAX - 1, file);
	      if (tmp_str != NULL)
		{
		  char *key;
		  tmp_str = strpbrk(url, "\r\n");
		  if (tmp_str != NULL)
		    *tmp_str = '\0';
		  key = g_strdup(url);
		  g_hash_table_insert(a_bone_url_table, key, key);
		}
	    }
	  fclose(file);
	}
      else
	close(fd);
    }


  /* ܡ󥵡ФΥꥹȤե뤫ɤ */
  fd = ochusha_config_open_file(&application->config,
				OCHUSHA_A_BONE_SERVER_TXT, NULL, O_RDONLY);
  if (fd >= 0)
    {
      FILE *file = fdopen(fd, "r");
      if (file != NULL)
	{
	  char server[PATH_MAX];
	  server[PATH_MAX - 1] = '\0';
	  while (!feof(file))
	    {
	      char *tmp_str = fgets(server, PATH_MAX - 1, file);
	      if (tmp_str != NULL)
		{
		  char *key;
		  tmp_str = strpbrk(server, "\r\n");
		  if (tmp_str != NULL)
		    *tmp_str = '\0';
		  key = g_strdup(server);
		  g_hash_table_insert(a_bone_server_table, key, key);
		}
	    }
	  fclose(file);
	}
      else
	close(fd);
    }
}


static gboolean
output_a_bone_key(gpointer key, gpointer value, gpointer user_data)
{
  FILE *file = (FILE *)user_data;

  fprintf(file, "%s\n", (const char *)key);

  return TRUE;
}


void
finalize_image_ui(OchushaApplication *application)
{
  int fd = ochusha_config_open_file(&application->config,
				    OCHUSHA_A_BONE_IMAGE_TXT, NULL,
				    O_WRONLY | O_TRUNC | O_CREAT);
  if (fd >= 0)
    {
      FILE *file = fdopen(fd, "w");
      /* ܡURLꥹȤե˽񤭽Ф */
      if (file != NULL)
	{
	  g_hash_table_foreach(a_bone_url_table,
			       (GHFunc)output_a_bone_key, file);
	  fclose(file);
	}
      else
	close(fd);
    }

  fd = ochusha_config_open_file(&application->config,
				OCHUSHA_A_BONE_SERVER_TXT, NULL,
				O_WRONLY | O_TRUNC | O_CREAT);
  if (fd >= 0)
    {
      FILE *file = fdopen(fd, "w");
      /* ܡURLꥹȤե˽񤭽Ф */
      if (file != NULL)
	{
	  g_hash_table_foreach(a_bone_server_table,
			       (GHFunc)output_a_bone_key, file);
	  fclose(file);
	}
      else
	close(fd);
    }
}


static void
a_bone_image_shown(GtkImage *image, gpointer unused)
{
  g_return_if_fail(GTK_IS_IMAGE(image));
  gtk_image_set_from_stock(image, GTK_STOCK_DIALOG_WARNING,
			   GTK_ICON_SIZE_DIALOG);
}


static void
a_bone_image_in_table(ImageInfo *info, gpointer unused)
{
  g_slist_foreach(info->widget_list, (GFunc)a_bone_image_shown, NULL);
}


void
ochusha_a_bone_image(OchushaApplication *application, const char *url)
{
  /* gdk_threads_enter()ĶƤ֤ */
  GSList *info_list;
  char *key;

  if (url == NULL || *url == '\0')
    return;

  key = g_strdup(url);

  g_hash_table_insert(a_bone_url_table, key, key);

  info_list = g_hash_table_lookup(image_info_table, key);
  g_slist_foreach(info_list, (GFunc)a_bone_image_in_table, NULL);

  ochusha_config_cache_unlink_file(&application->config, url);
  ochusha_config_image_cache_unlink_file(&application->config, url);
}


void
ochusha_a_bone_image_server(OchushaApplication *application, const char *url)
{
  /* gdk_threads_enter()ĶƤ֤ */
  char *server;

  if (url == NULL || *url == '\0')
    return;

  server = ochusha_utils_url_extract_http_server(url);
  if (server == NULL)
    return;

  g_hash_table_insert(a_bone_server_table, server, server);
}


static gboolean
set_pixbuf_on_idle(ImageInfo *image_info)
{
  GSList *widget_list;

  OCHU_THREADS_ENTER();

  widget_list = image_info->widget_list;

  if (image_info->animation != NULL)
    {
      GdkPixbufAnimation *animation = image_info->animation;
      while (widget_list != NULL)
	{
	  GtkImage *image = (GtkImage *)widget_list->data;
	  gtk_image_set_from_animation(image, animation);
	  widget_list = widget_list->next;
	}
    }
  else if (image_info->pixbuf != NULL)
    {
      GdkPixbuf *pixbuf = image_info->pixbuf;
      while (widget_list != NULL)
	{
	  GtkImage *image = (GtkImage *)widget_list->data;
	  gtk_image_set_from_pixbuf(image, pixbuf);
	  widget_list = widget_list->next;
	}
    }
  else
    {
      while (widget_list != NULL)
	{
	  GtkImage *image = (GtkImage *)widget_list->data;
	  gtk_image_set_from_stock(image, GTK_STOCK_MISSING_IMAGE,
				   GTK_ICON_SIZE_DIALOG);
	  widget_list = widget_list->next;
	}
    }

  if (image_info->widget_list == NULL
      && (image_info->animation != NULL || image_info->pixbuf != NULL))
    {
      /* 180øinfoơ֥뤫õ */
      image_info->timeout_id
	= g_timeout_add(IMAGE_CACHE_TIMEOUT,
			(GSourceFunc)image_info_timeout_cb, image_info);
    }

  image_info->idle_id = 0;

  OCHU_THREADS_LEAVE();

  return FALSE;
}


static void
set_pixbuf(ImageInfo *image_info)
{
  /* gdk_threads_enter()ƤĶ餷ƤӽФʤߤ */
  GdkPixbufLoader *loader = image_info->loader;
  GdkPixbufAnimation *animation;

  if (image_info->animation != NULL || image_info->pixbuf != NULL)
    return;

  if (g_hash_table_lookup(a_bone_url_table, image_info->url) != NULL)
    return;

  if (loader == NULL)
    return;

  g_return_if_fail(image_info->idle_id == 0);

  animation = gdk_pixbuf_loader_get_animation(loader);
  if (animation != NULL)
    {
      if (image_info->animation == NULL)
	{
	  image_info->animation = animation;
	  g_object_ref(animation);
	}
    }
  else
    {
      GdkPixbuf *pixbuf = gdk_pixbuf_loader_get_pixbuf(loader);
      if (pixbuf != NULL)
	{
	  if (image_info->pixbuf == NULL)
	    {
	      image_info->pixbuf = pixbuf;
	      g_object_ref(pixbuf);
	    }
	}
    }

  image_info->idle_id = g_idle_add_full(G_PRIORITY_HIGH_IDLE + 13,
					(GSourceFunc)set_pixbuf_on_idle,
					image_info, NULL);

  image_info->loader = NULL;
  g_object_unref(loader);
}


static void
size_prepared_cb(GdkPixbufLoader *loader, int width, int height,
		 ImageInfo *image_info)
{
  double width_request;
  double height_request;
  double width_scale;
  double height_scale;

  if (g_hash_table_lookup(a_bone_url_table, image_info->url) != NULL)
    return;

  width_request = image_info->width;
  height_request = image_info->height;

  if (width <= width_request && height <= height_request)
    return;	/* Ϥʤ */

  width_scale = width_request / width;
  height_scale = height_request / height;
  if (width_scale > height_scale)
    {
      width *= height_scale;
      height *= height_scale;
    }
  else
    {
      width *= width_scale;
      height *= width_scale;
    }
  if (width < 0 || height < 0)
    return;

  gdk_pixbuf_loader_set_size(loader, width, height);

  if (gdk_pixbuf_loader_get_animation(loader) != NULL
      || gdk_pixbuf_loader_get_pixbuf(loader) != NULL)
    set_pixbuf(image_info);
}


static void
area_prepared_cb(GdkPixbufLoader *loader, ImageInfo *image_info)
{
  set_pixbuf(image_info);
}


static void
area_updated_cb(GdkPixbufLoader *loader, int x, int y, int width, int height,
		ImageInfo *image_info)
{
  set_pixbuf(image_info);
}


static void
download_image(WorkerThread *employee, gpointer args)
{
  ImageInfo *image_info = (ImageInfo *)args;
  GdkPixbufLoader *loader = image_info->loader;
  OchushaAsyncBuffer *buffer = image_info->buffer;
  gboolean image_shown = FALSE;

  if (image_info->width >= 0 || image_info->height >= 0)
    g_signal_connect(G_OBJECT(loader), "size_prepared",
		     G_CALLBACK(size_prepared_cb), image_info);

  g_signal_connect(G_OBJECT(loader), "area_prepared",
		   G_CALLBACK(area_prepared_cb), image_info);

  g_signal_connect(G_OBJECT(loader), "area_updated",
		   G_CALLBACK(area_updated_cb), image_info);

  if (ochusha_async_buffer_active_ref(buffer))
    {
      size_t offset = 0;

      ochusha_async_buffer_lock(buffer);
      {
	while (TRUE)
	  {
	    size_t length = buffer->length;
	    if (length > 0 && length > offset)
	      {
		size_t updated_len = length - offset;
		OCHU_THREADS_ENTER();
		image_shown
		  |= gdk_pixbuf_loader_write(loader,
					     ((const guchar *)buffer->buffer
					      + offset), updated_len, NULL);
		OCHU_THREADS_LEAVE();
	      }
	    if (buffer->fixed)
	      goto terminated;
	    offset = length;
	    do
	      {
		if (!ochusha_async_buffer_wait(buffer))
		  goto terminated;
	      } while (length >= buffer->length && !buffer->fixed);
	  }
      }
    terminated:
      ochusha_async_buffer_unlock(buffer);
      ochusha_async_buffer_active_unref(buffer);
    }

  OCHU_THREADS_ENTER();
  if (!image_shown)
    set_pixbuf(image_info);

  gdk_pixbuf_loader_close(loader, NULL);

  image_info->loader = NULL;
  image_info->buffer = NULL;

  OCHU_THREADS_LEAVE();

  if (image_shown)
    {
      /* ХåեƤå¸ */
      OchushaApplication *application
	= g_object_get_data(G_OBJECT(buffer), "application");
      if (application != NULL)
	{
	  int fd = ochusha_config_image_cache_open_file(&application->config,
						image_info->url,
						O_WRONLY | O_TRUNC | O_CREAT);
	  if (fd >= 0)
	    {
	      ssize_t len = write(fd, (void *)buffer->buffer, buffer->length);
	      close(fd);
	      if (len != buffer->length)
		{
		  ochusha_config_image_cache_unlink_file(&application->config,
							 image_info->url);
		}
	    }
	}
    }

  OCHU_OBJECT_UNREF(buffer);

  g_object_unref(loader);
}


GtkWidget *
ochusha_download_image(OchushaApplication *application, const char *url,
		       int width, int height)
{
  OchushaAsyncBuffer *buffer;
  GtkWidget *image;
  WorkerJob *job = NULL;
  GdkPixbufLoader *loader;
  char *scheme;
  char *server;

  ImageInfo *image_info;
  GSList *info_list;

  g_return_val_if_fail(url != NULL, NULL);

  if (*url == '\0')
    return NULL;

  if (g_hash_table_lookup(a_bone_url_table, url) != NULL)
    return NULL;

  server = ochusha_utils_url_extract_http_server(url);
  if (server == NULL)
    return NULL;
  if (g_hash_table_lookup(a_bone_server_table, server) != NULL)
    {
      G_FREE(server);
      return NULL;
    }
  G_FREE(server);

  scheme = ochusha_utils_url_extract_scheme(url);
  if (scheme == NULL)
    return NULL;
  if (strcmp(scheme, "http") != 0)
    {
      G_FREE(scheme);
      return NULL;
    }
  G_FREE(scheme);

  image_info = lookup_image_info(url, width, height);
  if (image_info != NULL)
    {
      if (image_info->animation != NULL)
	image = gtk_image_new_from_animation(image_info->animation);
      else if (image_info->pixbuf != NULL)
	image = gtk_image_new_from_pixbuf(image_info->pixbuf);
      else if (image_info->loader != NULL)
	image = gtk_image_new_from_stock(OCHUSHA_STOCK_MISSING_IMAGE,
					 GTK_ICON_SIZE_DIALOG);
      else
	image = gtk_image_new_from_stock(GTK_STOCK_MISSING_IMAGE,
					 GTK_ICON_SIZE_DIALOG);
      if (image_info->timeout_id != 0)
	{
	  g_source_remove(image_info->timeout_id);
	  image_info->timeout_id = 0;
	}

      image_info->widget_list = g_slist_append(image_info->widget_list, image);
      g_signal_connect(G_OBJECT(image), "destroy",
		       G_CALLBACK(image_destroy_cb), image_info);

      return image;
    }

  info_list = g_hash_table_lookup(image_info_table, url);
  buffer = NULL;
  while (info_list != NULL)
    {
      image_info = (ImageInfo *)info_list->data;
      if (image_info->loader == NULL
	  && image_info->animation == NULL && image_info->pixbuf == NULL)
	{
	  /* Ǥ뤳ȤȽƤ */
	  image = gtk_image_new_from_stock(GTK_STOCK_MISSING_IMAGE,
					   GTK_ICON_SIZE_DIALOG);
	  image_info = image_info_new(url, NULL, image);
	  image_info->width = width;
	  image_info->height = height;

	  info_list = g_hash_table_lookup(image_info_table, url);
	  if (info_list != NULL)
	    info_list = g_slist_append(info_list, image_info);
	  else
	    g_hash_table_insert(image_info_table, image_info->url,
				g_slist_append(NULL, image_info));
	  return image;
	}

      if (image_info->buffer != NULL)
	{
	  buffer = image_info->buffer;
	  OCHU_OBJECT_REF(buffer);
	  break;
	}
      info_list = info_list->next;
    }

  if (buffer == NULL)
    {
      /* ߴΤ */
      buffer = ochusha_network_broker_read_from_url(application->broker,
					NULL, url, NULL,
					OCHUSHA_NETWORK_BROKER_CACHE_ONLY,
					TRUE,
					application->image_chunksize);
    }

  if (buffer == NULL)
    {
      buffer = ochusha_network_broker_read_from_url(application->broker,
					NULL, url, NULL,
					OCHUSHA_NETWORK_BROKER_IMAGE_CACHE,
					TRUE,
					application->image_chunksize);
    }

  if (buffer == NULL)
    {
      buffer = ochusha_network_broker_read_from_url(application->broker,
					NULL, url, NULL,
					OCHUSHA_NETWORK_BROKER_CACHE_IGNORE,
					TRUE,
					application->image_chunksize);
      if (buffer != NULL)
	g_object_set_data(G_OBJECT(buffer), "application", application);
    }

  if (buffer == NULL)
    return NULL;

  loader = gdk_pixbuf_loader_new();
  image = gtk_image_new_from_stock(OCHUSHA_STOCK_MISSING_IMAGE,
				   GTK_ICON_SIZE_DIALOG);
  image_info = image_info_new(url, loader, image);
  image_info->width = width;
  image_info->height = height;
  image_info->buffer = buffer;

  info_list = g_hash_table_lookup(image_info_table, url);
  if (info_list != NULL)
    info_list = g_slist_append(info_list, image_info);
  else
    g_hash_table_insert(image_info_table, image_info->url,
			g_slist_append(NULL, image_info));


  job = G_NEW0(WorkerJob, 1);
  job->canceled = FALSE;
  job->job = download_image;
  job->args = image_info;

  commit_modest_job(job);

  return image;
}
