//
//  NSFileManager+OpenMetaKitPrivate.m
//  OpenMetaKit
//
//  Created by s-take on 09/02/08.
//  
//  The MIT License
//  
//  Copyright (c) 2009 s-take.
//  
//  Permission is hereby granted, free of charge, to any person obtaining a copy
//  of this software and associated documentation files (the "Software"), to deal
//  in the Software without restriction, including without limitation the rights
//  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//  copies of the Software, and to permit persons to whom the Software is
//  furnished to do so, subject to the following conditions:
//  
//  The above copyright notice and this permission notice shall be included in
//  all copies or substantial portions of the Software.
//  
//  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
//  THE SOFTWARE.
//  

#import "NSFileManager+OpenMetaKitPrivate.h"
#import "NSArray+OpenMetaKitPrivate.h"
#import "NSFileManager+OpenMetaKit.h"
#import "OMKConstants.h"
#import <sys/xattr.h>


@implementation NSFileManager (OpenMetaKitPrivate)

- (NSString *)namespacedAttributeNameFromLocalAttributeName:(NSString *)attribute
{
	return [NSString stringWithFormat:@"com.apple.metadata:%@", attribute];
}

- (NSString *)localAttributeNameFromNamespacedAttributeName:(NSString *)attribute
{
	NSArray *subarray = [attribute componentsSeparatedByString:@":"];
	return [[subarray subarrayWithRange:NSMakeRange(1, [subarray count] - 1)] componentsJoinedByString:@""];
}

- (NSData *)dataForExtendedAttribute:(NSString *)attribute ofFileAtPath:(NSString *)path error:(NSError **)error
{
	const char *pathString = [path fileSystemRepresentation];
	const char *attributeString= [[self namespacedAttributeNameFromLocalAttributeName:attribute] UTF8String];
	// examine metadata size
	size_t byteSize = getxattr(pathString, attributeString, NULL, 0, 0, 0);
	if(byteSize > 0 ) {
		// get actual data
		NSMutableData *data = [NSMutableData dataWithLength:byteSize];
		size_t realSize = getxattr(pathString, attributeString, [data mutableBytes], byteSize, 0, 0);
		if(realSize > 0) {
			return data;
		}
	}
	if(error) {
		*error = [NSError errorWithDomain:NSPOSIXErrorDomain code:errno userInfo:nil];
	}
	return nil;
}

- (NSError *)setData:(NSData *)data forExtendedAttribute:(NSString *)attribute ofFileAtPath:(NSString *)path
{
	const char *pathString = [path fileSystemRepresentation];
	const char *attributeString = [[self namespacedAttributeNameFromLocalAttributeName:attribute] UTF8String];
	int result = setxattr(pathString, attributeString, [data bytes], [data length], 0, 0);
	if(result < 0) {
		return [NSError errorWithDomain:NSPOSIXErrorDomain code:errno userInfo:nil];
	}
	return nil;
}

- (NSArray *)extendenAttributesOfFileAtPath:(NSString *)path error:(NSError **)error
{
	const char *pathString = [path fileSystemRepresentation];
	ssize_t byteSize = listxattr(pathString, NULL, 0, 0);
	// check error
	if(byteSize < 0) {
		if(error) {
			*error = [NSError errorWithDomain:NSPOSIXErrorDomain code:errno userInfo:nil];
			return nil;
		}
	}
	NSMutableData *data = [NSMutableData dataWithLength:byteSize];
	ssize_t realSize = listxattr(pathString, [data mutableBytes], byteSize, 0);
	// check error
	if(realSize < 0) {
		if(error) {
			*error = [NSError errorWithDomain:NSPOSIXErrorDomain code:errno userInfo:nil];
			return nil;
		}
	}
	NSMutableArray *attrs = [NSMutableArray array];
	char *current = [data mutableBytes];
	for(NSUInteger i = 0; i < realSize; i++) {
		if(((char *)[data mutableBytes])[i] == '\0') {
			NSString *attr = [NSString stringWithUTF8String:current];
			if([attr hasPrefix:@"com.apple.metadata:kOM"]) {
				[attrs addObject:[self localAttributeNameFromNamespacedAttributeName:attr]];
			}
			current += i + 1;
		}
	}
	return attrs;
}

- (void)updateTagHistoryWithTags:(NSArray *)tags
{
	if(!tags || ![tags count]) return;
	NSMutableArray *recents = [NSMutableArray arrayWithArray:tags];
	NSArray *olds = [self tagHistory];
	for(NSUInteger i = 0; i < 200; i++) {
		NSString *oldTag = [olds objectAtIndex:i];
		if(![recents containsString:oldTag]) {
			[recents addObject:oldTag];
		}
	}
	while([recents count] > 200) {
		[recents removeLastObject];
	}
	CFPreferencesSetAppValue(CFSTR("recentlyEnteredTags"), (CFPropertyListRef)recents, CFSTR("com.openmeta.shared"));
	CFPreferencesAppSynchronize(CFSTR("com.openmeta.shared"));
}

- (void)touchUserTaggedDateOfFileAtPath:(NSString *)path withDate:(NSDate *)date
{
	if(!date) {
		date = [NSDate date];
	}
	NSData *dateData = [NSPropertyListSerialization dataFromPropertyList:date
																  format:NSPropertyListBinaryFormat_v1_0
														errorDescription:NULL];
	[self setData:dateData forExtendedAttribute:OMKUserTaggedDateAttributeName ofFileAtPath:path];
}

@end
