//
// tkstream.h -- stream for tokenizer of Open Web Middleware
//
//      Version 1.1
//
//      Copyright (C) 2006 Kazunari Saitoh
//
//      This library is free software; you can redistribute it and/or
//      modify it under the terms of the GNU Lesser General Public
//      License as published by the Free Software Foundation; either
//      version 2.1 of the License, or (at your option) any later version.
//
//      This library is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//      Lesser General Public License for more details.
//
//      You should have received a copy of the GNU Lesser General Public
//      License along with this library; if not, write to the Free Software
//      Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//

#ifndef __OMT_TKSTREAM_H
#define __OMT_TKSTREAM_H

#include <omt/xbuf.h>
#include <omt/gstream.h>

#ifndef NO_NAMESPACE
namespace omt {
#endif

class itkstream : public gistream
{
        gistream*       m_in;
        unsigned        m_line;

	// for partial stream keeping
        bool    m_keep;
        xstr    m_buf;

    public:
        explicit itkstream( FILE* in )
                : m_in( new gin<FILE*>( in )), m_line( 1 ), m_keep( false ) { }
        explicit itkstream( const char* in )
                : m_in( new gin<char*>( in )), m_line( 1 ), m_keep( false ) { }
        explicit itkstream( const xstr& in )
                : m_in( new gin<xstr>( in )), m_line( 1 ), m_keep( false ) { }
#ifndef NO_CPPSTD
        explicit itkstream( std::istream& in )
                : m_in( new gin<std::istream>( in )), m_line( 1 ), m_keep( false ) { }
        explicit itkstream( const std::string& in )
                : m_in( new gin<std::string>( in )), m_line( 1 ), m_keep( false ) { }
#endif

        ~itkstream() { delete m_in; }

        int unget( int a )
        {
                if ( a == '\n' ) m_line--;
                return unkeeping( m_in->unget( a ));
        }
        int get()
        {
                int     r = m_in->get();

                if ( r == '\n' ) m_line++;
                return keeping( r );
        }
        int peek() { return m_in->peek(); }
        bool is_eof() const { return m_in->is_eof(); }

        unsigned line() const { return m_line; }

        int skip( int end = '\n' )
        {
                int     r;

                while (( r = get()) >= 0 && r != end ) ;
                return r;
        }
        int skip( const char* pat )
        {
                int     r;

                while (( r = get()) >= 0 ) {
                        const char* p;
                        for ( p = pat; *p == r && *++p; ) r = get();
                        if ( *p == '\0' ) break;
                }
                return r;
        }

        void keep()
	{
		m_buf.clear();
		m_keep = true;
	}
        const char* kept()
	{
		m_keep = false;
		return m_buf.c_str();
	}

    private:
        itkstream( const itkstream& );
        itkstream& operator=( const itkstream& );

	// wrapper for partial stream keeping
        int keeping( int a )
        {
                if ( m_keep ) m_buf.put( a );
                return a;
        }
        int unkeeping( int a )
	{
		if ( m_keep ) m_buf.pop();
		return a;
	}
};

#ifndef NO_NAMESPACE
}
#endif

#endif // __OMT_TKSTREAM_H

