//
// utest.h --- unit test utility for Open Web Middleware
//
//	Version 1.1
//
//      Copyright (C) 2003, 2005 Kazunari Saitoh
//
//      This library is free software; you can redistribute it and/or
//      modify it under the terms of the GNU Lesser General Public
//      License as published by the Free Software Foundation; either
//      version 2.1 of the License, or (at your option) any later version.
//
//      This library is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//      Lesser General Public License for more details.
//
//      You should have received a copy of the GNU Lesser General Public
//      License along with this library; if not, write to the Free Software
//      Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//

#ifndef __OMT_UTEST_H
#define __OMT_UTEST_H

#include <cstdio>
#include <cstdlib>
#include <omt/xbuf.h>

#ifndef NO_NAMESPACE
namespace omt {
#endif

// macros
// -----------------------------------------------------------------------------
#define AssertEqual unittpos( __FILE__, __LINE__ ).assertEqual
#define AssertZero unittpos( __FILE__, __LINE__ ).assertZero
#define AssertNonZero unittpos( __FILE__, __LINE__ ).assertNonZero
#define AssertTrue unittpos( __FILE__, __LINE__ ).assertTrue

// -----------------------------------------------------------------------------
class unittpos
{
	const char*	m_file;
	size_t     	m_line;

    public:
	unittpos( const char* fn, size_t ln ) : m_file( fn ), m_line( ln ) { }
	
	template <typename T> void assertEqual( const T& a, const T& b ) { assertEqual( "-", a, b ); }
	template <typename T> void assertEqual( unsigned n, const T& a, const T& b )
	{
		char	buf[ 16 ];

		snprintf( buf, 16, "%u", n );
		assertEqual( buf, a, b );
	}
	template <typename T> void assertEqual( const char* msg, const T& a, const T& b )
	{
		if ( a == b ) {
			printf( "ok: %s %s(%d)\n", msg, m_file, m_line );
		} else {
			printf( "fail: %s %s(%d)\n", msg, m_file, m_line );
			exit( 0 );
		}
	}

	template <typename T> void assertZero( const T& a ) { assertZero( "-", a ); }
	template <typename T> void assertZero( unsigned n, const T& a )
	{
		char	buf[ 16 ];

		snprintf( buf, 16, "%u", n );
		assertZero( buf, a );
	}
	template <typename T> void assertZero( const char* msg, const T& a )
	{
		if ( a == 0 ) {
			printf( "ok: %s %s(%d)\n", msg, m_file, m_line );
		} else {
			printf( "fail: %s %s(%d)\n", msg, m_file, m_line );
			exit( 0 );
		}
	}

	template <typename T> void assertNonZero( const T& a ) { assertNonZero( "-", a ); }
	template <typename T> void assertNonZero( unsigned n, const T& a )
	{
		char	buf[ 16 ];

		snprintf( buf, 16, "%u", n );
		assertNonZero( buf, a );
	}
	template <typename T> void assertNonZero( const char* msg, const T& a )
	{
		if ( a != 0 ) {
			printf( "ok: %s %s(%d)\n", msg, m_file, m_line );
		} else {
			printf( "fail: %s %s(%d)\n", msg, m_file, m_line );
			exit( 0 );
		}
	}

	template <typename T> void assertTrue( const T& a ) { assertTrue( "-", a ); }
	template <typename T> void assertTrue( unsigned n, const T& a )
	{
		char	buf[ 16 ];

		snprintf( buf, 16, "%u", n );
		assertTrue( buf, a );
	}
	template <typename T> void assertTrue( const char* msg, const T& a )
	{
		if ( a ) {
			printf( "ok: %s %s(%d)\n", msg, m_file, m_line );
		} else {
			printf( "fail: %s %s(%d)\n", msg, m_file, m_line );
			exit( 0 );
		}
	}
};

// base for unittest class
// -----------------------------------------------------------------------------
class unittest
{
	xstr	m_testname;
	xstr	m_subtitle;

    public:
	unittest() : m_testname( "Unit Test" ) { }
	unittest( const char* title ) : m_testname( title )
	{
		printf( "\n*** %s start. ***\n", m_testname.c_str());
	}
	virtual ~unittest()
	{
		printf( "\n*** %s done. ***\n", m_testname.c_str());
	}

	void set_subtitle( const char* s )
	{
		m_subtitle = s;
		printf( "\n*** %s ***\n", s );
	}

	const char* get_testname() { return m_testname.c_str(); }
	const char* get_subtitle() { return m_subtitle.c_str(); }

	// need to implement
	virtual void run() = 0;
	
    private:
	unittest( const unittest& );
	unittest& operator=( const unittest& );
};

#ifndef NO_NAMESPACE
}
#endif
#endif
