//
// phashtest.cpp
//
//	Copyright (C) Kazunari Saitoh, 2003, 2006 all right reserved.
//

#include <cstdio>
#include <cstdarg>
#include <omt/common.h>
#include <omt/phash.h>
#include <omt/utest.h>

using namespace std;
using namespace omt;

// Test Data Class
// -----------------------------------------------------------------------------
class test {
        int     m_int;
        char*   m_str;

        static int cntr;

    public:
        test( int n, const char* b ) : m_int( n ), m_str( strdup( b ))
        {
                ++cntr;
#ifdef _DEBUG
                printf( "%03d: ==> alloc test(%d %s)\n", cntr, m_int, m_str ? m_str : "<null>" );
#endif
        }
        ~test()
        {
#ifdef _DEBUG
                printf( "%03d: <== free test(%d %s)\n", cntr, m_int, m_str ? m_str : "<null>" );
#endif
                --cntr;
                if ( m_str ) delete [] m_str;
        }

        void print() const
        {
                printf( "(%d, \"%s\")", m_int, m_str ? m_str : "<null>" );
        }

        const char*     cstr() const { return m_str; }
        int             num()  const { return m_int; }

        static int getCntr() { return cntr; }

    private:
        test( const test& t );
        test& operator=( const test& t );
};

int test::cntr = 0;

// -----------------------------------------------------------------------------
bool operator== ( const test& a, const test& b )
{
        return a.num() == b.num() && eql_fn<const char*>()( a.cstr(), b.cstr());
}

// -----------------------------------------------------------------------------
test* dup( const test* p )
{
        return p ? new test( p->num(), p->cstr()) : 0;
}

// -----------------------------------------------------------------------------
namespace omt {
	template<> struct hash_fn<const test*>
	{
       		unsigned long operator()( const test* p ) const
        	{
			hash_fn<const char*>	fs;
			hash_fn<unsigned long>	fn;

               		return fs( p->cstr()) + fn( p->num()) * 7; 
        	}
	};

	template<> struct eql_fn<const test*>
	{
       		bool operator()( const test* p, const test* q ) const
        	{
			return !p && !q || p && q && *p == *q; 
        	}
	};

}


// Utilities
// -----------------------------------------------------------------------------
template<typename P, size_t Z, typename Q>
void print( const phash<test*,P,Z,const char*,Q>& h )
{
	typename phash<test*,P,Z,const char*,Q>::const_itor i( h );
	test*	p;

	printf( "---> " );
	for ( i.init(); i.cont(); i.next()) {
		if (( p = i.get()) != 0 ) {
			printf( "%s=", i.key()); p->print(); printf( ", " );
		} else {
			printf( "%s=null, ", i.key());
		}
	}
	printf( " <---\n" );
	fflush( stdout );
}

// -----------------------------------------------------------------------------
template<typename P, size_t Z, typename Q>
void print( const phash<const char*,P,Z,const char*,Q>& h )
{
	typename phash<const char*,P,Z,const char*,Q>::const_itor i( h );
	const char*	p;

	printf( "---> " );
	for ( i.init(); i.cont(); i.next()) {
		if (( p = i.get()) != 0 ) {
			printf( "%s=\"%s\", ", i.key(), p );
		} else {
			printf( "%s=null, ", i.key());
		}
	}
	printf( " <---\n" );
	fflush( stdout );
}

// -----------------------------------------------------------------------------
template<size_t Z, typename Q>
void print( const phash<int,refer_policy<int>,Z,const char*,Q>& h )
{
	typename phash<const char*,refer_policy<int>,Z,const char*,Q>::const_itor i( h );
	const char*	p;

	printf( "---> " );
	for ( i.init(); i.cont(); i.next()) printf( "%s=%d, ", i.key(), p );
	printf( " <---\n" );
	fflush( stdout );
}

// -----------------------------------------------------------------------------
template<typename P, size_t Z, typename Q>
void print( const phash<test*,P,Z,const test*,Q>& h )
{
	typename phash<test*,P,Z,const test*,Q>::const_itor i( h );

	printf( "---> " );
	for ( i.init(); i.cont(); i.next()) {
		const test*	k = i.key();
		test*		p = i.get();
		if ( p && k ) {
			k->print(); putc( '=', stdout ); p->print(); printf( ", " );
		} else if ( k ) {
			k->print(); printf( "=null, " );
		} else {
			printf( "***illegal null key*** ( p=" );
			if ( p ) p->print(); else printf( "null" );
			printf( " ), " );
		}
	}
	printf( " <---\n" );
	fflush( stdout );
}

// -----------------------------------------------------------------------------
template<typename P, size_t Z, typename Q>
void print( const phash<const char*,P,Z,const test*,Q>& h )
{
	typename phash<const char*,P,Z,const test*,Q>::const_itor i( h );

	printf( "---> " );
	for ( i.init(); i.cont(); i.next()) {
		const test*	k = i.key();
		const char*	p = i.get();
		if ( p && k ) {
			k->print(); printf( "=\"%s\", ", p );
		} else if ( k ) {
			k->print(); printf( "=null, " );
		} else {
			printf( "***illegal null key*** ( p=%s ), ", p ? p : "null" );
		}
	}
	printf( " <---\n" );
	fflush( stdout );
}

// -----------------------------------------------------------------------------
template<typename P, size_t Z, typename Q>
bool check_testhash( const phash<test*,P,Z,const char*,Q>& hs, size_t n, ... )
{
	va_list							ap;
	typename phash<test*,P,Z,const char*,Q>::const_itor	i( hs );
	size_t							j;

	va_start( ap, n );
	for ( i.init(), j = 0; i.cont() && j < n; i.next(), ++j ) {
		const char*	k = va_arg( ap, const char* );
		test*		p = va_arg( ap, test* );
		test*		q = hs.find( k );

		if ( p && q == 0 || p == 0 && q || p && q && !( *p == *q )) break;
	}
	va_end( ap );

	return !i.cont() && j == n;
}

// -----------------------------------------------------------------------------
template<typename P, size_t Z, typename Q>
bool check_wtesthash( const phash<test*,P,Z,const test*,Q>& hs, size_t n, ... )
{
	va_list							ap;
	typename phash<test*,P,Z,const test*,Q>::const_itor	i( hs );
	size_t							j;

	va_start( ap, n );
	for ( i.init(), j = 0; i.cont() && j < n; i.next(), ++j ) {
		const test*	k = va_arg( ap, const test* );
		test*		p = va_arg( ap, test* );
		test*		q = hs.find( k );

		if ( p && q == 0 || p == 0 && q || p && q && !( *p == *q )) break;
	}
	va_end( ap );

	return !i.cont() && j == n;
}

// -----------------------------------------------------------------------------
template<typename P, size_t Z, typename Q>
bool check_objkyhash( const phash<const char*,P,Z,const test*,Q>& hs, size_t n, ... )
{
	va_list								ap;
	typename phash<const char*,P,Z,const test*,Q>::const_itor	i( hs );
	size_t								j;

	va_start( ap, n );
	for ( i.init(), j = 0; i.cont() && j < n; i.next(), ++j ) {
		const test*	k = va_arg( ap, const test* );
		const char*	p = va_arg( ap, const char* );
		const char*	q = hs.find( k );

		if ( p && q == 0 || p == 0 && q || p && q && !( *p == *q )) break;
	}
	va_end( ap );

	return !i.cont() && j == n;
}

// -----------------------------------------------------------------------------
template<typename P, size_t Z>
bool check_intkyhash( const phash<test*,P,Z,int,refer_policy<int> >& hs, size_t n, ... )
{
	va_list									ap;
	typename phash<test*,P,Z,int,refer_policy<int> >::const_itor	i( hs );
	size_t									j;

	va_start( ap, n );
	for ( i.init(), j = 0; i.cont() && j < n; i.next(), ++j ) {
		int		k = va_arg( ap, int );
		test*		p = va_arg( ap, test* );
		test*		q = hs.find( k );

		if ( p && q == 0 || p == 0 && q || p && q && !( *p == *q )) break;
	}
	va_end( ap );

	return !i.cont() && j == n;
}

// Unit Test Class
// -----------------------------------------------------------------------------
class phashtest : public unittest
{
	// test data
        int i1;
        int i2;
        int i3;
        int i4;
        int i5;
        int i6;
        int i7;
        int i8;

        const char* s0;
        const char* s1;
        const char* s2;
        const char* s3;
        const char* s4;
        const char* s5;
        const char* s6;
        const char* s7;
        const char* s8;

	test t0;
        test t1;
        test t2;
        test t3;
        test t4;
        test t5;
        test t6;
        test t7;
        test t8;

	int	m_cnt;
   
    public:
	void test11();
        void test12();
        void test13();
        void test14();
        void test15();
        void test16();
        void test17();

        void test21();
        void test22();
        void test23();
        void test24();
        void test25();
        void test26();
        void test27();

        void test31();
        void test32();
        void test33();
        void test34();
        void test35();
        void test36();
        void test37();
        void test38();

        void test41();
        void test42();
        void test43();
        void test44();
        void test45();
        void test46();
        void test47();

	phashtest() : unittest( "test for phash.h" ),

                i1( 1 ),
                i2( 2 ),
                i3( 3 ),
                i4( 4 ),
                i5( 5 ),
                i6( 6 ),
                i7( 7 ),
                i8( 8 ),

		s0( "" ),
                s1( "Hello" ),
                s2( "World" ),
                s3( "Open" ),
                s4( "Client" ),
                s5( "Server" ),
                s6( "Open" ),		// same value with s3
                s7( "Middleware" ),
                s8( "Web site" ),

		t0( 0, "" ),
                t1( i1, s1 ),
                t2( i2, s2 ),
                t3( i3, s3 ),
                t4( i4, s4 ),
                t5( i5, s5 ),
                t6( i3, s6 ),		// same value with s3
                t7( i7, s7 ),
                t8( i8, s8 )
	{
		m_cnt = test::getCntr();
	}
	~phashtest()
	{
		this->set_subtitle( "test class allocation" );
		AssertEqual( m_cnt, test::getCntr());
	}
	void run();
};


// phash operations
// -----------------------------------------------------------------------------
void phashtest::test11()
{
	this->set_subtitle( "miscellaneous operations to phash - refer/copy policy" );

	typedef phash<test*,refer_policy<test*>,16> hash;

	hash		ht;

	AssertZero( ht.length());
	AssertZero( ht.set( s1, &t1 ));	
	AssertEqual( ht.length(), 1u );
	AssertZero( ht.set( s2, &t2 ));	
	AssertZero( ht.set( s3, &t3 ));	
	AssertZero( ht.set( s0, 0 ));	
	AssertEqual( &t3, ht.set( s6, &t6 ));	
	AssertZero( ht.set( s5, &t5 ));
	AssertEqual( ht.length(), 5u );
	AssertEqual( ht.size(), 16u );
	AssertNonZero( check_testhash( ht, 5, s1, &t1, s2, &t2, s6, &t6, s0, 0, s5, &t5 ));

	AssertEqual( *ht.get( s3 ), t6 );
	AssertZero( ht.get( s4 ));

	AssertEqual( *ht.find( s3 ), t6 );
	AssertZero( ht.find( s4 ));

	AssertEqual( &t2, ht.remove( s2 ));
	AssertEqual( &t6, ht.remove( s3 )); 
	AssertZero( ht.remove( s4 )); 
	AssertEqual( &t5, ht.remove( s5 ));
	AssertNonZero( check_testhash( ht, 2, s1, &t1, s0, 0 ));

	ht.clear();
	AssertNonZero( check_testhash( ht, 0 ));
	AssertZero( ht.set( s6, &t6 ));
	AssertZero( ht.set( s7, &t7 ));
	AssertZero( ht.set( s8, &t8 ));
	AssertNonZero( check_testhash( ht, 3, s6, &t6, s7, &t7, s8, &t8 ));

	AssertNonZero( ht.replace( s7, &t4 ));
	AssertZero( ht.replace( s5, &t5 ));
	AssertNonZero( check_testhash( ht, 3, s6, &t6, s7, &t4, s8, &t8 ));

	AssertZero( ht.insert( s8, &t3 ));
	AssertNonZero( ht.insert( s5, &t5 ));
	AssertNonZero( check_testhash( ht, 4, s5, &t5, s6, &t6, s7, &t4, s8, &t8 ));
}

// -----------------------------------------------------------------------------
void phashtest::test12()
{
	this->set_subtitle( "miscellaneous operations to phash - store/copy policy" );

	typedef phash<test*,store_policy<test*>,16> hash;

	hash		ht;

	AssertZero( ht.length());
	AssertZero( ht.set( s1, dup( &t1 )));	
	AssertEqual( ht.length(), 1u );
	AssertZero( ht.set( s2, dup( &t2 )));	
	AssertZero( ht.set( s3, dup( &t3 )));	
	AssertZero( ht.set( s0, 0 ));	
	AssertZero( ht.set( s6, dup( &t6 )));	
	AssertZero( ht.set( s5, dup( &t5 )));
	AssertEqual( ht.length(), 5u );
	AssertEqual( ht.size(), 16u );
	AssertNonZero( check_testhash( ht, 5, s1, &t1, s2, &t2, s6, &t6, s0, 0, s5, &t5 ));

	AssertEqual( *ht.get( s3 ), t6 );
	AssertZero( ht.get( s4 ));

	AssertEqual( *ht.find( s3 ), t6 );
	AssertZero( ht.find( s4 ));

	AssertZero( ht.remove( s2 ));
	AssertZero( ht.remove( s3 )); 
	AssertZero( ht.remove( s4 )); 
	AssertZero( ht.remove( s5 ));
	AssertNonZero( check_testhash( ht, 2, s1, &t1, s0, 0 ));

	ht.clear();
	AssertNonZero( check_testhash( ht, 0 ));
	AssertZero( ht.set( s6, dup( &t6 )));
	AssertZero( ht.set( s7, dup( &t7 )));
	AssertZero( ht.set( s8, dup( &t8 )));
	AssertNonZero( check_testhash( ht, 3, s6, &t6, s7, &t7, s8, &t8 ));

	AssertNonZero( ht.replace( s7, dup( &t4 )));
	AssertZero( ht.replace( s5, dup( &t5 )));
	AssertNonZero( check_testhash( ht, 3, s6, &t6, s7, &t4, s8, &t8 ));

	AssertZero( ht.insert( s8, dup( &t3 )));
	AssertNonZero( ht.insert( s5, dup( &t5 )));
	AssertNonZero( check_testhash( ht, 4, s5, &t5, s6, &t6, s7, &t4, s8, &t8 ));
}

// -----------------------------------------------------------------------------
void phashtest::test13()
{
	this->set_subtitle( "miscellaneous operations to phash - copy/copy policy" );

	typedef phash<test*,copy_policy<test*>,16> hash;

	hash		ht;

	AssertZero( ht.length());
	AssertZero( ht.set( s1, &t1 ));	
	AssertEqual( ht.length(), 1u );
	AssertZero( ht.set( s2, &t2 ));	
	AssertZero( ht.set( s3, &t3 ));	
	AssertZero( ht.set( s0, 0 ));	
	AssertZero( ht.set( s6, &t6 ));	
	AssertZero( ht.set( s5, &t5 ));
	AssertEqual( ht.length(), 5u );
	AssertEqual( ht.size(), 16u );
	AssertNonZero( check_testhash( ht, 5, s1, &t1, s2, &t2, s6, &t6, s0, 0, s5, &t5 ));

	AssertEqual( *ht.get( s3 ), t6 );
	AssertZero( ht.get( s4 ));

	AssertEqual( *ht.find( s3 ), t6 );
	AssertZero( ht.find( s4 ));

	AssertZero( ht.remove( s2 ));
	AssertZero( ht.remove( s3 )); 
	AssertZero( ht.remove( s4 )); 
	AssertZero( ht.remove( s5 ));
	AssertNonZero( check_testhash( ht, 2, s1, &t1, s0, 0 ));

	ht.clear();
	AssertNonZero( check_testhash( ht, 0 ));
	AssertZero( ht.set( s6, &t6 ));
	AssertZero( ht.set( s7, &t7 ));
	AssertZero( ht.set( s8, &t8 ));
	AssertNonZero( check_testhash( ht, 3, s6, &t6, s7, &t7, s8, &t8 ));

	AssertNonZero( ht.replace( s7, &t4 ));
	AssertZero( ht.replace( s5, &t5 ));
	AssertNonZero( check_testhash( ht, 3, s6, &t6, s7, &t4, s8, &t8 ));

	AssertZero( ht.insert( s8, &t3 ));
	AssertNonZero( ht.insert( s5, &t5 ));
	AssertNonZero( check_testhash( ht, 4, s5, &t5, s6, &t6, s7, &t4, s8, &t8 ));
}

// -----------------------------------------------------------------------------
void phashtest::test14()
{
	this->set_subtitle( "miscellaneous operations to phash - store/refer policy" );

	typedef phash<test*,store_policy<test*>,16,const char*,refer_policy<const char*> > hash;

	hash		ht;

	AssertZero( ht.length());
	AssertZero( ht.set( t1.cstr(), dup( &t1 )));	
	AssertEqual( ht.length(), 1u );
	AssertZero( ht.set( t2.cstr(), dup( &t2 )));	
	AssertZero( ht.set( t3.cstr(), dup( &t3 )));	
	AssertZero( ht.set( t0.cstr(), 0 ));	
	AssertZero( ht.set( t6.cstr(), dup( &t6 )));	
	AssertZero( ht.set( t5.cstr(), dup( &t5 )));
	AssertEqual( ht.length(), 5u );
	AssertEqual( ht.size(), 16u );
	AssertNonZero( check_testhash( ht, 5,
		 t1.cstr(), &t1, t2.cstr(), &t2, t6.cstr(), &t6, t0.cstr(), 0, t5.cstr(), &t5 ));

	AssertEqual( *ht.get( t3.cstr() ), t6 );
	AssertZero( ht.get( t4.cstr() ));

	AssertEqual( *ht.find( t3.cstr() ), t6 );
	AssertZero( ht.find( t4.cstr() ));

	AssertZero( ht.remove( t2.cstr() ));
	AssertZero( ht.remove( t3.cstr() )); 
	AssertZero( ht.remove( t4.cstr() )); 
	AssertZero( ht.remove( t5.cstr() ));
	AssertNonZero( check_testhash( ht, 2, t1.cstr(), &t1, t0.cstr(), 0 ));

	ht.clear();
	AssertNonZero( check_testhash( ht, 0 ));
	AssertZero( ht.set( t6.cstr(), dup( &t6 )));
	AssertZero( ht.set( t7.cstr(), dup( &t7 )));
	AssertZero( ht.set( t8.cstr(), dup( &t8 )));
	AssertNonZero( check_testhash( ht, 3, t6.cstr(), &t6, t7.cstr(), &t7, t8.cstr(), &t8 ));

	AssertNonZero( ht.replace( t7.cstr(), dup( &t4 )));
	AssertZero( ht.replace( t5.cstr(), dup( &t5 )));
	AssertNonZero( check_testhash( ht, 3, t6.cstr(), &t6, t7.cstr(), &t4, t8.cstr(), &t8 ));

	AssertZero( ht.insert( t8.cstr(), dup( &t3 )));
	AssertNonZero( ht.insert( t5.cstr(), dup( &t5 )));
	AssertNonZero( check_testhash( ht, 4,
		 t5.cstr(), &t5, t6.cstr(), &t6, t7.cstr(), &t4, t8.cstr(), &t8 ));
}

// -----------------------------------------------------------------------------
void phashtest::test15()
{
	this->set_subtitle( "miscellaneous operations to phash - store/refer<obj> policy" );

	typedef phash<test*,store_policy<test*>,16,const test*,refer_policy<const test*> > hash;

	hash	ht;
	test*	p0 = 0;

	AssertZero( ht.length());
	AssertZero( ht.set( &t1, dup( &t1 )));	
	AssertEqual( ht.length(), 1u );
	AssertZero( ht.set( &t2, dup( &t2 )));	
	AssertZero( ht.set( &t3, dup( &t3 )));	
	AssertZero( ht.set( &t0, p0 ));	
	AssertZero( ht.set( &t6, dup( &t6 )));	
	AssertZero( ht.set( &t5, dup( &t5 )));
	AssertEqual( ht.length(), 5u );
	AssertEqual( ht.size(), 16u );
	AssertNonZero( check_wtesthash( ht, 5, &t1, &t1, &t2, &t2, &t6, &t6, &t0, p0, &t5, &t5 ));

	AssertEqual( *ht.get( &t3 ), t6 );
	AssertZero( ht.get( &t4 ));

	AssertEqual( *ht.find( &t3 ), t6 );
	AssertZero( ht.find( &t4 ));

	AssertZero( ht.remove( &t2 ));
	AssertZero( ht.remove( &t3 )); 
	AssertZero( ht.remove( &t4 )); 
	AssertZero( ht.remove( &t5 ));
	AssertNonZero( check_wtesthash( ht, 2, &t1, &t1, &t0, p0 ));

	ht.clear();
	AssertNonZero( check_wtesthash( ht, 0 ));
	AssertZero( ht.set( &t6, dup( &t6 )));
	AssertZero( ht.set( &t7, dup( &t7 )));
	AssertZero( ht.set( &t8, dup( &t8 )));
	AssertNonZero( check_wtesthash( ht, 3, &t6, &t6, &t7, &t7, &t8, &t8 ));

	AssertNonZero( ht.replace( &t7, dup( &t4 )));
	AssertZero( ht.replace( &t5, dup( &t5 )));
	AssertNonZero( check_wtesthash( ht, 3, &t6, &t6, &t7, &t4, &t8, &t8 ));

	AssertZero( ht.insert( &t8, dup( &t3 )));
	AssertNonZero( ht.insert( &t5, dup( &t5 )));
	AssertNonZero( check_wtesthash( ht, 4, &t5, &t5, &t6, &t6, &t7, &t4, &t8, &t8 ));
}

// -----------------------------------------------------------------------------
void phashtest::test16()
{
	this->set_subtitle( "miscellaneous operations to phash - store/store<obj> policy" );

	typedef phash<test*,store_policy<test*>,16,const test*,store_policy<const test*> > hash;

	hash	ht;
	test*	p0 = 0;

	AssertZero( ht.length());
	AssertZero( ht.set( dup( &t1 ), dup( &t1 )));	
	AssertEqual( ht.length(), 1u );
	AssertZero( ht.set( dup( &t2 ), dup( &t2 )));	
	AssertZero( ht.set( dup( &t3 ), dup( &t3 )));	
	AssertZero( ht.set( dup( &t0 ), p0 ));	
	AssertZero( ht.set( dup( &t6 ), dup( &t6 )));	
	AssertZero( ht.set( dup( &t5 ), dup( &t5 )));
	AssertEqual( ht.length(), 5u );
	AssertEqual( ht.size(), 16u );
	AssertNonZero( check_wtesthash( ht, 5, &t1, &t1, &t2, &t2, &t6, &t6, &t0, p0, &t5, &t5 ));

	AssertEqual( *ht.get( dup( &t3 )), t6 );
	AssertZero( ht.get( dup( &t4 )));

	AssertEqual( *ht.find( &t3 ), t6 );
	AssertZero( ht.find( &t4 ));

	AssertZero( ht.remove( &t2 ));
	AssertZero( ht.remove( &t3 )); 
	AssertZero( ht.remove( &t4 )); 
	AssertZero( ht.remove( &t5 ));
	AssertNonZero( check_wtesthash( ht, 2, &t1, &t1, &t0, p0 ));

	ht.clear();
	AssertNonZero( check_wtesthash( ht, 0 ));
	AssertZero( ht.set( dup( &t6 ), dup( &t6 )));
	AssertZero( ht.set( dup( &t7 ), dup( &t7 )));
	AssertZero( ht.set( dup( &t8 ), dup( &t8 )));
	AssertNonZero( check_wtesthash( ht, 3, &t6, &t6, &t7, &t7, &t8, &t8 ));

	AssertNonZero( ht.replace( &t7, dup( &t4 )));		// no need for dup( k ) for replace
	AssertZero( ht.replace( &t5, dup( &t5 )));
	AssertNonZero( check_wtesthash( ht, 3, &t6, &t6, &t7, &t4, &t8, &t8 ));

	AssertZero( ht.insert( dup( &t8 ), dup( &t3 )));
	AssertNonZero( ht.insert( dup( &t5 ), dup( &t5 )));
	AssertNonZero( check_wtesthash( ht, 4, &t5, &t5, &t6, &t6, &t7, &t4, &t8, &t8 ));
}

// -----------------------------------------------------------------------------
void phashtest::test17()
{
	this->set_subtitle( "miscellaneous operations to phash - store/copy<obj> policy" );

	typedef phash<test*,store_policy<test*>,16,const test*,copy_policy<const test*> > hash;

	hash	ht;
	test*	p0 = 0;

	AssertZero( ht.length());
	AssertZero( ht.set( &t1, dup( &t1 )));	
	AssertEqual( ht.length(), 1u );
	AssertZero( ht.set( &t2, dup( &t2 )));	
	AssertZero( ht.set( &t3, dup( &t3 )));	
	AssertZero( ht.set( &t0, p0 ));	
	AssertZero( ht.set( &t6, dup( &t6 )));	
	AssertZero( ht.set( &t5, dup( &t5 )));
	AssertEqual( ht.length(), 5u );
	AssertEqual( ht.size(), 16u );
	AssertNonZero( check_wtesthash( ht, 5, &t1, &t1, &t2, &t2, &t6, &t6, &t0, p0, &t5, &t5 ));

	AssertEqual( *ht.get( &t3 ), t6 );
	AssertZero( ht.get( &t4 ));

	AssertEqual( *ht.find( &t3 ), t6 );
	AssertZero( ht.find( &t4 ));

	AssertZero( ht.remove( &t2 ));
	AssertZero( ht.remove( &t3 )); 
	AssertZero( ht.remove( &t4 )); 
	AssertZero( ht.remove( &t5 ));
	AssertNonZero( check_wtesthash( ht, 2, &t1, &t1, &t0, p0 ));

	ht.clear();
	AssertNonZero( check_wtesthash( ht, 0 ));
	AssertZero( ht.set( &t6, dup( &t6 )));
	AssertZero( ht.set( &t7, dup( &t7 )));
	AssertZero( ht.set( &t8, dup( &t8 )));
	AssertNonZero( check_wtesthash( ht, 3, &t6, &t6, &t7, &t7, &t8, &t8 ));

	AssertNonZero( ht.replace( &t7, dup( &t4 )));
	AssertZero( ht.replace( &t5, dup( &t5 )));
	AssertNonZero( check_wtesthash( ht, 3, &t6, &t6, &t7, &t4, &t8, &t8 ));

	AssertZero( ht.insert( &t8, dup( &t3 )));
	AssertNonZero( ht.insert( &t5, dup( &t5 )));
	AssertNonZero( check_wtesthash( ht, 4, &t5, &t5, &t6, &t6, &t7, &t4, &t8, &t8 ));
}

// -----------------------------------------------------------------------------
void phashtest::test21()
{
	this->set_subtitle( "iterator operation - refer/copy policy" );

	typedef phash<test*,refer_policy<test*>,16>	hash;

	hash		ht;
	hash*		d;
	hash::itor	i( ht );
	int		n;

	ht.set( s1, &t1 );
	ht.set( s2, &t2 );
	ht.set( s3, &t3 );
	ht.set( s0, 0 );
	ht.set( s6, &t4 );
	ht.set( s5, &t5 );
	ht.set( s7, 0 );

	d = dup( &ht );
	for ( i.init(), n = 0; i.cont(); i.next(), ++n ) {
		test*		p = i.get();
		const char*	k = i.key();

		if ( p ) AssertEqual( n, *p, *d->get( k ));
		else 	 AssertZero( n, p );
	}
	delete d;

	// -------------------------------------------
	for ( i.init(); i.cont(); i.next()) 
		if ( ! i.get()) AssertZero( i.set( &t6 ));
	AssertNonZero( check_testhash( ht, 6, 
		s1, &t1, s2, &t2, s0, &t6, s6, &t4, s5, &t5, s7, &t6 ));

	// -------------------------------------------
	for ( i.init(); i.cont(); i.next()) {
		test* p = i.get();
		if ( p != 0 && p->num() % 2 == 1 ) AssertEqual( *i.del(), *p );
	}
	AssertNonZero( check_testhash( ht, 2, s2, &t2, s6, &t4 ));
}

// -----------------------------------------------------------------------------
void phashtest::test22()
{
	this->set_subtitle( "iterator operation - store/copy policy" );

	typedef phash<test*,store_policy<test*>,16>	hash;

	hash		ht;
	hash*		d;
	hash::itor	i( ht );
	int		n;

	ht.set( s1, dup( &t1 ));
	ht.set( s2, dup( &t2 ));
	ht.set( s3, dup( &t3 ));
	ht.set( s0, 0 );
	ht.set( s6, dup( &t4 ));
	ht.set( s5, dup( &t5 ));
	ht.set( s7, 0 );

	d = dup( &ht );
	for ( i.init(), n = 0; i.cont(); i.next(), ++n ) {
		test*		p = i.get();
		const char*	k = i.key();

		if ( p ) AssertEqual( n, *p, *d->get( k ));
		else	 AssertZero( n, p );
	}
	delete d;

	// -------------------------------------------
	for ( i.init(); i.cont(); i.next()) 
		if ( ! i.get()) AssertZero( i.set( dup( &t6 )));

	AssertNonZero( check_testhash( ht, 6, 
		s1, &t1, s2, &t2, s0, &t6, s6, &t4, s5, &t5, s7, &t6 ));

	// -------------------------------------------
	for ( i.init(); i.cont(); i.next()) {
		test* p = i.get();
		if ( p != 0 && p->num() % 2 == 1 ) AssertEqual( *i.del(), *p );
	}
	AssertNonZero( check_testhash( ht, 2, s2, &t2, s6, &t4 ));
}

// -----------------------------------------------------------------------------
void phashtest::test23()
{
	this->set_subtitle( "iterator operation - copy/copy policy" );

	typedef phash<test*,copy_policy<test*>,16>	hash;

	hash		ht;
	hash*		d;
	hash::itor	i( ht );
	int		n;

	ht.set( s1, &t1 );
	ht.set( s2, &t2 );
	ht.set( s3, &t3 );
	ht.set( s0, 0 );
	ht.set( s6, &t4 );
	ht.set( s5, &t5 );
	ht.set( s7, 0 );

	d = dup( &ht );
	for ( i.init(), n = 0; i.cont(); i.next(), ++n ) {
		test*		p = i.get();
		const char*	k = i.key();

		if ( p ) AssertEqual( n, *p, *d->get( k ));
		else	 AssertZero( n, p );
	}
	delete d;

	// -------------------------------------------
	for ( i.init(); i.cont(); i.next()) 
		if ( ! i.get()) AssertZero( i.set( &t6 ));

	AssertNonZero( check_testhash( ht, 6, 
		s1, &t1, s2, &t2, s0, &t6, s6, &t4, s5, &t5, s7, &t6 ));

	// -------------------------------------------
	for ( i.init(); i.cont(); i.next()) {
		test* p = i.get();
		if ( p != 0 && p->num() % 2 == 1 ) AssertEqual( *i.del(), *p );
	}
	AssertNonZero( check_testhash( ht, 2, s2, &t2, s6, &t4 ));
}

// -----------------------------------------------------------------------------
void phashtest::test24()
{
        this->set_subtitle( "iterator operation - store/refer policy" );

        typedef phash<test*,store_policy<test*>,16>     hash;

        hash            ht;
        hash*           d;
        hash::itor      i( ht );
        int             n;

        ht.set( t1.cstr(), dup( &t1 ));
        ht.set( t2.cstr(), dup( &t2 ));
        ht.set( t3.cstr(), dup( &t3 ));
        ht.set( t4.cstr(), dup( &t4 ));
        ht.set( t5.cstr(), dup( &t5 ));
        ht.set( t6.cstr(), dup( &t6 ));
        ht.set( t7.cstr(), dup( &t7 ));

        d = dup( &ht );
        for ( i.init(), n = 0; i.cont(); i.next(), ++n ) {
                test*           p = i.get();
                const char*     k = i.key();

                if ( p ) AssertEqual( n, *p, *d->get( k ));
                else	 AssertZero( n, p );
        }
        delete d;

        // -------------------------------------------
        for ( i.init(); i.cont(); i.next())
                if ( strcmp( "Open", i.key()) == 0 ) AssertZero( i.set( dup( &t8 )));

        AssertNonZero( check_testhash( ht, 6,
                s1, &t1, s2, &t2, s4, &t4, s5, &t5, s6, &t8, s7, &t7 ));

        // -------------------------------------------
        for ( i.init(); i.cont(); i.next()) {
                test* p = i.get();
                if ( p != 0 && p->num() % 2 == 0 ) AssertEqual( *i.del(), *p );
        }
        AssertNonZero( check_testhash( ht, 3, s1, &t1, s5, &t5, s7, &t7 ));
}

// -----------------------------------------------------------------------------
void phashtest::test25()
{
	this->set_subtitle( "iterator operation - store<const char*>/refer<obj*> policy" );

	typedef phash<const char*,store_policy<const char*>,16,const test*,refer_policy<const test*> > hash;

	hash		ht;
	hash*		d;
	hash::itor	i( ht );
	int		n;

	ht.set( &t1, dup( s1 ));
	ht.set( &t2, dup( s2 ));
	ht.set( &t3, dup( s3 ));
	ht.set( &t4, 0 );
	ht.set( &t5, dup( s5 ));
	ht.set( &t6, dup( s6 ));
	ht.set( &t7, 0 );

	d = dup( &ht );
	for ( i.init(), n = 0; i.cont(); i.next(), ++n ) {
		const char*	p = i.get();
		const test*	k = i.key();

		if ( p ) AssertEqual( n, *p, *d->get( k ));
		else	 AssertZero( n, p );
	}
	delete d;

	// -------------------------------------------
	for ( i.init(); i.cont(); i.next()) 
		if ( ! i.get()) AssertZero( i.set( dup( s8 )));

	AssertNonZero( check_objkyhash( ht, 6, 
		 &t1, s1, &t2, s2, &t4, s8, &t5, s5, &t6, s6, &t7, s8 ));

	// -------------------------------------------
	for ( i.init(); i.cont(); i.next()) {
		const char* p = i.get();
		if ( p != 0 && strlen( p ) == 6 ) AssertZero( strcmp( i.del(), p ));
	}
	AssertNonZero( check_objkyhash( ht, 5, 
		 &t1, s1, &t2, s2, &t4, s8, &t6, s6, &t7, s8 ));
}

// -----------------------------------------------------------------------------
void phashtest::test26()
{
	this->set_subtitle( "iterator operation - store<const char*>/store<obj*> policy" );

	typedef phash<const char*,store_policy<const char*>,16,const test*,store_policy<const test*> > hash;

	hash		ht;
	hash*		d;
	hash::itor	i( ht );
	int		n;

	ht.set( dup( &t1 ), dup( s1 ));
	ht.set( dup( &t2 ), dup( s2 ));
	ht.set( dup( &t3 ), dup( s3 ));
	ht.set( dup( &t4 ), 0 );
	ht.set( dup( &t5 ), dup( s5 ));
	ht.set( dup( &t6 ), dup( s6 ));
	ht.set( dup( &t7 ), 0 );

	d = dup( &ht );
	for ( i.init(), n = 0; i.cont(); i.next(), ++n ) {
		const char*	p = i.get();
		const test*	k = i.key();

		if ( p ) AssertEqual( n, *p, *d->get( dup( k )));
		else	 AssertZero( n, p );
	}
	delete d;

	// -------------------------------------------
	for ( i.init(); i.cont(); i.next()) 
		if ( ! i.get()) AssertZero( i.set( dup( s8 )));

	AssertNonZero( check_objkyhash( ht, 6, 
		 &t1, s1, &t2, s2, &t4, s8, &t5, s5, &t6, s6, &t7, s8 ));

	// -------------------------------------------
	for ( i.init(); i.cont(); i.next()) {
		const char* p = i.get();
		if ( p != 0 && strlen( p ) == 5 ) AssertZero( strcmp( i.del(), p ));
	}
	AssertNonZero( check_objkyhash( ht, 4, 
		 &t4, s8, &t5, s5, &t6, s6, &t7, s8 ));
}

// -----------------------------------------------------------------------------
void phashtest::test27()
{
	this->set_subtitle( "iterator operation - store<const char*>/copy<obj*> policy" );

	typedef phash<const char*,store_policy<const char*>,16,const test*,copy_policy<const test*> > hash;

	hash		ht;
	hash*		d;
	hash::itor	i( ht );
	int		n;

	ht.set( &t1, dup( s1 ));
	ht.set( &t2, dup( s2 ));
	ht.set( &t3, dup( s3 ));
	ht.set( &t4, 0 );
	ht.set( &t5, dup( s5 ));
	ht.set( &t6, dup( s6 ));
	ht.set( &t7, 0 );

	d = dup( &ht );
	for ( i.init(), n = 0; i.cont(); i.next(), ++n ) {
		const char*	p = i.get();
		const test*	k = i.key();

		if ( p ) AssertEqual( n, *p, *d->get( k ));
		else	 AssertZero( n, p );
	}
	delete d;

	// -------------------------------------------
	for ( i.init(); i.cont(); i.next()) 
		if ( ! i.get()) AssertZero( i.set( dup( s8 )));

	AssertNonZero( check_objkyhash( ht, 6, 
		 &t1, s1, &t2, s2, &t4, s8, &t5, s5, &t6, s6, &t7, s8 ));

	// -------------------------------------------
	for ( i.init(); i.cont(); i.next()) {
		const char* p = i.get();
		if ( p != 0 && strlen( p ) == 6 ) AssertZero( strcmp( i.del(), p ));
	}
	AssertNonZero( check_objkyhash( ht, 5, 
		 &t1, s1, &t2, s2, &t4, s8, &t6, s6, &t7, s8 ));
}

// -----------------------------------------------------------------------------
void phashtest::test31()
{
	this->set_subtitle( "dref operation - refer/copy policy" );

	typedef phash<test*,refer_policy<test*>,16>	hash;

	hash		ht;

	ht.set( s1, &t1 );
	ht.set( s2, &t2 );
	ht.set( s3, &t3 );
	ht.set( s0, 0 );
	ht.set( s6, &t4 );
	ht.set( s5, &t5 );
	ht.set( s7, 0 );

	// -------------------------------------------
	hash::dref	d = ht.get( s5 );
	hash::dref	e = ht.get( s4 );

	AssertNonZero( d );
	AssertZero( e );

	AssertZero( strcmp( s5, d.key()));
	AssertEqual( &t5, d.get());
	AssertZero( e.key() );
	AssertZero( e.get() );
	AssertEqual( &t5, d.set( &t7 ));
	AssertNonZero( check_testhash( ht, 6, s1, &t1, s2, &t2, s0, 0, s6, &t4, s5, &t7, s7, 0 ));
	e = d = &t8;
	AssertNonZero( check_testhash( ht, 7, s1, &t1, s2, &t2, s0, 0, s6, &t4, s5, &t8, s7, 0, s4, &t8 ));

	d.del();
	e.del();
	AssertNonZero( check_testhash( ht, 5, s1, &t1, s2, &t2, s0, 0, s6, &t4, s7, 0 ));
}

// -----------------------------------------------------------------------------
void phashtest::test32()
{
	this->set_subtitle( "dref operation - store/copy policy" );

	typedef phash<test*,store_policy<test*>,16>	hash;

	hash		ht;

	ht.set( s1, dup( &t1 ));
	ht.set( s2, dup( &t2 ));
	ht.set( s3, dup( &t3 ));
	ht.set( s0, 0 );
	ht.set( s6, dup( &t4 ));
	ht.set( s5, dup( &t5 ));
	ht.set( s7, 0 );

	// -------------------------------------------
	hash::dref	d = ht.get( s5 );
	hash::dref	e = ht.get( s4 );

	AssertNonZero( d );
	AssertZero( e );

	AssertZero( strcmp( s5, d.key()));
	AssertEqual( t5, *d.get());
	AssertZero( e.key() );
	AssertZero( e.get() );
	AssertZero( d.set( dup( &t7 )));
	AssertNonZero( check_testhash( ht, 6, s1, &t1, s2, &t2, s0, 0, s6, &t4, s5, &t7, s7, 0 ));
	e = dup( d = dup( &t8 ));
	AssertNonZero( check_testhash( ht, 7, s1, &t1, s2, &t2, s0, 0, s6, &t4, s5, &t8, s7, 0, s4, &t8 ));

	d.del();
	e.del();
	AssertNonZero( check_testhash( ht, 5, s1, &t1, s2, &t2, s0, 0, s6, &t4, s7, 0 ));
}

// -----------------------------------------------------------------------------
void phashtest::test33()
{
	this->set_subtitle( "dref operation - copy/copy policy" );

	typedef phash<test*,copy_policy<test*>,16>	hash;

	hash		ht;

	ht.set( s1, &t1 );
	ht.set( s2, &t2 );
	ht.set( s3, &t3 );
	ht.set( s0, 0 );
	ht.set( s6, &t4 );
	ht.set( s5, &t5 );
	ht.set( s7, 0 );

	// -------------------------------------------
	hash::dref	d = ht.get( s5 );
	hash::dref	e = ht.get( s4 );

	AssertNonZero( d );
	AssertZero( e );

	AssertZero( strcmp( s5, d.key()));
	AssertEqual( t5, *d.get());
	AssertZero( e.key() );
	AssertZero( e.get() );
	AssertZero( d.set( &t7 ));
	AssertNonZero( check_testhash( ht, 6, s1, &t1, s2, &t2, s0, 0, s6, &t4, s5, &t7, s7, 0 ));
	e = d = &t8;
	AssertNonZero( check_testhash( ht, 7, s1, &t1, s2, &t2, s0, 0, s6, &t4, s5, &t8, s7, 0, s4, &t8 ));

	d.del();
	e.del();
	AssertNonZero( check_testhash( ht, 5, s1, &t1, s2, &t2, s0, 0, s6, &t4, s7, 0 ));
}

// -----------------------------------------------------------------------------
void phashtest::test34()
{
        this->set_subtitle( "dref operation - store/refer policy" );

        typedef phash<test*,store_policy<test*>,16,const char*,refer_policy<const char*> > hash;

        hash            ht;

        ht.set( t1.cstr(), dup( &t1 ));
        ht.set( t2.cstr(), dup( &t2 ));
        ht.set( t3.cstr(), dup( &t3 ));
        ht.set( t0.cstr(), dup( &t0 ));
        ht.set( t5.cstr(), dup( &t5 ));
        ht.set( t6.cstr(), dup( &t6 ));
        ht.set( t7.cstr(), dup( &t7 ));

	// -------------------------------------------
	hash::dref	d = ht.get( t5.cstr());
	hash::dref	e = ht.get( t4.cstr());

	AssertNonZero( d );
	AssertZero( e );

	AssertZero( strcmp( s5, d.key()));
	AssertEqual( t5, *d.get());
	AssertZero( e.key() );
	AssertZero( e.get() );
	AssertZero( d.set( dup( &t7 )));
	AssertNonZero( check_testhash( ht, 6,
		t1.cstr(), &t1, t2.cstr(), &t2, t0.cstr(), &t0, t5.cstr(), &t7, t6.cstr(), &t6, t7.cstr(), &t7 ));

	e = dup( d = dup( &t8 ));
	AssertNonZero( check_testhash( ht, 7,
		t1.cstr(), &t1, t2.cstr(), &t2, t0.cstr(), &t0, t4.cstr(), &t8, t5.cstr(), &t8, t6.cstr(), &t6, t7.cstr(), &t7 ));

	d.del();
	e.del();
	AssertNonZero( check_testhash( ht, 5,
		t1.cstr(), &t1, t2.cstr(), &t2, t0.cstr(), &t0, t6.cstr(), &t6, t7.cstr(), &t7 ));
}

// -----------------------------------------------------------------------------
void phashtest::test35()
{
	this->set_subtitle( "dref operation - store<const char*>/refer<obj*> policy" );

	typedef phash<const char*,store_policy<const char*>,16,const test*,refer_policy<const test*> > hash;

	hash		ht;

	ht.set( &t1, dup( s1 ));
	ht.set( &t2, dup( s2 ));
	ht.set( &t3, dup( s3 ));
	ht.set( &t0, 0 );
	ht.set( &t6, dup( s4 ));
	ht.set( &t5, dup( s5 ));
	ht.set( &t7, 0 );

	// -------------------------------------------
	hash::dref	d = ht.get( &t5 );
	hash::dref	e = ht.get( &t4 );

	AssertNonZero( d );
	AssertZero( e );

	AssertEqual( t5, *d.key());
	AssertZero( strcmp( s5, d.get()));
	AssertZero( e.key() );
	AssertZero( e.get() );
	AssertZero( d.set( dup( s7 )));
	AssertNonZero( check_objkyhash( ht, 6, &t1, s1, &t2, s2, &t0, 0, &t6, s4, &t5, s7, &t7, 0 ));
	e = dup( d = dup( s8 ));
	AssertNonZero( check_objkyhash( ht, 7, &t1, s1, &t2, s2, &t0, 0, &t4, s8, &t6, s4, &t5, s8, &t7, 0 ));

	d.del();
	e.del();
	AssertNonZero( check_objkyhash( ht, 5, &t1, s1, &t2, s2, &t0, 0, &t6, s4, &t7, 0 ));
}

// -----------------------------------------------------------------------------
void phashtest::test36()
{
	this->set_subtitle( "dref operation - store<const char*>/store<obj*> policy" );

	typedef phash<const char*,store_policy<const char*>,16,const test*,store_policy<const test*> > hash;

	hash		ht;

	ht.set( dup( &t1 ), dup( s1 ));
	ht.set( dup( &t2 ), dup( s2 ));
	ht.set( dup( &t3 ), dup( s3 ));
	ht.set( dup( &t0 ), 0 );
	ht.set( dup( &t6 ), dup( s4 ));
	ht.set( dup( &t5 ), dup( s5 ));
	ht.set( dup( &t7 ), 0 );

	// -------------------------------------------
	hash::dref	d = ht.get( dup( &t5 ));
	hash::dref	e = ht.get( dup( &t4 ));

	AssertNonZero( d );
	AssertZero( e );

	AssertEqual( t5, *d.key());
	AssertZero( strcmp( s5, d.get()));
	AssertZero( e.key() );
	AssertZero( e.get() );
	AssertZero( d.set( dup( s7 )));
	AssertNonZero( check_objkyhash( ht, 6, &t1, s1, &t2, s2, &t0, 0, &t6, s4, &t5, s7, &t7, 0 ));
	e = dup( d = dup( s8 ));
	AssertNonZero( check_objkyhash( ht, 7, &t1, s1, &t2, s2, &t0, 0, &t4, s8, &t6, s4, &t5, s8, &t7, 0 ));

	d.del();
	e.del();
	AssertNonZero( check_objkyhash( ht, 5, &t1, s1, &t2, s2, &t0, 0, &t6, s4, &t7, 0 ));
}

// -----------------------------------------------------------------------------
void phashtest::test37()
{
	this->set_subtitle( "dref operation - store<const char*>/copy<obj*> policy" );

	typedef phash<const char*,store_policy<const char*>,16,const test*,copy_policy<const test*> > hash;

	hash		ht;

	ht.set( &t1, dup( s1 ));
	ht.set( &t2, dup( s2 ));
	ht.set( &t3, dup( s3 ));
	ht.set( &t0, 0 );
	ht.set( &t6, dup( s4 ));
	ht.set( &t5, dup( s5 ));
	ht.set( &t7, 0 );

	// -------------------------------------------
	hash::dref	d = ht.get( &t5 );
	hash::dref	e = ht.get( &t4 );

	AssertNonZero( d );
	AssertZero( e );

	AssertEqual( t5, *d.key());
	AssertZero( strcmp( s5, d.get()));
	AssertZero( e.key() );
	AssertZero( e.get() );
	AssertZero( d.set( dup( s7 )));
	AssertNonZero( check_objkyhash( ht, 6, &t1, s1, &t2, s2, &t0, 0, &t6, s4, &t5, s7, &t7, 0 ));
	e = dup( d = dup( s8 ));
	AssertNonZero( check_objkyhash( ht, 7, &t1, s1, &t2, s2, &t0, 0, &t4, s8, &t6, s4, &t5, s8, &t7, 0 ));

	d.del();
	e.del();
	AssertNonZero( check_objkyhash( ht, 5, &t1, s1, &t2, s2, &t0, 0, &t6, s4, &t7, 0 ));
}

// -----------------------------------------------------------------------------
void phashtest::test38()
{
	this->set_subtitle( "dref operation - hash collision" );

	typedef phash<test*,store_policy<test*>,1,const test*> hash;

	hash		ht;
	hash::dref	d, e, f;

	// -------------------------------------------
	this->set_subtitle( "dref/hash collision: same value, hit" );

	ht.set( &t1, dup( &t1 ));
	ht.set( &t2, dup( &t2 ));
	ht.set( &t3, dup( &t3 ));

	d = e = f = ht.get( &t2 );

	AssertNonZero( d );
	AssertNonZero( e );
	AssertNonZero( f );

	AssertZero( d.set( dup( &t4 )));
	AssertNonZero( check_wtesthash( ht, 3, &t1, &t1, &t2, &t4, &t3, &t3 ));
	AssertZero( f.set( dup( &t5 )));	
	AssertNonZero( check_wtesthash( ht, 3, &t1, &t1, &t2, &t5, &t3, &t3 ));
	AssertZero( e.del());
	AssertNonZero( check_wtesthash( ht, 2, &t1, &t1, &t3, &t3 ));
	AssertZero( f.del());
	AssertNonZero( check_wtesthash( ht, 2, &t1, &t1, &t3, &t3 ));
	AssertZero( d.set( dup( &t4 )));	
	AssertNonZero( check_wtesthash( ht, 3, &t1, &t1, &t3, &t3, &t2, &t4 ));
	AssertZero( f.set( dup( &t5 )));
	AssertNonZero( check_wtesthash( ht, 3, &t1, &t1, &t3, &t3, &t2, &t5 ));
	AssertZero( d.del());
	AssertNonZero( check_wtesthash( ht, 2, &t1, &t1, &t3, &t3 ));
	AssertZero( e.del());
	AssertNonZero( check_wtesthash( ht, 2, &t1, &t1, &t3, &t3 ));
	AssertZero( e.set( dup( &t6 )));	
	AssertNonZero( check_wtesthash( ht, 3, &t1, &t1, &t3, &t3, &t2, &t6 ));

	ht.clear();

	// -------------------------------------------
	this->set_subtitle( "dref/hash collision: same value, not hit" );

	ht.set( &t1, dup( &t1 ));
	ht.set( &t2, dup( &t2 ));
	ht.set( &t3, dup( &t3 ));

	d = e = f = ht.get( &t4 );

	AssertZero( d );
	AssertZero( e );
	AssertZero( f );

	AssertZero( d.set( dup( &t4 )));
	AssertNonZero( check_wtesthash( ht, 4, &t1, &t1, &t2, &t2, &t3, &t3, &t4, &t4 ));
	AssertZero( f.set( dup( &t5 )));	
	AssertNonZero( check_wtesthash( ht, 4, &t1, &t1, &t2, &t2, &t3, &t3, &t4, &t5 ));
	AssertZero( e.del());
	AssertNonZero( check_wtesthash( ht, 3, &t1, &t1, &t2, &t2, &t3, &t3 ));
	AssertZero( f.del());
	AssertNonZero( check_wtesthash( ht, 3, &t1, &t1, &t2, &t2, &t3, &t3 ));
	AssertZero( d.set( dup( &t4 )));
	AssertNonZero( check_wtesthash( ht, 4, &t1, &t1, &t2, &t2, &t3, &t3, &t4, &t4 ));
	AssertZero( f.set( dup( &t5 )));	
	AssertNonZero( check_wtesthash( ht, 4, &t1, &t1, &t2, &t2, &t3, &t3, &t4, &t5 ));
	AssertZero( d.del());
	AssertNonZero( check_wtesthash( ht, 3, &t1, &t1, &t2, &t2, &t3, &t3 ));
	AssertZero( e.del());
	AssertNonZero( check_wtesthash( ht, 3, &t1, &t1, &t2, &t2, &t3, &t3 ));
	AssertZero( e.set( dup( &t6 )));	
	AssertNonZero( check_wtesthash( ht, 4, &t1, &t1, &t2, &t2, &t3, &t3, &t4, &t6 ));

	ht.clear();

	// -------------------------------------------
	this->set_subtitle( "dref/hash collision: different value" );

	ht.set( &t1, dup( &t1 ));
	ht.set( &t2, dup( &t2 ));
	ht.set( &t3, dup( &t3 ));

	d = ht.get( &t4 );
	e = ht.get( &t5 );
	f = ht.get( &t3 );

	AssertZero( d );
	AssertZero( e );
	AssertNonZero( f );

	AssertZero( d.set( dup( &t4 )));
	AssertNonZero( check_wtesthash( ht, 4, &t1, &t1, &t2, &t2, &t3, &t3, &t4, &t4 ));
	AssertZero( f.set( dup( &t5 )));	
	AssertNonZero( check_wtesthash( ht, 4, &t1, &t1, &t2, &t2, &t3, &t5, &t4, &t4 ));
	AssertZero( e.del());
	AssertNonZero( check_wtesthash( ht, 4, &t1, &t1, &t2, &t2, &t3, &t5, &t4, &t4 ));
	AssertZero( f.del());
	AssertNonZero( check_wtesthash( ht, 3, &t1, &t1, &t2, &t2, &t4, &t4 ));
	AssertZero( d.set( dup( &t6 )));
	AssertNonZero( check_wtesthash( ht, 3, &t1, &t1, &t2, &t2, &t4, &t6 ));
	AssertZero( f.set( dup( &t5 )));	
	AssertNonZero( check_wtesthash( ht, 4, &t1, &t1, &t2, &t2, &t4, &t6, &t3, &t5 ));
	AssertZero( d.del());
	AssertNonZero( check_wtesthash( ht, 3, &t1, &t1, &t2, &t2, &t3, &t5 ));
	AssertZero( e.del());
	AssertNonZero( check_wtesthash( ht, 3, &t1, &t1, &t2, &t2, &t3, &t5 ));
	AssertZero( d.set( dup( &t6 )));	
	AssertNonZero( check_wtesthash( ht, 4, &t1, &t1, &t2, &t2, &t3, &t5, &t4, &t6 ));
	AssertZero( f.set( dup( &t3 )));	
	AssertNonZero( check_wtesthash( ht, 4, &t1, &t1, &t2, &t2, &t3, &t3, &t4, &t6 ));

	ht.clear();

	// -------------------------------------------
	this->set_subtitle( "dref/hash collision: different value, w/ remove()" );

	ht.set( &t1, dup( &t1 ));
	ht.set( &t2, dup( &t2 ));
	ht.set( &t3, dup( &t3 ));

	d = ht.get( &t4 );
	e = ht.get( &t5 );
	f = ht.get( &t3 );

	AssertZero( d );
	AssertZero( e );
	AssertNonZero( f );

	AssertZero( d.set( dup( &t4 )));
	AssertNonZero( check_wtesthash( ht, 4, &t1, &t1, &t2, &t2, &t3, &t3, &t4, &t4 ));
	AssertZero( f.set( dup( &t5 )));	
	AssertNonZero( check_wtesthash( ht, 4, &t1, &t1, &t2, &t2, &t3, &t5, &t4, &t4 ));
	AssertZero( e.del());
	AssertNonZero( check_wtesthash( ht, 4, &t1, &t1, &t2, &t2, &t3, &t5, &t4, &t4 ));
	// delete by phash::remove
	ht.remove( &t3 );
	AssertNonZero( check_wtesthash( ht, 3, &t1, &t1, &t2, &t2, &t4, &t4 ));
	AssertZero( d.set( dup( &t6 )));
	AssertNonZero( check_wtesthash( ht, 3, &t1, &t1, &t2, &t2, &t4, &t6 ));
	AssertZero( f.set( dup( &t5 )));	
	AssertNonZero( check_wtesthash( ht, 4, &t1, &t1, &t2, &t2, &t4, &t6, &t3, &t5 ));
	AssertZero( d.del());
	AssertNonZero( check_wtesthash( ht, 3, &t1, &t1, &t2, &t2, &t3, &t5 ));
	AssertZero( e.del());
	AssertNonZero( check_wtesthash( ht, 3, &t1, &t1, &t2, &t2, &t3, &t5 ));
	AssertZero( d.set( dup( &t6 )));	
	AssertNonZero( check_wtesthash( ht, 4, &t1, &t1, &t2, &t2, &t3, &t5, &t4, &t6 ));
	AssertZero( f.set( dup( &t3 )));	
	AssertNonZero( check_wtesthash( ht, 4, &t1, &t1, &t2, &t2, &t3, &t3, &t4, &t6 ));

	ht.clear();

	// -------------------------------------------
	this->set_subtitle( "dref/hash collision: different value, w/ itor.del()" );

	ht.set( &t1, dup( &t1 ));
	ht.set( &t2, dup( &t2 ));
	ht.set( &t3, dup( &t3 ));

	d = ht.get( &t4 );
	e = ht.get( &t5 );
	f = ht.get( &t3 );

	AssertZero( d );
	AssertZero( e );
	AssertNonZero( f );

	AssertZero( d.set( dup( &t4 )));
	AssertNonZero( check_wtesthash( ht, 4, &t1, &t1, &t2, &t2, &t3, &t3, &t4, &t4 ));
	AssertZero( f.set( dup( &t5 )));	
	AssertNonZero( check_wtesthash( ht, 4, &t1, &t1, &t2, &t2, &t3, &t5, &t4, &t4 ));
	AssertZero( e.del());
	AssertNonZero( check_wtesthash( ht, 4, &t1, &t1, &t2, &t2, &t3, &t5, &t4, &t4 ));
	// delete by iterator
	hash::itor i( ht );
	for ( i.init(); i.cont(); i.next()) { if ( *i.key() == t3 ) i.del(); }
	AssertNonZero( check_wtesthash( ht, 3, &t1, &t1, &t2, &t2, &t4, &t4 ));
	AssertZero( d.set( dup( &t6 )));
	AssertNonZero( check_wtesthash( ht, 3, &t1, &t1, &t2, &t2, &t4, &t6 ));
	AssertZero( f.set( dup( &t5 )));	
	AssertNonZero( check_wtesthash( ht, 4, &t1, &t1, &t2, &t2, &t4, &t6, &t3, &t5 ));
	AssertZero( d.del());
	AssertNonZero( check_wtesthash( ht, 3, &t1, &t1, &t2, &t2, &t3, &t5 ));
	AssertZero( e.del());
	AssertNonZero( check_wtesthash( ht, 3, &t1, &t1, &t2, &t2, &t3, &t5 ));
	AssertZero( d.set( dup( &t6 )));	
	AssertNonZero( check_wtesthash( ht, 4, &t1, &t1, &t2, &t2, &t3, &t5, &t4, &t6 ));
	AssertZero( f.set( dup( &t3 )));	
	AssertNonZero( check_wtesthash( ht, 4, &t1, &t1, &t2, &t2, &t3, &t3, &t4, &t6 ));
}

// -----------------------------------------------------------------------------
void phashtest::test41()
{
	this->set_subtitle( "dup items - refer/copy" );

	phash<test*,refer_policy<test*> > 	ht;
	phash<test*,refer_policy<test*> >*	p;

	p = dup( &ht );
	AssertNonZero( check_testhash( *p, 0 ));	
	delete p;

	ht.set( s1, &t1 );
	p = dup( &ht );
	AssertNonZero( check_testhash( *p, 1, s1, &t1 ));	
	delete p;

	ht.set( s2, &t2 );
	ht.set( s3, &t3 );
	ht.set( s4, &t4 );
	ht.set( s5, &t5 );
	p = dup( &ht );
	AssertNonZero( check_testhash( *p, 5, s1, &t1, s2, &t2, s3, &t3, s4, &t4, s5, &t5 ));	
	delete p;
}

// -----------------------------------------------------------------------------
void phashtest::test42()
{
	this->set_subtitle( "dup items - store/copy" );

	phash<test*,store_policy<test*> > 	ht;
	phash<test*,store_policy<test*> >*	p;

	p = dup( &ht );
	AssertNonZero( check_testhash( *p, 0 ));	
	delete p;

	ht.set( s1, dup( &t1 ));
	p = dup( &ht );
	AssertNonZero( check_testhash( *p, 1, s1, &t1 ));	
	delete p;

	ht.set( s2, dup( &t2 ));
	ht.set( s3, dup( &t3 ));
	ht.set( s4, dup( &t4 ));
	ht.set( s5, dup( &t5 ));
	p = dup( &ht );
	AssertNonZero( check_testhash( *p, 5, s1, &t1, s2, &t2, s3, &t3, s4, &t4, s5, &t5 ));	
	delete p;
}

// -----------------------------------------------------------------------------
void phashtest::test43()
{
	this->set_subtitle( "dup items - copy/copy" );

	phash<test*,copy_policy<test*> > 	ht;
	phash<test*,copy_policy<test*> >*	p;

	p = dup( &ht );
	AssertNonZero( check_testhash( *p, 0 ));	
	delete p;

	ht.set( s1, &t1 );
	p = dup( &ht );
	AssertNonZero( check_testhash( *p, 1, s1, &t1 ));	
	delete p;

	ht.set( s2, &t2 );
	ht.set( s3, &t3 );
	ht.set( s4, &t4 );
	ht.set( s5, &t5 );
	p = dup( &ht );
	AssertNonZero( check_testhash( *p, 5, s1, &t1, s2, &t2, s3, &t3, s4, &t4, s5, &t5 ));	
	delete p;
}

// -----------------------------------------------------------------------------
void phashtest::test44()
{
	this->set_subtitle( "dup items - store/refer<int>" );

	phash<test*,store_policy<test*>,16,int,refer_policy<int> > 	ht;
	phash<test*,store_policy<test*>,16,int,refer_policy<int> >*	p;

	p = dup( &ht );
	AssertNonZero( check_intkyhash( *p, 0 ));	
	delete p;

	ht.set( i1, dup( &t1 ));
	p = dup( &ht );
	AssertNonZero( check_intkyhash( *p, 1, i1, &t1 ));	
	delete p;

	ht.set( i2, dup( &t2 ));
	ht.set( i3, dup( &t3 ));
	ht.set( i4, dup( &t4 ));
	ht.set( i5, dup( &t5 ));
	p = dup( &ht );
	AssertNonZero( check_intkyhash( *p, 5, i1, &t1, i2, &t2, i3, &t3, i4, &t4, i5, &t5 ));	
	delete p;
}

// -----------------------------------------------------------------------------
void phashtest::test45()
{
	this->set_subtitle( "dup items - store/refer" );

	phash<test*,store_policy<test*>,16,const test*,refer_policy<const test*> > 	ht;
	phash<test*,store_policy<test*>,16,const test*,refer_policy<const test*> >*	p;

	p = dup( &ht );
	AssertNonZero( check_wtesthash( *p, 0 ));	
	delete p;

	ht.set( &t1, dup( &t1 ));
	p = dup( &ht );
	AssertNonZero( check_wtesthash( *p, 1, &t1, &t1 ));	
	delete p;

	ht.set( &t2, dup( &t2 ));
	ht.set( &t3, dup( &t3 ));
	ht.set( &t4, dup( &t4 ));
	ht.set( &t5, dup( &t5 ));
	p = dup( &ht );
	AssertNonZero( check_wtesthash( *p, 5, &t1, &t1, &t2, &t2, &t3, &t3, &t4, &t4, &t5, &t5 ));	
	delete p;
}

// -----------------------------------------------------------------------------
void phashtest::test46()
{
	this->set_subtitle( "dup items - store/store" );

	phash<test*,store_policy<test*>,16,const test*,store_policy<const test*> > 	ht;
	phash<test*,store_policy<test*>,16,const test*,store_policy<const test*> >*	p;

	p = dup( &ht );
	AssertNonZero( check_wtesthash( *p, 0 ));	
	delete p;

	ht.set( dup( &t1 ), dup( &t1 ));
	p = dup( &ht );
	AssertNonZero( check_wtesthash( *p, 1, &t1, &t1 ));	
	delete p;

	ht.set( dup( &t2 ), dup( &t2 ));
	ht.set( dup( &t3 ), dup( &t3 ));
	ht.set( dup( &t4 ), dup( &t4 ));
	ht.set( dup( &t5 ), dup( &t5 ));
	p = dup( &ht );
	AssertNonZero( check_wtesthash( *p, 5, &t1, &t1, &t2, &t2, &t3, &t3, &t4, &t4, &t5, &t5 ));	
	delete p;
}

// -----------------------------------------------------------------------------
void phashtest::test47()
{
	this->set_subtitle( "dup items - store/copy" );

	phash<test*,store_policy<test*>,16,const test*,copy_policy<const test*> > 	ht;
	phash<test*,store_policy<test*>,16,const test*,copy_policy<const test*> >*	p;

	p = dup( &ht );
	AssertNonZero( check_wtesthash( *p, 0 ));	
	delete p;

	ht.set( &t1, dup( &t1 ));
	p = dup( &ht );
	AssertNonZero( check_wtesthash( *p, 1, &t1, &t1 ));	
	delete p;

	ht.set( &t2, dup( &t2 ));
	ht.set( &t3, dup( &t3 ));
	ht.set( &t4, dup( &t4 ));
	ht.set( &t5, dup( &t5 ));
	p = dup( &ht );
	AssertNonZero( check_wtesthash( *p, 5, &t1, &t1, &t2, &t2, &t3, &t3, &t4, &t4, &t5, &t5 ));	
	delete p;
}

// -----------------------------------------------------------------------------
void phashtest::run()
{
	test11();
	test12();
	test13();
	test14();
	test15();
	test16();
	test17();

	test21();
	test22();
	test23();
	test24();
	test25();
	test26();
	test27();

	test31();
	test32();
	test33();
	test34();
	test35();
	test36();
	test37();
	test38();

	test41();
	test42();
	test43();
	test44();
	test45();
	test46();
	test47();
}

// -----------------------------------------------------------------------------
int main( int argc, char **argv )
{
	phashtest().run();

	return 0;
}

